/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.config;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.CacheSPI;
import org.jboss.cache.CacheStatus;
import org.jboss.cache.factories.ComponentRegistry;
import org.jboss.cache.factories.annotations.Inject;
import org.jboss.cache.factories.annotations.Start;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * Base superclass of Cache configuration classes that expose some properties
 * that can be changed after the cache is started.
 *
 * @author <a href="brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision: 5898 $
 * @see #testImmutability(String)
 */
public abstract class ConfigurationComponent implements Serializable, Cloneable
{
   private static final long serialVersionUID = 4879873994727821938L;

   protected transient Log log = LogFactory.getLog(getClass());
   private transient CacheSPI cache; // back-reference to test whether the cache is running.
   private final Set<ConfigurationComponent> children = Collections.synchronizedSet(new HashSet<ConfigurationComponent>());
   private transient ComponentRegistry cr;
   // a workaround to get over immutability checks
   private boolean accessible;

   protected ConfigurationComponent()
   {
   }

   public void passCacheToChildConfig(ConfigurationComponent child)
   {
      if (child != null)
      {
         child.setCache(cache);
      }
   }

   protected void addChildConfig(ConfigurationComponent child)
   {
      if (child != null && children.add(child))
         child.setCache(cache);
   }

   protected void addChildConfigs(Collection<? extends ConfigurationComponent> toAdd)
   {
      if (toAdd != null)
      {
         for (ConfigurationComponent child : toAdd)
            addChildConfig(child);
      }
   }

   protected void removeChildConfig(ConfigurationComponent child)
   {
      children.remove(child);
   }

   protected void removeChildConfigs(Collection<? extends ConfigurationComponent> toRemove)
   {
      if (toRemove != null)
      {
         for (ConfigurationComponent child : toRemove)
            removeChildConfig(child);
      }
   }

   protected void replaceChildConfig(ConfigurationComponent oldConfig, ConfigurationComponent newConfig)
   {
      removeChildConfig(oldConfig);
      addChildConfig(newConfig);
   }

   protected void replaceChildConfigs(Collection<? extends ConfigurationComponent> oldConfigs,
                                      Collection<? extends ConfigurationComponent> newConfigs)
   {
      synchronized (children)
      {
         removeChildConfigs(oldConfigs);
         addChildConfigs(newConfigs);
      }
   }

   /**
    * Checks field modifications via setters
    *
    * @param fieldName
    */
   protected void testImmutability(String fieldName)
   {
      try
      {
         if (!accessible && cache != null && cache.getCacheStatus() != null && cache.getCacheStatus() == CacheStatus.STARTED && !getClass().getDeclaredField(fieldName).isAnnotationPresent(Dynamic.class))
         {
            throw new ConfigurationException("Attempted to modify a non-Dynamic configuration element [" + fieldName + "] after the cache has started!");
         }
      }
      catch (NoSuchFieldException e)
      {
         log.warn("Field " + fieldName + " not found!!");
      }
      finally
      {
         accessible = false;
      }
   }

   /**
    * Sets a back-reference to the cache associated with this configuration
    *
    * @param cache
    */
   public void setCache(CacheSPI cache)
   {
      this.cache = cache;
      synchronized (children)
      {
         for (ConfigurationComponent child : children)
         {
            child.setCache(cache);
         }
      }
   }

   @Inject
   private void injectDependencies(ComponentRegistry cr)
   {
      this.cr = cr;
   }

   @Start
   private void start()
   {
      setCache(cr.getComponent(CacheSPI.class));
   }

   @Override
   public ConfigurationComponent clone() throws CloneNotSupportedException
   {
      ConfigurationComponent c = (ConfigurationComponent) super.clone();
      c.setCache(null);
      return c;
   }
}
