#  holidays
#  --------
#  A fast, efficient Python library for generating country, province and state
#  specific sets of holidays on the fly. It aims to make determining whether a
#  specific date is a holiday as fast and flexible as possible.
#
#  Authors: Vacanza Team and individual contributors (see CONTRIBUTORS file)
#           dr-prodigy <dr.prodigy.github@gmail.com> (c) 2017-2023
#           ryanss <ryanssdev@icloud.com> (c) 2014-2017
#  Website: https://github.com/vacanza/holidays
#  License: MIT (see LICENSE file)

from unittest import TestCase

from holidays.countries.israel import Israel
from tests.common import CommonCountryTests


class TestIsrael(CommonCountryTests, TestCase):
    @classmethod
    def setUpClass(cls):
        super().setUpClass(Israel)

    def test_rosh_hashanah(self):
        name = "ראש השנה"
        self.assertHolidayName(
            name,
            "2020-09-19",
            "2020-09-20",
            "2021-09-07",
            "2021-09-08",
            "2022-09-26",
            "2022-09-27",
            "2023-09-16",
            "2023-09-17",
            "2024-10-03",
            "2024-10-04",
            "2025-09-23",
            "2025-09-24",
        )
        self.assertHolidayNameCount(name, 2, self.full_range)

    def test_yom_kippur(self):
        name = "יום כיפור"
        self.assertHolidayName(
            name,
            "2020-09-28",
            "2021-09-16",
            "2022-10-05",
            "2023-09-25",
            "2024-10-12",
            "2025-10-02",
        )
        self.assertHolidayName(name, self.full_range)

    def test_sukhot(self):
        name = "סוכות"
        name_holiday = f"חול המועד {name}"

        self.assertHolidayName(
            name,
            "2020-10-03",
            "2021-09-21",
            "2022-10-10",
            "2023-09-30",
            "2024-10-17",
            "2025-10-07",
        )
        self.assertHolidayName(name, self.full_range)

        dts_holiday = (
            "2020-10-04",
            "2020-10-05",
            "2020-10-06",
            "2020-10-07",
            "2020-10-08",
            "2021-09-22",
            "2021-09-23",
            "2021-09-24",
            "2021-09-25",
            "2021-09-26",
            "2022-10-11",
            "2022-10-12",
            "2022-10-13",
            "2022-10-14",
            "2022-10-15",
            "2023-10-01",
            "2023-10-02",
            "2023-10-03",
            "2023-10-04",
            "2023-10-05",
            "2024-10-18",
            "2024-10-19",
            "2024-10-20",
            "2024-10-21",
            "2024-10-22",
            "2025-10-08",
            "2025-10-09",
            "2025-10-10",
            "2025-10-11",
            "2025-10-12",
        )
        self.assertNoHolidayName(name_holiday)
        self.assertOptionalHolidayName(name_holiday, dts_holiday)
        self.assertOptionalHolidayNameCount(name_holiday, 5, self.full_range)
        self.assertSchoolHolidayName(name_holiday, dts_holiday)
        self.assertSchoolHolidayNameCount(name_holiday, 5, self.full_range)

    def test_simchat_torah_shemini_atzeret(self):
        name = "שמחת תורה/שמיני עצרת"
        self.assertHolidayName(
            name,
            "2020-10-10",
            "2021-09-28",
            "2022-10-17",
            "2023-10-07",
            "2024-10-24",
            "2025-10-14",
        )
        self.assertHolidayName(name, self.full_range)

    def test_sigd(self):
        name = "סיגד"
        self.assertNoHolidayName(name)
        self.assertOptionalHolidayName(
            name,
            "2020-11-16",
            "2021-11-04",
            "2022-11-23",
            "2023-11-13",
            "2024-11-30",
            "2025-11-20",
        )
        self.assertOptionalHolidayName(name, range(2008, self.end_year))
        self.assertNoOptionalHolidayName(name, range(self.start_year, 2008))

    def test_hanukkah(self):
        name = "חנוכה"
        self.assertNoHolidayName(name)
        self.assertSchoolHolidayName(
            name,
            "2020-12-11",
            "2020-12-12",
            "2020-12-13",
            "2020-12-14",
            "2020-12-15",
            "2020-12-16",
            "2020-12-17",
            "2020-12-18",
            "2021-11-29",
            "2021-11-30",
            "2021-12-01",
            "2021-12-02",
            "2021-12-03",
            "2021-12-04",
            "2021-12-05",
            "2021-12-06",
            "2022-12-19",
            "2022-12-20",
            "2022-12-21",
            "2022-12-22",
            "2022-12-23",
            "2022-12-24",
            "2022-12-25",
            "2022-12-26",
            "2023-12-08",
            "2023-12-09",
            "2023-12-10",
            "2023-12-11",
            "2023-12-12",
            "2023-12-13",
            "2023-12-14",
            "2023-12-15",
            "2024-12-26",
            "2024-12-27",
            "2024-12-28",
            "2024-12-29",
            "2024-12-30",
            "2024-12-31",
            "2025-01-01",
            "2025-01-02",
            "2025-12-15",
            "2025-12-16",
            "2025-12-17",
            "2025-12-18",
            "2025-12-19",
            "2025-12-20",
            "2025-12-21",
            "2025-12-22",
        )
        self.assertSchoolHolidayName(name, self.full_range)

    def test_taanit_ester(self):
        name = "תענית אסתר"
        dts = (
            "2000-03-20",
            "2001-03-08",
            "2002-02-25",
            "2003-03-17",
            "2005-03-24",
            "2006-03-13",
            "2008-03-20",
            "2009-03-09",
            "2012-03-07",
            "2015-03-04",
            "2016-03-23",
            "2018-02-28",
            "2019-03-20",
            "2020-03-09",
            "2021-02-25",
            "2022-03-16",
            "2023-03-06",
        )
        obs_dts = (
            "2004-03-04",
            "2007-03-01",
            "2010-02-25",
            "2011-03-17",
            "2013-02-21",
            "2014-03-13",
            "2017-03-09",
        )
        non_obs_dts = (
            "2004-03-06",
            "2007-03-03",
            "2010-02-27",
            "2011-03-19",
            "2013-02-23",
            "2014-03-15",
            "2017-03-11",
        )
        self.assertNoHolidayName(name)
        self.assertSchoolHolidayName(name, dts)
        self.assertSchoolNonObservedHolidayName(name, self.full_range)
        self.assertSchoolHolidayName(f"{name} (נצפה)", obs_dts)
        self.assertSchoolNonObservedHolidayName(name, dts, non_obs_dts)
        self.assertNoSchoolNonObservedHoliday(obs_dts)

    def test_purim(self):
        name = "פורים"
        dts = (
            "2020-03-10",
            "2021-02-26",
            "2022-03-17",
            "2023-03-07",
            "2024-03-24",
            "2025-03-14",
        )
        self.assertNoHolidayName(name)
        self.assertOptionalHolidayName(name, dts)
        self.assertOptionalHolidayName(name, self.full_range)
        self.assertSchoolHolidayName(name, dts)
        self.assertSchoolHolidayName(name, self.full_range)

    def test_pesach(self):
        name = "פסח"
        name_7th_day = f"שביעי של {name}"
        name_holiday = f"חול המועד {name}"

        self.assertHolidayName(
            name,
            "2020-04-09",
            "2021-03-28",
            "2022-04-16",
            "2023-04-06",
            "2024-04-23",
            "2025-04-13",
        )
        self.assertHolidayName(name, self.full_range)

        self.assertHolidayName(
            name_7th_day,
            "2020-04-15",
            "2021-04-03",
            "2022-04-22",
            "2023-04-12",
            "2024-04-29",
            "2025-04-19",
        )
        self.assertHolidayName(name_7th_day, self.full_range)

        dts = (
            "2020-04-10",
            "2020-04-11",
            "2020-04-12",
            "2020-04-13",
            "2020-04-14",
            "2021-03-29",
            "2021-03-30",
            "2021-03-31",
            "2021-04-01",
            "2021-04-02",
            "2022-04-17",
            "2022-04-18",
            "2022-04-19",
            "2022-04-20",
            "2022-04-21",
            "2023-04-07",
            "2023-04-08",
            "2023-04-09",
            "2023-04-10",
            "2023-04-11",
            "2024-04-24",
            "2024-04-25",
            "2024-04-26",
            "2024-04-27",
            "2024-04-28",
            "2025-04-14",
            "2025-04-15",
            "2025-04-16",
            "2025-04-17",
            "2025-04-18",
        )
        self.assertNoHolidayName(name_holiday)
        self.assertOptionalHolidayName(name_holiday, dts)
        self.assertOptionalHolidayNameCount(name_holiday, 5, self.full_range)
        self.assertSchoolHolidayName(name_holiday, dts)
        self.assertSchoolHolidayNameCount(name_holiday, 5, self.full_range)

    def test_remembrance_day(self):
        name = "יום הזיכרון לחללי מערכות ישראל ונפגעי פעולות האיבה"
        dts = (
            "2000-05-09",
            "2002-04-16",
            "2003-05-06",
            "2006-05-02",
            "2009-04-28",
            "2020-04-28",
            "2023-04-25",
        )
        obs_dts = (
            "2001-04-25",
            "2004-04-26",
            "2005-05-11",
            "2007-04-23",
            "2008-05-07",
            "2010-04-19",
            "2011-05-09",
            "2012-04-25",
            "2013-04-15",
            "2014-05-05",
            "2015-04-22",
            "2016-05-11",
            "2017-05-01",
            "2018-04-18",
            "2019-05-08",
            "2021-04-14",
            "2022-05-04",
        )
        non_obs_dts = (
            "2001-04-27",
            "2004-04-25",
            "2005-05-13",
            "2007-04-22",
            "2008-05-09",
            "2010-04-18",
            "2011-05-08",
            "2012-04-26",
            "2013-04-14",
            "2014-05-04",
            "2015-04-23",
            "2016-05-12",
            "2017-04-30",
            "2018-04-19",
            "2019-05-09",
            "2021-04-16",
            "2022-05-05",
        )
        self.assertNoHolidayName(name)
        self.assertOptionalHolidayName(name, dts)
        self.assertOptionalNonObservedHolidayName(name, range(1963, self.end_year))
        self.assertOptionalHolidayName(f"{name} (נצפה)", obs_dts)
        self.assertOptionalNonObservedHolidayName(name, dts, non_obs_dts)
        self.assertNoOptionalNonObservedHoliday(obs_dts)
        self.assertNoOptionalHolidayName(name, range(self.start_year, 1963))

    def test_independence_day(self):
        name = "יום העצמאות"
        dts = (
            "2000-05-10",
            "2002-04-17",
            "2003-05-07",
            "2006-05-03",
            "2009-04-29",
            "2020-04-29",
            "2023-04-26",
        )
        obs_dts = (
            "2001-04-26",
            "2004-04-27",
            "2005-05-12",
            "2007-04-24",
            "2008-05-08",
            "2010-04-20",
            "2011-05-10",
            "2012-04-26",
            "2013-04-16",
            "2014-05-06",
            "2015-04-23",
            "2016-05-12",
            "2017-05-02",
            "2018-04-19",
            "2019-05-09",
            "2021-04-15",
            "2022-05-05",
        )
        non_obs_dts = (
            "2001-04-28",
            "2004-04-26",
            "2005-05-14",
            "2007-04-23",
            "2008-05-10",
            "2010-04-19",
            "2011-05-09",
            "2012-04-27",
            "2013-04-15",
            "2014-05-05",
            "2015-04-24",
            "2016-05-13",
            "2017-05-01",
            "2018-04-20",
            "2019-05-10",
            "2021-04-17",
            "2022-05-06",
        )
        self.assertHolidayName(name, dts)
        self.assertNonObservedHolidayName(name, self.full_range)
        self.assertHolidayName(f"{name} (נצפה)", obs_dts)
        self.assertNonObservedHolidayName(name, dts, non_obs_dts)
        self.assertNoNonObservedHoliday(obs_dts)

    def test_lag_baomer(self):
        name = 'ל"ג בעומר'
        self.assertNoHolidayName(name)
        self.assertSchoolHolidayName(
            name,
            "2020-05-12",
            "2021-04-30",
            "2022-05-19",
            "2023-05-09",
            "2024-05-26",
            "2025-05-16",
        )
        self.assertSchoolHolidayName(name, self.full_range)

    def test_jerusalem_day(self):
        name = "יום ירושלים"
        self.assertNoHolidayName(name)
        self.assertOptionalHolidayName(
            name,
            "2020-05-22",
            "2021-05-10",
            "2022-05-29",
            "2023-05-19",
            "2024-06-05",
            "2025-05-26",
        )
        self.assertOptionalHolidayName(name, range(1998, self.end_year))
        self.assertNoOptionalHolidayName(name, range(self.start_year, 1998))

    def test_shavuot(self):
        name = "שבועות"
        self.assertHolidayName(
            name,
            "2020-05-29",
            "2021-05-17",
            "2022-06-05",
            "2023-05-26",
            "2024-06-12",
            "2025-06-02",
        )
        self.assertHolidayName(name, self.full_range)

    def test_tisha_bav(self):
        name = "תשעה באב"
        dts = (
            "2000-08-10",
            "2001-07-29",
            "2002-07-18",
            "2003-08-07",
            "2004-07-27",
            "2005-08-14",
            "2006-08-03",
            "2007-07-24",
            "2008-08-10",
            "2009-07-30",
            "2010-07-20",
            "2011-08-09",
            "2013-07-16",
            "2014-08-05",
            "2017-08-01",
            "2020-07-30",
            "2021-07-18",
            "2023-07-27",
        )
        obs_dts = (
            "2012-07-29",
            "2015-07-26",
            "2016-08-14",
            "2018-07-22",
            "2019-08-11",
            "2022-08-07",
        )
        non_obs_dts = (
            "2012-07-28",
            "2015-07-25",
            "2016-08-13",
            "2018-07-21",
            "2019-08-10",
            "2022-08-06",
        )
        self.assertNoHolidayName(name)
        self.assertOptionalHolidayName(name, dts)
        self.assertOptionalNonObservedHolidayName(name, self.full_range)
        self.assertOptionalHolidayName(f"{name} (נצפה)", obs_dts)
        self.assertOptionalNonObservedHolidayName(name, dts, non_obs_dts)
        self.assertNoOptionalNonObservedHoliday(obs_dts)

    def test_2021(self):
        self.assertHolidaysInYear(
            2021,
            ("2021-03-28", "פסח"),
            ("2021-04-03", "שביעי של פסח"),
            ("2021-04-15", "יום העצמאות (נצפה)"),
            ("2021-05-17", "שבועות"),
            ("2021-09-07", "ראש השנה"),
            ("2021-09-08", "ראש השנה"),
            ("2021-09-16", "יום כיפור"),
            ("2021-09-21", "סוכות"),
            ("2021-09-28", "שמחת תורה/שמיני עצרת"),
        )

    def test_2022(self):
        self.assertHolidaysInYear(
            2022,
            ("2022-04-16", "פסח"),
            ("2022-04-22", "שביעי של פסח"),
            ("2022-05-05", "יום העצמאות (נצפה)"),
            ("2022-06-05", "שבועות"),
            ("2022-09-26", "ראש השנה"),
            ("2022-09-27", "ראש השנה"),
            ("2022-10-05", "יום כיפור"),
            ("2022-10-10", "סוכות"),
            ("2022-10-17", "שמחת תורה/שמיני עצרת"),
        )

    def test_2023(self):
        self.assertHolidaysInYear(
            2023,
            ("2023-04-06", "פסח"),
            ("2023-04-12", "שביעי של פסח"),
            ("2023-04-26", "יום העצמאות"),
            ("2023-05-26", "שבועות"),
            ("2023-09-16", "ראש השנה"),
            ("2023-09-17", "ראש השנה"),
            ("2023-09-25", "יום כיפור"),
            ("2023-09-30", "סוכות"),
            ("2023-10-07", "שמחת תורה/שמיני עצרת"),
        )

    def test_2021_optional(self):
        self.assertOptionalHolidaysInYear(
            2021,
            ("2021-02-26", "פורים"),
            ("2021-03-29", "חול המועד פסח"),
            ("2021-03-30", "חול המועד פסח"),
            ("2021-03-31", "חול המועד פסח"),
            ("2021-04-01", "חול המועד פסח"),
            ("2021-04-02", "חול המועד פסח"),
            ("2021-04-14", "יום הזיכרון לחללי מערכות ישראל ונפגעי פעולות האיבה (נצפה)"),
            ("2021-05-10", "יום ירושלים"),
            ("2021-07-18", "תשעה באב"),
            ("2021-09-22", "חול המועד סוכות"),
            ("2021-09-23", "חול המועד סוכות"),
            ("2021-09-24", "חול המועד סוכות"),
            ("2021-09-25", "חול המועד סוכות"),
            ("2021-09-26", "חול המועד סוכות"),
            ("2021-11-04", "סיגד"),
        )

    def test_2022_optional(self):
        self.assertOptionalHolidaysInYear(
            2022,
            ("2022-03-17", "פורים"),
            ("2022-04-17", "חול המועד פסח"),
            ("2022-04-18", "חול המועד פסח"),
            ("2022-04-19", "חול המועד פסח"),
            ("2022-04-20", "חול המועד פסח"),
            ("2022-04-21", "חול המועד פסח"),
            ("2022-05-04", "יום הזיכרון לחללי מערכות ישראל ונפגעי פעולות האיבה (נצפה)"),
            ("2022-05-29", "יום ירושלים"),
            ("2022-08-07", "תשעה באב (נצפה)"),
            ("2022-10-11", "חול המועד סוכות"),
            ("2022-10-12", "חול המועד סוכות"),
            ("2022-10-13", "חול המועד סוכות"),
            ("2022-10-14", "חול המועד סוכות"),
            ("2022-10-15", "חול המועד סוכות"),
            ("2022-11-23", "סיגד"),
        )

    def test_2023_optional(self):
        self.assertOptionalHolidaysInYear(
            2023,
            ("2023-03-07", "פורים"),
            ("2023-04-07", "חול המועד פסח"),
            ("2023-04-08", "חול המועד פסח"),
            ("2023-04-09", "חול המועד פסח"),
            ("2023-04-10", "חול המועד פסח"),
            ("2023-04-11", "חול המועד פסח"),
            ("2023-04-25", "יום הזיכרון לחללי מערכות ישראל ונפגעי פעולות האיבה"),
            ("2023-05-19", "יום ירושלים"),
            ("2023-07-27", "תשעה באב"),
            ("2023-10-01", "חול המועד סוכות"),
            ("2023-10-02", "חול המועד סוכות"),
            ("2023-10-03", "חול המועד סוכות"),
            ("2023-10-04", "חול המועד סוכות"),
            ("2023-10-05", "חול המועד סוכות"),
            ("2023-11-13", "סיגד"),
        )

    def test_2021_school(self):
        self.assertSchoolHolidaysInYear(
            2021,
            ("2021-02-25", "תענית אסתר"),
            ("2021-02-26", "פורים"),
            ("2021-03-29", "חול המועד פסח"),
            ("2021-03-30", "חול המועד פסח"),
            ("2021-03-31", "חול המועד פסח"),
            ("2021-04-01", "חול המועד פסח"),
            ("2021-04-02", "חול המועד פסח"),
            ("2021-04-30", 'ל"ג בעומר'),
            ("2021-09-22", "חול המועד סוכות"),
            ("2021-09-23", "חול המועד סוכות"),
            ("2021-09-24", "חול המועד סוכות"),
            ("2021-09-25", "חול המועד סוכות"),
            ("2021-09-26", "חול המועד סוכות"),
            ("2021-11-29", "חנוכה"),
            ("2021-11-30", "חנוכה"),
            ("2021-12-01", "חנוכה"),
            ("2021-12-02", "חנוכה"),
            ("2021-12-03", "חנוכה"),
            ("2021-12-04", "חנוכה"),
            ("2021-12-05", "חנוכה"),
            ("2021-12-06", "חנוכה"),
        )

    def test_2022_school(self):
        self.assertSchoolHolidaysInYear(
            2022,
            ("2022-03-16", "תענית אסתר"),
            ("2022-03-17", "פורים"),
            ("2022-04-17", "חול המועד פסח"),
            ("2022-04-18", "חול המועד פסח"),
            ("2022-04-19", "חול המועד פסח"),
            ("2022-04-20", "חול המועד פסח"),
            ("2022-04-21", "חול המועד פסח"),
            ("2022-05-19", 'ל"ג בעומר'),
            ("2022-10-11", "חול המועד סוכות"),
            ("2022-10-12", "חול המועד סוכות"),
            ("2022-10-13", "חול המועד סוכות"),
            ("2022-10-14", "חול המועד סוכות"),
            ("2022-10-15", "חול המועד סוכות"),
            ("2022-12-19", "חנוכה"),
            ("2022-12-20", "חנוכה"),
            ("2022-12-21", "חנוכה"),
            ("2022-12-22", "חנוכה"),
            ("2022-12-23", "חנוכה"),
            ("2022-12-24", "חנוכה"),
            ("2022-12-25", "חנוכה"),
            ("2022-12-26", "חנוכה"),
        )

    def test_2023_school(self):
        self.assertSchoolHolidaysInYear(
            2023,
            ("2023-03-06", "תענית אסתר"),
            ("2023-03-07", "פורים"),
            ("2023-04-07", "חול המועד פסח"),
            ("2023-04-08", "חול המועד פסח"),
            ("2023-04-09", "חול המועד פסח"),
            ("2023-04-10", "חול המועד פסח"),
            ("2023-04-11", "חול המועד פסח"),
            ("2023-05-09", 'ל"ג בעומר'),
            ("2023-10-01", "חול המועד סוכות"),
            ("2023-10-02", "חול המועד סוכות"),
            ("2023-10-03", "חול המועד סוכות"),
            ("2023-10-04", "חול המועד סוכות"),
            ("2023-10-05", "חול המועד סוכות"),
            ("2023-12-08", "חנוכה"),
            ("2023-12-09", "חנוכה"),
            ("2023-12-10", "חנוכה"),
            ("2023-12-11", "חנוכה"),
            ("2023-12-12", "חנוכה"),
            ("2023-12-13", "חנוכה"),
            ("2023-12-14", "חנוכה"),
            ("2023-12-15", "חנוכה"),
        )

    def test_l10n_default(self):
        self.assertLocalizedHolidays(
            ("2021-02-25", "תענית אסתר"),
            ("2021-02-26", "פורים"),
            ("2021-03-28", "פסח"),
            ("2021-03-29", "חול המועד פסח"),
            ("2021-03-30", "חול המועד פסח"),
            ("2021-03-31", "חול המועד פסח"),
            ("2021-04-01", "חול המועד פסח"),
            ("2021-04-02", "חול המועד פסח"),
            ("2021-04-03", "שביעי של פסח"),
            ("2021-04-14", "יום הזיכרון לחללי מערכות ישראל ונפגעי פעולות האיבה (נצפה)"),
            ("2021-04-15", "יום העצמאות (נצפה)"),
            ("2021-04-30", 'ל"ג בעומר'),
            ("2021-05-10", "יום ירושלים"),
            ("2021-05-17", "שבועות"),
            ("2021-07-18", "תשעה באב"),
            ("2021-09-07", "ראש השנה"),
            ("2021-09-08", "ראש השנה"),
            ("2021-09-16", "יום כיפור"),
            ("2021-09-21", "סוכות"),
            ("2021-09-22", "חול המועד סוכות"),
            ("2021-09-23", "חול המועד סוכות"),
            ("2021-09-24", "חול המועד סוכות"),
            ("2021-09-25", "חול המועד סוכות"),
            ("2021-09-26", "חול המועד סוכות"),
            ("2021-09-28", "שמחת תורה/שמיני עצרת"),
            ("2021-11-04", "סיגד"),
            ("2021-11-29", "חנוכה"),
            ("2021-11-30", "חנוכה"),
            ("2021-12-01", "חנוכה"),
            ("2021-12-02", "חנוכה"),
            ("2021-12-03", "חנוכה"),
            ("2021-12-04", "חנוכה"),
            ("2021-12-05", "חנוכה"),
            ("2021-12-06", "חנוכה"),
        )

    def test_l10n_en_us(self):
        self.assertLocalizedHolidays(
            "en_US",
            ("2021-02-25", "Ta'anit Ester"),
            ("2021-02-26", "Purim"),
            ("2021-03-28", "Pesach"),
            ("2021-03-29", "Pesach holiday"),
            ("2021-03-30", "Pesach holiday"),
            ("2021-03-31", "Pesach holiday"),
            ("2021-04-01", "Pesach holiday"),
            ("2021-04-02", "Pesach holiday"),
            ("2021-04-03", "Seventh day of Pesach"),
            ("2021-04-14", "Remembrance Day (observed)"),
            ("2021-04-15", "Independence Day (observed)"),
            ("2021-04-30", "Lag BaOmer"),
            ("2021-05-10", "Jerusalem Day"),
            ("2021-05-17", "Shavuot"),
            ("2021-07-18", "Tisha B'Av"),
            ("2021-09-07", "Rosh Hashanah"),
            ("2021-09-08", "Rosh Hashanah"),
            ("2021-09-16", "Yom Kippur"),
            ("2021-09-21", "Sukkot"),
            ("2021-09-22", "Sukkot holiday"),
            ("2021-09-23", "Sukkot holiday"),
            ("2021-09-24", "Sukkot holiday"),
            ("2021-09-25", "Sukkot holiday"),
            ("2021-09-26", "Sukkot holiday"),
            ("2021-09-28", "Simchat Torah / Shemini Atzeret"),
            ("2021-11-04", "Sigd"),
            ("2021-11-29", "Hanukkah"),
            ("2021-11-30", "Hanukkah"),
            ("2021-12-01", "Hanukkah"),
            ("2021-12-02", "Hanukkah"),
            ("2021-12-03", "Hanukkah"),
            ("2021-12-04", "Hanukkah"),
            ("2021-12-05", "Hanukkah"),
            ("2021-12-06", "Hanukkah"),
        )

    def test_l10n_th(self):
        self.assertLocalizedHolidays(
            "th",
            ("2021-02-25", "วันทาอานิต เอสเธอร์"),
            ("2021-02-26", "เทศกาลปูริม"),
            ("2021-03-28", "วันเพสสะห์"),
            ("2021-03-29", "เทศกาลเพสสะห์"),
            ("2021-03-30", "เทศกาลเพสสะห์"),
            ("2021-03-31", "เทศกาลเพสสะห์"),
            ("2021-04-01", "เทศกาลเพสสะห์"),
            ("2021-04-02", "เทศกาลเพสสะห์"),
            ("2021-04-03", "วันเพสสะห์วันที่เจ็ด"),
            ("2021-04-14", "ชดเชยวันรำลึกถึงทหารผู้สละชีพและเหยื่อการก่อการร้าย"),
            ("2021-04-15", "ชดเชยวันชาติอิสราเอล"),
            ("2021-04-30", "วันแล็ก บาโอเมอร์"),
            ("2021-05-10", "วันเยรูซาเล็ม"),
            ("2021-05-17", "วันชาวูโอท"),
            ("2021-07-18", "วันทิชอา เบอัฟ"),
            ("2021-09-07", "เทศกาลรอช ฮาชานาห์ (วันปีใหม่ยิว)"),
            ("2021-09-08", "เทศกาลรอช ฮาชานาห์ (วันปีใหม่ยิว)"),
            ("2021-09-16", "วันยม คิปปูร์"),
            ("2021-09-21", "วันสุคคต"),
            ("2021-09-22", "เทศกาลสุคคต"),
            ("2021-09-23", "เทศกาลสุคคต"),
            ("2021-09-24", "เทศกาลสุคคต"),
            ("2021-09-25", "เทศกาลสุคคต"),
            ("2021-09-26", "เทศกาลสุคคต"),
            ("2021-09-28", "วันซิมหัต โทราห์ / วันเชมินี อัตเซเรต"),
            ("2021-11-04", "เทศกาลซิกด์"),
            ("2021-11-29", "เทศกาลฮานุกกะห์"),
            ("2021-11-30", "เทศกาลฮานุกกะห์"),
            ("2021-12-01", "เทศกาลฮานุกกะห์"),
            ("2021-12-02", "เทศกาลฮานุกกะห์"),
            ("2021-12-03", "เทศกาลฮานุกกะห์"),
            ("2021-12-04", "เทศกาลฮานุกกะห์"),
            ("2021-12-05", "เทศกาลฮานุกกะห์"),
            ("2021-12-06", "เทศกาลฮานุกกะห์"),
        )

    def test_l10n_uk(self):
        self.assertLocalizedHolidays(
            "uk",
            ("2021-02-25", "Тааніт-Естер"),
            ("2021-02-26", "Пурім"),
            ("2021-03-28", "Песах"),
            ("2021-03-29", "Свято Песах"),
            ("2021-03-30", "Свято Песах"),
            ("2021-03-31", "Свято Песах"),
            ("2021-04-01", "Свято Песах"),
            ("2021-04-02", "Свято Песах"),
            ("2021-04-03", "Сьомий день Песаха"),
            ("2021-04-14", "День памʼяті (вихідний)"),
            ("2021-04-15", "День незалежності (вихідний)"),
            ("2021-04-30", "Лаг ба-Омер"),
            ("2021-05-10", "День Єрусалиму"),
            ("2021-05-17", "Шавуот"),
            ("2021-07-18", "Тиша Бе-Ав"),
            ("2021-09-07", "Рош га-Шана"),
            ("2021-09-08", "Рош га-Шана"),
            ("2021-09-16", "Йом Кіпур"),
            ("2021-09-21", "Суккот"),
            ("2021-09-22", "Свято Суккот"),
            ("2021-09-23", "Свято Суккот"),
            ("2021-09-24", "Свято Суккот"),
            ("2021-09-25", "Свято Суккот"),
            ("2021-09-26", "Свято Суккот"),
            ("2021-09-28", "Сімхат Тора / Шміні Ацерет"),
            ("2021-11-04", "Сігд"),
            ("2021-11-29", "Ханука"),
            ("2021-11-30", "Ханука"),
            ("2021-12-01", "Ханука"),
            ("2021-12-02", "Ханука"),
            ("2021-12-03", "Ханука"),
            ("2021-12-04", "Ханука"),
            ("2021-12-05", "Ханука"),
            ("2021-12-06", "Ханука"),
        )
