;; Scheme frontends for various programs: patch, tar

;;  Copyright (C) 2013 Aljosha Papsch <misc@rpapsch.de>
;;
;;  This file is part of Upmf.
;;
;;  Upmf is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  Upmf is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with Upmf.  If not, see <http://www.gnu.org/licenses/>.

(define errormsg
  (lambda* (message #:optional append-str fatal)
	   (if (eq? append-str #f)
	       (begin (display (string-append (list-ref (command-line) 0)
					      ": " message)))
	       (begin (display (string-append (list-ref (command-line) 0)
					      ": " message ": "
					      append-str))))
	   (newline)
	   (if (eq? fatal 'fatal)
	       (quit))))

(define statusmsg
  (lambda* (message #:optional append-str)
	   (if (eq? append-str #f)
	       (begin (display (string-append ">> " message))
		      (newline))
	       (begin (display (string-append ">> " message
					      ": "  append-str))
		      (newline)))))

;; Execute shell COMMAND
(define exec
  (lambda (command)
    (let ((pipe (open-pipe command OPEN_WRITE)))
      (if (eq? (close-pipe pipe) 0)
	  #t
	  #f))))

(define (exec-list directory
		   cmd-list)
  (do ((k 0 (1+ k)))
      ((> k (- (length cmd-list) 1)))
    (exec (string-append "cd "
			 directory
			 "; "
			 (list-ref cmd-list k)))))

(define patch
  (lambda (directory patch)
    (statusmsg (_ "Applying patch") patch)
    (newline)
    (if (exec (string-append "patch -d " directory
			     " -Np1 " patch ".patch"))
	(statusmsg (_ "Patch successfully applied."))
	(errormsg (_ "Failed to apply patch") patch 'fatal))))

(define tar-extract
  (lambda (archive-file dest-dir)
    (statusmsg (_ "Extracting archive") archive-file)
    (if (exec (string-append "tar -xf " archive-file " -C " dest-dir))
	(statusmsg (_ "Archive file successfully extracted."))
	(errormsg (_ "Failed to extract archive") archive-file 'fatal))))

;; This procedure requires Guile >= 2.0.7 (string-split with
;; character sets).
(define compare-version
  (lambda (lv-str rv-str)
    (let* ((lv (string-split lv-str (char-set #\- #\.)))
	   (rv (string-split rv-str (char-set #\- #\.))))
      (do ((k 1 (1+ k)))
	  ((> k (- (length lv) 1)))
	(if (and (eq? (+ k 1) (length lv))
		 (> (length rv) (length lv)))
	    ;; return
	    '())
	(if (and (eq? (+ k 1) (length rv))
		 (> (length lv) (length rv)))
	    ;; return
	    '())))))

(define install-info
  (lambda (pkg-dest-dir)
    (let* ((dir-file (string-append pkg-dest-dir "/" package-prefix
				    "/share/info/dir"))
	   (info-dir (string-append pkg-dest-dir "/" package-prefix
				    "/share/info"))
	   (info-dir-stream (if (access? info-dir R_OK)
				(opendir info-dir)
				#f)))
      (if (not (exec (string-append "rm " dir-file)))
	      (errormsg (_ "Could not remove the directory file") dir-file))

      (do ((k (readdir info-dir-stream) (readdir info-dir-stream)))
	  ((eof-object? k))
	(if (not (or (string=? k ".")
                    (string=? k "..")))
            (if (string-suffix? ".info" k)
                (let ((info-file (string-append pkg-dest-dir "/" package-prefix
                                                "/share/info/" k)))
                  (if (not (exec (string-append "install-info " info-file
                                               " " package-prefix
                                               "/share/info/dir")))
                      (errormsg (_ "Could not install info file")
                                info-file)))))))))
