;;  Copyright (C) 2013 Aljosha Papsch <misc@rpapsch.de>
;;
;;  This file is part of Upmf.
;;
;;  Upmf is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  Upmf is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with Upmf.  If not, see <http://www.gnu.org/licenses/>.

(use-modules (ice-9 getopt-long))
(use-modules (ice-9 popen))

(load "utils.scm")
(load "install-modes.scm")
(load "find-utils.scm")
(load "dependency.scm")
(load "fetch.scm")
(load "package.scm")
(load "patch.scm")
(load "stow.scm")

(define (_ msg) (gettext msg))
(define help-message (_ "\
upmf [options]
  -i, --install PACKAGELIST            Install PACKAGELIST (comma separated list)
  -g, --install-group PACKAGEGROUPLIST Install groups of packages (comma separated list)
  -r, --remove  PACKAGELIST            Remove PACKAGELIST (comma separated list)
  -s, --search  PACKAGE                Search for PACKAGE
  -V, --version                        Display version
  -h, --help                           Display this help

Other options:
  -b, --bootstrap                      Enable bootstrap mode.  Some
                                       packages want to be build
                                       differently when creating a
                                       toolchain.  That's this option
                                       is for.

A package string looks like: SECTION/NAME[:VERSION]
For example: core/autoconf:2.69
A package group string looks like: NAME
For example: core, gnome, x11))"))
(define license-message
  (string-append "Upmf 0.7.1\n"
		 (_ "\
Copyright (C) 2013 Aljosha Papsch
License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>.
This is free software: you are free to change and redistribute it.
There is NO WARRANTY, to the extent permitted by law.")))

;; install-package adds a package string after successful installation
;; and install-handler displays these strings
(define installed-packages '())

;; Set whether bootstrap mode is enabled
(define bootstrap-mode #f)

(define install-package
  (lambda (package-string)
    (if (eq? package-string '())
	'()
    (begin (validate-package-string package-string 'no-version)
	   (statusmsg (_ "Installing package") package-string)
	   (let* ((package-with-ver (string-split package-string #\:))
		  (package-name (list-ref package-with-ver 0))
		  (package-release (if (eq? (length package-with-ver) 2)
				       (list-ref package-with-ver 1)
				       "latest"))
		  (package-file (find-package-file package-name))
		  (package-obj (load package-file))
		  (release-to-install (get-release-for package-obj package-release 'verbose))
		  (extracted-directory (fetch-and-extract (cdr release-to-install)))
		  (pkg-base-dir (string-append package-dest-dir "/"
					       (assq-ref package-obj 'section)
					       "/" (assq-ref package-obj 'name)))
		  (pkg-dest-dir (string-append pkg-base-dir
					       "/" (car release-to-install)
					       (if (eq? bootstrap-mode #f)
						   ""
						   "-bootstrap")))
		  (installed-list '())
		  (current-installed #f))
	     (if (access? pkg-dest-dir R_OK)
		 (if (exec (string-append "rm -rf " pkg-dest-dir))
		     (display "")
		     (errormsg (_ "Could not wipe installation directory")
			       pkg-dest-dir)))
	     (check-dependencies (assq-ref package-obj 'dependencies))
	     (apply-patches extracted-directory (assq-ref package-obj 'patches)
			    (car release-to-install))
	     (if (install-with-mode (assq-ref package-obj 'mode)
				    package-obj extracted-directory
				    pkg-dest-dir)
		 (begin (unless (eq? (assq-ref package-obj 'install-info) #f)
				(install-info pkg-dest-dir))
			(stow-package pkg-base-dir (car release-to-install))
			(set! installed-packages
			      (append installed-packages (list package-string))))))))))

(define remove-package
  (lambda (package-string)
    (validate-package-string package-string 'with-version)
    (let* ((pkg-and-ver (string-split package-string #\:))
	   (dir-to-remove (string-append package-dest-dir "/"
					 (list-ref pkg-and-ver 0) "/"
					 (list-ref pkg-and-ver 1)))
	   (package-name (list-ref pkg-and-ver 0))
	   (package-file (find-package-file package-name))
	   (package-obj (load package-file))
	   (pkg-base-dir (string-append package-dest-dir "/"
					(assq-ref package-obj 'section)
					"/" (assq-ref package-obj 'name))))
      (statusmsg (_ "Removing package") package-string)
      (unstow-package pkg-base-dir (list-ref pkg-and-ver 1))
      ;; remove installation directory
      (if (access? dir-to-remove F_OK)
	  (statusmsg (_ "Removing directory") dir-to-remove)
	  (errormsg (_ "Release not installed") package-string 'fatal))
      (if (exec (string-append "rm -r " dir-to-remove))
	  (statusmsg (_ "Package successfully removed") package-string)
	  (errormsg (_ "Unable to remove directory") dir-to-remove)))))

(define print-release
  (lambda (release package-obj)
    (newline)
    (display (string-append (_ "Release:\t") (car release)
			    " (" (cdr release) ")"))
    (if (is-installed? package-obj (car release))
	(display (_ " (installed)")))))

;; Print information about the package on the standard port
(define print-package-stat
  (lambda (package-obj)
    (display (string-append (_ "Package:\t") (assq-ref package-obj 'section)
			    "/" (assq-ref package-obj 'name)
			    (_ "\nDescription:\t")
			    (assq-ref package-obj 'description)))
    (do ((k 0 (1+ k)))
	((> k (- (length (assq-ref package-obj 'releases)) 1)))
      (print-release (list-ref (assq-ref package-obj 'releases) k)
		     package-obj))
    (newline)(newline)))

(define install-handler
  (lambda (install-list)
    (do ((k 0 (1+ k)))
	((> k (- (length install-list) 1)))
      (install-package (list-ref install-list k)))
      
    (newline)
    (statusmsg (_ "Successfully installed packages:"))
    (do ((k 0 (1+ k)))
	((> k (- (length installed-packages) 1)))
      (statusmsg (_ "Package") (list-ref installed-packages k)))
    (newline)))

(define remove-handler
  (lambda (remove-list)
    (do ((k 0 (1+ k)))
	((> k (- (length remove-list) 1)))
      (remove-package (list-ref remove-list k)))))

(define (upmf-start)
  (let* ((option-spec '((install (single-char #\i) (value #t))
			(install-group (single-char #\g) (value #t))
			(remove (single-char #\r) (value #t))
			(search (single-char #\s) (value #t))
			(bootstrap (single-char #\b) (value #f))
			(version (single-char #\V) (value #f))
			(help (single-char #\h) (value #f))))
	 (options (getopt-long (command-line) option-spec))
	 (install-wanted (option-ref options 'install #f))
	 (group-wanted (option-ref options 'install-group #f))
	 (remove-wanted (option-ref options 'remove #f))
	 (search-wanted (option-ref options 'search #f))
	 (bootstrap-wanted (option-ref options 'bootstrap #f))
	 (help-wanted (option-ref options 'help #f))
	 (version-wanted (option-ref options 'version #f)))
    (if (or install-wanted group-wanted remove-wanted search-wanted
	    bootstrap-wanted help-wanted version-wanted)
	(begin
	  (if bootstrap-wanted
	      (set! bootstrap-mode #t))
	  (if remove-wanted
	      (remove-handler (string-split remove-wanted #\,)))
	  (if group-wanted
	      (install-group (string-split group-wanted #\,)))
	  (if install-wanted
	      (install-handler (string-split install-wanted #\,)))
	  (if search-wanted
	      (find-package search-wanted))
	  (if help-wanted
	      (begin (display help-message)(newline)(quit)))
	  (if version-wanted
	      (begin (display license-message)(newline)(quit))))
	(begin (display help-message)(newline)))))
