/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   inputmonitor.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: This class is responsible for listening to keyboard events.
 *              It gets key-down and key-up signals, and updates the
 *              array of pressed-down keys. Then repetition is handled
 *              separately by other classes. 
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "inputmonitor.h"

/*---------------------------------------------------------------------------
 static variables
 ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
 methods
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * Creates a default input monitor
 */ 
U61_InputMonitor::U61_InputMonitor()
{
  int i;

  for (i=0;i<96;++i)
    {
      ascii_labels[2*i]=32+i;
      ascii_labels[2*i+1]=0;
      ascii[i].define(32+i,ascii_labels+2*i);
    }
  ascii[0].define(' ',"SPACE");

  for (i=0;i<CL_NUM_KEYS;++i)
    {
      scancode[i].define(' ',"unknown");
    }

  scancode[CL_KEY_A].define('A',"A");
  scancode[CL_KEY_B].define('B',"B");
  scancode[CL_KEY_C].define('C',"C");
  scancode[CL_KEY_D].define('D',"D");
  scancode[CL_KEY_E].define('E',"E");
  scancode[CL_KEY_F].define('F',"F");
  scancode[CL_KEY_G].define('G',"G");
  scancode[CL_KEY_H].define('H',"H");
  scancode[CL_KEY_I].define('I',"I");
  scancode[CL_KEY_J].define('J',"J");
  scancode[CL_KEY_K].define('K',"K");
  scancode[CL_KEY_L].define('L',"L");
  scancode[CL_KEY_M].define('M',"M");
  scancode[CL_KEY_N].define('N',"N");
  scancode[CL_KEY_O].define('O',"O");
  scancode[CL_KEY_P].define('P',"P");
  scancode[CL_KEY_Q].define('Q',"Q");
  scancode[CL_KEY_R].define('R',"R");
  scancode[CL_KEY_S].define('S',"S");
  scancode[CL_KEY_T].define('T',"T");
  scancode[CL_KEY_U].define('U',"U");
  scancode[CL_KEY_V].define('V',"V");
  scancode[CL_KEY_W].define('W',"W");
  scancode[CL_KEY_X].define('X',"X");
  scancode[CL_KEY_Y].define('Y',"Y");
  scancode[CL_KEY_Z].define('Z',"Z");
  scancode[CL_KEY_0].define('0',"0");
  scancode[CL_KEY_1].define('1',"1");
  scancode[CL_KEY_2].define('2',"2");
  scancode[CL_KEY_3].define('3',"3");
  scancode[CL_KEY_4].define('4',"4");
  scancode[CL_KEY_5].define('5',"5");
  scancode[CL_KEY_6].define('6',"6");
  scancode[CL_KEY_7].define('7',"7");
  scancode[CL_KEY_8].define('8',"8");
  scancode[CL_KEY_9].define('9',"9");
  scancode[CL_KEY_F1].define(' ',"F1");
  scancode[CL_KEY_F2].define(' ',"F2");
  scancode[CL_KEY_F3].define(' ',"F3");
  scancode[CL_KEY_F4].define(' ',"F4");
  scancode[CL_KEY_F5].define(' ',"F5");
  scancode[CL_KEY_F6].define(' ',"F6");
  scancode[CL_KEY_F7].define(' ',"F7");
  scancode[CL_KEY_F8].define(' ',"F8");
  scancode[CL_KEY_F9].define(' ',"F9");
  scancode[CL_KEY_F10].define(' ',"F10");
  scancode[CL_KEY_F11].define(' ',"F11");
  scancode[CL_KEY_F12].define(' ',"F12");
  scancode[CL_KEY_ESCAPE].define(' ',"ESCAPE");
  scancode[CL_KEY_LEFT].define(' ',"LEFT");
  scancode[CL_KEY_RIGHT].define(' ',"RIGHT");
  scancode[CL_KEY_UP].define(' ',"UP");
  scancode[CL_KEY_DOWN].define(' ',"DOWN");
  scancode[CL_KEY_LCTRL].define(' ',"LCTRL");
  scancode[CL_KEY_RCTRL].define(' ',"RCTRL");
  scancode[CL_KEY_LSHIFT].define(' ',"LSHIFT");
  scancode[CL_KEY_RSHIFT].define(' ',"RSHIFT");
  scancode[CL_KEY_ALT].define(' ',"ALT");
  scancode[CL_KEY_ALTGR].define(' ',"ALTGR");
  scancode[CL_KEY_TAB].define(' ',"TAB");
  scancode[CL_KEY_ENTER].define(' ',"ENTER");
  scancode[CL_KEY_SPACE].define(' ',"SPACE");
  scancode[CL_KEY_BACKSPACE].define(' ',"BACKSPACE");
  scancode[CL_KEY_INSERT].define(' ',"INSERT");
  scancode[CL_KEY_DELETE].define(' ',"DELETE");
  scancode[CL_KEY_HOME].define(' ',"HOME");
  scancode[CL_KEY_END].define(' ',"END");
  scancode[CL_KEY_PAGEUP].define(' ',"PAGEUP");
  scancode[CL_KEY_PAGEDOWN].define(' ',"PAGEDOWN");
  scancode[CL_KEY_CAPSLOCK].define(' ',"CAPSLOCK");
  scancode[CL_KEY_NUMLOCK].define(' ',"NUMLOCK");
  scancode[CL_KEY_SCRLOCK].define(' ',"SCRLOCK");
  scancode[CL_KEY_PRINT].define(' ',"PRINT");
  scancode[CL_KEY_PAUSE].define(' ',"PAUSE");
  scancode[CL_KEY_KP_DIV].define(' ',"KP_DIV");
  scancode[CL_KEY_KP_MULT].define(' ',"KP_MULT");
  scancode[CL_KEY_KP_MINUS].define(' ',"KP_MINUS");
  scancode[CL_KEY_KP_PLUS].define(' ',"KP_PLUS");
  scancode[CL_KEY_KP_ENTER].define(' ',"KP_ENTER");

  reset();
}

/*--------------------------------------------------------------------------*/
/* 
 * Binds the keyboard to the input monitor
 * Could not be done in the constructor since this object is global/static
 */ 
void U61_InputMonitor::init()
{
  device = CL_Input::keyboards[0];

  slots.add_slot(CL_Input::sig_button_press.connect
		 (thCreateSlot(this, &U61_InputMonitor::on_button_press)));
  slots.add_slot(CL_Input::sig_button_release.connect
		 (thCreateSlot(this, &U61_InputMonitor::on_button_release)));
}

/*--------------------------------------------------------------------------*/
/* 
 * Destroys the input monitor
 */ 
U61_InputMonitor::~U61_InputMonitor()
{
}

/*--------------------------------------------------------------------------*/
/* 
 * Resets all the keys.
 */ 
void U61_InputMonitor::reset()
{
  int i;

  for (i=0;i<96;++i)
    {
      ascii[i].reset();
      ascii_time[i]=0;
    }
  for (i=0;i<CL_NUM_KEYS;++i)
    {
      scancode[i].reset();
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns the state of a given key (pressed or not)
 */ 
bool U61_InputMonitor::is_pressed(int key)
{
  bool result=false;

  if (key<CL_NUM_KEYS)
    {
      result=scancode[key].get_state();
    }
  else
    {
      result=ascii[key-CL_NUM_KEYS].get_state();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Converts an internal keycode to a printable ascii code
 */ 
char U61_InputMonitor::to_ascii(int key)
{
  char result=' ';

  if (key>=CL_NUM_KEYS)
    {
      result=ascii[key-CL_NUM_KEYS].get_ascii();
    }
  else
    {
      result=scancode[key].get_ascii();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Converts an internal scancode to a ClanLib scancode
 */ 
int U61_InputMonitor::to_scancode(int key)
{
  int result=CL_KEY_NONE_OF_THE_ABOVE;

  if (key<CL_NUM_KEYS)
    {
      return key;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Converts an internal keycode to a readable label
 */ 
char *U61_InputMonitor::to_label(int key)
{
  char *result="?";

  if (key>=CL_NUM_KEYS)
    {
      result=ascii[key-CL_NUM_KEYS].get_label();
    }
  else
    {
      result=scancode[key].get_label();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Callback called when a key is pressed down
 */ 
void U61_InputMonitor::on_button_press(CL_InputDevice *device,
				       const CL_Key &key)
{
  if (this->device != device) return; // wrong device.

  if (key.ascii>=32 && key.ascii<=127)
    {
      ascii[key.ascii-32].press();
      ascii_time[key.ascii-32]=CL_System::get_time();
    }
  else if (key.id<CL_KEY_NONE_OF_THE_ABOVE)
    {
      scancode[key.id].press();
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * Callback called when a key is released
 */ 
void U61_InputMonitor::on_button_release(CL_InputDevice *device,
					 const CL_Key &key)
{
  if (this->device != device) return; // wrong device.

  if (key.ascii>=32 && key.ascii<=127)
    {
      ascii[key.ascii-32].release();
      ascii_time[key.ascii-32]=0;
    }
  else if (key.id<CL_KEY_NONE_OF_THE_ABOVE)
    {
      scancode[key.id].release();
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * Callback called when a key is released
 */ 
void U61_InputMonitor::check()
{
  int i;
  bool reset_needed=false;

  for (i=0;i<96;++i)
    {
      if (ascii_time[i]>0 &&
	  CL_System::get_time()-ascii_time[i]>1000)
	{
	  reset_needed=true;
	}
    }
  if (reset_needed)
    {
      reset();
    }
}


