
/*
 *  bltDragDrop.c --
 *
 *	This module implements a drag-and-drop mechanism for the Tk
 *	Toolkit.  Allows widgets to be registered as drag&drop sources
 *	and targets for handling "drag-and-drop" operations between
 *	Tcl/Tk applications.
 *
 * Copyright 1993-1998 Lucent Technologies, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the names
 * of Lucent Technologies any of their entities not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and
 * fitness.  In no event shall Lucent Technologies be liable for any
 * special, indirect or consequential damages or any damages
 * whatsoever resulting from loss of use, data or profits, whether in
 * an action of contract, negligence or other tortuous action, arising
 * out of or in connection with the use or performance of this
 * software.
 *
 * The "drag&drop" command was created by Michael J. McLennan.  
 */
#include "bltInt.h"

#ifndef NO_DRAGDROP
#include <X11/Xatom.h>

#if HAVE_NAMESPACES
static char dragDropCmd[] = "blt::drag&drop";
#else
static char dragDropCmd[] = "drag&drop";
#endif

/* Error Proc used to report drag&drop background errors */
#define DEF_ERROR_PROC              "tkerror"

static char className[] = "DragDrop";	/* CLASS NAME for token window */
static char propName[] = "DragDropInfo";	/* Property name */

/*
 *  DRAG&DROP ROOT WINDOW HIERARCHY (cached during "drag" operations)
 */

typedef struct TreeNode {
    Window window;		/* X window for this record */
    int initialized;		/* non-zero => rest of info is valid */

    struct TreeNode *parent;	/* Node containing this as a child */
    int x0, y0;			/* upper-left corner of window */
    int x1, y1;			/* lower-right corner of window */
    char *ddprop;		/* drag&drop property info */

    Tk_Uid interpName;		/* interp name within ddprop */
    char *targetName;
    Blt_LinkedList *handlerListPtr;	
				/* List of handlers within the drag&drop
				 * property. NULL if no handlers are
				 * defined. */
    Blt_LinkedList *childListPtr; 
				/* List of this window's children */
} TreeNode;

static Tcl_HashTable sourceTable;
static Tcl_HashTable targetTable;
static Window root;
static char *errorCmd;
static int numActive;
static int locX, locY;
static int initialized = FALSE;

/*
 *  DRAG&DROP REGISTRATION DATA
 */
typedef struct Token {
    Tk_Window tkwin;		/* Window representing drag item */
    int lastX, lastY;		/* last position of token window */
    int overTarget;		/* non-zero => over target window */
    Tk_TimerToken timerToken;	/* token for routine to hide tokenwin */
    GC rejectFgGC;		/* GC used to draw rejection fg: (\) */
    GC rejectBgGC;		/* GC used to draw rejection bg: (\) */
    Window Xid;			/* X Window identifier for token */

    /* User-configurable fields */

    Tk_Anchor anchor;		/* Position of token win relative to mouse */
    Tk_Cursor cursor;		/* Cursor used when dragging token */
    Tk_3DBorder outline;	/* Outline border around token window */
    Tk_3DBorder normalBorder;	/* Border/background for token window */
    Tk_3DBorder activeBorder;	/* Border/background for token window */
    int relief;
    int activeRelief;
    int borderWidth;		/* Border width in pixels */
    int activeBorderWidth;	/* Border width in pixels */
    XColor *rejectFg;		/* Color used to draw rejection fg: (\) */
    XColor *rejectBg;		/* Color used to draw rejection bg: (\) */
    Pixmap rejectStipple;	/* Stipple used to draw rejection: (\) */

} Token;

typedef struct {
    Tcl_Interp *interp;
    Tcl_Command cmdToken;
    Tk_Window tkwin;		/* drag&drop source window */
    Display *display;		/* drag&drop source window display */
    Blt_LinkedList *listPtr;

    Atom ddAtom;		/* X atom referring to "DragDropInfo" */
    int button;			/* button used to invoke drag for sources */

    Token token;		/* Information about drag&drop token window */

    int pkgCmdInProg;		/* non-zero => executing pkgCmd */
    char *pkgCmd;		/* cmd executed on start of drag op */
    char *pkgCmdResult;		/* result returned by recent pkgCmd */
    char *sitecmd;		/* cmd executed to update token win */

    TreeNode *allwins;		/* window info (used during "drag") */
    int selfTarget;		/* non-zero => source can drop onto itself */
    Tk_Cursor normalCursor;	/* cursor restored after dragging */

    char *send;			/* list of data handler names or "all" */
} Source;

/*
 *  STACK INFO
 */
typedef struct DD_Stack {
    ClientData *values;		/* values on stack */
    int len;			/* number of values on stack */
    int max;			/* maximum size of stack */
    ClientData space[5];	/* initial space for stack data */
} DD_Stack;

/*
 *  PERCENT SUBSTITUTIONS
 */
typedef struct DD_PercentSubst {
    char letter;		/* character like 'x' in "%x" */
    char *value;		/* value to be substituted in place of "%x" */
} DD_PercentSubst;

typedef struct {
    Tcl_Interp *interp;
    Tk_Window tkwin;		/* drag&drop target window */
    Display *display;		/* drag&drop target window display */
    Blt_LinkedList *listPtr;
} Target;

/*
 * Each "drag&drop" widget window is tagged with a "DragDropInfo"
 * property in XA_STRING format.  This property identifies the
 * window as a "drag&drop" widget, and contains the following:
 *
 *     "<interp-name>]<drag&drop-path>]<handler-list>"
 *
 * The <drag&drop-path> is the window path name of the drag&drop
 * widget, <interp-name> is the name of the interpreter controlling
 * the widget (useful for the "send" command), and <handler-list>
 * is the list of handler types recognized by the widget.
 *
 * When the user invokes the "drag" operation, a snapshot of the
 * entire window hierarchy is made, and windows carrying a
 * "DragDropInfo" property are identified.  As the token window is
 * dragged around, * this snapshot can be queried to determine when
 * the token is over a valid target window.  When the token is
 * dropped over a valid site, the drop information is sent to the
 * application via the usual "send" command.  If communication fails,
 * the drag&drop facility automatically posts a rejection symbol on
 * the token window.
 */

/*
 *  Maximum size property that can be read at one time:
 */
#define MAX_PROP_SIZE 1000

/*
 *  CONFIG PARAMETERS
 */

#define DEF_DND_BUTTON_BG_COLOR		RGB_COLOR_YELLOW
#define DEF_DND_BUTTON_BG_MONO		STD_MONO_NORMAL_BG
#define DEF_DND_BUTTON_NUMBER		"3"
#define DEF_DND_PACKAGE_COMMAND		(char *)NULL
#define DEF_DND_REJECT_BG_COLOR		STD_COLOR_NORMAL_BG
#define DEF_DND_REJECT_BG_MONO		RGB_COLOR_WHITE
#define DEF_DND_REJECT_FG_COLOR		RGB_COLOR_RED
#define DEF_DND_REJECT_FG_MONO		RGB_COLOR_BLACK
#define DEF_DND_REJECT_STIPPLE_COLOR	(char *)NULL
#define DEF_DND_REJECT_STIPPLE_MONO	RGB_COLOR_GREY50
#define DEF_DND_SELF_TARGET		"no"
#define DEF_DND_SEND			"all"
#define DEF_DND_SITE_COMMAND		(char *)NULL
#define DEF_DND_TOKEN_ACTIVE_BG_COLOR	STD_COLOR_ACTIVE_BG
#define DEF_DND_TOKEN_ACTIVE_BG_MONO	STD_MONO_ACTIVE_BG
#define DEF_DND_TOKEN_ACTIVE_BORDERWIDTH	"3"
#define DEF_DND_TOKEN_ACTIVE_RELIEF	"sunken"
#define DEF_DND_TOKEN_ANCHOR		"se"
#define DEF_DND_TOKEN_BG_COLOR		STD_COLOR_NORMAL_BG
#define DEF_DND_TOKEN_BG_MONO		STD_MONO_NORMAL_BG
#define DEF_DND_TOKEN_BORDERWIDTH	"3"
#define DEF_DND_TOKEN_CURSOR		"top_left_arrow"
#define DEF_DND_TOKEN_OUTLINE_COLOR	RGB_COLOR_BLACK
#define DEF_DND_TOKEN_OUTLINE_MONO	RGB_COLOR_BLACK
#define DEF_DND_TOKEN_RELIEF		"raised"

static Tk_ConfigSpec configSpecs[] =
{
    {TK_CONFIG_INT, "-button", "buttonBinding", "ButtonBinding",
	DEF_DND_BUTTON_NUMBER, Tk_Offset(Source, button), 0},
    {TK_CONFIG_STRING, "-packagecmd", "packageCommand", "Command",
	DEF_DND_PACKAGE_COMMAND, Tk_Offset(Source, pkgCmd), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-rejectbg", "rejectBackground", "Background",
	DEF_DND_REJECT_BG_COLOR, Tk_Offset(Source, token.rejectBg),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-rejectbg", "rejectBackground", "Background",
	DEF_DND_REJECT_BG_MONO, Tk_Offset(Source, token.rejectBg),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_COLOR, "-rejectfg", "rejectForeground", "Foreground",
	DEF_DND_REJECT_FG_COLOR, Tk_Offset(Source, token.rejectFg),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-rejectfg", "rejectForeground", "Foreground",
	DEF_DND_REJECT_BG_COLOR, Tk_Offset(Source, token.rejectFg),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BITMAP, "-rejectstipple", "rejectStipple", "Stipple",
	DEF_DND_REJECT_STIPPLE_COLOR, Tk_Offset(Source, token.rejectStipple),
	TK_CONFIG_COLOR_ONLY | TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-rejectstipple", "rejectStipple", "Stipple",
	DEF_DND_REJECT_STIPPLE_MONO, Tk_Offset(Source, token.rejectStipple),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BOOLEAN, "-selftarget", "selfTarget", "SelfTarget",
	DEF_DND_SELF_TARGET, Tk_Offset(Source, selfTarget), 0},
    {TK_CONFIG_STRING, "-send", "send", "Send",
	DEF_DND_SEND, Tk_Offset(Source, send), TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-sitecmd", "siteCommand", "Command",
	DEF_DND_SITE_COMMAND, Tk_Offset(Source, sitecmd), TK_CONFIG_NULL_OK},
    {TK_CONFIG_ANCHOR, "-tokenanchor", "tokenAnchor", "Anchor",
	DEF_DND_TOKEN_ANCHOR, Tk_Offset(Source, token.anchor), 0},
    {TK_CONFIG_BORDER, "-tokenactivebackground",
	"tokenActiveBackground", "ActiveBackground",
	DEF_DND_TOKEN_ACTIVE_BG_COLOR, Tk_Offset(Source, token.activeBorder),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-tokenactivebackground",
	"tokenActiveBackground", "ActiveBackground",
	DEF_DND_TOKEN_ACTIVE_BG_MONO, Tk_Offset(Source, token.activeBorder),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BORDER, "-tokenbg", "tokenBackground", "Background",
	DEF_DND_TOKEN_BG_COLOR, Tk_Offset(Source, token.normalBorder),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-tokenbg", "tokenBackground", "Background",
	DEF_DND_TOKEN_BG_MONO, Tk_Offset(Source, token.normalBorder),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BORDER, "-tokenoutline", "tokenOutline", "Outline",
	DEF_DND_TOKEN_OUTLINE_COLOR, Tk_Offset(Source, token.outline),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-tokenoutline", "tokenOutline", "Outline",
	DEF_DND_TOKEN_OUTLINE_MONO, Tk_Offset(Source, token.outline),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_PIXELS, "-tokenborderwidth", "tokenBorderWidth", "BorderWidth",
	DEF_DND_TOKEN_BORDERWIDTH, Tk_Offset(Source, token.borderWidth), 0},
    {TK_CONFIG_CURSOR, "-tokencursor", "tokenCursor", "Cursor",
	DEF_DND_TOKEN_CURSOR, Tk_Offset(Source, token.cursor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL, (char *)NULL, 0, 0},
};

static Tk_ConfigSpec tokenConfigSpecs[] =
{
    {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "ActiveBackground",
	DEF_DND_TOKEN_ACTIVE_BG_COLOR, Tk_Offset(Token, activeBorder),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "ActiveBackground",
	DEF_DND_TOKEN_ACTIVE_BG_MONO, Tk_Offset(Token, activeBorder),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_RELIEF, "-activerelief", "activeRelief", "activeRelief",
	DEF_DND_TOKEN_ACTIVE_RELIEF, Tk_Offset(Token, activeRelief), 0},
    {TK_CONFIG_ANCHOR, "-anchor", "anchor", "Anchor",
	DEF_DND_TOKEN_ANCHOR, Tk_Offset(Token, anchor), 0},
    {TK_CONFIG_PIXELS, "-activeborderwidth", "activeBorderWidth",
	"ActiveBorderWidth",
	DEF_DND_TOKEN_ACTIVE_BORDERWIDTH, Tk_Offset(Token, activeBorderWidth), 0},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_DND_TOKEN_BG_COLOR, Tk_Offset(Token, normalBorder),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_DND_TOKEN_BG_MONO, Tk_Offset(Token, normalBorder),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_DND_TOKEN_BORDERWIDTH, Tk_Offset(Token, borderWidth), 0},
    {TK_CONFIG_CURSOR, "-cursor", "cursor", "Cursor",
	DEF_DND_TOKEN_CURSOR, Tk_Offset(Token, cursor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BORDER, "-outline", "outline", "Outline",
	DEF_DND_TOKEN_OUTLINE_COLOR, Tk_Offset(Token, outline),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-outline", "outline", "Outline",
	DEF_DND_TOKEN_OUTLINE_MONO, Tk_Offset(Token, outline), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_COLOR, "-rejectbg", "rejectBackground", "Background",
	DEF_DND_REJECT_BG_COLOR, Tk_Offset(Token, rejectBg), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-rejectbg", "rejectBackground", "Background",
	DEF_DND_REJECT_BG_MONO, Tk_Offset(Token, rejectBg), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_COLOR, "-rejectfg", "rejectForeground", "Foreground",
	DEF_DND_REJECT_FG_COLOR, Tk_Offset(Token, rejectFg), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-rejectfg", "rejectForeground", "Foreground",
	DEF_DND_REJECT_BG_COLOR, Tk_Offset(Token, rejectFg), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BITMAP, "-rejectstipple", "rejectStipple", "Stipple",
	DEF_DND_REJECT_STIPPLE_COLOR, Tk_Offset(Token, rejectStipple),
	TK_CONFIG_COLOR_ONLY | TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-rejectstipple", "rejectStipple", "Stipple",
	DEF_DND_REJECT_STIPPLE_MONO, Tk_Offset(Token, rejectStipple),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
	DEF_DND_TOKEN_RELIEF, Tk_Offset(Token, relief), 0},
    {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL, (char *)NULL, 0, 0},
};

/*
 *  FORWARD DECLARATIONS
 */
static void DragDrop_Delete _ANSI_ARGS_((ClientData clientData));
static int DragDrop_Cmd _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, int argc, char **argv));

static Source *CreateSource _ANSI_ARGS_((char *pathname, int *newEntry));
static Source *FindSource _ANSI_ARGS_((char *pathname));
static void DestroySource _ANSI_ARGS_((Source *srcPtr));
static int ConfigureSource _ANSI_ARGS_((Tcl_Interp *interp, Source *srcPtr,
	int argc, char **argv, int flags));
static int ConfigureToken _ANSI_ARGS_((Tcl_Interp *interp, Source *srcPtr,
	int argc, char **argv));
static char *FindSourceHandler _ANSI_ARGS_((Source *srcPtr, char *dtname));
static void PutSourceHandler _ANSI_ARGS_((Source *srcPtr, char *dtname,
	char *cmd));
static SourceHndl *CreateSourceHandler _ANSI_ARGS_((char *dtname, char *cmd));
static void UnregSource _ANSI_ARGS_((ClientData clientData, XEvent *eventPtr));

static Target *CreateTarget _ANSI_ARGS_((char *pathname, int *newEntry));
static void DestroyTarget _ANSI_ARGS_((Target *targetPtr));
static void UnregTarget _ANSI_ARGS_((ClientData clientData, XEvent *eventPtr));

static void DragDropSend _ANSI_ARGS_((Source *srcPtr));
static char *DragDropSendHndlr _ANSI_ARGS_((Source *srcPtr,
	char *interpName, char *ddName));

static TreeNode *GetTreeInfo _ANSI_ARGS_((Source *srcPtr));
static TreeNode *FindTargetWin _ANSI_ARGS_((Source *srcPtr, int x, int y));
static void TreeRelease _ANSI_ARGS_((TreeNode *wr));
static void QueryTree _ANSI_ARGS_((TreeNode *wr, Source *srcPtr));

static void AddPropertyToTarget _ANSI_ARGS_((Target *targetPtr));
static void TokenEventProc _ANSI_ARGS_((ClientData clientData, XEvent *eventPtr));
static void MoveToken _ANSI_ARGS_((Source *srcPtr, Token *tokenPtr));
static void UpdateToken _ANSI_ARGS_((ClientData clientData));
static void HideToken _ANSI_ARGS_((Token *tokenPtr));
static void RejectToken _ANSI_ARGS_((Token *tokenPtr));

static void StackInit _ANSI_ARGS_((DD_Stack *stack));
static void StackDelete _ANSI_ARGS_((DD_Stack *stack));
static void StackPush _ANSI_ARGS_((ClientData cdata, DD_Stack *stack));
static ClientData StackPop _ANSI_ARGS_((DD_Stack *stack));

static char *ExpandPercents _ANSI_ARGS_((char *str, DD_PercentSubst *subs,
	int nsubs, Tcl_DString * dStrPtr));


/*
 * ------------------------------------------------------------------------
 *  Blt_DragDropInit()
 *
 *  Adds the drag&drop command to the given interpreter.  Should be
 *  invoked to properly install the command whenever a new interpreter
 *  is created.
 * ------------------------------------------------------------------------
 */
int
Blt_DragDropInit(interp)
    Tcl_Interp *interp;		/* interpreter to be updated */
{
    static Blt_CmdSpec cmdSpec = {
	"drag&drop", DragDrop_Cmd, DragDrop_Delete,
    };
    if (!initialized) {
	Tcl_InitHashTable(&sourceTable, TCL_STRING_KEYS);
	Tcl_InitHashTable(&targetTable, TCL_STRING_KEYS);
	errorCmd = strdup(DEF_ERROR_PROC);
	numActive = 0;
	locX = locY = 0;
	initialized = TRUE;
    }
    if (Blt_InitCmd(interp, "blt", &cmdSpec) == NULL) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *  DragDrop_Delete()
 *
 *  Invoked when the drag&drop command is removed from an interpreter
 *  to free up allocated memory.
 * ------------------------------------------------------------------------
 */
static void
DragDrop_Delete(cdata)
    ClientData cdata;		/* client data for drag&drop command */
{
    TreeNode *wrpool, *wrnext;

    Tcl_DeleteHashTable(&sourceTable);
    Tcl_DeleteHashTable(&targetTable);

    if (errorCmd != NULL) {
	free((char *)errorCmd);
    }
}

static void
RaiseToken(srcPtr, tokenPtr)
    Source *srcPtr;
    Token *tokenPtr;
{
    if (tokenPtr->Xid == 0) {
	tokenPtr->Xid = Blt_WindowId(tokenPtr->tkwin);
    }
#ifdef WIN32
    XRaiseWindow(srcPtr->display, tokenPtr->Xid);
#else
    {
	XWindowChanges changes;

	changes.stack_mode = Above;
	changes.sibling = None;
	XReconfigureWMWindow(srcPtr->display, tokenPtr->Xid,
	     Tk_ScreenNumber(tokenPtr->tkwin), CWStackMode, &changes);
    }
#endif
}

/*
 *  Create the window for the drag&drop token...
 */
static int
CreateToken(interp, srcPtr)
    Tcl_Interp *interp;
    Source *srcPtr;
{
    XSetWindowAttributes atts;
    Tk_Window tkwin;
    char tokenName[200];
    static int nextTokenId = 0;
    Token *tokenPtr = &(srcPtr->token);

    sprintf(tokenName, "dd-token%d", ++nextTokenId);
    tkwin = Tk_CreateWindow(interp, srcPtr->tkwin, tokenName, "");
    if (tkwin == NULL) {
	Tcl_AppendResult(interp, "can't token window \"", tokenName, "\"",
	 (char *)NULL);
	return TCL_ERROR;
    }
    Tk_SetClass(tkwin, className);
    Tk_CreateEventHandler(tkwin, ExposureMask | StructureNotifyMask,
	TokenEventProc, (ClientData)tokenPtr);

    atts.override_redirect = True;
    atts.save_under = True;
    Tk_ChangeWindowAttributes(tkwin, CWOverrideRedirect | CWSaveUnder, &atts);
    Tk_SetInternalBorder(tkwin, tokenPtr->borderWidth + 2);
    tokenPtr->tkwin = tkwin;

    if (srcPtr->button > 0) {
	if (Tcl_VarEval(interp, "blt::InitDrag&DropBindings ", 
		Tk_PathName(srcPtr->tkwin), " ", Blt_Int(srcPtr->button), " ",
	    Tcl_GetCommandName(srcPtr->cmdToken), (char *)NULL) != TCL_OK) {
	    Tk_DestroyWindow(tokenPtr->tkwin);
	    return TCL_ERROR;
	}
    }
    /*
     *  Arrange for the window to unregister itself when it
     *  is destroyed.
     */
    Tk_CreateEventHandler(srcPtr->tkwin, StructureNotifyMask, UnregSource,
	(ClientData)srcPtr);
    return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *  DragDrop_Cmd()
 *
 *  Invoked by TCL whenever the user issues a drag&drop command.
 *  Handles the following syntax:
 *
 *    drag&drop source
 *    drag&drop source <pathName> ?options...?
 *    drag&drop source <pathName> handler ?<dataType>? ?<cmd> <arg>...?
 *
 *    drag&drop target
 *    drag&drop target <pathName> handler ?<dataType> <cmd> <arg>...?
 *    drag&drop target <pathName> handle <dataType> ?<value>?
 *
 *    drag&drop token <pathName>
 *    drag&drop drag <pathName> <x> <y>
 *    drag&drop drop <pathName> <x> <y>
 *
 *    drag&drop errors ?<proc>?
 *    drag&drop active
 *    drag&drop location ?<x> <y>?
 *
 * ------------------------------------------------------------------------
 */
static int
DragDrop_Cmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Registration information */
    Tcl_Interp *interp;		/* current interpreter */
    int argc;			/* number of arguments */
    char **argv;		/* argument strings */
{
    register Source *srcPtr;
    register Target *targetPtr;
    Token *tokenPtr;
    int status, length, x, y, isNew;
    char c, *cmd;
    DD_PercentSubst subst[2];
    Tk_Window tkwin;
    char buffer[1024];

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    " oper ?args?\"", (char *)NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);

    /*
     *  HANDLE:  drag&drop source
     *           drag&drop source <pathName> ?options...?
     *           drag&drop source <pathName> handler ?<data>? ?<scmd> <arg>...?
     */
    if ((c == 's') && strncmp(argv[1], "source", length) == 0) {
	/*
         *  HANDLE:  drag&drop source
         */
	if (argc == 2) {
	    Tcl_HashSearch cursor;
	    Tcl_HashEntry *hPtr;
	    char *name;

	    for (hPtr = Tcl_FirstHashEntry(&sourceTable, &cursor);
		hPtr != NULL; hPtr = Tcl_NextHashEntry(&cursor)) {
		name = Tcl_GetHashKey(&sourceTable, hPtr);
		Tcl_AppendElement(interp, name);
	    }
	    return TCL_OK;
	}
	/*
         *  Find or create source info...
         */
	srcPtr = CreateSource(argv[2], &isNew);
	srcPtr->cmdToken = (Tcl_Command)clientData;
	if (srcPtr == NULL) {
	    return TCL_ERROR;
	}
	tokenPtr = &(srcPtr->token);
	if (argc > 3) {
	    /*
	     *  HANDLE:  drag&drop source <pathName> ?options...?
	     */
	    c = argv[3][0];
	    length = strlen(argv[3]);

	    if (c == '-') {
		if (argc == 3) {
		    status = Tk_ConfigureInfo(interp, tokenPtr->tkwin, configSpecs,
			(char *)srcPtr, (char *)NULL, 0);
		} else if (argc == 4) {
		    status = Tk_ConfigureInfo(interp, tokenPtr->tkwin, configSpecs,
			(char *)srcPtr, argv[3], 0);
		} else {
		    status = ConfigureSource(interp, srcPtr, argc - 3, argv + 3,
			TK_CONFIG_ARGV_ONLY);
		}
		if (status != TCL_OK) {
		    return TCL_ERROR;
		}
	    } else if ((c == 'h') && strncmp(argv[3], "handler", length) == 0) {
		return HandlerOpOp(srcPtr, interp, argc, argv);
#ifdef notdef		
		/*
		 *  HANDLE:  drag&drop source <pathName> handler \
		 *             ?<data>? ?<scmd>...?
		 */
		if (argc == 4) {
		    /* Show source handler data types */ 
		    for(item = Blt_ListFirstItem(srcPtr->listPtr); 
			item != NULL; item = Blt_NextListItem(item)) {
			Tcl_AppendElement(interp, Blt_GetListKey(item));
		    }
		    return TCL_OK;

		} else if (argc == 5) {
		    /*
		     *  HANDLE:  drag&drop source <pathName> handler <data>
		     *
		     *    Create the new <data> type if it doesn't already
		     *    exist, and return the code associated with it.
		     */
		    item = Blt_ListFindItem(srcPtr->listPtr, argv[4]);
		    if (item) {
			char *cmd = Blt_ListGetValue(item);

			Tcl_SetResult(interp, cmd, TCL_STATIC);
			return TCL_OK;
		    }
		    if (strstr(argv[4], " ")) {
			Tcl_AppendResult(interp, "bad source handler name \"",
			    argv[4], "\": should not contain spaces",
			    (char *)NULL);
			return TCL_ERROR;
		    } 
		    /* Create NULL handler for data type */
		    PutSourceHandler(srcPtr, argv[4], "");
		    return TCL_OK;

		} else {
		    /*
		     *  HANDLE:  drag&drop source <pathName> handler \
		     *               <data> <cmd> ?<arg>...?
		     *
		     *    Create the new <data> type and set its command
		     */
		    if (strstr(argv[4], " ")) {
			Tcl_AppendResult(interp,
			    "bad source handler name \"",
			    argv[4], "\": should not contain spaces",
			    (char *)NULL);
			return TCL_ERROR;
		    }
		    cmd = Tcl_Concat(argc - 5, argv + 5);
		    PutSourceHandler(srcPtr, argv[4], cmd);
		    ckfree(cmd);
		    return TCL_OK;
		}
#endif 
	    } else {
		Tcl_AppendResult(interp, "bad operation \"", argv[3],
		    "\": must be \"handler\" or a configuration option",
		    (char *)NULL);
		return TCL_ERROR;
	    }
	}
	if (isNew) {
	    /*
             *  Create the window for the drag&drop token...
             */
	    if (CreateToken(interp, srcPtr) != TCL_OK) {
		DestroySource(srcPtr);
		return TCL_ERROR;
	    }
	}
    } else if ((c == 't') && (length >= 2) &&
	(strncmp(argv[1], "target", length) == 0)) {
	/*
	 *  HANDLE:  drag&drop target ?<pathName>? ?handling info...?
	 */
	if (argc == 2) {
	    Tcl_HashSearch cursor;
	    Tcl_HashEntry *hPtr;

	    for (hPtr = Tcl_FirstHashEntry(&targetTable, &cursor);
		hPtr != NULL; hPtr = Tcl_NextHashEntry(&cursor)) {
		Tcl_AppendElement(interp,
		    Tcl_GetHashKey(&targetTable, hPtr));
	    }
	    return TCL_OK;
	}
	tkwin = Tk_NameToWindow(interp, argv[2], Tk_MainWindow(interp));
	if (tkwin == NULL) {
	    Tcl_AppendResult(interp, "window does not exist: ", argv[2],
		(char *)NULL);
	    return TCL_ERROR;
	}
	targetPtr = CreateTarget(argv[2], &isNew);
	targetPtr->tkwin = tkwin;
	targetPtr->display = Tk_Display(tkwin);

	/*
         *  If this is a new target, attach a property to identify
         *  window as "drag&drop" target, and arrange for the window
         *  to un-register itself when it is destroyed.
         */
	if (isNew) {
	    Tk_MakeWindowExist(targetPtr->tkwin);
	    AddPropertyToTarget(interp, targetPtr);

	    /*
             *  Arrange for the window to unregister itself when it
             *  is destroyed.
             */
	    Tk_CreateEventHandler(targetPtr->tkwin, StructureNotifyMask,
		UnregTarget, (ClientData)targetPtr);
	}
	if ((argc >= 4) && (strcmp(argv[3], "handler") == 0)) {
	    /*
	     *  HANDLE:  drag&drop target <pathName> handler
	     *           drag&drop target <pathName> handler ?<data> <cmd> <arg>...?
	     */
	    if (argc == 4) {
		Blt_ListItem;

		for(item = Blt_ListFirstItem(targetPtr->listPtr); item != NULL;
		    item = Blt_ListNextItem(item)) {
		    Tcl_AppendElement(interp, Blt_GetListKey(item));
		}
		return TCL_OK;
	    } else if (argc >= 6) {
		char *cmd;
		Blt_ListItem item;
		
		/*
		 *  Process handler definition
		 */
		if (strstr(argv[4], " ")) {
		    Tcl_AppendResult(interp,
			"bad source handler name \"",
			argv[4], "\": should not contain spaces",
			(char *)NULL);
		    return TCL_ERROR;
		}
		item = Blt_FindListItem(targetPtr->listPtr, dtName);
		cmd = Tcl_Concat(argc - 5, argv + 5);
		if (item == NULL) {
		    item = Blt_ListAppend(targetPtr->listPtr, dtName, 
				  (ClientData)cmd);
		} else {
		    oldCmd = (char *)Blt_ListGetValue(item);
		    if (oldCmd != NULL) {
			free ((char *)oldCmd);
		    }
		    Blt_ListSetValue(item, (ClientData)cmd);
		}
		AddPropertyToTarget(interp, targetPtr);
		return TCL_OK;
	    }
	    Tcl_AppendResult(interp,
		"wrong # args: should be \"", argv[0], " ", argv[1],
		" ", argv[2], " ", argv[3], " data command ?arg arg...?",
		(char *)NULL);
	    return TCL_ERROR;
	} else if ((argc >= 4) && (strcmp(argv[3], "handle") == 0)) {
	    /*
	     *  HANDLE:  drag&drop target <pathName> handle <data> ?<value>?
	     */
	    Tcl_DString cmdStr;
	    int result;

	    if (argc < 5 || argc > 6) {
		Tcl_AppendResult(interp,
		    "wrong # args: should be \"", argv[0], " ", argv[1],
		    " ", argv[2], " handle data ?value?",
		    (char *)NULL);
		return TCL_ERROR;
	    }
	    item = Blt_ListFindItem(targetPtr->listPtr, argv[4]);
	    if (item == NULL) {
		Tcl_AppendResult(interp, "target cannot handle datatype: ",
		    argv[4], (char *)NULL);
		return TCL_ERROR;	/* no handler found */
	    }
	    cmd = (char *)Blt_ListGetValue(item);
	    if (cmd != NULL) {
		subst[0].letter = 'W';
		subst[0].value = Tk_PathName(targetPtr->tkwin);
		subst[1].letter = 'v';
		if (argc > 5) {
		    subst[1].value = argv[5];
		} else {
		    subst[1].value = "";
		}
		Tcl_DStringInit(&cmdStr);
		result = Tcl_Eval(interp, 
				  ExpandPercents(cmd, subst, 2, &cmdStr));
		Tcl_DStringFree(&cmdStr);
		return result;
	    }
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, "usage: ", argv[0], " target ", argv[2],
		" handler ?data command arg arg...?\n   or: ",
		argv[0], " target ", argv[2], " handle <data>",
		(char *)NULL);
	    return TCL_ERROR;
	}
    } else if ((c == 't') && (length >= 2) &&
	(strncmp(argv[1], "token", length) == 0)) {
	/*
	 *  HANDLE:  drag&drop token <pathName>
	 */
	srcPtr = FindSource(argv[2]);
	if (srcPtr == NULL) {
	    Tcl_AppendResult(interp, "window \"", argv[2],
		"\" has not been initialized as a drag&drop source",
		(char *)NULL);
	    return TCL_ERROR;
	}
	if (argc > 3) {
	    if (ConfigureToken(interp, srcPtr, argc - 3, argv + 3) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
	Tcl_SetResult(interp, Tk_PathName(srcPtr->token.tkwin), TCL_STATIC);
	return TCL_OK;
    } else if ((c == 'd') && strncmp(argv[1], "drag", length) == 0) {
	/*
	 *  HANDLE:  drag&drop drag <path> <x> <y>
	 */
	if (argc < 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " drag pathname x y\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
	srcPtr = FindSource(argv[2]);
	if (srcPtr == NULL) {
	    Tcl_AppendResult(interp, "not a drag&drop source: ", argv[2],
		(char *)NULL);
	    return TCL_ERROR;
	}
	if ((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	    (Tcl_GetInt(interp, argv[4], &y) != TCL_OK))
	    return TCL_ERROR;

	tokenPtr = &(srcPtr->token);
	locX = x;	/* save drag&drop location */
	locY = y;
	tokenPtr->lastX = x;
	tokenPtr->lastY = y;

	/*
         *  If HideToken() is pending, then do it now!
         */
	if (tokenPtr->timerToken != 0) {
	    Tk_DeleteTimerHandler(tokenPtr->timerToken);
	    HideToken(tokenPtr);
	}
	/*
         *  If pkgCmd is in progress, then ignore subsequent calls
         *  until it completes.  Only perform drag if pkgCmd
         *  completed successfully and token window is mapped.
         */
	if ((!Tk_IsMapped(tokenPtr->tkwin)) && (!srcPtr->pkgCmdInProg)) {
	    Tcl_DString cmdStr;

	    /*
             *  No list of send handlers?  Then source is disabled.
             *  Abort drag quietly.
             */
	    if (srcPtr->send == NULL)
		return TCL_OK;

	    /*
             *  No token command?  Then cannot build token.
             *  Signal error.
             */
	    if (srcPtr->pkgCmd == NULL) {
		Tcl_AppendResult(interp, "missing -packagecmd: ", argv[2],
		    (char *)NULL);
		return TCL_ERROR;
	    }
	    /*
             *  Execute token command to initialize token window.
             */
	    srcPtr->pkgCmdInProg = ~0;
	    subst[0].letter = 'W';
	    subst[0].value = Tk_PathName(srcPtr->tkwin);
	    subst[1].letter = 't';
	    subst[1].value = Tk_PathName(tokenPtr->tkwin);

	    Tcl_DStringInit(&cmdStr);
	    status = Tcl_Eval(srcPtr->interp,
		ExpandPercents(srcPtr->pkgCmd, subst, 2, &cmdStr));
	    Tcl_DStringFree(&cmdStr);

	    srcPtr->pkgCmdInProg = 0;

	    /*
             *  Null string from the package command?
             *  Then quietly abort the drag&drop operation.
             */
	    if (*Tcl_GetStringResult(interp) == '\0') {
		return TCL_OK;
	    }
	    /*
             *  Save result of token command for send command.
             */
	    if (srcPtr->pkgCmdResult != NULL) {
		free(srcPtr->pkgCmdResult);
	    }
	    srcPtr->pkgCmdResult = strdup(Tcl_GetStringResult(interp));

	    /*
             *  Token building failed?  If an error handler is defined,
             *  then signal the error.  Otherwise, abort quietly.
             */
	    if (status != TCL_OK) {
		if (errorCmd && *errorCmd) {
		    return Tcl_VarEval(srcPtr->interp, errorCmd, " {", 
			Tcl_GetStringResult(srcPtr->interp), "}",
			(char *)NULL);
		} else
		    return TCL_OK;
	    }
	    /*
             *  Install token cursor...
             */
	    if (tokenPtr->cursor != None) {
		status = Tcl_VarEval(srcPtr->interp,
		    Tk_PathName(srcPtr->tkwin), " configure -cursor",
		    (char *)NULL);

		if (status == TCL_OK) {
		    char *cname = Tcl_GetStringResult(interp);
		    while (*cname != '\0') {
			cname++;
		    }
		    while ((cname > Tcl_GetStringResult(interp)) && 
			(*(cname - 1) != ' ')) {
			cname--;
		    }
		    if (srcPtr->normalCursor != None) {
			Tk_FreeCursor(srcPtr->display, srcPtr->normalCursor);
			srcPtr->normalCursor = None;
		    }
		    if (strcmp(cname, "{}") != 0) {
			srcPtr->normalCursor = Tk_GetCursor(interp,
			    srcPtr->tkwin, Tk_GetUid(cname));
		    }
		}
		Tk_DefineCursor(srcPtr->tkwin, tokenPtr->cursor);
	    }
	    /*
             *  Get ready to drag token window...
             *  1) Cache info for all windows on root
             *  2) Map token window to begin drag operation
             */
	    if (srcPtr->allwins) {
		TreeRelease(srcPtr->allwins);
	    }
	    srcPtr->allwins = GetTreeInfo(srcPtr);

	    numActive++;	/* one more drag&drop window active */

	    if (!Tk_IsMapped(tokenPtr->tkwin)) {
		Tk_MapWindow(tokenPtr->tkwin);
	    }
	    if (Tk_WindowId(tokenPtr->tkwin) == None) {
		Tk_MakeWindowExist(tokenPtr->tkwin);
	    }
	    RaiseToken(srcPtr, tokenPtr);
	}
	/*
         *  Arrange to update status of token window...
         */
	Tk_CancelIdleCall(UpdateToken, (ClientData)srcPtr);
	Tk_DoWhenIdle(UpdateToken, (ClientData)srcPtr);

	/*
         *  Move the token window to the current drag point...
         */
	MoveToken(srcPtr, tokenPtr);
    } else if ((c == 'd') && strncmp(argv[1], "drop", length) == 0) {
	/*
	 *  HANDLE:  drag&drop drop <path> <x> <y>
	 */
	if (argc < 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " drop pathname x y\"", (char *)NULL);
	    return TCL_ERROR;
	}
	srcPtr = FindSource(argv[2]);
	if (srcPtr == NULL) {
	    Tcl_AppendResult(interp, "not a drag&drop source: ", argv[2],
		(char *)NULL);
	    return TCL_ERROR;
	}
	if ((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	    (Tcl_GetInt(interp, argv[4], &y) != TCL_OK))
	    return TCL_ERROR;

	locX = x;	/* save drag&drop location */
	locY = y;
	srcPtr->token.lastX = x;
	srcPtr->token.lastY = y;

	/*
         *  Put the cursor back to its usual state.
         */
	if (srcPtr->normalCursor == None) {
	    Tk_UndefineCursor(srcPtr->tkwin);
	} else {
	    Tk_DefineCursor(srcPtr->tkwin, srcPtr->normalCursor);
	}
	Tk_CancelIdleCall(UpdateToken, (ClientData)srcPtr);

	/*
         *  Make sure that token window was not dropped before it
         *  was either mapped or packed with info.
         */
	if (Tk_IsMapped(srcPtr->token.tkwin) && !srcPtr->pkgCmdInProg) {
	    UpdateToken((ClientData)srcPtr);

	    if (srcPtr->send) {
		if (srcPtr->token.overTarget) {
		    DragDropSend(srcPtr);
		} else {
		    HideToken(&(srcPtr->token));
		}
	    }
	    numActive--;	/* one fewer active token window */
	}
    } else if ((c == 'e') && strncmp(argv[1], "errors", length) == 0) {
	/*
	 *  HANDLE:  drag&drop errors ?<proc>?
	 */
	if (argc == 3) {
	    if (errorCmd) {
		free((char *)errorCmd);
	    }
	    errorCmd = strdup(argv[2]);
	} else if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " errors ?proc?\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, errorCmd, TCL_VOLATILE);
	return TCL_OK;
    } else if ((c == 'a') && strncmp(argv[1], "active", length) == 0) {
	/*
	 *  HANDLE:  drag&drop active
	 */
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " active\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, (numActive > 0) ? "1" : "0", TCL_STATIC);
	return TCL_OK;
    } else if ((c == 'l') && strncmp(argv[1], "location", length) == 0) {
	/*
	 *  HANDLE:  drag&drop location ?<x> <y>?
	 */
	if (argc == 2) {
	    sprintf(buffer, "%d %d", locX, locY);
	    Tcl_SetResult(interp, buffer, TCL_VOLATILE);
	    return TCL_OK;
	} else if ((argc == 4) && (Tcl_GetInt(interp, argv[2], &x) == TCL_OK) &&
	    (Tcl_GetInt(interp, argv[3], &y) == TCL_OK)) {
	    locX = x;
	    locY = y;
	    sprintf(buffer, "%d %d", locX, locY);
	    Tcl_SetResult(interp, buffer, TCL_VOLATILE);
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " location ?x y?\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	/*
	 *  Report improper command arguments
	 */
	Tcl_AppendResult(interp, "bad operation \"", argv[1],
	    "\": must be active, drag, drop, errors, location, ",
	    "source, target or token",
	    (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *
 *  CreateSource --
 *
 *	Looks for a Source record in the hash table for drag&drop source
 *	widgets.  Creates a new record if the widget name is not already
 *	registered.  Returns a pointer to the desired record.
 *
 * ------------------------------------------------------------------------
 */
static Source *
CreateSource(interp, pathName, newPtr)
    Tcl_Interp *interp;
    char *pathName;		/* widget pathname for desired record */
    int *newPtr;		/* returns non-zero => new record created */
{
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_CreateHashEntry(&sourceTable, pathName, newPtr);
    if (*newPtr) {
	Source *srcPtr;
	Tk_Window tkwin;

	tkwin = Tk_NameToWindow(interp, pathName, Tk_MainWindow(interp));
	if (tkwin == NULL) {
	    Tcl_AppendResult(interp, "window does not exist: ", pathName,
		(char *)NULL);
	    return NULL;
	}
	srcPtr = (Source *)calloc(1, sizeof(Source));
	assert(srcPtr);
	srcPtr->tkwin = tkwin;
	srcPtr->display = Tk_Display(tkwin);
	srcPtr->ddAtom = XInternAtom(srcPtr->display, propName, False);
	srcPtr->interp = interp;
	srcPtr->token.anchor = TK_ANCHOR_SE;	/* Used to be "center". */
	srcPtr->token.relief = TK_RELIEF_RAISED;
	srcPtr->token.activeRelief = TK_RELIEF_SUNKEN;
	srcPtr->token.borderWidth = srcPtr->token.activeBorderWidth = 3;
	if (ConfigureSource(interp, srcPtr, 0, (char **)NULL, 0) != TCL_OK) {
	    DestroySource(srcPtr);
	    return NULL;
	}
	Tcl_SetHashValue(hPtr, (ClientData)srcPtr);
    }
    return (Source *) Tcl_GetHashValue(hPtr);
}

/*
 * ------------------------------------------------------------------------
 *  DestroySource --
 *
 *  Looks for a Source record in the hash table for drag&drop source
 *  widgets.  Destroys the record if found.
 * ------------------------------------------------------------------------
 */
static void
DestroySource(srcPtr)
    Source *srcPtr;
{
    SourceHndl *dsHndl, *next;
    Tcl_HashEntry *hPtr;

    Tk_CancelIdleCall(UpdateToken, (ClientData)srcPtr);
    if (srcPtr->token.timerToken) {
	Tk_DeleteTimerHandler(srcPtr->token.timerToken);
    }
    Tk_FreeOptions(configSpecs, (char *)srcPtr, srcPtr->display, 0);
    
    if (srcPtr->token.rejectFgGC != NULL) {
	Tk_FreeGC(srcPtr->display, srcPtr->token.rejectFgGC);
    }
    if (srcPtr->token.rejectBgGC != NULL) {
	Tk_FreeGC(srcPtr->display, srcPtr->token.rejectBgGC);
    }
    if (srcPtr->pkgCmdResult) {
	free(srcPtr->pkgCmdResult);
    }
    if (srcPtr->allwins) {
	TreeRelease(srcPtr->allwins);
    }
    if (srcPtr->normalCursor != None) {
	Tk_FreeCursor(srcPtr->display, srcPtr->normalCursor);
    }
    for(item = Blt_ListFirstItem(srcPtr->listPtr); item != NULL;
	item = Blt_ListNextItem(item)) {
	cmd = (char *)Blt_ListGetValue(item);
	if (cmd != NULL) {
	    free((char *)cmd);
	}
    }
    Blt_ResetList(srcPtr->listPtr);
    free((char *)srcPtr);
}

/*
 * ------------------------------------------------------------------------
 *
 *  FindSource --
 *
 *	Looks for a Source record in the hash table for drag&drop source
 *	widgets.  Returns a pointer to the desired record.
 *
 * ------------------------------------------------------------------------
 */
static Source *
FindSource(pathname)
    char *pathname;		/* widget pathname for desired record */
{
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_FindHashEntry(&sourceTable, pathname);
    if (hPtr == NULL) {
	return NULL;
    }
    return (Source *) Tcl_GetHashValue(hPtr);
}

/*
 * ------------------------------------------------------------------------
 *
 *  ConfigureSource --
 *
 *	Called to process an (argc,argv) list to configure (or
 *	reconfigure) a drag&drop source widget.
 *
 * ------------------------------------------------------------------------
 */
static int
ConfigureSource(interp, srcPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* current interpreter */
    register Source *srcPtr;/* drag&drop source widget record */
    int argc;			/* number of arguments */
    char **argv;		/* argument strings */
    int flags;			/* flags controlling interpretation */
{
    unsigned long gcMask;
    XGCValues gcValues;
    GC newGC;

    /*
     *  Handle the bulk of the options...
     */
    if (Tk_ConfigureWidget(interp, srcPtr->tkwin, configSpecs, argc, argv,
	    (char *)srcPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     *  Check the button binding for valid range (0 or 1-5)
     */
    if ((srcPtr->button < 0) || (srcPtr->button > 5)) {
	Tcl_SetResult(interp, "button number must be 1-5, or 0 for no bindings",
	    TCL_STATIC);
	return TCL_ERROR;
    }
    /*
     *  Set up the rejection foreground GC for the token window...
     */
    gcValues.foreground = srcPtr->token.rejectFg->pixel;
    gcValues.subwindow_mode = IncludeInferiors;
    gcValues.graphics_exposures = False;
    gcMask = GCForeground | GCSubwindowMode | GCGraphicsExposures;

    if (srcPtr->token.rejectStipple != None) {
	gcValues.stipple = srcPtr->token.rejectStipple;
	gcValues.fill_style = FillStippled;
	gcMask |= GCForeground | GCStipple | GCFillStyle;
    }
    newGC = Tk_GetGC(srcPtr->tkwin, gcMask, &gcValues);

    if (srcPtr->token.rejectFgGC != NULL) {
	Tk_FreeGC(srcPtr->display, srcPtr->token.rejectFgGC);
    }
    srcPtr->token.rejectFgGC = newGC;

    /*
     *  Set up the rejection background GC for the token window...
     */
    gcValues.foreground = srcPtr->token.rejectBg->pixel;
    gcValues.subwindow_mode = IncludeInferiors;
    gcValues.graphics_exposures = False;
    gcMask = GCForeground | GCSubwindowMode | GCGraphicsExposures;

    newGC = Tk_GetGC(srcPtr->tkwin, gcMask, &gcValues);

    if (srcPtr->token.rejectBgGC != NULL) {
	Tk_FreeGC(srcPtr->display, srcPtr->token.rejectBgGC);
    }
    srcPtr->token.rejectBgGC = newGC;

    /*
     *  Reset the border width in case it has changed...
     */
    if (srcPtr->token.tkwin) {
	Tk_SetInternalBorder(srcPtr->token.tkwin, 
		     srcPtr->token.borderWidth + 2);
    }
    return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *  UnregSource()
 *
 *  Invoked by Tk_HandleEvent whenever a DestroyNotify event is received
 *  on a registered drag&drop source widget.
 * ------------------------------------------------------------------------
 */
static void
UnregSource(clientData, eventPtr)
    ClientData clientData;	/* drag&drop registration list */
    XEvent *eventPtr;		/* event description */
{
    Source *srcPtr = (Source *)clientData;

    if (eventPtr->type == DestroyNotify) {
	DestroySource(srcPtr);
    }
}

#ifdef notdef
/*
 * ------------------------------------------------------------------------
 *
 *  FindTarget --
 *
 *	Looks for a Target record in the hash table for drag&drop
 *	target widgets.  Creates a new record if the widget name is
 *	not already registered.  Returns a pointer to the desired
 *	record.
 *
 * ------------------------------------------------------------------------
 */
static Target *
FindTarget(pathname)
    char *pathname;		/* widget pathname for desired record */
{
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_FindHashEntry(&targetTable, pathname);
    if (hPtr == NULL) {
	return NULL;
    }
    return (Target *) Tcl_GetHashValue(hPtr);
}

#endif
/*
 * ------------------------------------------------------------------------
 *
 *  CreateTarget --
 *
 *	Looks for a Target record in the hash table for drag&drop
 *	target widgets.  Creates a new record if the widget name is
 *	not already registered.  Returns a pointer to the desired
 *	record.
 *
 * ------------------------------------------------------------------------
 */
static Target *
CreateTarget(pathname, newPtr)
    char *pathname;		/* widget pathname for desired record */
    int *newPtr;		/* returns non-zero => new record created */
{
    Target *targetPtr;
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_CreateHashEntry(&targetTable, pathname, newPtr);
    if (*newPtr) {
	/*
         *  Initialize a data structure for the widget...
         */
	targetPtr = (Target *)calloc(1, sizeof(Target));
	Tcl_SetHashValue(hPtr, (ClientData)targetPtr);
    }
    return (Target *) Tcl_GetHashValue(hPtr);
}

/*
 * ------------------------------------------------------------------------
 *  DestroyTarget --
 *
 *  Looks for a Target record in the hash table for drag&drop target
 *  widgets.  Destroys the record if found.
 * ------------------------------------------------------------------------
 */
static void
DestroyTarget(targetPtr)
    Target *targetPtr;
{
    register Blt_ListItem item;
    char *cmd;
    
    for(item = Blt_ListFirstItem(targetPtr->listPtr); item != NULL;
	item = Blt_ListNextItem(item)) {
	cmd = (char *)Blt_ListGetValue(item);
	if (cmd != NULL) {
	    free((char *)cmd);
	}
    }
    Blt_ResetList(targetPtr->listPtr);
    Blt_DestroyList(targetPtr->listPtr);
    free((char *)targetPtr);
}


/*
 * ------------------------------------------------------------------------
 *  UnregTarget()
 *
 *  Invoked by Tk_HandleEvent whenever a DestroyNotify event is received
 *  on a registered drag&drop target widget.
 * ------------------------------------------------------------------------
 */

#ifdef	WIN32
/*ARGSUSED*/
static BOOL CALLBACK
EnumDeletePropProc(HWND hWnd, LPCTSTR string, HANDLE hData) 
{
    if (strcmp(bltPropName, string) == 0)) {
	RemoveProp(hWnd, string);
    }
    return TRUE;
}
#endif /* WIN32 */

static void
UnregTarget(clientData, eventPtr)
    ClientData clientData;	/* drag&drop registration list */
    XEvent *eventPtr;		/* event description */
{
    Target *targetPtr = (Target *)clientData;

    if (eventPtr->type == DestroyNotify) {
#ifdef	WIN32
	HWND hWnd;

	hWnd = Tk_GetHWND(targetPtr->tkwin);
	EnumProps(hWnd, EnumDeletePropProc);
#endif
	DestroyTarget(targetPtr);
    }
}

/*
 * ------------------------------------------------------------------------
 *  DragDropSend()
 *
 *  Invoked after a drop operation to send data to the drop application.
 * ------------------------------------------------------------------------
 */
static void
DragDropSend(srcPtr)
    register Source *srcPtr; /* drag&drop source record */
{
    char *datatype = NULL;
    int status;
    char *sendcmd;
    TreeNode *target;
    Tcl_DString buffer;
    DD_PercentSubst subst[3];

    treePtr = FindTargetWin(srcPtr, srcPtr->token.lastX, srcPtr->token.lastY);
    if (treePtr == NULL) {
	return;
    }
    Tcl_DStringInit(&buffer);
    Tcl_DStringAppend(&buffer, srcPtr->pkgCmdResult, -1);
    /*
     *  See if current position is over drop point.
     */
    status = Tcl_VarEval(srcPtr->interp, "send {", treePtr->interpName, "} 
	", dragDropCmd, " location ", Blt_Int(srcPtr->token.lastX), " ", 
	Blt_Int(srcPtr->token.lastY), (char *)NULL);
    if (status != TCL_OK) {
	goto reject;
    }
    datatype = DragDropSendHndlr(srcPtr, treePtr->interpName, 
	treePtr->targetName);
    if (datatype != NULL) {
	sendcmd = FindSourceHandler(srcPtr, datatype);
	if (sendcmd && *sendcmd) {
	    Tcl_DString cmdStr;

	    subst[0].letter = 'i';
	    subst[0].value = treePtr->interpName;
	    subst[1].letter = 'w';
	    subst[1].value = treePtr->targetName;
	    subst[2].letter = 'v';
	    subst[2].value = srcPtr->pkgCmdResult;
	    
	    Tcl_DStringInit(&cmdStr);
	    status = Tcl_Eval(srcPtr->interp, 
		ExpandPercents(sendcmd, subst, 3, &cmdStr));
	    Tcl_DStringFree(&cmdStr);
	    
	    Tcl_DStringSetLength(&buffer, 0);
	    Tcl_DStringAppend(&buffer, 
			      Tcl_GetStringResult(srcPtr->interp), -1);
	}
    } else {
	Tcl_AppendResult(srcPtr->interp, "target \"", 
		 treePtr->targetName, 
		"\" does not recognize handlers for source \"",
		Tk_PathName(srcPtr->tkwin), "\"", (char *)NULL);
	goto reject;
    }
    /*
     *  If the "send" command was successful, then tell the
     *  target application to handle the data.
     */
    status = Tcl_VarEval(srcptr->interp, "send {", treePtr->interpName, "} ",
		dragDropCmd, " target ", Tk_PathName(treePtr->tkwin), 
		" handle ", datatype, " {", Tcl_DStringValue(&buffer), "}",
		(char *)NULL);
    if (status != TCL_OK) {
	goto reject;
    }
    HideToken(&(srcPtr->token));
    Tcl_DStringFree(&buffer);
    return;
 reject:
    /* 
     * Give failure information to user.  If an error occurred and an
     * error proc is defined, then use it to handle the error.  
     */
    RejectToken(&(srcPtr->token));
    Tcl_DStringFree(&buffer);
    if (errorCmd != NULL) {
	Tcl_VarEval(srcPtr->interp, errorCmd, " {", 
		Tcl_GetStringResult(srcPtr->interp), "}", (char *)NULL);
    }
}


/*
 * ------------------------------------------------------------------------
 *  DragDropSendHndlr()
 *
 *  Queries the drag&drop target under the specified interpreter for a
 *  handler that is compatible with one of the handlers defined for the
 *  source.  Returns a pointer to the appropriate data type, or
 *  NULL if none is found.
 * ------------------------------------------------------------------------
 */
static char *
DragDropSendHndlr(srcPtr, interpName, ddName)
    register Source *srcPtr;/* drag&drop source record */
    char *interpName;		/* interpreter containing drag&drop target */
    char *ddName;		/* drag&drop target pathname */
{
    char *retn = NULL;		/* no handler found yet */
    Tcl_Interp *interp = srcPtr->interp;

    int hndlc, hi, ei;
    char **hndlv, *hlist;
    SourceHndl *dsHndl;
    char buffer[1024];

    /*
     *  Query the drag&drop target for its list of known handlers.
     */
    Tcl_ResetResult(interp);	/* for Tcl_AppendResult() below */
    if (Tcl_VarEval(interp, "send {", interpName, "} ", dragDropCmd, 
		    " target {", ddName, "} handler",
	    (char *)NULL) != TCL_OK) {
	return NULL;
    }
    hlist = strdup(Tcl_GetStringResult(interp));
    if (Tcl_SplitList(interp, hlist, &elemArr, &numElem) == TCL_OK) {
	/*
         *  If the list of send handlers is specified as "all", then
         *  search through the handlers in order.
         */
	if (strcmp(srcPtr->send, "all") == 0) {
	    for (i = 0; i < hndlc; i++) {
		item = Blt_ListFindItem(srcPtr->listPtr, elemArr[i]);
		if (item != NULL) {
		    free((char *)elemArr);
		    return Blt_ListGetValue(item);
		}
	    }
	    /*
	     *  Otherwise, search through the specified send handlers.
	     */
	} else {
	    int elemc;
	    char **elemv;

	    if (Tcl_SplitList(interp, srcPtr->send, &elemc, &elemv) == TCL_OK) {
		for (ei = 0; (ei < elemc) && !retn; ei++) {
		    for (hi = 0; (hi < hndlc) && !retn; hi++) {
			if (strcmp(elemv[ei], hndlv[hi]) == 0) {
			    item = Blt_ListFindItem(srcPtr->listPtr, elemv[ei]);
			    if (item != NULL) {
				break;
			    }
			    cmd = Blt_ListGetValue(item);
			    if (cmd != NULL) {
				retn = dsHndl->dataType;
			    } else {
				sprintf(buffer, "unknown handler \"%.50s\" requested for drag&drop source \"%.200s\"", elemv[ei], Tk_PathName(srcPtr->tkwin));
				Tcl_ResetResult(interp);
				Tcl_AddErrorInfo(interp, buffer);
				Tk_BackgroundError(interp);
			    }
			}
		    }
		}
		free((char *)elemv);
	    } else {
		sprintf(buffer, "drag&drop source has invalid -send: %.200s",
		    srcPtr->send);
		Tcl_ResetResult(interp);
		Tcl_AddErrorInfo(interp, buffer);
		Tk_BackgroundError(interp);
	    }
	}
	free((char *)hndlv);
    }
    free(hlist);

    return retn;
}

/*
 * ------------------------------------------------------------------------
 *  GetTreeInfo()
 *
 *  Invoked at the start of a "drag" operation to capture the positions
 *  of all windows on the current root.  Queries the entire window
 *  hierarchy and determines the placement of each window.  Queries
 *  "DragDropInfo" property info where appropriate.  This information
 *  is used during the drag operation to determine when the drag&drop
 *  token is over a valid drag&drop target.
 *
 *  Returns the record for the root window, which contains records for
 *  all other windows as children.
 * ------------------------------------------------------------------------
 */
static TreeNode *
GetTreeInfo(srcPtr)
    Source *srcPtr;		/* drag&drop source window */
{
    TreeNode *treePtr;

    treePtr = (TreeNode *)calloc(1, sizeof(TreeNode));
    assert(treePtr);
    treePtr->window = DefaultRootWindow(srcPtr->display);
    QueryTree(treePtr, srcPtr);
    return treePtr;
}

/*
 * ------------------------------------------------------------------------
 *  FindTargetWin()
 *
 *  Checks to see if a compatible drag&drop target exists at the given
 *  position.  A target is "compatible" if it is a drag&drop window,
 *  and if it has a handler that is compatible with the current source
 *  window.
 *
 *  Returns a pointer to a structure describing the target, or NULL
 *  if no compatible target is found.
 * ------------------------------------------------------------------------
 */
static TreeNode *
FindTargetWin(srcPtr, x, y)
    Source *srcPtr;		/* drag&drop source window */ 
    int x, y;			/* current drag&drop location
				 * (in virtual coords) */
{
    int vx, vy, vw, vh;
    register char *type;

    TreeNode *treePtr, *childPtr;
    DD_Stack stack;
    SourceHndl *shandl;

    /*
     *  If window representations have not yet been built, then
     *  abort this call.  This probably means that the token is being
     *  moved before it has been properly built.
     */
    if (!srcPtr->allwins)
	return NULL;

    /*
     *  Adjust current location for virtual root windows.
     */
    Tk_GetVRootGeometry(srcPtr->tkwin, &vx, &vy, &vw, &vh);
    x += vx;
    y += vy;

    /*
     *  Build a stack of all windows containing the given point,
     *  in order from least to most specific.
     */
    StackInit(&stack);

#ifdef	__WIN32__
    StackPush((ClientData)treePtr, &stack);

    while ((treePtr = (TreeNode *)StackPop(&stack))) {
	if (!treePtr->initialized) {
	    treeInit(treePtr, dsPtr);
	}
	if ((x >= treePtr->x0) && (x <= treePtr->x1) && 
	    (y >= treePtr->y0) && (y <= treePtr->y1)
	    && treePtr->ddprop && (treePtr->handlerListPtr != NULL)) {
	    for(ddItem = Blt_ListFirstItem(treePtr->handleListPtr); ddItem != NULL;
		ddItem = Blt_ListNextItem(item)) {
		type = Blt_ListGetKey(item);
		srcItem = Blt_ListFindItem(dsPtr->listPtr, type);
		if (item != NULL) {
		    goto L_done;
		} 
	    }
	}
	for (item = Blt_ListFirstItem(treePtr->childListPtr); item != NULL; 
	     item = Blt_ListNextItem(item)) {
	    childPtr = (TreeNode *)Blt_ListGetValue(item);
	    StackPush((ClientData)childPtr, &stack);
	}
    }
L_done:

#else /* !__WIN32__ */

    if ((x >= treePtr->x0) && (x <= treePtr->x1) && 
	(y >= treePtr->y0) && (y <= treePtr->y1)) {
	StackPush((ClientData)treePtr, &stack);
    }

    while (treePtr) {
	for (item = Blt_ListFirstItem(treePtr->childListPtr); item != NULL; 
	     item = Blt_ListNextItem(item)) {
	    childPtr = Blt_ListNextItem(item);
	    if (!childPtr->initialized) {
		QueryTree(childPtr, dsPtr);
	    }
	    if ((x >= childPtr->x0) && (x <= childPtr->x1) && 
		(y >= childPtr->y0) && (y <= childPtr->y1)) {
		StackPush((ClientData)childPtr, &stack);
		break;
	    }
	}
	treePtr = childPtr;  /* continue search */
    }
    /*
     *  Pop windows from the stack until one containing a
     *  "DragDropInfo" property is found.  See if the handlers
     *  listed in this property are compatible with the
     *  given source.
     */
    for(;;) {
	treePtr = (TreeNode *)StackPop(&stack);
	if ((treePtr == NULL) || (treePtr->ddProp != NULL)) {
	    break;
	}
    }
    if ((treePtr != NULL) && (treePtr->ddListPtr != NULL)) {
	for(ddItem = Blt_ListFirstItem(treePtr->ddListPtr); ddItem != NULL;
	    ddItem = Blt_ListNextItem(item)) {
	    type = Blt_ListGetValue(item);
	    item = Blt_ListFindItem(dsPtr->listPtr, Blttype);
	    if (item != NULL) {
		break;
	    }
	}
	if (item == NULL) {
	    treePtr = NULL;  /* no handler match; return NULL */
	}
    }
#endif /* !__WIN32__ */

    treePtr = srcPtr->allwins;
    if ((x >= treePtr->x0) && (x <= treePtr->x1) &&
	(y >= treePtr->y0) && (y <= treePtr->y1)) {
	StackPush((ClientData)treePtr, &stack);
    }

    while (treePtr) {
	for (item = Blt_ListFirstItem(treePtr->childListPtr); item != NULL;
	     item = Blt_ListNextItem(item)) {
	    childPtr = (TreeNode *)Blt_ListGetValue(item);
	    if (!childPtr->initialized) {
		QueryTree(childPtr, srcPtr);
	    }
	    if ((x >= childPtr->x0) && (x <= childPtr->x1) &&
		(y >= childPtr->y0) && (y <= childPtr->y1)) {
		StackPush((ClientData)childPtr, &stack);
		break;
	    }
	}
	treePtr = childPtr;		/* continue search */
    }
    /*
     *  Pop windows from the stack until one containing a
     *  "DragDropInfo" property is found.  See if the handlers
     *  listed in this property are compatible with the
     *  given source.
     */
    for(;;) {
	treePtr = (TreeNode *)StackPop(&stack);
	if ((treePtr == NULL) || (treePtr->ddProp != NULL)) {
	    break;
	}
    }
    if ((treePtr != NULL) && (treePtr->ddListPtr != NULL)) {
	type = treePtr->ddhandlers;
	while (*type != '\0') {
	    for (shandl = srcPtr->handlers; shandl; shandl = shandl->next)
		if (strcmp(shandl->dataType, type) == 0)
		    break;

	    if (shandl)		/* found a match? */
		break;		/* then stop searching */
	    else {		/* otherwise, move to next handler type */
		while (*type++ != '\0');
	    }
	}
	if (*type == '\0')	/* no handler match? */
	    treePtr = NULL;		/* then return NULL */
    }
    StackDelete(&stack);

    return treePtr;
}

/*
 * ------------------------------------------------------------------------
 *  TreeRelease()
 * ------------------------------------------------------------------------
 */
static void
TreeRelease(treePtr)
    TreeNode *treePtr;		/* window rep to be freed */
{
    TreeNode *childPtr;
    Blt_ListItem item;

    for (item = Blt_ListFirstItem(treePtr->childListPtr); item != NULL;
	 item = Blt_ListNextItem(item)) {
	childPtr = (TreeNode *)Blt_ListGetValue(item);
	TreeRelease(childPtr);
    }
    Blt_DestroyList(treePtr->childListPtr);
    treePtr->childListPtr = NULL;
    Blt_ListDeleteItem(treePtr->item);
}

/*
 * ------------------------------------------------------------------------
 *  QueryTree()
 *
 *  Invoked during "drag" operations to dig a little deeper into the
 *  root window hierarchy and cache the resulting information.  If a
 *  point coordinate lies within an uninitialized TreeNode, this
 *  routine is called to query window coordinates and drag&drop info.
 *  If this particular window has any children, more uninitialized
 *  TreeNode structures are allocated.  Further queries will cause
 *  these structures to be initialized in turn.
 * ------------------------------------------------------------------------
 */
static void
QueryTree(treePtr, srcPtr)
    TreeNode *treePtr;		/* window rep to be initialized */
    Source *srcPtr;		/* drag&drop source managing win rep */
{
    TreeNode *childPtr;
    Window root, parent, *childArr;
    unsigned int numChildren;
    XWindowAttributes winInfo;
    Window srcWindow, tokenWindow;
    char *propInfo;
    int i, result, actualFormat;
    Atom actualType;
    unsigned long numItems, bytesAfter;

    srcWindow = Tk_WindowId(srcPtr->tkwin);
    tokenWindow = Tk_WindowId(srcPtr->token.tkwin);
    /*
     *  If the self-target flag is set, allow the source window to
     *  drop onto itself.  Do not ignore source window during search.
     */
    if (srcPtr->selfTarget) {
	srcWindow = None;
    }
    if (treePtr->initialized) {
	return;
    }
    /*
     *  Query for the window coordinates.
     */
    if (XGetWindowAttributes(srcPtr->display, treePtr->window, &winInfo) &&
	(winInfo.map_state == IsViewable) && (treePtr->window != tokenWindow) &&
	(treePtr->window != srcWindow)) {
	treePtr->x0 = winInfo.x;
	treePtr->y0 = winInfo.y;
	treePtr->x1 = winInfo.x + winInfo.width;
	treePtr->y1 = winInfo.y + winInfo.height;
	
	if (treePtr->parent) {	/* offset by parent coords */
	    treePtr->x0 += treePtr->parent->x0;
	    treePtr->y0 += treePtr->parent->y0;
	    treePtr->x1 += treePtr->parent->x0;
	    treePtr->y1 += treePtr->parent->y0;
	}
    } else {
	treePtr->x0 = treePtr->y0 = -1;
	treePtr->x1 = treePtr->y1 = -1;
    }
    /*
     *  See if this window has a "DragDropInfo" property.
     */
    result = XGetWindowProperty(srcPtr->display, treePtr->win, srcPtr->ddAtom, 
	0, MAX_PROP_SIZE, False, XA_STRING, &actualType, &actualFormat, 
	&numItems, &bytesAfter, (unsigned char **)&propInfo);
    if ((result != Success) || (actualFormat != 8) || 
	(actualType != XA_STRING)) {
	if (propInfo != NULL) {
	    XFree((char *) propInfo);
	}
	propInfo = NULL;
    }
    treePtr->ddprop = propInfo;
    if (treePtr->ddprop != NULL) {
	char **argv;
	int argc;

	if (Tcl_SplitList(interp, treePtr->ddprop, &argv, &argc) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (treePtr->interpName != NULL) {
	    Blt_FreeUid(treePtr->interpName);
	} 
	treePtr->interpName = Blt_GetUid(argv[0]);
	Blt_DestroyList(treePtr->handlerListPtr);
	treePtr->handlerListPtr = Blt_CreateList(TCL_STRING_KEYS);
	for(i = 1; i < argc; i++) {
	    Blt_ListAppend(treePtr->handlerListPtr, argv[i], 
			   (ClientData)argv[i]);
	}
	free((char *)argv);
    }
    /*
     *  If this window has any children, then create TreeNodes
     *  for them as well.
     */
    if ((XQueryTree(srcPtr->display, treePtr->window, &root, &parent, 
	   &childArr, &numChildren)) && (numChildren > 0)) {
	if (treePtr->childListPtr != NULL) {
	    Blt_DestroyList(treePtr->childListPtr);
	}
	treePtr->childListPtr = Blt_CreateList(TCL_ONE_WORD_KEYS);
	for(i = numChildren - 1; i >= 0; i--) {
	    childPtr = (TreeNode *)calloc(1, sizeof(TreeNode));
	    assert(childPtr);
	    childPtr->window = childArr[i];
	    childPtr->parent = treePtr;
	    Blt_ListAppend(treePtr->childListPtr, (char *)childPtr, 
		   (ClientData)childPtr);
	}
	if (childArr != NULL) {
	    XFree((char *) childArr);	/* done with list of kids */
	}
    }
    treePtr->initialized = TRUE;
}

/*
 * ------------------------------------------------------------------------
 *
 *  AddPropertyToTarget --
 *
 *	Attaches a "DragDropInfo" property to the given target window.
 *	This property allows the drag&drop mechanism to recognize the
 *	window as a valid target, and stores important information
 *	including the interpreter managing the target and the pathname
 *	for the target window.  Usually invoked when the target is
 *	first registered or first exposed (so that the X-window really
 *	exists).
 *
 * ------------------------------------------------------------------------
 */
static void
AddPropertyToTarget(interp, targetPtr) 
    Tcl_Interp *interp;
    Target *targetPtr;	/* drag&drop target window data */
{
    Tk_Window mainWindow;
    Atom dndProperty;
    char *result;
    Tcl_DString dString;
    register Blt_ListItem item;
    unsigned int numBytes;

    if (targetPtr->tkwin == NULL) {
	return;
    }
    mainWindow = Tk_MainInterp(interp);
    Tcl_DStringInit(&dString);
    Tcl_DStringAppend(&dString, Tk_Name(mainWindow), -1);
    Tcl_DStringAppend(&dString, "]", -1);
    Tcl_DStringAppend(&dString, Tk_PathName(targetPtr->tkwin), -1);
    Tcl_DStringAppend(&dString, "]", -1);
    for (item = Blt_ListFirstItem(targetPtr->handlerListPtr); item != NULL; 
	 item = Blt_ListNextItem(item)) {
	Tcl_DStringAppendElement(&dString, Blt_ListGetKey(item));
    }
    dndProperty = XInternAtom(targetPtr->display, propName, False);
    result = Tcl_DStringValue(&dString);
    numBytes = strlen(result) + 1;
    XChangeProperty(targetPtr->display, Tk_WindowId(targetPtr->tkwin), 
	    dndProperty, XA_STRING, 8, PropModeReplace, (unsigned char *)result,
	    numBytes);
    Tcl_DStringFree(&dString);
}


static void
ActivateToken(tokenPtr, active)
    Token *tokenPtr;
    int active;
{
    int relief;
    Tk_3DBorder *borderPtr;
    int borderWidth;

    Tk_Fill3DRectangle(tokenPtr->tkwin, Tk_WindowId(tokenPtr->tkwin),
	tokenPtr->outline, 0, 0, Tk_Width(tokenPtr->tkwin),
	Tk_Height(tokenPtr->tkwin), 0, TK_RELIEF_FLAT);
    if (active) {
	relief = tokenPtr->activeRelief;
	borderPtr = &(tokenPtr->activeBorder);
	borderWidth = tokenPtr->activeBorderWidth;
    } else {
	relief = tokenPtr->relief;
	borderPtr = &(tokenPtr->normalBorder);
	borderWidth = tokenPtr->borderWidth;
    }
    Tk_Fill3DRectangle(tokenPtr->tkwin, Tk_WindowId(tokenPtr->tkwin),
	*borderPtr, 2, 2, Tk_Width(tokenPtr->tkwin) - 4,
	Tk_Height(tokenPtr->tkwin) - 4, borderWidth, relief);
}

/*
 * ------------------------------------------------------------------------
 *
 *  TokenEventProc --
 *
 *	Invoked by the Tk dispatcher to handle widget events.
 *	Manages redraws for the drag&drop token window.
 *
 * ------------------------------------------------------------------------
 */
static void
TokenEventProc(clientData, eventPtr)
    ClientData clientData;	/* data associated with widget */
    XEvent *eventPtr;		/* information about event */
{
    Token *tokenPtr = (Token *)clientData;

    if ((eventPtr->type == Expose) && (eventPtr->xexpose.count == 0)) {
	if (tokenPtr->tkwin != NULL) {
	    ActivateToken(tokenPtr, tokenPtr->overTarget);
	}
    } else if (eventPtr->type == DestroyNotify) {
	tokenPtr->tkwin = NULL;
    }
}

/*
 * ------------------------------------------------------------------------
 *
 *  MoveToken --
 *
 *	Invoked during "drag" operations to move a token window to its
 *	current "drag" coordinate.
 *
 * ------------------------------------------------------------------------
 */
static void
MoveToken(srcPtr, tokenPtr)
    Source *srcPtr;		/* drag&drop source window data */
    Token *tokenPtr;
{
    int x, y;
    int maxWidth, maxHeight;
    int vx, vy, vw, vh;

    /*
     *  Adjust current location for virtual root windows.
     */
    Tk_GetVRootGeometry(srcPtr->tkwin, &vx, &vy, &vw, &vh);
    x = tokenPtr->lastX + vx;
    y = tokenPtr->lastY + vy;

    maxWidth = WidthOfScreen(Tk_Screen(srcPtr->tkwin)) - Tk_Width(tokenPtr->tkwin);
    maxHeight =
	HeightOfScreen(Tk_Screen(srcPtr->tkwin)) - Tk_Height(tokenPtr->tkwin);
    Blt_TranslateAnchor(x, y, Tk_Width(tokenPtr->tkwin),
	Tk_Height(tokenPtr->tkwin), tokenPtr->anchor, &x, &y);

    if (x > maxWidth) {
	x = maxWidth;
    } else if (x < 0) {
	x = 0;
    }
    if (y > maxHeight) {
	y = maxHeight;
    } else if (y < 0) {
	y = 0;
    }
    if ((x != Tk_X(tokenPtr->tkwin)) || (y != Tk_Y(tokenPtr->tkwin))) {
	Tk_MoveToplevelWindow(tokenPtr->tkwin, x, y);
    }
    RaiseToken(srcPtr, tokenPtr);
}

/*
 * ------------------------------------------------------------------------
 *  UpdateToken --
 *
 *	Invoked when the event loop is idle to determine whether or not
 *	the current drag&drop token position is over another drag&drop
 *	target.
 *
 * ------------------------------------------------------------------------
 */
static void
UpdateToken(clientData)
    ClientData clientData;	/* widget data */
{
    register Source *srcPtr = (Source *)clientData;
    Token *tokenPtr = &(srcPtr->token);
    int status, result;
    DD_PercentSubst subst[2];

    status = (FindTargetWin(srcPtr, tokenPtr->lastX, tokenPtr->lastY) != NULL);

    if (tokenPtr->overTarget ^ status) {
	ActivateToken(tokenPtr, status);
	/*
         *  If the source has a site command, then invoke it to
         *  modify the appearance of the token window.  Pass any
         *  errors onto the drag&drop error handler.
         */
	if (srcPtr->sitecmd) {
	    char buffer[200];
	    Tcl_DString cmdStr;

	    sprintf(buffer, "%d", status);
	    subst[0].letter = 's';
	    subst[0].value = buffer;
	    subst[1].letter = 't';
	    subst[1].value = Tk_PathName(tokenPtr->tkwin);

	    Tcl_DStringInit(&cmdStr);
	    result = Tcl_Eval(srcPtr->interp,
		ExpandPercents(srcPtr->sitecmd, subst, 2, &cmdStr));
	    Tcl_DStringFree(&cmdStr);

	    if ((result != TCL_OK) && (errorCmd != NULL) && (*errorCmd)) {

		(void)Tcl_VarEval(srcPtr->interp, errorCmd, " {",
		    Tcl_GetStringResult(srcPtr->interp), "}",
		    (char *)NULL);
	    }
	}
    }
    tokenPtr->overTarget = status;
}

/*
 * ------------------------------------------------------------------------
 *
 *  HideToken --
 *
 *	Unmaps the drag&drop token.  Invoked directly at the end of a
 *	successful communication, or after a delay if the communication
 *	fails (allowing the user to see a graphical picture of failure).
 *
 * ------------------------------------------------------------------------
 */
static void
HideToken(tokenPtr)
    Token *tokenPtr;
{
    if (tokenPtr->tkwin != NULL) {
	Tk_UnmapWindow(tokenPtr->tkwin);
    }
    tokenPtr->timerToken = NULL;
}

/*
 * ------------------------------------------------------------------------
 *
 *  RejectToken --
 *
 *	Draws a rejection mark on the current drag&drop token, and arranges
 *	for the token to be unmapped after a small delay.
 *
 * ------------------------------------------------------------------------
 */
static void
RejectToken(tokenPtr)
    Token *tokenPtr;
{
    int divisor = 6;		/* controls size of rejection symbol */
    int w, h, lineWidth, x, y, margin;

    margin = 2 * tokenPtr->borderWidth;
    w = Tk_Width(tokenPtr->tkwin) - 2 * margin;
    h = Tk_Height(tokenPtr->tkwin) - 2 * margin;
    lineWidth = (w < h) ? w / divisor : h / divisor;
    lineWidth = (lineWidth < 1) ? 1 : lineWidth;

    w = h = lineWidth * (divisor - 1);
    x = (Tk_Width(tokenPtr->tkwin) - w) / 2;
    y = (Tk_Height(tokenPtr->tkwin) - h) / 2;

    /*
     *  Draw the rejection symbol background (\) on the token window...
     */
    XSetLineAttributes(Tk_Display(tokenPtr->tkwin), tokenPtr->rejectBgGC,
	lineWidth + 4, LineSolid, CapButt, JoinBevel);

    XDrawArc(Tk_Display(tokenPtr->tkwin), Tk_WindowId(tokenPtr->tkwin),
	tokenPtr->rejectBgGC, x, y, w, h, 0, 23040);

    XDrawLine(Tk_Display(tokenPtr->tkwin), Tk_WindowId(tokenPtr->tkwin),
	tokenPtr->rejectBgGC, x + lineWidth, y + lineWidth, x + w - lineWidth,
	y + h - lineWidth);

    /*
     *  Draw the rejection symbol foreground (\) on the token window...
     */
    XSetLineAttributes(Tk_Display(tokenPtr->tkwin), tokenPtr->rejectFgGC,
	lineWidth, LineSolid, CapButt, JoinBevel);

    XDrawArc(Tk_Display(tokenPtr->tkwin), Tk_WindowId(tokenPtr->tkwin),
	tokenPtr->rejectFgGC, x, y, w, h, 0, 23040);

    XDrawLine(Tk_Display(tokenPtr->tkwin), Tk_WindowId(tokenPtr->tkwin),
	tokenPtr->rejectFgGC, x + lineWidth, y + lineWidth, x + w - lineWidth,
	y + h - lineWidth);

    /*
     *  Arrange for token window to disappear eventually.
     */
    tokenPtr->timerToken = Tk_CreateTimerHandler(1000, (Tcl_TimerProc *) HideToken,
	(ClientData)tokenPtr);
}

/*
 * ------------------------------------------------------------------------
 *  StackInit()
 *
 *  Initializes a stack structure, allocating a certain amount of memory
 *  for the stack and setting the stack length to zero.
 * ------------------------------------------------------------------------
 */
static void
StackInit(stack)
    DD_Stack *stack;		/* stack to be initialized */
{
    stack->values = stack->space;
    stack->max = sizeof(stack->space) / sizeof(ClientData);
    stack->len = 0;
}

/*
 * ------------------------------------------------------------------------
 *  StackDelete()
 *
 *  Destroys a stack structure, freeing any memory that may have been
 *  allocated to represent it.
 * ------------------------------------------------------------------------
 */
static void
StackDelete(stack)
    DD_Stack *stack;		/* stack to be deleted */
{
    if (stack->values != stack->space)	/* allocated extra memory? */
	free((char *)stack->values);	/* then free it */

    stack->values = NULL;
    stack->len = stack->max = 0;
}

/*
 * ------------------------------------------------------------------------
 *  StackPush()
 *
 *  Pushes a piece of client data onto the top of the given stack.
 * ------------------------------------------------------------------------
 */
static void
StackPush(cdata, stack)
    ClientData cdata;		/* data to be pushed onto stack */
    DD_Stack *stack;		/* stack */
{
    ClientData *newStack;

    if (stack->len + 1 >= stack->max) {
	stack->max = (stack->max == 0) ? 5 : 2 * stack->max;
	newStack = (ClientData *)
	    malloc((unsigned)(stack->max * sizeof(ClientData)));

	if (stack->values) {
	    memcpy((char *)newStack, (char *)stack->values,
		stack->len * sizeof(ClientData));

	    if (stack->values != stack->space)
		free((char *)stack->values);
	}
	stack->values = newStack;
    }
    stack->values[stack->len++] = cdata;
}

/*
 * ------------------------------------------------------------------------
 *  StackPop()
 *
 *  Pops a bit of client data from the top of the given stack.
 * ------------------------------------------------------------------------
 */
static ClientData
StackPop(stack)
    DD_Stack *stack;		/* stack to be manipulated */
{
    if ((stack->values != NULL) && (stack->len > 0)) {
	return (stack->values[--stack->len]);
    }
    return (ClientData) 0;
}

/*
 * ------------------------------------------------------------------------
 *  ExpandPercents(str,subs,nsubs, dStrPtr)
 *
 *  Expands all percent substitutions found in the input "str" that
 *  match specifications in the "subs" list.  Any percent field that
 *  is not found in the "subs" list is left alone.  Returns a string
 *  that remains valid until the next call to this routine.
 * ------------------------------------------------------------------------
 */
static char *
ExpandPercents(string, subs, nsubs, dStrPtr)
    char *string;		/* incoming string */
    DD_PercentSubst *subs;	/* array of known substitutions */
    int nsubs;			/* number of elements in subs array */
    Tcl_DString *dStrPtr;
{
    register char *chunk, *p;
    char letter;
    char percentSign;
    int i;

    /*
     *  Scan through the copy of the input string, look for
     *  the next '%' character, and try to make the substitution.
     *  Continue doing this to the end of the string.
     */
    chunk = p = string;
    while ((p = strchr(p, '%')) != NULL) {

	/* Copy up to the percent sign.  Repair the string afterwards */
	percentSign = *p;
	*p = '\0';
	Tcl_DStringAppend(dStrPtr, chunk, -1);
	*p = percentSign;

	/* Search for a matching substition rule */
	letter = *(p + 1);
	for (i = 0; i < nsubs; i++) {
	    if (subs[i].letter == letter) {
		break;
	    }
	}
	if (i < nsubs) {
	    /* Make the substitution */
	    Tcl_DStringAppend(dStrPtr, subs[i].value, -1);
	} else {
	    /* Copy in the %letter verbatim */
	    static char noSub[3] = "%?";

	    noSub[1] = letter;
	    Tcl_DStringAppend(dStrPtr, noSub, -1);
	}
	p += 2;			/* Skip % + letter */
	if (letter == '\0') {
	    p += 1;		/* Premature % substitution (end of string) */
	}
	chunk = p;
    }
    /* Pick up last chunk if a substition wasn't the last thing in the string */
    if (*chunk != '\0') {
	Tcl_DStringAppend(dStrPtr, chunk, -1);
    }
    return Tcl_DStringValue(dStrPtr);
}


static int
ConfigureToken(interp, srcPtr, argc, argv)
    Tcl_Interp *interp;
    Source *srcPtr;
    int argc;
    char **argv;
{
    Token *tokenPtr;

    tokenPtr = &(srcPtr->token);
    if (Tk_ConfigureWidget(interp, srcPtr->tkwin, tokenConfigSpecs, argc, argv,
	    (char *)tokenPtr, TK_CONFIG_ARGV_ONLY) != TCL_OK) {
	return TCL_ERROR;
    }
    return ConfigureSource(interp, srcPtr, 0, (char **)NULL, TK_CONFIG_ARGV_ONLY);
}
#endif /* NO_DRAGDROP */

#ifndef __WIN32__
static char propName[] = "DragDropInfo";	/* Property name */
#   define BLT_TK2PHYSWIN(w)			Tk_WindowId(w)
#   define DEFAULTROOTWINDOW(d)			(Window)DefaultRootWindow(d)
#   define BLT_XQUERYTREE(d,w,r,p,k,nk)		XQueryTree(d,w,r,p,k,nk)
#   define BLT_XGETWINDOWATTRIBUTES(d, w, i)	XGetWindowAttributes(d,w,i)

#else /* __WIN32__ */

static char propName[] = "Tk_BLTDDI";		/* Property name */
#   define BLT_TK2PHYSWIN(w)		(Window)TkWinGetHWND(Tk_WindowId(w))
#   define DEFAULTROOTWINDOW(d)			(Window)GetDesktopWindow()
#   define BLT_XQUERYTREE(d,w,r,p,k,nk)		Blt_XQueryTree(d,w,r,p,k,nk)
#   define BLT_XGETWINDOWATTRIBUTES(d,w,i)	Blt_XGetWindowAttributes(d,w,i)

static char *BLTfProp;
static int BLTPropNL;

typedef	struct {
    int nk;
    Window *k;
} Blt_CWQ_t, *pBlt_CWQ_t;

typedef	struct {
    int	x, y, width, height;
    int	map_state;
} BLT_WI_t, *pBLT_WI_t;

/* Win32 only! */
static BOOL CALLBACK
FindBLTDDProp(HWND hwnd, LPCTSTR str, HANDLE hData)
{
    if (!strncmp(propName, str, BLTPropNL)) {
	BLTfProp = ckalloc(strlen(str) - BLTPropNL + 1);
	strcpy(BLTfProp, str + BLTPropNL);
	return FALSE;
    }
    return TRUE;
}

/* Win32 only! */
static char *
BLTGetProp(HWND w)
{
    BLTfProp = NULL;
    BLTPropNL = strlen(propName);
    EnumProps(w, FindBLTDDProp);
    BLT_WDEBUG(("BLTGetProp %X --> %s\n", w, BLTfProp);)
    return BLTfProp;
}

/* Win32 only! */
static BOOL CALLBACK
DeleteProperties(HWND hwnd, LPCTSTR str, HANDLE hData)
{
    if (!strncmp(propName, str, strlen(propName))) {
	RemoveProp(hwnd, str);
    }
    return TRUE;
}

/* Win32 only! */
static BOOL CALLBACK
BltGetNextChild(HWND cw, LPARAM p)
{
    pBlt_CWQ_t t = (pBlt_CWQ_t)p;

    if (t->k) {
	t->k[t->nk] = (Window)cw;
    }
    t->nk++;
    return TRUE;
}

/* Win32 only! */
static int
Blt_XQueryTree(Display *d, Window w, Window *r, Window *p, Window **k,
	unsigned int *nk)
{
    Blt_CWQ_t CWQ;
    HWND W = (HWND)w;

    CWQ.nk = 0;
    CWQ.k = NULL;

    /*
     *  first count children
     */
    if (W) {
	EnumChildWindows(W, BltGetNextChild, (LPARAM)&CWQ);
    } else {
	EnumWindows(BltGetNextChild, (LPARAM)&CWQ);
    }

    if (CWQ.nk) {
	CWQ.k = (Window *)ckalloc((CWQ.nk+10) * sizeof(w));
	CWQ.nk = 0;
	if (W) {
	    EnumChildWindows(W, BltGetNextChild, (LPARAM)&CWQ);
	} else {
	    EnumWindows(BltGetNextChild, (LPARAM)&CWQ);
	}
	if (!CWQ.nk) {
	    ckfree((char *) CWQ.k);
	    CWQ.k = NULL;
	}
    }

    *p = *r = (Window)NULL;
    *k = CWQ.k;
    *nk = CWQ.nk;
#ifdef	_BLT_WDEBUG
    BLT_WDEBUG(("BLT_XQuery(%x)\n",W);)
    {   int i;

	for (i = 0; i < CWQ.nk; i++) {
	    BLT_WDEBUG(("   %2d: %x\n",i, CWQ.k[i]);)
	}
    }
#endif
    return CWQ.nk;
}

/* Win32 only! */
static int
Blt_XGetWindowAttributes(Display *d, Window w, pBLT_WI_t i)
{
    if (w) {
	RECT r;
	BOOL z;
	HWND W = (HWND)w;

	i->map_state = IsWindowVisible(W) ? IsViewable : IsUnviewable;
	z = GetWindowRect(W, &r);
	i->x = r.left;
	i->y = r.top;
	i->width = r.right - r.left;
	i->height= r.bottom - r.top;
	BLT_WDEBUG(("Attr %x(%d,%d,%d,%d) %d %d\n", W,
	    r.left, r.top, i->width, i->height, z, i->map_state==IsViewable);)
	return z!=FALSE;
    }
    return 0;
}

/* Copied from Tk_Frame.c; must be kept in sync with Tk's notion of frames. */
typedef struct {
    Tk_Window tkwin;		/* Window that embodies the frame.  NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up. */
    Display *display;		/* Display containing widget.  Used, among
				 * other things, so that resources can be
				 * freed even after tkwin has gone away. */
    Tcl_Interp *interp;		/* Interpreter associated with widget.  Used
				 * to delete widget command. */
    Tcl_Command widgetCmd;	/* Token for frame's widget command. */
    char *className;		/* Class name for widget (from configuration
				 * option).  Malloc-ed. */
    int mask;			/* Either FRAME or TOPLEVEL;  used to select
				 * which configuration options are valid for
				 * widget. */
    char *screenName;		/* Screen on which widget is created.  Non-null
				 * only for top-levels.  Malloc-ed, may be
				 * NULL. */
    char *visualName;		/* Textual description of visual for window,
				 * from -visual option.  Malloc-ed, may be
				 * NULL. */
    char *colormapName;		/* Textual description of colormap for window,
				 * from -colormap option.  Malloc-ed, may be
				 * NULL. */
    char *menuName;		/* Textual description of menu to use for
				 * menubar. Malloc-ed, may be NULL. */
    Colormap colormap;		/* If not None, identifies a colormap
				 * allocated for this window, which must be
				 * freed when the window is deleted. */
    Tk_3DBorder border;		/* Structure used to draw 3-D border and
				 * background.  NULL means no background
				 * or border. */
    int borderWidth;		/* Width of 3-D border (if any). */
    int relief;			/* 3-d effect: TK_RELIEF_RAISED etc. */
    int highlightWidth;		/* Width in pixels of highlight to draw
				 * around widget when it has the focus.
				 * 0 means don't draw a highlight. */
    XColor *highlightBgColorPtr;
				/* Color for drawing traversal highlight
				 * area when highlight is off. */
    XColor *highlightColorPtr;	/* Color for drawing traversal highlight. */
    int width;			/* Width to request for window.  <= 0 means
				 * don't request any size. */
    int height;			/* Height to request for window.  <= 0 means
				 * don't request any size. */
    Tk_Cursor cursor;		/* Current cursor for window, or None. */
    char *takeFocus;		/* Value of -takefocus option;  not used in
				 * the C code, but used by keyboard traversal
				 * scripts.  Malloc'ed, but may be NULL. */
    int isContainer;		/* 1 means this window is a container, 0 means
				 * that it isn't. */
    char *useThis;		/* If the window is embedded, this points to
				 * the name of the window in which it is
				 * embedded (malloc'ed).  For non-embedded
				 * windows this is NULL. */
    int flags;			/* Various flags;  see below for
				 * definitions. */
} Frame;

static void Blt_CreateFrame(Tcl_Interp* interp, Tk_Window tkwin)
{
    Frame *framePtr = (Frame *) ckalloc(sizeof(Frame));
    framePtr->tkwin = tkwin;
    framePtr->display = Tk_Display(tkwin);
    framePtr->interp = interp;
    framePtr->widgetCmd = NULL;
    framePtr->className = NULL;
    framePtr->mask = (TK_CONFIG_USER_BIT << 1);
    framePtr->screenName = NULL;
    framePtr->visualName = NULL;
    framePtr->colormapName = NULL;
    framePtr->colormap = None;
    framePtr->border = NULL;
    framePtr->borderWidth = 0;
    framePtr->relief = TK_RELIEF_FLAT;
    framePtr->highlightWidth = 0;
    framePtr->highlightBgColorPtr = NULL;
    framePtr->highlightColorPtr = NULL;
    framePtr->width = 0;
    framePtr->height = 0;
    framePtr->cursor = None;
    framePtr->takeFocus = NULL;
    framePtr->isContainer = 0;
    framePtr->useThis = NULL;
    framePtr->flags = 0;
    framePtr->menuName = NULL;

    /*
     * Store backreference to frame widget in window structure.
     */
    TkSetClassProcs(tkwin, NULL, (ClientData) framePtr);
}
#endif /* __WIN32__ */


static int
HandlerOpOp (srcPtr, interp, argc, argv)
    Source *srcPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Blt_ListItem item;
    char *cmd;

    /*
     *  HANDLE:  drag&drop source <pathName> handler \
     *             ?<data>? ?<scmd>...?
     */
    if (argc == 4) {
	/* Show source handler data types */ 
	for(item = Blt_ListFirstItem(srcPtr->listPtr); 
	    item != NULL; item = Blt_NextListItem(item)) {
	    Tcl_AppendElement(interp, Blt_GetListKey(item));
	}
	return TCL_OK;
    }
    item = Blt_ListFindItem(srcPtr->listPtr, argv[4]);
    if (item == NULL) {
	if (strstr(argv[4], " ")) {
	    Tcl_AppendResult(interp, "bad source handler name \"",
		 argv[4], "\": should not contain spaces",
			     (char *)NULL);
	    return TCL_ERROR;
	}
	if (item == NULL) {
	    item = Blt_ListAppend(srcPtr->listPtr, argv[4], (ClientData)NULL);
	}
    }
    /*
     *  HANDLE:  drag&drop source <pathName> handler <data>
     *
     *    Create the new <data> type if it doesn't already
     *    exist, and return the code associated with it.
     */
    if (argc == 5) {
	cmd = (char *)Blt_ListGetValue(item);
	if (cmd == NULL) {
	    cmd = "";
	}
	Tcl_SetResult(interp, cmd, TCL_STATIC);
	return TCL_OK;
    }

    /*
     *  HANDLE:  drag&drop source <pathName> handler \
     *               <data> <cmd> ?<arg>...?
     *
     *    Create the new <data> type and set its command
     */
    cmd = Tcl_Concat(argc - 5, argv + 5);
    Blt_ListSetValue(item, (ClientData)cmd);
    return TCL_OK;
}





	    /*
	     *  HANDLE:  drag&drop source <pathName> ?options...?
	     */
	    c = argv[3][0];
	    length = strlen(argv[3]);

	    if (c == '-') {
		if (argc == 3) {
		    status = Tk_ConfigureInfo(interp, tokenPtr->tkwin, configSpecs,
			(char *)srcPtr, (char *)NULL, 0);
		} else if (argc == 4) {
		    status = Tk_ConfigureInfo(interp, tokenPtr->tkwin, configSpecs,
			(char *)srcPtr, argv[3], 0);
		} else {
		    status = ConfigureSource(interp, srcPtr, argc - 3, argv + 3,
			TK_CONFIG_ARGV_ONLY);
		}
		if (status != TCL_OK) {
		    return TCL_ERROR;
		}
