/*
 * bltBind.c --
 *
 *	This module implements object binding procedures for the BLT
 *	toolkit.
 *
 * Copyright 1998 Lucent Technologies, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the names
 * of Lucent Technologies any of their entities not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and
 * fitness.  In no event shall Lucent Technologies be liable for any
 * special, indirect or consequential damages or any damages
 * whatsoever resulting from loss of use, data or profits, whether in
 * an action of contract, negligence or other tortuous action, arising
 * out of or in connection with the use or performance of this
 * software.  
 */

#include "bltInt.h"
#include <ctype.h>
#include <X11/Xutil.h>

#if defined(__STDC__)
static Tk_EventProc BindProc;
#endif


/*
 * Binding table procedures.
 */
#define REPICK_IN_PROGRESS (1<<0)
#define LEFT_GRABBED_ITEM  (1<<1)

#define ALL_BUTTONS_MASK \
	(Button1Mask | Button2Mask | Button3Mask | Button4Mask | Button5Mask)

#ifndef VirtualEventMask
#define VirtualEventMask    (1L << 30)
#endif

#define ALL_VALID_EVENTS_MASK \
	(ButtonMotionMask | Button1MotionMask | Button2MotionMask | \
	 Button3MotionMask | Button4MotionMask | Button5MotionMask | \
	 ButtonPressMask | ButtonReleaseMask | EnterWindowMask | \
	 LeaveWindowMask | KeyPressMask | KeyReleaseMask | \
	 PointerMotionMask | VirtualEventMask) 


static int buttonMasks[] = {
    0, Button1Mask, Button2Mask, Button3Mask, Button4Mask, Button5Mask,
};



/*
 *--------------------------------------------------------------
 *
 * DoEvent --
 *
 *	This procedure is called to invoke binding processing
 *	for a new event that is associated with the current item
 *	for a legend.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Depends on the bindings for the legend.  A binding script
 *	could delete an entry, so callers should protect themselves
 *	with Tcl_Preserve and Tcl_Release.
 *
 *--------------------------------------------------------------
 */
static void
DoEvent(bindPtr, eventPtr)
    BindTable *bindPtr;		/* Binding information for widget in 
				 * which event occurred. */
    XEvent *eventPtr;		/* Real or simulated X event that
				 * is to be processed. */ 
{
    ClientData item;

    if (bindPtr->bindingTable == NULL) {
	return;
    }
    item = bindPtr->currentPtr;
    if ((eventPtr->type == KeyPress) || (eventPtr->type == KeyRelease)) {
	item = bindPtr->focusPtr;
    }
    if (item == NULL) {
	return;
    }
    /*
     * Invoke the binding system.
     */
    if (bindPtr->tkwin != NULL) {
	ClientData tagArr[10];
	int numTags;

	if (bindPtr->tagProc == NULL) {
	    tagArr[0] = (ClientData)Tk_GetUid("all");
	    tagArr[1] = item;
	    numTags = 2;
	} else {
	    (*bindPtr->tagProc)(bindPtr, item, tagArr, &numTags);
	}
	Tk_BindEvent(bindPtr->bindingTable, eventPtr, bindPtr->tkwin, 
	     numTags, tagArr);
    }
}

/*
 *--------------------------------------------------------------
 *
 * PickCurrentItem --
 *
 *	Find the topmost item in a legend that contains a given
 *	location and mark the the current item.  If the current
 *	item has changed, generate a fake exit event on the old
 *	current item and a fake enter event on the new current
 *	item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The current item for legendPtr may change.  If it does,
 *	then the commands associated with item entry and exit
 *	could do just about anything.  A binding script could
 *	delete the legend, so callers should protect themselves
 *	with Tcl_Preserve and Tcl_Release.
 *
 *--------------------------------------------------------------
 */
static void
PickCurrentItem(bindPtr, eventPtr)
    BindTable *bindPtr;		/* Graph widget in which to select
				 * current item. */
    XEvent *eventPtr;		/* Event describing location of
				 * mouse cursor.  Must be EnterWindow,
				 * LeaveWindow, ButtonRelease, or
				 * MotionNotify. */
{
    int buttonDown;
    static ClientData save = 0;
    /*
     * Check whether or not a button is down.  If so, we'll log entry
     * and exit into and out of the current item, but not entry into
     * any other item.  This implements a form of grabbing equivalent
     * to what the X server does for windows.
     */

    buttonDown = (bindPtr->state & ALL_BUTTONS_MASK);
    if (!buttonDown) {
	bindPtr->flags &= ~LEFT_GRABBED_ITEM;
    }

    /*
     * Save information about this event in the widget.  The event in
     * the widget is used for two purposes:
     *
     * 1. Event bindings: if the current item changes, fake events are
     *    generated to allow item-enter and item-leave bindings to trigger.
     * 2. Reselection: if the current item gets deleted, can use the
     *    saved event to find a new current item.
     * Translate MotionNotify events into EnterNotify events, since that's
     * what gets reported to item handlers.
     */

    if (eventPtr != &bindPtr->pickEvent) {
	if ((eventPtr->type == MotionNotify) || 
	    (eventPtr->type == ButtonRelease)) {
	    bindPtr->pickEvent.xcrossing.type = EnterNotify;
	    bindPtr->pickEvent.xcrossing.serial = eventPtr->xmotion.serial;
	    bindPtr->pickEvent.xcrossing.send_event = 
		eventPtr->xmotion.send_event;
	    bindPtr->pickEvent.xcrossing.display = eventPtr->xmotion.display;
	    bindPtr->pickEvent.xcrossing.window = eventPtr->xmotion.window;
	    bindPtr->pickEvent.xcrossing.root = eventPtr->xmotion.root;
	    bindPtr->pickEvent.xcrossing.subwindow = None;
	    bindPtr->pickEvent.xcrossing.time = eventPtr->xmotion.time;
	    bindPtr->pickEvent.xcrossing.x = eventPtr->xmotion.x;
	    bindPtr->pickEvent.xcrossing.y = eventPtr->xmotion.y;
	    bindPtr->pickEvent.xcrossing.x_root = eventPtr->xmotion.x_root;
	    bindPtr->pickEvent.xcrossing.y_root = eventPtr->xmotion.y_root;
	    bindPtr->pickEvent.xcrossing.mode = NotifyNormal;
	    bindPtr->pickEvent.xcrossing.detail = NotifyNonlinear;
	    bindPtr->pickEvent.xcrossing.same_screen
		    = eventPtr->xmotion.same_screen;
	    bindPtr->pickEvent.xcrossing.focus = False;
	    bindPtr->pickEvent.xcrossing.state = eventPtr->xmotion.state;
	} else  {
	    bindPtr->pickEvent = *eventPtr;
	}
    }
    bindPtr->activePick = TRUE;

    /*
     * If this is a recursive call (there's already a partially completed
     * call pending on the stack;  it's in the middle of processing a
     * Leave event handler for the old current item) then just return;
     * the pending call will do everything that's needed.
     */

    if (bindPtr->flags & REPICK_IN_PROGRESS) {
	return;
    }

    /*
     * A LeaveNotify event automatically means that there's no current
     * tab, so the check for closest item can be skipped.
     */

    if (bindPtr->pickEvent.type != LeaveNotify) {
	int x, y;

	x = bindPtr->pickEvent.xcrossing.x;
	y = bindPtr->pickEvent.xcrossing.y;
	bindPtr->newCurrentPtr = 
	    (*bindPtr->pickProc)(bindPtr->clientData, x, y);
    } else {
	bindPtr->newCurrentPtr = NULL;
    }
    if ((bindPtr->newCurrentPtr == bindPtr->currentPtr) && 
	!(bindPtr->flags & LEFT_GRABBED_ITEM)) {
	/*
	 * Nothing to do:  the current item hasn't changed.
	 */
	return;
    }
#ifdef xFULLY_SIMULATE_GRAB
    if ((bindPtr->newCurrentPtr != bindPtr->currentPtr) && (buttonDown)) {
	bindPtr->flags |= LEFT_GRABBED_ITEM;
	return;
    }
#endif
    /*
     * Simulate a LeaveNotify event on the previous current item and
     * an EnterNotify event on the new current item.  Remove the "current"
     * tag from the previous current item and place it on the new current
     * item.
     */
    if ((bindPtr->newCurrentPtr != bindPtr->currentPtr) && 
	(bindPtr->currentPtr != NULL) && 
	!(bindPtr->flags & LEFT_GRABBED_ITEM)) {
	XEvent event;
	ClientData hold;

	event = bindPtr->pickEvent;
	event.type = LeaveNotify;
	/*
	 * If the event's detail happens to be NotifyInferior the
	 * binding mechanism will discard the event.  To be consistent,
	 * always use NotifyAncestor.
	 */
	event.xcrossing.detail = NotifyAncestor;

	bindPtr->flags |= REPICK_IN_PROGRESS;
	hold = bindPtr->currentPtr;
	if (save != 0) {
	    bindPtr->currentPtr = save;
	}
	DoEvent(bindPtr, &event);
	bindPtr->currentPtr = hold;
	save = 0;

	bindPtr->flags &= ~REPICK_IN_PROGRESS;

	/*
	 * Note:  during DoEvent above, it's possible that
	 * bindPtr->newCurrentPtr got reset to NULL because the
	 * item was deleted.
	 */
    }
    if ((bindPtr->newCurrentPtr != bindPtr->currentPtr) && (buttonDown)) {
	bindPtr->flags |= LEFT_GRABBED_ITEM;
#ifdef notdef
	if (bindPtr->newCurrentPtr != NULL) {
	    XEvent event;
	    
	    if (save == 0) {
		save = bindPtr->currentPtr;
	    }
	    bindPtr->currentPtr = bindPtr->newCurrentPtr;
	    event = bindPtr->pickEvent;
	    event.type = EnterNotify;
	    event.xcrossing.detail = NotifyAncestor;
	    DoEvent(bindPtr, &event);
	}
#endif
	return;
    }

    /*
     * Special note:  it's possible that 
     *		bindPtr->newCurrentPtr == bindPtr->currentPtr 
     * here.  This can happen, for example, if LEFT_GRABBED_ITEM was set.
     */

    bindPtr->flags &= ~LEFT_GRABBED_ITEM;
    bindPtr->currentPtr = bindPtr->newCurrentPtr;
    if (bindPtr->currentPtr != NULL) {
	XEvent event;

	event = bindPtr->pickEvent;
	event.type = EnterNotify;
	event.xcrossing.detail = NotifyAncestor;
	DoEvent(bindPtr, &event);
    }
}


/*
 *--------------------------------------------------------------
 *
 * BindProc --
 *
 *	This procedure is invoked by the Tk dispatcher to handle
 *	events associated with bindings on items.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Depends on the command invoked as part of the binding
 *	(if there was any).
 *
 *--------------------------------------------------------------
 */
static void
BindProc(clientData, eventPtr)
    ClientData clientData;		/* Pointer to widget structure. */
    XEvent *eventPtr;			/* Pointer to X event that just
					 * happened. */
{
    BindTable *bindPtr = (BindTable *)clientData;
    int mask;

    Tcl_Preserve(bindPtr->clientData);

    /*
     * This code below keeps track of the current modifier state in
     * bindPtr->state.  This information is used to defer repicks of
     * the current item while buttons are down.
     */
    switch(eventPtr->type) {
    case ButtonPress:
    case ButtonRelease:
	mask = 0;
	if ((eventPtr->xbutton.button >= Button1) &&
	    (eventPtr->xbutton.button <= Button5)) {
	    mask = buttonMasks[eventPtr->xbutton.button];
	}
	
	/*
	 * For button press events, repick the current item using the
	 * button state before the event, then process the event.  For
	 * button release events, first process the event, then repick
	 * the current item using the button state *after* the event
	 * (the button has logically gone up before we change the
	 * current item).
	 */
	
	if (eventPtr->type == ButtonPress) {
	    
	    /*
	     * On a button press, first repick the current item using
	     * the button state before the event, the process the event.
	     */
	    
	    bindPtr->state = eventPtr->xbutton.state;
	    PickCurrentItem(bindPtr, eventPtr);
	    bindPtr->state ^= mask;
	    DoEvent(bindPtr, eventPtr);

	} else {

	    /*
	     * Button release: first process the event, with the button
	     * still considered to be down.  Then repick the current
	     * item under the assumption that the button is no longer down.
	     */
	    
	    bindPtr->state = eventPtr->xbutton.state;
	    DoEvent(bindPtr, eventPtr);
	    eventPtr->xbutton.state ^= mask;
	    bindPtr->state = eventPtr->xbutton.state;
	    PickCurrentItem(bindPtr, eventPtr);
	    eventPtr->xbutton.state ^= mask;
	}
	break;

    case EnterNotify:
    case LeaveNotify:
	bindPtr->state = eventPtr->xcrossing.state;
	PickCurrentItem(bindPtr, eventPtr);
	break;

    case MotionNotify:
	bindPtr->state = eventPtr->xmotion.state;
	PickCurrentItem(bindPtr, eventPtr);
	DoEvent(bindPtr, eventPtr);
	break;

    case KeyPress:
    case KeyRelease:
	bindPtr->state = eventPtr->xkey.state;
	PickCurrentItem(bindPtr, eventPtr);
	DoEvent(bindPtr, eventPtr);
	break;
    }
    Tcl_Release(bindPtr->clientData);
}

int
Blt_ConfigureBindings(interp, bindPtr, item, argc, argv)
    Tcl_Interp *interp;
    BindTable *bindPtr;
    ClientData item;
    int argc;
    char **argv;
{
    char *command;
    unsigned long mask;
    char *seq;

    switch (argc) {
    case 0:
	Tk_GetAllBindings(interp, bindPtr->bindingTable, item);
	break;

    case 1:
	command = Tk_GetBinding(interp, bindPtr->bindingTable, item, argv[0]);
	if (command == NULL) {
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, command, TCL_VOLATILE);
	break;
	
    case 2:
	seq = argv[0];
	command = argv[1];
	if (command[0] == '\0') {
	    return Tk_DeleteBinding(interp, bindPtr->bindingTable, item, seq);
	}
	if (command[0] == '+') {
	    mask = Tk_CreateBinding(interp, bindPtr->bindingTable, item, seq, 
		command + 1, TRUE);
	} else {
	    mask = Tk_CreateBinding(interp, bindPtr->bindingTable, item, seq, 
		command, FALSE);
	}
	if (mask == 0) {
	    return TCL_ERROR;
	}
	if (mask & (unsigned)~ALL_VALID_EVENTS_MASK) {
	    Tk_DeleteBinding(interp, bindPtr->bindingTable, item, seq);
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "requested illegal events; ",
		 "only key, button, motion, enter, leave, and virtual ",
		 "events may be used", (char *) NULL);
	    return TCL_ERROR;
	}
	break;
    }
    return TCL_OK;
}

BindTable *
Blt_CreateBindingTable(interp, tkwin, clientData, pickProc, tagProc)
    Tcl_Interp *interp;
    Tk_Window tkwin;
    ClientData clientData;
    BindPickProc *pickProc;
    BindTagProc *tagProc;
{
    unsigned int mask;
    BindTable *bindPtr;

    bindPtr = (BindTable *)calloc(1, sizeof(BindTable));
    assert(bindPtr);
    bindPtr->clientData = clientData;
    bindPtr->pickProc = pickProc;
    bindPtr->tagProc = tagProc;
    bindPtr->tkwin = tkwin;
    bindPtr->bindingTable = Tk_CreateBindingTable(interp);
    mask = (KeyPressMask | KeyReleaseMask | ButtonPressMask | 
	    ButtonReleaseMask | EnterWindowMask | LeaveWindowMask | 
	    PointerMotionMask);
    Tk_CreateEventHandler(tkwin, mask, BindProc, (ClientData)bindPtr);
    return bindPtr;
}


void
Blt_DestroyBindingTable(bindPtr)
    BindTable *bindPtr;
{
    unsigned int mask;

    Tk_DeleteBindingTable(bindPtr->bindingTable);
    mask = (KeyPressMask | KeyReleaseMask | ButtonPressMask | 
	    ButtonReleaseMask | EnterWindowMask | LeaveWindowMask | 
	    PointerMotionMask);
    Tk_DeleteEventHandler(bindPtr->tkwin, mask, BindProc, (ClientData)bindPtr);
    free((char *)bindPtr);
}


void 
Blt_PickCurrentItem(bindPtr)
    BindTable *bindPtr;
{
    if (bindPtr->activePick) {
	PickCurrentItem(bindPtr, &(bindPtr->pickEvent));
    }
}

void 
Blt_DeleteAllBindings(bindPtr, object) 
    BindTable *bindPtr;
    ClientData object;
{
    Tk_DeleteAllBindings(bindPtr->bindingTable, object);

    /*  
     * If this is the object currently picked, we need to repick one.
     */
    if (bindPtr->currentPtr == object) {
	bindPtr->currentPtr = NULL;
    }
    if (bindPtr->newCurrentPtr == object) {
	bindPtr->newCurrentPtr = NULL;
    }
    if (bindPtr->focusPtr == object) {
	bindPtr->focusPtr = NULL;
    }
}


