/*
 * bltBgexec.c --
 *
 *	This module implements a background "exec" command for the
 *	Tk toolkit.
 *
 * Copyright 1993-1998 Lucent Technologies, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the names
 * of Lucent Technologies any of their entities not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and
 * fitness.  In no event shall Lucent Technologies be liable for any
 * special, indirect or consequential damages or any damages
 * whatsoever resulting from loss of use, data or profits, whether in
 * an action of contract, negligence or other tortuous action, arising
 * out of or in connection with the use or performance of this
 * software.
 *
 * The "bgexec" command was created by George Howlett.  
 */

#include "bltInt.h"

#ifndef NO_BGEXEC

#include <ctype.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/param.h>
#include <sys/types.h>

#ifdef HAVE_WAITFLAGS_H
#   include <waitflags.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#   include <sys/wait.h>
#endif

#if (TCL_MAJOR_VERSION == 7) && (TCL_MINOR_VERSION > 4)
#define FILEHANDLER_USES_TCLFILES 1
#else
typedef int Tcl_File;
#endif

#if (TCL_MAJOR_VERSION == 7) && (TCL_MINOR_VERSION == 4)
#define Tcl_CreateFileHandler	Tk_CreateFileHander
#define Tcl_DeleteFileHandler	Tk_DeleteFileHander
#endif

/*
 *	As of Tcl 7.6, we're using our own version of the old
 *	Tcl_CreatePipeline routine.  I would have tried to use 
 *	Tcl_OpenCommandChannel but you can't get at the array of
 *	process ids, unless of course you pry open the undocumented
 *	structure PipeStatus as clientData.  Nor could I figure out
 *	how to set one side of the pipe to be non-blocking.  The whole
 *	channel API seems overly complex for what its supposed to
 *	do. [And maybe that's why it keeps changing every release.]
 */
extern int Blt_CreatePipeline _ANSI_ARGS_((Tcl_Interp *interp, int argc,
	char **argv, int **pidPtrPtr, int *inPipePtr, int *outPipePtr,
	int *errFilePtr));

#define READ_AGAIN	(0)
#define READ_EOF	(-1)
#define READ_ERROR	(-2)

/* The wait-related definitions are taken from tclUnix.h */

/*
 * Not all systems declare the errno variable in errno.h. so this
 * file does it explicitly.  The list of system error messages also
 * isn't generally declared in a header file anywhere.
 */

extern int errno;

/*
 * The type of the status returned by wait varies from UNIX system
 * to UNIX system.  The macro below defines it:
 */

#ifdef AIX
#   define WAIT_STATUS_TYPE pid_t
#else
#ifndef NO_UNION_WAIT
#   define WAIT_STATUS_TYPE union wait
#else
#   define WAIT_STATUS_TYPE int
#endif
#endif

/*
 * Supply definitions for macros to query wait status, if not already
 * defined in header files above.
 */

#ifndef WIFEXITED
#   define WIFEXITED(stat)  (((*((int *) &(stat))) & 0xff) == 0)
#endif

#ifndef WEXITSTATUS
#   define WEXITSTATUS(stat) (((*((int *) &(stat))) >> 8) & 0xff)
#endif

#ifndef WIFSIGNALED
#   define WIFSIGNALED(stat) (((*((int *) &(stat)))) && ((*((int *) &(stat))) == ((*((int *) &(stat))) & 0x00ff)))
#endif

#ifndef WTERMSIG
#   define WTERMSIG(stat)    ((*((int *) &(stat))) & 0x7f)
#endif

#ifndef WIFSTOPPED
#   define WIFSTOPPED(stat)  (((*((int *) &(stat))) & 0xff) == 0177)
#endif

#ifndef WSTOPSIG
#   define WSTOPSIG(stat)    (((*((int *) &(stat))) >> 8) & 0xff)
#endif


#define TRACE_FLAGS (TCL_TRACE_WRITES | TCL_TRACE_UNSETS | TCL_GLOBAL_ONLY)

#define BLOCK_SIZE	1024	/* Size of allocation blocks for string buffer */
#define DEF_BUFFER_SIZE	(BLOCK_SIZE * 8)
#define MAX_READS       100	/* Maximum number of successful reads
			         * before stopping to let Tk catch up
			         * on events */

#ifndef NSIG
#define NSIG 		32	/* Number of signals available */
#endif /*NSIG*/

typedef struct {
    int number;
    char *name;
} SignalId;

static SignalId signalIds[] =
{
#ifdef SIGABRT
    {SIGABRT, "SIGABRT"},
#endif
#ifdef SIGALRM
    {SIGALRM, "SIGALRM"},
#endif
#ifdef SIGBUS
    {SIGBUS, "SIGBUS"},
#endif
#ifdef SIGCHLD
    {SIGCHLD, "SIGCHLD"},
#endif
#if defined(SIGCLD) && (!defined(SIGCHLD) || (SIGCLD != SIGCHLD))
    {SIGCLD, "SIGCLD"},
#endif
#ifdef SIGCONT
    {SIGCONT, "SIGCONT"},
#endif
#if defined(SIGEMT) && (!defined(SIGXCPU) || (SIGEMT != SIGXCPU))
    {SIGEMT, "SIGEMT"},
#endif
#ifdef SIGFPE
    {SIGFPE, "SIGFPE"},
#endif
#ifdef SIGHUP
    {SIGHUP, "SIGHUP"},
#endif
#ifdef SIGILL
    {SIGILL, "SIGILL"},
#endif
#ifdef SIGINT
    {SIGINT, "SIGINT"},
#endif
#ifdef SIGIO
    {SIGIO, "SIGIO"},
#endif
#if defined(SIGIOT) && (!defined(SIGABRT) || (SIGIOT != SIGABRT))
    {SIGIOT, "SIGIOT"},
#endif
#ifdef SIGKILL
    {SIGKILL, "SIGKILL"},
#endif
#if defined(SIGLOST) && (!defined(SIGIOT) || (SIGLOST != SIGIOT)) && (!defined(SIGURG) || (SIGLOST != SIGURG))
    {SIGLOST, "SIGLOST"},
#endif
#ifdef SIGPIPE
    {SIGPIPE, "SIGPIPE"},
#endif
#if defined(SIGPOLL) && (!defined(SIGIO) || (SIGPOLL != SIGIO))
    {SIGPOLL, "SIGPOLL"},
#endif
#ifdef SIGPROF
    {SIGPROF, "SIGPROF"},
#endif
#if defined(SIGPWR) && (!defined(SIGXFSZ) || (SIGPWR != SIGXFSZ))
    {SIGPWR, "SIGPWR"},
#endif
#ifdef SIGQUIT
    {SIGQUIT, "SIGQUIT"},
#endif
#ifdef SIGSEGV
    {SIGSEGV, "SIGSEGV"},
#endif
#ifdef SIGSTOP
    {SIGSTOP, "SIGSTOP"},
#endif
#ifdef SIGSYS
    {SIGSYS, "SIGSYS"},
#endif
#ifdef SIGTERM
    {SIGTERM, "SIGTERM"},
#endif
#ifdef SIGTRAP
    {SIGTRAP, "SIGTRAP"},
#endif
#ifdef SIGTSTP
    {SIGTSTP, "SIGTSTP"},
#endif
#ifdef SIGTTIN
    {SIGTTIN, "SIGTTIN"},
#endif
#ifdef SIGTTOU
    {SIGTTOU, "SIGTTOU"},
#endif
#if defined(SIGURG) && (!defined(SIGIO) || (SIGURG != SIGIO))
    {SIGURG, "SIGURG"},
#endif
#if defined(SIGUSR1) && (!defined(SIGIO) || (SIGUSR1 != SIGIO))
    {SIGUSR1, "SIGUSR1"},
#endif
#if defined(SIGUSR2) && (!defined(SIGURG) || (SIGUSR2 != SIGURG))
    {SIGUSR2, "SIGUSR2"},
#endif
#ifdef SIGVTALRM
    {SIGVTALRM, "SIGVTALRM"},
#endif
#ifdef SIGWINCH
    {SIGWINCH, "SIGWINCH"},
#endif
#ifdef SIGXCPU
    {SIGXCPU, "SIGXCPU"},
#endif
#ifdef SIGXFSZ
    {SIGXFSZ, "SIGXFSZ"},
#endif
    {-1, "unknown signal"},
};

static int StringToSignal _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tk_Window tkwin, char *string, char *widgRec,
	int offset));
static char *SignalToString _ANSI_ARGS_((ClientData clientData, Tk_Window tkwin, 	char *widgRec, int offset, Tcl_FreeProc **freeProcPtr));

static Tk_CustomOption signalOption =
{
    StringToSignal, SignalToString, (ClientData)0
};

typedef struct {
    char *doneVar;		/* Name of a Tcl variable (malloc'ed)
				 * set to the collected data of the
				 * last UNIX subprocess. */

    char *updateVar;		/* Name of a Tcl variable (malloc'ed)
				 * updated as data is read from the
				 * pipe.  If NULL, no updates are made
				 * */

    Tcl_File file;		/* Used for backward compatability
				 * with Tcl 7.5 */
    int fd;			/* File descriptor of the pipe. */

    int count;
    int prevEnd;		/* Number of bytes read the last time a
				 * buffer was retrieved */
    int fixMark;		/* Index of fixed newline character in
				 * buffer.  If -1, no fix was made. */
    
    char *byteArr;		/* Stores command output (malloc-ed):
				 * Initially points to static storage
				 */
    int end;			/* Number of characters in the buffer */
    int arraySize;		/* Size of buffer allocated */

    char staticSpace[DEF_BUFFER_SIZE];	/* Static space */

} Sink;

typedef struct {
    Tk_Uid statVarId;		/* Name of a Tcl variable set to the
				 * exit status of the last
				 * process. Setting this variable
				 * triggers the termination of all
				 * subprocesses (regardless whether
				 * they have already completed) */

    int signalNum;		/* If non-zero, indicates the signal
				 * to send subprocesses when cleaning
				 * up.*/
    int keepTrailingNewLine;	/* If non-zero, indicates to set Tcl
				 * output variables with trailing
				 * newlines intact */
    int interval;		/* Interval to poll for the exiting
				 * processes */

    /* Private */
    Tk_Window tkwin;		/* Main window of interpreter. Used with
				 * Tk_ConfigureWidget to handle options */
    Tcl_Interp *interp;		/* Interpreter containing variables */

    int numPids;		/* Number of processes in pipeline */
    int *pidArr;		/* Array of process Ids from pipeline. */

    int detached;		/* Indicates that the pipeline is
				 * detached from standard I/O, running
				 * in the background. */
    Tk_TimerToken timerToken;	/* Token for timer handler which polls
				 * for the exit status of each
				 * sub-process. If zero, there's no
				 * timer handler queued. */

    int *exitCodePtr;		/* Pointer to a memory location to
				 * contain the last process' exit
				 * code. */
    int *donePtr;

    Sink outputSink, errorSink;
    
} BackgroundInfo;


static Tk_ConfigSpec configSpecs[] =
{
    {TK_CONFIG_STRING, "-output", (char *)NULL, (char *)NULL,
	(char *)NULL, Tk_Offset(BackgroundInfo, outputSink.doneVar),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-update", (char *)NULL, (char *)NULL,
	(char *)NULL, Tk_Offset(BackgroundInfo, outputSink.updateVar),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-error", (char *)NULL, (char *)NULL,
	(char *)NULL, Tk_Offset(BackgroundInfo, errorSink.doneVar),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-onerror", (char *)NULL, (char *)NULL,
	(char *)NULL, Tk_Offset(BackgroundInfo, errorSink.updateVar),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-onoutput", (char *)NULL, (char *)NULL,
	(char *)NULL, Tk_Offset(BackgroundInfo, outputSink.updateVar),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BOOLEAN, "-keepnewline", (char *)NULL, (char *)NULL,
	(char *)NULL, Tk_Offset(BackgroundInfo, keepTrailingNewLine), 0},
    {TK_CONFIG_CUSTOM, "-killsignal", (char *)NULL, (char *)NULL,
	(char *)NULL, Tk_Offset(BackgroundInfo, signalNum), 0, &signalOption},
    {TK_CONFIG_INT, "-check", (char *)NULL, (char *)NULL,
	(char *)NULL, Tk_Offset(BackgroundInfo, interval), 0},
    {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL,
	(char *)NULL, 0, 0}
};

static char *VariableProc _ANSI_ARGS_((ClientData clientData, 
	Tcl_Interp *interp, char *part1, char *part2, int flags));
static void TimerProc _ANSI_ARGS_((ClientData clientData));
static void OutputEventProc _ANSI_ARGS_((ClientData clientData, int mask));
static void ErrorEventProc _ANSI_ARGS_((ClientData clientData, int mask));

/*
 *----------------------------------------------------------------------
 *
 * StringToSignal --
 *
 *	Convert a string represent a signal number into its integer
 *	value.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
StringToSignal(clientData, interp, tkwin, string, widgRec, offset)
    ClientData clientData;	/* not used */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *string;		/* Signal number */
    char *widgRec;		/* Background info record */
    int offset;			/* Offset of vector in Element record */
{
    int *signalPtr = (int *)(widgRec + offset);
    int signalNum;

    if ((string == NULL) || (*string == '\0')) {
	*signalPtr = 0;
	return TCL_OK;
    }
    if (isdigit(UCHAR(string[0]))) {
	if (Tcl_GetInt(interp, string, &signalNum) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	char *name;
	register SignalId *sigPtr;

	name = string;

	/*  Clip off any "SIG" prefix from the signal name */
	if ((name[0] == 'S') && (name[1] == 'I') && (name[2] == 'G')) {
	    name += 3;
	}
	signalNum = -1;
	for (sigPtr = signalIds; sigPtr->number > 0; sigPtr++) {
	    if (strcmp(sigPtr->name + 3, name) == 0) {
		signalNum = sigPtr->number;
		break;
	    }
	}
	if (signalNum < 0) {
	    Tcl_AppendResult(interp, "unknown signal \"", string, "\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
    }
    if ((signalNum < 1) || (signalNum > NSIG)) {
	/* Outside range of signals */
	Tcl_AppendResult(interp, "signal number \"", string,
	    "\" is out of range", (char *)NULL);
	return TCL_ERROR;
    }
    *signalPtr = signalNum;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SignalToString --
 *
 *	Convert the integer signal number into an ASCII string.
 *
 * Results:
 *	The string representation of the kill signal is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
SignalToString(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* Not used */
    Tk_Window tkwin;		/* Not used */
    char *widgRec;		/* BackgroundInfo record */
    int offset;			/* Offset of signal number in record */
    Tcl_FreeProc **freeProcPtr;	/* Memory deallocation scheme to use */
{
    int signalNum = *(int *)(widgRec + offset);

    if (signalNum == 0) {
	return "";
    } else {
	char *result;
	char string[20];

	sprintf(string, "%d", signalNum);
	*freeProcPtr = (Tcl_FreeProc *)free;
	result = strdup(string);
	return result;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * GetSinkData --
 *
 *	Returns the data currently saved in the buffer
 *
 *----------------------------------------------------------------------
 */
static char *
GetSinkData(sinkPtr)
    Sink *sinkPtr;
{
    sinkPtr->byteArr[sinkPtr->end] = '\0';
    return sinkPtr->byteArr;
}

/*
 *----------------------------------------------------------------------
 *
 * FlushSink --
 *
 *	Flushes the buffer, resetting it to empty.
 *	This is used when we don't want to save all the data from 
 *	the pipeline.
 *
 *----------------------------------------------------------------------
 */

static void
FlushSink(sinkPtr)
    Sink *sinkPtr;
{
    sinkPtr->end = 0;
    sinkPtr->byteArr[0] = '\0';
    sinkPtr->count = 0;
    sinkPtr->prevEnd = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * InitSink --
 *
 *	Initializes the buffer's storage.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Storage is cleared.
 *
 *---------------------------------------------------------------------- 
 */
static void
InitSink(sinkPtr)
    Sink *sinkPtr;
{
    sinkPtr->fd = -1;
    sinkPtr->file = (Tcl_File)NULL;
    sinkPtr->byteArr = sinkPtr->staticSpace;
    sinkPtr->end = sinkPtr->prevEnd = 0;
    sinkPtr->count = 0;
    sinkPtr->fixMark = -1;
    sinkPtr->arraySize = DEF_BUFFER_SIZE;
    FlushSink(sinkPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ResetSinkBuffer --
 *
 *	Resets the buffer's storage, freeing any malloc'ed space.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static void
ResetSinkBuffer(sinkPtr)
    Sink *sinkPtr;
{
    if (sinkPtr->byteArr != sinkPtr->staticSpace) {
	free(sinkPtr->byteArr);
    }
    InitSink(sinkPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ExtendSinkBuffer --
 *
 *	Doubles the size of the current buffer.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static int
ExtendSinkBuffer(sinkPtr)
    Sink *sinkPtr;
{
    char *newPtr;

    /*
     * Allocate a new array, double the old size
     */
    sinkPtr->arraySize += sinkPtr->arraySize;
    newPtr = (char *)malloc(sizeof(char) * sinkPtr->arraySize);
    if (newPtr == NULL) {
	return TCL_ERROR;
    }
    strcpy(newPtr, sinkPtr->byteArr);
    if (sinkPtr->byteArr != sinkPtr->staticSpace) {
	free((char *)sinkPtr->byteArr);
    }
    sinkPtr->byteArr = newPtr;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ReadBytes --
 *
 *	Reads and appends any available data from a given file descriptor
 *	to the buffer.
 *
 * Results:
 *	Returns TCL_OK when EOF is found, TCL_RETURN if reading
 *	data would block, and TCL_ERROR if an error occurred.
 *
 *----------------------------------------------------------------------
 */
static int
ReadBytes(sinkPtr)
    Sink *sinkPtr;
{
    int numBytes, bytesLeft;
    register int i, n;
    char *array;

    /*
     * ------------------------------------------------------------------
     *
     * 	Worry about indefinite postponement.
     *
     * 	Typically we want to stay in the read loop as long as it takes
     * 	to collect all the data that's currently available.  But if
     * 	it's coming in at a constant high rate, we need to arbitrarily
     * 	break out at some point. This allows for both setting the
     * 	update variable and the Tk program to handle idle events.
     *
     * ------------------------------------------------------------------
     */

    for (i = 0; i < MAX_READS; i++) {

	/*
	 * Allocate a larger buffer when the number of remaining bytes
	 * is below the threshold BLOCK_SIZE.
	 */

	bytesLeft = sinkPtr->arraySize - sinkPtr->end;

	if (bytesLeft < BLOCK_SIZE) {
	    if (ExtendSinkBuffer(sinkPtr) != TCL_OK) {
		return READ_ERROR;
	    }
	    /* Size of buffer has changed. */
	    bytesLeft = sinkPtr->arraySize - sinkPtr->end;
	}
	array = sinkPtr->byteArr + sinkPtr->end;

	/*  
	 * Read into a buffer but make sure we leave room for a
	 * trailing NUL byte. 
	 */
	numBytes = read(sinkPtr->fd, array, bytesLeft - 1);
	if (numBytes == 0) {	/* EOF: break out of loop. */
	    return READ_EOF;
	}
	if (numBytes < 0) {

	    /*
	     * Either an error has occurred or no more data is
	     * currently available to read.
	     */
#ifdef O_NONBLOCK
	    if (errno == EAGAIN) {
#else
	    if (errno == EWOULDBLOCK) {
#endif /*O_NONBLOCK*/
		return READ_AGAIN;
	    }
	    sinkPtr->byteArr[0] = '\0';
	    return READ_ERROR;
	}
	/* Clean out NUL bytes, make spaces */
	for (n = 0; n < numBytes; n++) {
	    if (array[n] == 0) {
		array[n] = ' ';
	    }
	}
	sinkPtr->count += numBytes;
	sinkPtr->end += numBytes;
	sinkPtr->byteArr[sinkPtr->end] = '\0';
    }
    return numBytes;
}

/*
 *----------------------------------------------------------------------
 *
 * FixNewline --
 *
 *	Clips off the trailing newline in the buffer (if one exists).
 *	Saves the location in the buffer where the fix was made.
 *
 *---------------------------------------------------------------------- 
 */
static void
FixNewline(sinkPtr)
    Sink *sinkPtr;
{
    sinkPtr->fixMark = -1;
    if (sinkPtr->end > 0) {
	int mark = sinkPtr->end - 1;

	if (sinkPtr->byteArr[mark] == '\n') {
	    sinkPtr->byteArr[mark] = '\0';
	    sinkPtr->fixMark = mark;
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * UnfixNewline --
 *
 *	Restores the previously clipped newline in the buffer.
 *	The fixMark field indicates whether one was clipped.
 *
 *----------------------------------------------------------------------
 */
static void
UnfixNewline(sinkPtr)
    Sink *sinkPtr;
{
    if (sinkPtr->fixMark >= 0) {
	sinkPtr->byteArr[sinkPtr->fixMark] = '\n';
	sinkPtr->fixMark = -1;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * LastRead --
 *
 *	Returns the data saved from the last time this routine
 *	was called.
 *
 *----------------------------------------------------------------------
 */
static char *
LastRead(sinkPtr)
    Sink *sinkPtr;
{
    char *string;

    sinkPtr->byteArr[sinkPtr->end] = '\0';
    string = sinkPtr->byteArr + sinkPtr->prevEnd;
    sinkPtr->prevEnd = sinkPtr->end;
    sinkPtr->count = 0;
    return string;
}

static void
CloseSink(bgPtr, sinkPtr)
    BackgroundInfo *bgPtr;
    Sink *sinkPtr;
{
    if (sinkPtr->fd != -1) {
	close(sinkPtr->fd);
#ifdef FILEHANDLER_USES_TCLFILES
	Tcl_DeleteFileHandler(sinkPtr->file);
	Tcl_FreeFile(sinkPtr->file);
#else
	Tcl_DeleteFileHandler(sinkPtr->fd);
#endif
	sinkPtr->file = (Tcl_File)NULL;
	sinkPtr->fd = -1;

	if (sinkPtr->doneVar != NULL) {
	    /* 
	     * If data is to be collected, set the "done" variable with
	     * the contents of the buffer.
	     */
	    if (!bgPtr->keepTrailingNewLine) {
		FixNewline(sinkPtr);
	    }
	    if (Tcl_SetVar(bgPtr->interp, sinkPtr->doneVar, 
			   GetSinkData(sinkPtr), TCL_GLOBAL_ONLY) == NULL) {
		Tk_BackgroundError(bgPtr->interp);
	    }
	}
    }
}

static int
CollectData(bgPtr, sinkPtr)
    BackgroundInfo *bgPtr;
    Sink *sinkPtr;
{
    int status;
    int flags;

    /* 
     * If there is no output variable (-update used alone) there's no
     * need to accumulate the output in memory.  Reset the counters so
     * we only keep what's last read.  
     */
    if ((bgPtr->detached) && (sinkPtr->doneVar == NULL)) {
	FlushSink(sinkPtr);
    }
    flags = (TCL_GLOBAL_ONLY | TCL_APPEND_VALUE | TCL_LEAVE_ERR_MSG);
    status = ReadBytes(sinkPtr);
    if ((sinkPtr->updateVar != NULL) && (sinkPtr->prevEnd < sinkPtr->end)) {
	char *data;
	int p1, p2;
	
	if (!bgPtr->keepTrailingNewLine) {
	    FixNewline(sinkPtr);
	}
	p1 = sinkPtr->prevEnd, p2 = sinkPtr->end;
	data = LastRead(sinkPtr);
	if (data[0] != '\0') {
	    if (Tcl_SetVar(bgPtr->interp, 
			   sinkPtr->updateVar, data, flags) == NULL) {
		Tk_BackgroundError(bgPtr->interp);
	    }
	}
	if (!bgPtr->keepTrailingNewLine) {
	    UnfixNewline(sinkPtr);
	}
    }
    if (status >= 0) {
	return TCL_RETURN;
    }
    if (status == READ_ERROR) {
	Tcl_AppendResult(bgPtr->interp, "can't read data from sink: ",
	    Tcl_PosixError(bgPtr->interp), (char *)NULL);
	Tk_BackgroundError(bgPtr->interp);
    }
    /*   
     * Either EOF or an error has occurred.  In either case,
     * close the sink.  
     */
    CloseSink(bgPtr, sinkPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * CreateSinkHandler --
 *
 *	Creates a file handler for the given sink.  The file
 *	descriptor is also set for non-blocking I/O.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The memory allocated to the BackgroundInfo structure released.
 *
 *---------------------------------------------------------------------- 
 */
static int
CreateSinkHandler(bgPtr, sinkPtr, proc)
    BackgroundInfo *bgPtr;
    Sink *sinkPtr;
    Tcl_FileProc *proc;
{
    int flags;

    flags = fcntl(sinkPtr->fd, F_GETFL);
#ifdef O_NONBLOCK
    flags |= O_NONBLOCK;
#else
    flags |= O_NDELAY;
#endif
    if (fcntl(sinkPtr->fd, F_SETFL, flags) < 0) {
	Tcl_AppendResult(bgPtr->interp, "can't set file descriptor ", 
		Blt_Int(sinkPtr->fd), " to non-blocking:", 
		Tcl_PosixError(bgPtr->interp), (char *)NULL);
	return TCL_ERROR;
    }
#ifdef FILEHANDLER_USES_TCLFILES
    sinkPtr->file = Tcl_GetFile((ClientData)sinkPtr->fd, TCL_UNIX_FD);
    Tcl_CreateFileHandler(sinkPtr->file, TK_READABLE, proc, (ClientData)bgPtr);
#else
    Tcl_CreateFileHandler(sinkPtr->fd, TK_READABLE, proc, (ClientData)bgPtr);
#endif /* FILEHANDLER_USES_TCLFILES */
    return TCL_OK;
}


static void
DisableTriggers(bgPtr)
    BackgroundInfo *bgPtr;	/* Background info record. */
{

    Tcl_UntraceVar(bgPtr->interp, bgPtr->statVarId, TRACE_FLAGS, VariableProc,
	(ClientData)bgPtr);

    CloseSink(bgPtr, &(bgPtr->outputSink));
    CloseSink(bgPtr, &(bgPtr->errorSink));

    if (bgPtr->timerToken != (Tk_TimerToken) 0) {
	Tk_DeleteTimerHandler(bgPtr->timerToken);
	bgPtr->timerToken = 0;
    }
    if (bgPtr->donePtr != NULL) {
	*bgPtr->donePtr = TRUE;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyBackgroundInfo --
 *
 * 	This procedure is invoked by Tk_EventuallyFree or Tk_Release
 * 	to clean up the internal structure (BackgroundInfo) at a safe
 * 	time (when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The memory allocated to the BackgroundInfo structure released.
 *
 *---------------------------------------------------------------------- 
 */
/* ARGSUSED */
static void
DestroyBackgroundInfo(bgPtr)
    BackgroundInfo *bgPtr;	/* Background info record. */
{
    DisableTriggers(bgPtr);
    if ((bgPtr->pidArr != NULL) && (bgPtr->signalNum > 0)) {
	register int i;

	for (i = 0; i < bgPtr->numPids; i++) {
	    kill(bgPtr->pidArr[i], (int)bgPtr->signalNum);
	}
    }
    ResetSinkBuffer(&(bgPtr->errorSink));
    ResetSinkBuffer(&(bgPtr->outputSink));
    Tk_FreeOptions(configSpecs, (char *)bgPtr, Tk_Display(bgPtr->tkwin), 0);
    if (bgPtr->pidArr != NULL) {
	if (bgPtr->numPids > 0) {
#if (TCL_MAJOR_VERSION >= 8)
	    Tcl_DetachPids(bgPtr->numPids, (Tcl_Pid *)bgPtr->pidArr);
#else
	    Tcl_DetachPids(bgPtr->numPids, (int *)bgPtr->pidArr);
#endif
	}
	free((char *)bgPtr->pidArr);
    }
    Blt_FreeUid(bgPtr->statVarId);
    Tcl_ReapDetachedProcs();
    free((char *)bgPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * VariableProc --
 *
 *	This procedure cleans up the BackgroundInfo data structure
 *	associated with the detached subprocesses.  It is called when
 *	the variable associated with UNIX subprocesses has been
 *	overwritten.  This usually occurs when the subprocesses have
 *	completed or an error was detected.  However, it may be used
 *	to terminate the detached processes from the Tcl program by
 *	setting the associated variable.
 *
 * Results:
 *	Always returns NULL.  Only called from a variable trace.
 *
 * Side effects:
 *	The output descriptor is closed and the variable trace is
 *	deleted.  In addition, the subprocesses are signaled for
 *	termination.
 *
 * ----------------------------------------------------------------------
 */
/* ARGSUSED */
static char *
VariableProc(clientData, interp, part1, part2, flags)
    ClientData clientData;	/* File output information. */
    Tcl_Interp *interp;
    char *part1, *part2;	/* Unused */
    int flags;
{
    if (flags & TRACE_FLAGS) {
	BackgroundInfo *bgPtr = (BackgroundInfo *)clientData;

	DisableTriggers(bgPtr);
	if (bgPtr->detached) {
	    DestroyBackgroundInfo(bgPtr);
	}
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * TimerProc --
 *
 *	This is a timer handler procedure which gets called
 *	periodically to reap any of the sub-processes if they have
 *	terminated.  After the last process has terminated, the
 *	contents of standard output are stored
 *	in the output variable, which triggers the cleanup proc (using
 *	a variable trace). The status the last process to exit is
 *	written to the status variable.
 *
 * Results:
 *	None.  Called from the Tk event loop.
 *
 * Side effects:
 *	Many. The contents of pidArr is shifted, leaving only those
 *	sub-processes which have not yet terminated.  If there are
 *	still subprocesses left, this procedure is placed in the timer
 *	queue again. Otherwise the output and possibly the status
 *	variables are updated.  The former triggers the cleanup
 *	routine which will destroy the information and resources
 *	associated with these background processes.
 *
 *---------------------------------------------------------------------- 
 */
static void
TimerProc(clientData)
    ClientData clientData;
{
    BackgroundInfo *bgPtr = (BackgroundInfo *)clientData;
    register int i;
    int code;
    int lastPid;
    WAIT_STATUS_TYPE waitStatus, lastStatus;
    int numLeft;		/* Number of processes still not reaped */
#ifdef notdef
    fprintf(stderr, "in TimerProc(numPids=%d)\n", bgPtr->numPids);
#endif

    lastPid = -1;
    *((int *)&waitStatus) = 0;
    *((int *)&lastStatus) = 0;
    numLeft = 0;
    for (i = 0; i < bgPtr->numPids; i++) {
	code = waitpid(bgPtr->pidArr[i], (int *)&waitStatus, WNOHANG);
	if (code == 0) {	/*  Process has not terminated yet */
	    if (numLeft < i) {
		bgPtr->pidArr[numLeft] = bgPtr->pidArr[i];
	    }
	    numLeft++;		/* Count the number of processes left */
	} else if (code > 0) {
	    /*
	     * Save the status information associated with the subprocess.
	     * We'll use it only if this is the last subprocess to be reaped.
	     */
	    lastStatus = waitStatus;
	    lastPid = bgPtr->pidArr[i];
	}
    }
    bgPtr->numPids = numLeft;

    if (numLeft > 0) {
	/* Keep polling for the status of the children that are left */
	bgPtr->timerToken = Tk_CreateTimerHandler(bgPtr->interval, TimerProc,
	    (ClientData)bgPtr);
    } else {
	char *merged;		/* List formed by merging msgArr components */
	char codeStr[20];
	char pidStr[20];
	char textStr[200];
	int exitCode;
	enum StatusComponents {
	    STATUS_TOKEN, STATUS_PID, STATUS_EXITCODE, STATUS_TEXT
	};
	char *msgArr[4];
	char *result;

	/*
	 * Set the status variable with the status of the last process
	 * reaped.  The status is a list of an error token, the exit
	 * status, and a message.
	 */
	exitCode = WEXITSTATUS(lastStatus);
	if (WIFEXITED(lastStatus)) {
	    msgArr[STATUS_TOKEN] = "CHILDSTATUS";
	    msgArr[STATUS_TEXT] = "child completed normally";
	} else if (WIFSIGNALED(lastStatus)) {
	    msgArr[STATUS_TOKEN] = "CHILDKILLED";
	    msgArr[STATUS_TEXT] = Tcl_SignalMsg((int)(WTERMSIG(lastStatus)));
	    exitCode = -1;
	} else if (WIFSTOPPED(lastStatus)) {
	    msgArr[STATUS_TOKEN] = "CHILDSUSP";
	    msgArr[STATUS_TEXT] = Tcl_SignalMsg((int)(WSTOPSIG(lastStatus)));
	    exitCode = -1;
	} else {
	    msgArr[STATUS_TOKEN] = "UNKNOWN";
	    sprintf(textStr, "child completed with unknown status 0x%x",
		*((int *)&lastStatus));
	    msgArr[STATUS_TEXT] = textStr;
	}
	sprintf(pidStr, "%d", lastPid);
	sprintf(codeStr, "%d", exitCode);
	msgArr[STATUS_PID] = pidStr;
	msgArr[STATUS_EXITCODE] = codeStr;
	if (bgPtr->exitCodePtr != NULL) {
	    *bgPtr->exitCodePtr = exitCode;
	}
	merged = Tcl_Merge(4, msgArr);

	DisableTriggers(bgPtr);
	result = Tcl_SetVar(bgPtr->interp, bgPtr->statVarId, merged,
	    TCL_GLOBAL_ONLY);
	free(merged);
	if (result == NULL) {
	    Tk_BackgroundError(bgPtr->interp);
	}
	if (bgPtr->detached) {
	    DestroyBackgroundInfo(bgPtr);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * OutputEventProc --
 *
 *	This procedure is called when output from the detached command
 *	is available.  The output is read and saved in a buffer in the
 *	BackgroundInfo structure.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Data is stored in the buffer.  This character array may
 *	be increased as more space is required to contain the output
 *	of the command.
 *
 *---------------------------------------------------------------------- 
 */
/* ARGSUSED */
static void
OutputEventProc(clientData, mask)
    ClientData clientData;	/* File output information. */
    int mask;			/* Not used. */
{
    BackgroundInfo *bgPtr = (BackgroundInfo *)clientData;

    if (CollectData(bgPtr, &(bgPtr->outputSink)) != TCL_RETURN) {
	/* 
	 * We're here if we've seen EOF or an error has occurred.  In
	 * either case, set up a timer handler to periodically poll
	 * for exit status of each process.  Initially check at the
	 * next idle interval.  
	 */
	bgPtr->timerToken = Tk_CreateTimerHandler(0, TimerProc, 
		  (ClientData)bgPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ErrorEventProc --
 *
 *	This procedure is called when error from the detached command
 *	is available.  The error is read and saved in a buffer in the
 *	BackgroundInfo structure.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Data is stored in the buffer.  This character array may
 *	be increased as more space is required to contain the stderr
 *	of the command.
 *
 *---------------------------------------------------------------------- 
 */
/* ARGSUSED */
static void
ErrorEventProc(clientData, mask)
    ClientData clientData;	/* File output information. */
    int mask;			/* Not used. */
{
    BackgroundInfo *bgPtr = (BackgroundInfo *)clientData;

    CollectData(bgPtr, &(bgPtr->errorSink));
}

/*
 *----------------------------------------------------------------------
 *
 * BgexecCmd --
 *
 *	This procedure is invoked to process the "bgexec" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */
/* ARGSUSED */
static int
BgexecCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window of interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    int *errFdPtr;
    int numPids, *pidPtr;
    char *lastArg;
    BackgroundInfo *bgPtr;
    int i;
    int detached;

    if (argc < 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    " varName ?options? command ?arg...?\"", (char *)NULL);
	return TCL_ERROR;
    }
    /*
     * Check if the command is to be run detached (indicated a '&' as
     * the last argument of the command)
     */
    lastArg = argv[argc - 1];
    detached = ((lastArg[0] == '&') && (lastArg[1] == '\0'));
    if (detached) {
	argc--;
	argv[argc] = NULL;	/* Remove the '&' argument */
    }
    for (i = 2; i < argc; i += 2) {
	/* Count the number of option-value pairs */
	if ((argv[i][0] != '-') || (argv[i][1] == '-')) {
	    break;		/* Not an option or "--" */
	}
    }
    if (i > argc) {
	i = argc;
    }
    bgPtr = (BackgroundInfo *)calloc(1, sizeof(BackgroundInfo));
    assert(bgPtr);

    /* Initialize the background information record */
    bgPtr->interp = interp;
    bgPtr->tkwin = (Tk_Window)clientData;	/* Main window of interpreter */
    bgPtr->signalNum = SIGKILL;
    bgPtr->numPids = -1;
    bgPtr->interval = 1000;
    bgPtr->detached = detached;
    bgPtr->statVarId = Blt_GetUid(argv[1]);
    InitSink(&(bgPtr->outputSink));
    InitSink(&(bgPtr->errorSink));

    if (Tk_ConfigureWidget(interp, bgPtr->tkwin, configSpecs, i - 2, argv + 2,
	    (char *)bgPtr, 0) != TCL_OK) {
	free((char *)bgPtr);
	return TCL_ERROR;
    }
    if (argc <= i) {
	Tcl_AppendResult(interp, "missing command to execute: should be \"",
	    argv[0], " varName ?options? command ?arg...?\"", (char *)NULL);
	/* Try to clean up any detached processes */
	Tcl_ReapDetachedProcs();
	Tk_FreeOptions(configSpecs, (char *)bgPtr, Tk_Display(bgPtr->tkwin), 0);
	free((char *)bgPtr);
	return TCL_ERROR;
    }
    if (argv[i][0] == '-') {
	i++;			/* If the last option was "--", skip it. */
    }
    /*
     * Put a trace on the status variable.  The user can prematurely terminate
     * the pipeline of subprocesses by simply setting it.
     */
    Tcl_TraceVar(interp, bgPtr->statVarId, TRACE_FLAGS, VariableProc,
	(ClientData)bgPtr);

    errFdPtr = (int *)NULL;
    if ((bgPtr->errorSink.doneVar != NULL) || 
	(bgPtr->errorSink.updateVar != NULL)) {
	errFdPtr = &(bgPtr->errorSink.fd);
    }
    numPids = Blt_CreatePipeline(interp, argc - i, argv + i, &pidPtr,
	(int *)NULL, &(bgPtr->outputSink.fd), errFdPtr);
    if (numPids < 0) {
	/* Error message put into interp->result by Blt_CreatePipeline */
	goto error;
    }
    bgPtr->pidArr = pidPtr;
    bgPtr->numPids = numPids;

    if (bgPtr->outputSink.fd == -1) {

	/*
	 * If output has been redirected, start polling immediately
	 * for the exit status of each process.  Normally, this is
	 * done only after stdout output has been closed by the last
	 * process.  The default polling interval is every 1 second.
	 */

	bgPtr->timerToken = Tk_CreateTimerHandler(bgPtr->interval, TimerProc,
	    (ClientData)bgPtr);

    } else if (CreateSinkHandler(bgPtr, &(bgPtr->outputSink), 
		    OutputEventProc) != TCL_OK) {
	goto error;
    }
    if ((bgPtr->errorSink.fd != -1)  && 
	(CreateSinkHandler(bgPtr, &(bgPtr->errorSink), 
	       ErrorEventProc) != TCL_OK)) {
	goto error;
    }
    if (bgPtr->detached) {
	/* Return a list of the child process ids */
	for (i = 0; i < numPids; i++) {
	    Tcl_AppendElement(interp, Blt_Int(bgPtr->pidArr[i]));
	}
	return TCL_OK;
    } else {
	int exitCode;
	int done;

	bgPtr->exitCodePtr = &exitCode;
	bgPtr->donePtr = &done;

	exitCode = done = 0;
	while (!done) {
	    Tk_DoOneEvent(0);
	}
	DisableTriggers(bgPtr);
	if (exitCode == 0) {
	    /* Return the output of the command */
	    Tcl_SetResult(interp, GetSinkData(&(bgPtr->outputSink)), 
			  TCL_VOLATILE);
	}
	DestroyBackgroundInfo(bgPtr);
	if (exitCode != 0) {
	    Tcl_AppendResult(interp, "child process exited abnormally",
		(char *)NULL);
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
  error:
    DisableTriggers(bgPtr);
    DestroyBackgroundInfo(bgPtr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_BgexecInit --
 *
 *	This procedure is invoked to initialize the "bgexec" Tcl
 *	command.  See the user documentation for details on what it
 *	does.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	See the user documentation.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_BgexecInit(interp)
    Tcl_Interp *interp;
{
    static Blt_CmdSpec cmdSpec =
    {"bgexec", BgexecCmd,};

    if (Blt_InitCmd(interp, "blt", &cmdSpec) == NULL) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

#endif /* NO_BGEXEC */
