/*
 * $Id: pst-helpers.c,v 1.17 2004/03/29 15:20:50 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <Python.h>
#include <streamtuner.h>
#include "gettext.h"
#include "pst-helpers.h"
#include "pst-stream.h"
#include "pst-image.h"

/*** function declarations ***************************************************/

static gboolean pst_streams_check (PyObject *streams, GError **err);

/*** implementation **********************************************************/

/*
 * Convert a sequence of Python strings to a GSList of newly-allocated
 * C strings.
 *
 * Return true if succeeds, false otherwise (in such case, a Python
 * exception will be raised).
 */
gboolean
pst_strings_as_gslist (PyObject *strings, GSList **gslist)
{
  int i;
  int len;
  GSList *l;

  g_return_val_if_fail(strings != NULL, FALSE);
  g_return_val_if_fail(gslist != NULL, FALSE);

  if (! PySequence_Check(strings))
    {
      PyErr_SetString(PyExc_TypeError, _("expected sequence"));
      return FALSE;
    }

  *gslist = NULL;
  len = PySequence_Length(strings);

  for (i = 0; i < len; i++)
    {
      PyObject *item;
      const char *str;
      
      item = PySequence_GetItem(strings, i);
      Py_DECREF(item);

      str = PyString_AsString(item);
      if (! str)		/* exception raised by PyString_AsString() */
	goto exception;

      *gslist = g_slist_append(*gslist, g_strdup(str));
    }
  
  return TRUE;

 exception:
  for (l = *gslist; l; l = l->next)
    g_free(l->data);
  g_slist_free(*gslist);

  return FALSE;
}

/*
 * Convert a GSList of C strings to a tuple of Python strings.
 */
PyObject *
pst_strings_from_gslist (GSList *gslist)
{
  PyObject *tuple;
  GSList *l;
  int i = 0;

  tuple = PyTuple_New(g_slist_length(gslist));
  for (l = gslist; l; l = l->next)
    {
      PyObject *str;

      str = PyString_FromString(l->data);
      PyTuple_SetItem(tuple, i++, str);
    }

  return tuple;
}

PyObject *
pst_object_from_gvalue (const GValue *gvalue)
{
  PyObject *object;

  g_return_val_if_fail(G_IS_VALUE(gvalue), NULL);

  if (G_VALUE_HOLDS_INT(gvalue)) /* also covers G_TYPE_BOOLEAN */
    object = PyInt_FromLong(g_value_get_int(gvalue));
  else if (G_VALUE_HOLDS_STRING(gvalue))
    {
      const char *str = g_value_get_string(gvalue);
      object = PyString_FromString(str ? str : "");
    }
  else if (G_VALUE_HOLDS(gvalue, G_TYPE_VALUE_ARRAY))
    {
      GValueArray *value_array = g_value_get_boxed(gvalue);
      int i;

      object = PyList_New(value_array->n_values);
      for (i = 0; i < value_array->n_values; i++)
	{
	  GValue *elem = g_value_array_get_nth(value_array, i);
	  PyObject *pelem = pst_object_from_gvalue(elem);

	  PyList_SetItem(object, i, pelem);
	}
    }
  else if (G_VALUE_HOLDS(gvalue, GDK_TYPE_PIXBUF))
    object = (PyObject *) PSTImage_New(g_value_get_object(gvalue));
  else
    g_return_val_if_reached(NULL);

  return object;
}

void
pst_object_as_gvalue (PyObject *object, GValue *gvalue)
{
  g_return_if_fail(object != NULL);
  g_return_if_fail(gvalue != NULL);

  if (PyInt_Check(object))
    {
      g_value_init(gvalue, G_TYPE_INT);
      g_value_set_int(gvalue, PyInt_AsLong(object));
    }
  else if (PyString_Check(object))
    {
      char *str = PyString_AsString(object);

      g_value_init(gvalue, G_TYPE_STRING);
      g_value_set_string(gvalue, *str ? str : NULL);
    }
  else if (PySequence_Check(object))
    {
      GValueArray *value_array;
      int size;
      int i;

      size = PySequence_Length(object);
      value_array = g_value_array_new(size);

      for (i = 0; i < size; i++)
	{
	  PyObject *pelem = PySequence_GetItem(object, i);
	  GValue elem = { 0, };

	  pst_object_as_gvalue(pelem, &elem);
	  Py_DECREF(pelem);

	  g_value_array_append(value_array, &elem);
	  g_value_unset(&elem);
	}

      g_value_init(gvalue, G_TYPE_VALUE_ARRAY);
      g_value_set_boxed_take_ownership(gvalue, value_array);
    }
  else if (PyObject_IsInstance(object, (PyObject *) &PSTImage_Type))
    {
      g_value_init(gvalue, GDK_TYPE_PIXBUF);
      g_value_set_object(gvalue, ((PSTImage *) object)->pixbuf);
    }
  else
    {
      PyObject *pstr;

      g_value_init(gvalue, G_TYPE_STRING);

      pstr = PyObject_Str(object);
      g_value_set_string(gvalue, PyString_AsString(pstr));
      Py_DECREF(pstr);
    }
}

PyObject *
pst_none (void)
{
  Py_INCREF(Py_None);
  return Py_None;
}

PyObject *
pst_string_getter (const char *str)
{
  return str ? PyString_FromString(str) : pst_none();
}

int
pst_string_setter (char **ptr, PyObject *value)
{
  const char *str = NULL;

  g_return_val_if_fail(ptr != NULL, -1);

  if (value)
    {
      str = PyString_AsString(value);
      if (! str)
	return -1;
    }
  
  g_free(*ptr);
  *ptr = g_strdup(str);

  return 0;
}

static gboolean
pst_streams_check (PyObject *streams, GError **err)
{
  int i;
  int len;

  g_return_val_if_fail(streams != NULL, FALSE);

  len = PySequence_Length(streams);
  for (i = 0; i < len; i++)
    {
      PyObject *item;

      item = PySequence_GetItem(streams, i);
      Py_DECREF(item);

      if (! PyObject_IsInstance(item, (PyObject *) &PSTStream_Type))
	{
	  g_set_error(err, 0, 0, _("element #%i is not a ST.Stream object"), i + 1);
	  return FALSE;
	}
    }

  return TRUE;
}

gboolean
pst_streams_as_glist (PyObject *streams, GList **list, GError **err)
{
  int i;
  int len;

  g_return_val_if_fail(streams != NULL, FALSE);
  g_return_val_if_fail(list != NULL, FALSE);

  if (! pst_streams_check(streams, err))
    return FALSE;

  *list = NULL;
  len = PySequence_Length(streams);

  for (i = 0; i < len; i++)
    {
      PyObject *item;

      item = PySequence_GetItem(streams, i);
      Py_DECREF(item);

      *list = g_list_append(*list, pst_stream_copy(((PSTStream *) item)->stream));
    }

  return TRUE;
}

GHashTable *
pst_streams_mapping_as_ghashtable (PyObject *streams, GError **err)
{
  int i;
  int len;
  PyObject *items;
  PyObject *item;
  PyObject *key;
  PyObject *value;
  GHashTable *hash = NULL;

  g_return_val_if_fail(streams != NULL, NULL);

  items = PyMapping_Items(streams);
  g_return_val_if_fail(items != NULL, NULL);

  /* first pass, check */

  len = PySequence_Length(items);
  for (i = 0; i < len; i++)
    {
      GError *tmp_err = NULL;

      item = PySequence_GetItem(items, i);
      g_return_val_if_fail(item != NULL, NULL);
      Py_DECREF(item);

      /* check validity of key */

      key = PySequence_GetItem(item, 0);
      g_return_val_if_fail(key != NULL, NULL);
      Py_DECREF(key);

      if (! PyString_Check(key))
	{
	  g_set_error(err, 0, 0, _("key #%i is not a string"), i + 1);
	  goto end;
	}

      /* check validity of value */

      value = PySequence_GetItem(item, 1);
      g_return_val_if_fail(value != NULL, NULL);
      Py_DECREF(value);

      if (! PySequence_Check(value))
	{
	  g_set_error(err, 0, 0, _("value #%i is not a sequence"), i + 1);
	  goto end;
	}

      if (! pst_streams_check(value, &tmp_err))
	{
	  g_set_error(err, 0, 0, _("in value #%i: %s"), i + 1, tmp_err->message);
	  g_error_free(tmp_err);
	  goto end;
	}
    }

  /* second pass, convert */
  
  hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
  
  for (i = 0; i < len; i++)
    {
      GList *list;

      item = PySequence_GetItem(items, i);
      g_return_val_if_fail(item != NULL, NULL);
      Py_DECREF(item);

      key = PySequence_GetItem(item, 0);
      g_return_val_if_fail(key != NULL, NULL);
      Py_DECREF(key);

      value = PySequence_GetItem(item, 1);
      g_return_val_if_fail(value != NULL, NULL);
      Py_DECREF(value);

      pst_streams_as_glist(value, &list, NULL);
      g_hash_table_insert(hash, g_strdup(PyString_AsString(key)), list);
    }

 end:
  Py_DECREF(items);

  return hash;
}

void
pst_set_error (GError **err)
{
  PyErr_Print();
  g_set_error(err, 0, 0, _("a Python exception has occurred (see standard error output for the full Python traceback)"));
}
