/* 
 * $Id: local.c,v 1.69 2004/03/29 11:50:00 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include "local.h"
#ifdef WITH_ID3
#include "local-id3.h"
#endif
#ifdef WITH_VC
#include "local-vc.h"
#endif
#include "art/icon.h"
#include "gettext.h"

/*** cpp *********************************************************************/

#define COPYRIGHT		"Copyright \302\251 2002, 2003, 2004 Jean-Yves Lefort"

/*** variable declarations ***************************************************/

static const char *local_root;

/*** function declarations ***************************************************/

static LocalStream *stream_new_cb (gpointer data);
static void stream_field_get_cb (LocalStream *stream,
				 STHandlerField *field,
				 GValue *value,
				 gpointer data);
static void stream_field_set_cb (LocalStream *stream,
				 STHandlerField *field,
				 const GValue *value,
				 gpointer data);
static void stream_stock_field_get_cb (LocalStream *stream,
				       STHandlerStockField stock_field,
				       GValue *value,
				       gpointer data);
static gboolean stream_modify_cb (LocalStream *stream,
				  GSList *fields,
				  GSList *values,
				  gpointer data,
				  GError **err);
static gboolean stream_delete_cb (LocalStream *stream,
				  gpointer data,
				  GError **err);
static void stream_free_cb (LocalStream *stream, gpointer data);

static gboolean stream_rename (LocalStream *stream,
			       const GValue *new_filename,
			       GError **err);

static gboolean stream_tune_in_multiple_cb (GSList *streams,
					    gpointer data,
					    GError **err);
static gboolean stream_browse_cb (LocalStream *stream,
				  gpointer data,
				  GError **err);

static gboolean refresh_cb (STCategory *category,
			    GNode **categories,
			    GList **streams,
			    gpointer data,
			    GError **err);

static gboolean refresh_categories (GNode *root, GError **err);
static gboolean refresh_streams (STCategory *category,
				 GList **streams,
				 GError **err);

static LocalType get_file_type (const char *filename);

/*** implementation **********************************************************/

static LocalStream *
stream_new_cb (gpointer data)
{
  return g_new0(LocalStream, 1);
}

static void
stream_field_get_cb (LocalStream *stream,
		     STHandlerField *field,
		     GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case LOCAL_FIELD_PATHNAME:
      g_value_set_string(value, stream->pathname);
      break;

    case LOCAL_FIELD_TYPE:
      g_value_set_int(value, stream->type);
      break;

    case LOCAL_FIELD_FILENAME:
      g_value_set_string(value, stream->filename);
      break;

    case LOCAL_FIELD_TITLE:
      g_value_set_string(value, stream->title);
      break;

    case LOCAL_FIELD_ARTIST:
      g_value_set_string(value, stream->artist);
      break;

    case LOCAL_FIELD_ALBUM:
      g_value_set_string(value, stream->album);
      break;
      
    case LOCAL_FIELD_YEAR:
      g_value_set_string(value, stream->year);
      break;

    case LOCAL_FIELD_GENRE:
      g_value_set_string(value, stream->genre);
      break;

    case LOCAL_FIELD_COMMENT:
      g_value_set_string(value, stream->comment);
      break;

    case LOCAL_FIELD_DURATION:
      g_value_set_string(value, stream->duration);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_field_set_cb (LocalStream *stream,
		     STHandlerField *field,
		     const GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case LOCAL_FIELD_PATHNAME:
      stream->pathname = g_value_dup_string(value);
      break;

    case LOCAL_FIELD_TYPE:
      stream->type = g_value_get_int(value);
      break;

    case LOCAL_FIELD_FILENAME:
      stream->filename = g_value_dup_string(value);
      break;

    case LOCAL_FIELD_TITLE:
      stream->title = g_value_dup_string(value);
      break;

    case LOCAL_FIELD_ARTIST:
      stream->artist = g_value_dup_string(value);
      break;

    case LOCAL_FIELD_ALBUM:
      stream->album = g_value_dup_string(value);
      break;
      
    case LOCAL_FIELD_YEAR:
      stream->year = g_value_dup_string(value);
      break;

    case LOCAL_FIELD_GENRE:
      stream->genre = g_value_dup_string(value);
      break;

    case LOCAL_FIELD_COMMENT:
      stream->comment = g_value_dup_string(value);
      break;

    case LOCAL_FIELD_DURATION:
      stream->duration = g_value_dup_string(value);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_stock_field_get_cb (LocalStream *stream,
			   STHandlerStockField stock_field,
			   GValue *value,
			   gpointer data)
{
  switch (stock_field)
    {
    case ST_HANDLER_STOCK_FIELD_NAME:
      {
	char *name;

	if (stream->artist && stream->title)
	  name = g_strdup_printf("%s - %s", stream->artist, stream->title);
	else if (stream->title)
	  name = g_strdup(stream->title);
	else
	  name = g_strdup(stream->filename);

	g_value_set_string(value, name);
	g_free(name);

	break;
      }

    case ST_HANDLER_STOCK_FIELD_GENRE:
      g_value_set_string(value, stream->genre);
      break;
    }
}

static gboolean
stream_modify_cb (LocalStream *stream,
		  GSList *fields,
		  GSList *values,
		  gpointer data,
		  GError **err)
{
  GSList *f = fields;
  GSList *v = values;
  gboolean modify_file = FALSE;

  while (f && v)
    {
      STHandlerField *field = f->data;
      const GValue *value = v->data;

      switch (field->id)
	{
	case LOCAL_FIELD_FILENAME:
	  if (! stream_rename(stream, value, err))
	    return FALSE;
	  break;

	case LOCAL_FIELD_TITLE:
	case LOCAL_FIELD_ARTIST:
	case LOCAL_FIELD_ALBUM:
	case LOCAL_FIELD_YEAR:
	case LOCAL_FIELD_GENRE:
	case LOCAL_FIELD_COMMENT:
	  modify_file = TRUE;
	  break;

	default:
	  g_assert_not_reached();
	}

      f = f->next;
      v = v->next;
    }

  if (modify_file)
    {
      switch (stream->type)
	{
	case LOCAL_TYPE_MP3:
#ifdef WITH_ID3
	  if (! local_id3_modify(stream, fields, values, err))
	    return FALSE;
#else
	  g_set_error(err, 0, 0, _("ID3 tag support is disabled"));
	  return FALSE;
#endif /* WITH_ID3 */
	  break;

	case LOCAL_TYPE_OGG:
#ifdef WITH_VC
	  if (! local_vc_modify(stream, fields, values, err))
	    return FALSE;
#else
	  g_set_error(err, 0, 0, _("Vorbis comment support is disabled"));
	  return FALSE;
#endif /* WITH_VC */
	  break;
	  
	case LOCAL_TYPE_PLAYLIST:
	  g_set_error(err, 0, 0, _("playlists can't be modified"));
	  return FALSE;
	  
	default:
	  g_return_val_if_reached(FALSE);
	}
    }

  return TRUE;
}

static gboolean
stream_delete_cb (LocalStream *stream, gpointer data, GError **err)
{
  if (unlink(stream->pathname) < 0)
    {
      g_set_error(err, 0, 0, "%s", g_strerror(errno));
      return FALSE;
    }
  else
    return TRUE;
}

static void
stream_free_cb (LocalStream *stream, gpointer data)
{
  g_free(stream->pathname);
  g_free(stream->filename);
  g_free(stream->title);
  g_free(stream->artist);
  g_free(stream->album);
  g_free(stream->year);
  g_free(stream->genre);
  g_free(stream->comment);
  g_free(stream->duration);

  st_stream_free((STStream *) stream);
}

static gboolean
stream_rename (LocalStream *stream, const GValue *new_filename, GError **err)
{
  GError *tmp_err = NULL;
  char *filename;
  char *directory;
  char *new_pathname;

  g_return_val_if_fail(stream != NULL, FALSE);
  g_return_val_if_fail(G_IS_VALUE(new_filename), FALSE);

  filename = g_filename_from_utf8(g_value_get_string(new_filename), -1, NULL, NULL, &tmp_err);
  if (! filename)
    {
      g_set_error(err, 0, 0, _("unable to convert filename from UTF-8: %s"), tmp_err->message);
      g_error_free(tmp_err);

      return FALSE;
    }

  directory = g_path_get_dirname(stream->pathname);
  new_pathname = g_build_filename(directory, filename, NULL);
  g_free(directory);

  if (g_file_test(new_pathname, G_FILE_TEST_EXISTS))
    {
      g_set_error(err, 0, 0, _("target file already exists"));
      g_free(filename);
      g_free(new_pathname);
      
      return FALSE;
    }

  if (rename(stream->pathname, new_pathname) < 0)
    {
      g_set_error(err, 0, 0, "%s", g_strerror(errno));
      g_free(filename);
      g_free(new_pathname);

      return FALSE;
    }

  /* success */

  stream->pathname = new_pathname;
  stream->filename = g_value_dup_string(new_filename);

  return TRUE;
}

static gboolean
stream_tune_in_multiple_cb (GSList *streams, gpointer data, GError **err)
{
  char *m3uname;
  GSList *filenames = NULL;
  GSList *l;
  gboolean status;

  /* create a list of filenames from STREAMS */

  for (l = streams; l; l = l->next)
    {
      LocalStream *stream = l->data;
      filenames = g_slist_append(filenames, stream->pathname);
    }

  /* write the .m3u */

  m3uname = st_m3u_mktemp("streamtuner.local.XXXXXX", filenames, err);
  g_slist_free(filenames);

  if (! m3uname)
    return FALSE;

  /* open the .m3u */

  status = st_action_run("play-m3u", m3uname, err);
  g_free(m3uname);

  return status;
}

static gboolean
stream_browse_cb (LocalStream *stream, gpointer data, GError **err)
{
  char *url;
  char *s;
  gboolean status;

  if (stream->album)
    url = g_strconcat("http://www.allmusic.com/cg/amg.dll?p=amg&opt1=2&sql=", stream->album, NULL);
  else if (stream->title)
    url = g_strconcat("http://www.allmusic.com/cg/amg.dll?p=amg&opt1=3&sql=", stream->title, NULL);
  else if (stream->artist)
    url = g_strconcat("http://www.allmusic.com/cg/amg.dll?p=amg&opt1=1&sql=", stream->artist, NULL);
  else
    {
      g_set_error(err, 0, 0, _("file has no album, title or artist information"));
      return FALSE;
    }

  /* allmusic.com needs this */
  for (s = url; *s; s++)
    if (*s == ' ')
      *s = '|';

  status = st_action_run("view-web", url, err);
  g_free(url);
  
  return status;
}

static gboolean
refresh_cb (STCategory *category,
	    GNode **categories,
	    GList **streams,
	    gpointer data,
	    GError **err)
{
  *categories = g_node_new(NULL);

  if (! refresh_categories(*categories, err))
    return FALSE;

  if (! refresh_streams(category, streams, err))
    return FALSE;

  return TRUE;
}

static gboolean
refresh_categories (GNode *root, GError **err)
{
  GDir *dir;
  char *dirname;
  const char *filename;
  gboolean status = TRUE;
  GError *tmp_err = NULL;

  g_return_val_if_fail(root != NULL, FALSE);

  dirname = root->data
    ? g_build_filename(local_root, ((STCategory *) root->data)->url_postfix, NULL)
    : g_strdup(local_root);
  
  dir = g_dir_open(dirname, 0, &tmp_err);
  if (! dir)
    {
      g_set_error(err, 0, 0, _("unable to open directory %s: %s"), dirname, tmp_err->message);
      g_error_free(tmp_err);

      status = FALSE;
      goto end;
    }
  
  while ((filename = g_dir_read_name(dir)))
    {
      GNode *node;
      char *pathname;
      
      if (st_is_aborted())
	{
	  status = FALSE;
	  goto end;
	}
      
      if (filename[0] == '.')
	continue;
      
      pathname = g_build_filename(dirname, filename, NULL);
      
      if (g_file_test(pathname, G_FILE_TEST_IS_DIR))
	{
	  STCategory *category;
	
	  category = st_category_new();
	
	  category->name = root->data
	    ? g_build_filename(((STCategory *) root->data)->url_postfix, filename, NULL)
	    : g_strdup(filename);

	  category->label = g_filename_to_utf8(filename, -1, NULL, NULL, &tmp_err);
	  if (! category->label)
	    {
	      st_notice(_("Local: %s: unable to convert directory name to UTF-8: %s"), pathname, tmp_err->message);
	      g_clear_error(&tmp_err);
	    }
	
	  category->url_postfix = g_strdup(category->name);
	
	  node = g_node_append_data(root, category);
	  if (! refresh_categories(node, err))
	    {
	      status = FALSE;
	      goto end;
	    }
	}
      
      g_free(pathname);
    }

 end:
  g_dir_close(dir);
  g_free(dirname);
  
  return status;
}

static gboolean
refresh_streams (STCategory *category, GList **streams, GError **err)
{
  GDir *dir;
  char *dirname;
  const char *filename;
  gboolean status = TRUE;
  GError *tmp_err = NULL;

  g_return_val_if_fail(category != NULL, FALSE);
  g_return_val_if_fail(streams != NULL, FALSE);

  dirname = category->url_postfix
    ? g_build_filename(local_root, category->url_postfix, NULL)
    : g_strdup(local_root);

  dir = g_dir_open(dirname, 0, &tmp_err);
  if (! dir)
    {
      g_set_error(err, 0, 0, _("unable to open directory %s: %s"), dirname, tmp_err->message);
      g_error_free(tmp_err);

      status = FALSE;
      goto end;
    }

  while ((filename = g_dir_read_name(dir)))
    {
      LocalType type;
      LocalStream *stream;
      
      if (st_is_aborted())
	{
	  status = FALSE;
	  goto end;
	}
      
      if (filename[0] == '.')
	continue;
      
      type = get_file_type(filename);
      if (type == LOCAL_TYPE_UNKNOWN)
	continue;

      stream = stream_new_cb(NULL);
      
      stream->pathname = g_build_filename(dirname, filename, NULL);
      stream->type = type;

      ((STStream *) stream)->name = g_strdup(filename);

      stream->filename = g_filename_to_utf8(filename, -1, NULL, NULL, &tmp_err);
      if (! stream->filename)
	{
	  st_notice(_("Local: %s: unable to convert filename to UTF-8: %s"), stream->pathname, tmp_err->message);
	  g_clear_error(&tmp_err);
	}
      
      switch (stream->type)
	{
#ifdef WITH_ID3
	case LOCAL_TYPE_MP3:	local_id3_read(stream); 	break;
#endif
#ifdef WITH_VC
	case LOCAL_TYPE_OGG:	local_vc_read(stream); 		break;
#endif
	default:		/* nop */
	}
      
      *streams = g_list_append(*streams, stream);
    }
  
 end:
  g_dir_close(dir);
  g_free(dirname);
  
  return status;
}

static LocalType
get_file_type (const char *filename)
{
  char *extension;

  g_return_val_if_fail(filename != NULL, -1);

  extension = strrchr(filename, '.');
  if (! extension++)
    goto unknown;

  if (! strcasecmp(extension, "mp3"))
    return LOCAL_TYPE_MP3;
  else if (! strcasecmp(extension, "ogg"))
    return LOCAL_TYPE_OGG;
  else if (! strcasecmp(extension, "m3u") || ! strcasecmp(extension, "pls"))
    return LOCAL_TYPE_PLAYLIST;

 unknown:
  return LOCAL_TYPE_UNKNOWN;
}

static void
init_handler (void)
{
  STHandler *handler;
  char *home;
  GNode *stock_categories;
  STCategory *category;

  handler = st_handler_new("local");

  st_handler_set_label(handler, _("Local"));
  st_handler_set_copyright(handler, COPYRIGHT);
  st_handler_set_description(handler, _("Local Music Collection"));

  home = g_strconcat("file://", local_root, NULL);
  st_handler_set_home(handler, home);
  g_free(home);

  st_handler_set_icon_from_inline(handler, sizeof(art_icon), art_icon);

  stock_categories = g_node_new(NULL);

  category = st_category_new();
  category->name = "__main";
  category->label = _("Root");
  
  g_node_append_data(stock_categories, category);
  
  st_handler_set_stock_categories(handler, stock_categories);
  st_handler_set_flags(handler, ST_HANDLER_CONFIRM_DELETION);

  st_handler_bind(handler, ST_HANDLER_EVENT_REFRESH, refresh_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_NEW, stream_new_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_GET, stream_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_SET, stream_field_set_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_STOCK_FIELD_GET, stream_stock_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_MODIFY, stream_modify_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_DELETE, stream_delete_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FREE, stream_free_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE, stream_tune_in_multiple_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_BROWSE, stream_browse_cb, NULL);

  st_handler_add_field(handler,
		       st_handler_field_new(LOCAL_FIELD_PATHNAME,
					    _("Pathname"),
					    G_TYPE_STRING,
					    0));
  st_handler_add_field(handler,
		       st_handler_field_new(LOCAL_FIELD_TYPE,
					    _("Type"),
					    G_TYPE_INT,
					    0));
  st_handler_add_field(handler,
		       st_handler_field_new(LOCAL_FIELD_FILENAME,
					    _("Filename"),
					    G_TYPE_STRING,
					    ST_HANDLER_FIELD_VISIBLE
					    | ST_HANDLER_FIELD_EDITABLE));
  st_handler_add_field(handler,
		       st_handler_field_new(LOCAL_FIELD_TITLE,
					    _("Title"),
					    G_TYPE_STRING,
					    ST_HANDLER_FIELD_VISIBLE
					    | ST_HANDLER_FIELD_EDITABLE));
  st_handler_add_field(handler,
		       st_handler_field_new(LOCAL_FIELD_ARTIST,
					    _("Artist"),
					    G_TYPE_STRING,
					    ST_HANDLER_FIELD_VISIBLE
					    | ST_HANDLER_FIELD_EDITABLE));
  st_handler_add_field(handler,
		       st_handler_field_new(LOCAL_FIELD_ALBUM,
					    _("Album"),
					    G_TYPE_STRING,
					    ST_HANDLER_FIELD_VISIBLE
					    | ST_HANDLER_FIELD_EDITABLE));
  st_handler_add_field(handler,
		       st_handler_field_new(LOCAL_FIELD_YEAR,
					    _("Year"),
					    G_TYPE_STRING,
					    ST_HANDLER_FIELD_VISIBLE
					    | ST_HANDLER_FIELD_EDITABLE));
  st_handler_add_field(handler,
		       st_handler_field_new(LOCAL_FIELD_GENRE,
					    _("Genre"),
					    G_TYPE_STRING,
					    ST_HANDLER_FIELD_VISIBLE
					    | ST_HANDLER_FIELD_EDITABLE
					    | ST_HANDLER_FIELD_START_HIDDEN));
  st_handler_add_field(handler,
		       st_handler_field_new(LOCAL_FIELD_COMMENT,
					    _("Comment"),
					    G_TYPE_STRING,
					    ST_HANDLER_FIELD_VISIBLE
					    | ST_HANDLER_FIELD_EDITABLE
					    | ST_HANDLER_FIELD_START_HIDDEN));
  st_handler_add_field(handler,
		       st_handler_field_new(LOCAL_FIELD_DURATION,
					    _("Duration"),
					    G_TYPE_STRING,
					    ST_HANDLER_FIELD_VISIBLE));

  st_handlers_add(handler);
}

gboolean
plugin_init (GError **err)
{
  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");

  if (! st_check_api_version(5, 5))
    {
      g_set_error(err, 0, 0, _("API version mismatch"));
      return FALSE;
    }

  if (! (local_root = g_getenv("STREAMTUNER_LOCAL_ROOT")))
    {
      g_set_error(err, 0, 0, _("You must point the STREAMTUNER_LOCAL_ROOT environment variable to your discotheque before using the Local plugin."));
      return FALSE;
    }
  
  init_handler();

  st_action_register("play-m3u", _("Listen to a .m3u file"), "xmms %q");
  st_action_register("view-web", _("Open a web page"), "epiphany %q");

  return TRUE;
}
