/*
 * $Id: st-applications-tab.c,v 1.7.2.2 2004/05/11 15:40:41 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <gtk/gtk.h>
#include "art/applications_preferences.h"
#include "gettext.h"
#include "sg-util.h"
#include "st-applications-tab.h"
#include "st-preferences-tab.h"
#include "st-action.h"

/*** type definitions ********************************************************/

enum {
  /* invisible columns */
  COLUMN_ID,

  /* visible columns */
  COLUMN_LABEL,
  COLUMN_COMMAND,

  N_COLUMNS
};

struct _STApplicationsTabPrivate
{
  GtkListStore *store;
};

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_applications_tab_class_init	(STApplicationsTabClass *class);
static void st_applications_tab_init		(STApplicationsTab    *tab);
static void st_applications_tab_finalize	(GObject              *object);

static void st_applications_tab_cell_edited_h	(GtkCellRendererText  *cell,
						 const char    *path_string,
						 const char	  *new_text,
						 gpointer	       data);
static void st_applications_tab_fill		(STApplicationsTab    *tab);

static gboolean st_applications_tab_search_equal_func (GtkTreeModel *model,
						       int          column,
						       const char   *key,
						       GtkTreeIter  *iter,
						       gpointer     search_data);

/*** implementation **********************************************************/

GType
st_applications_tab_get_type (void)
{
  static GType applications_tab_type = 0;
  
  if (! applications_tab_type)
    {
      static const GTypeInfo applications_tab_info = {
	sizeof(STApplicationsTabClass),
	NULL,
	NULL,
	(GClassInitFunc) st_applications_tab_class_init,
	NULL,
	NULL,
	sizeof(STApplicationsTab),
	0,
	(GInstanceInitFunc) st_applications_tab_init,
      };

      applications_tab_type = g_type_register_static(ST_TYPE_PREFERENCES_TAB,
						     "STApplicationsTab",
						     &applications_tab_info,
						     0);
    }

  return applications_tab_type;
}

static void
st_applications_tab_class_init (STApplicationsTabClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);
  STPreferencesTabClass *preferences_tab_class = ST_PREFERENCES_TAB_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  object_class->finalize = st_applications_tab_finalize;

  preferences_tab_class->link_id = "preferences-applications";
}

static void
st_applications_tab_init (STApplicationsTab *tab)
{
  GdkPixbuf *pixbuf;
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GtkWidget *scrolled;
  GtkWidget *view;

  tab->priv = g_new0(STApplicationsTabPrivate, 1);

  pixbuf = gdk_pixbuf_new_from_inline(sizeof(art_applications_preferences),
				      art_applications_preferences,
				      FALSE,
				      NULL);

  st_preferences_tab_construct_label(ST_PREFERENCES_TAB(tab), pixbuf, _("Applications"));
  g_object_unref(pixbuf);

  /* construct containers */
  
  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);

  /* construct the view */

  tab->priv->store = gtk_list_store_new(N_COLUMNS,
					G_TYPE_STRING,
					G_TYPE_STRING,
					G_TYPE_STRING);
  view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(tab->priv->store));

  /* construct the label column */

  column = gtk_tree_view_column_new_with_attributes(_("Action"),
						    gtk_cell_renderer_text_new(),
						    "text", COLUMN_LABEL,
						    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_LABEL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

  /* construct the command column */

  renderer = gtk_cell_renderer_text_new();

  g_object_set(renderer, "editable", TRUE, NULL);
  g_signal_connect(G_OBJECT(renderer), "edited",
		   G_CALLBACK(st_applications_tab_cell_edited_h), tab);

  column = gtk_tree_view_column_new_with_attributes(_("Command"),
						    renderer,
						    "text", COLUMN_COMMAND,
						    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_COMMAND);
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

  /* widget setup and packing */

  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(view), TRUE);
  gtk_tree_view_set_search_equal_func(GTK_TREE_VIEW(view),
				      st_applications_tab_search_equal_func,
				      NULL,
				      NULL);
  
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(tab->priv->store),
				       COLUMN_LABEL,
				       GTK_SORT_ASCENDING);
  
  gtk_container_add(GTK_CONTAINER(scrolled), view);
  gtk_container_add(GTK_CONTAINER(tab), scrolled);
  gtk_widget_show_all(scrolled);

  /* fill it */

  st_applications_tab_fill(tab);
}

static void
st_applications_tab_finalize (GObject *object)
{
  STApplicationsTab *tab = ST_APPLICATIONS_TAB(object);

  g_object_unref(tab->priv->store);
  g_free(tab->priv);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
st_applications_tab_cell_edited_h (GtkCellRendererText *cell,
				   const char *path_string,
				   const char *new_text,
				   gpointer data)
{
  STApplicationsTab *tab = data;
  GtkTreePath *path;
  GtkTreeIter iter;
  char *id;
  
  g_return_if_fail(ST_IS_APPLICATIONS_TAB(tab));

  path = gtk_tree_path_new_from_string(path_string);
  gtk_tree_model_get_iter(GTK_TREE_MODEL(tab->priv->store), &iter, path);
  gtk_tree_path_free(path);

  gtk_tree_model_get(GTK_TREE_MODEL(tab->priv->store), &iter,
		     COLUMN_ID, &id,
		     -1);
  gtk_list_store_set(tab->priv->store, &iter,
		     COLUMN_COMMAND, new_text,
		     -1);
  st_action_associate(id, new_text);

  g_free(id);
}

static void
st_applications_tab_fill (STApplicationsTab *tab)
{
  GSList *actions;
  GSList *l;

  g_return_if_fail(ST_IS_APPLICATIONS_TAB(tab));

  actions = st_action_list();

  SG_LIST_FOREACH(l, actions)
    {
      const char *id = l->data;
      STAction *action;
      GtkTreeIter iter;

      action = st_action_get(id);
      g_return_if_fail(action != NULL);

      gtk_list_store_append(tab->priv->store, &iter);
      gtk_list_store_set(tab->priv->store, &iter,
			 COLUMN_ID, id,
			 COLUMN_LABEL, action->label,
			 COLUMN_COMMAND, action->command,
			 -1);
    }

  g_slist_free(actions);
}

static gboolean
st_applications_tab_search_equal_func (GtkTreeModel *model,
				       int column,
				       const char *key,
				       GtkTreeIter *iter,
				       gpointer search_data)
{
  char *label;
  char *command;
  gboolean equal = FALSE;

  gtk_tree_model_get(model, iter,
		     COLUMN_LABEL, &label,
		     COLUMN_COMMAND, &command,
		     -1);

  equal = sg_utf8_strcasecontains(label, key) || sg_utf8_strcasecontains(command, key);

  g_free(label);
  g_free(command);

  return ! equal;
}

GtkWidget *
st_applications_tab_new (void)
{
  return g_object_new(ST_TYPE_APPLICATIONS_TAB, NULL);
}
