/*
 * $Id: st-config-save.c,v 1.21 2004/01/30 15:04:36 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <glib.h>
#include "gettext.h"
#include "sg-dumper.h"
#include "sg-util.h"
#include "st-handler.h"
#include "st-handler-field-api.h"
#include "st-handlers.h"
#include "st-action.h"
#include "st-settings.h"

/*** type definitions ********************************************************/

typedef struct
{
  SGDumper *dumper;
  GError **err;
} STConfigSaveInfo;

/*** function declarations ***************************************************/

static void st_config_save_handler_cb (STHandler	*handler,
				       STConfigSaveInfo	*info);

/*** implementation **********************************************************/

gboolean
st_config_save (const char *filename, GError **err)
{
  GSList *actions;
  GSList *l;
  SGDumper *dumper;
  STConfigSaveInfo info;

  filename = g_build_filename(st_settings.private_dir, "config", NULL);

  if (! (dumper = sg_dumper_new(filename, err)))
    return FALSE;

  if (! sg_dumper_write(dumper, err,
			"#", _("streamtuner configuration file, automatically generated"),
			"",
			"k{", "options",
			"kb", "view_toolbar", st_settings.view_toolbar,
			"kb", "view_tabs", st_settings.view_tabs,
			"kb", "view_tab_icons", st_settings.view_tab_icons,
			"kb", "view_statusbar", st_settings.view_statusbar,
			"ki", "toolbar_style", st_settings.toolbar_style,
			"ki", "toolbar_size", st_settings.toolbar_size,
			"kb", "save_automatic", st_settings.save_automatic,
			"kb", "always_refresh", st_settings.always_refresh,
			"ki", "main_window_width", st_settings.main_window_width,
			"ki", "main_window_height", st_settings.main_window_height,
			"ki", "preferences_window_width", st_settings.preferences_window_width,
			"ki", "preferences_window_height", st_settings.preferences_window_height,
			"ki", "stream_properties_window_width", st_settings.stream_properties_window_width,
			"ki", "selected_preferences_page", st_settings.selected_preferences_page,
			"kb", "proxy_enabled", st_settings.proxy_enabled,
			"ks", "proxy_url", st_settings.proxy_url,
			"kb", "proxy_auth_enabled", st_settings.proxy_auth_enabled,
			"ks", "proxy_auth_name", st_settings.proxy_auth_name,
			"ks", "proxy_auth_password", st_settings.proxy_auth_password,
			NULL))
    goto error;

  if (st_settings.selected_handler_name)
    {
      if (! sg_dumper_write(dumper, err,
			    "ks", "selected_handler", st_settings.selected_handler_name,
			    NULL))
	goto error;
    }
  
  if (! sg_dumper_write(dumper, err, "}", NULL))
    goto error;
  
  actions = st_action_list();
  SG_LIST_FOREACH(l, actions)
    {
      const char *id = l->data;
      STAction *action;

      action = st_action_get(id);
      g_return_val_if_fail(action != NULL, FALSE);

      if (! sg_dumper_write(dumper, err,
			    "ks{", "action", id,
			    "ks", "command", action->command,
			    "}",
			    NULL))
	goto error;
    }
  
  info.dumper = dumper;
  info.err = err;

  g_slist_foreach(st_handlers_list, (GFunc) st_config_save_handler_cb, &info);
  if (*err)
    goto error;
  
  /* everything went fine */

  if (! sg_dumper_free(dumper, err))
    return FALSE;

  return TRUE;

 error:
  sg_dumper_free(dumper, NULL);	/* ignore errors since ERR is already set */
  return FALSE;
}

static void
st_config_save_handler_cb (STHandler *handler, STConfigSaveInfo *info)
{
  GSList *l;

  /* A GFunc callback can't return false to stop the traversal, so
     we have to check if a previous save_handler_callback() invocation
     failed */
  if (*info->err)
    return;
  
  if (! sg_dumper_write(info->dumper, info->err,
			"ks{", "handler", handler->name,
			"ki", "fields_sort_index", handler->fields_sort_index,
			"ki", "fields_sort_order", handler->fields_sort_order,
			"ki", "paned_position", handler->paned_position,
			"ks", "selected_category", handler->selected_category->name,
			NULL))
    return;

  if (! sg_dumper_write(info->dumper, info->err,
			"k{", "selected_streams",
			NULL))
    return;

  SG_LIST_FOREACH(l, handler->selected_streams)
    {
      STStream *stream = l->data;
      
      if (! sg_dumper_write(info->dumper, info->err,
			    "s", stream->name,
			    NULL))
	return;
    }

  if (! sg_dumper_write(info->dumper, info->err, "}", NULL))
    return;

  SG_LIST_FOREACH(l, handler->fields)
    {
      STHandlerField *field = l->data;

      if (field->visible)
	{
	  if (! sg_dumper_write(info->dumper, info->err,
				"k{", "field",
				"kb", "visible", field->user_visible,
				"ki", "width", field->width,
				"}",
				NULL))
	    return;
	}
    }
  
  if (handler->expanded_categories)
    {
      if (! sg_dumper_write(info->dumper, info->err,
			    "k{", "expanded_categories",
			    NULL))
	return;

      SG_LIST_FOREACH(l, handler->expanded_categories)
	if (! sg_dumper_write(info->dumper, info->err,
			      "s", (const char *) l->data,
			      NULL))
	  return;

      if (! sg_dumper_write(info->dumper, info->err, "}", NULL))
	return;
    }
  
  if (! sg_dumper_write(info->dumper, info->err, "}", NULL))
    return;
}
