-- Copyright (C) 2010-2011 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

if options.debug then
   options.drawvolumes = true
   options.drawjoints = true
   options.external = true
end

require "units"
require "resources"
require "derived"
require "frames"
require "switches"
require "dynamics"
require "network"
require "plotting"
require "bookmarks"
require "morewidgets"

require "recorder"

common.iterate = true

network.http = 29176
      
plotting.colors = {
   {1.0000, 0.0141, 0.0149},
   {0, 1, 0},
   {0.0374, 0.2031, 0.4535},
   {1.0000, 0.4892, 0.0013},
   {0.5812, 0.3987, 0.1811},
   {0.2156, 0.1479, 0.0659},
}
      
plotting.styles = {1, 1, 1, 1, 1, 1, 1, 1, 1, 1}

graphics.window = {1024, 640}
graphics.title = "Slipstream"
graphics.hide = false
graphics.cursor = false

graphics.close = function ()
		    if not options.interactive then
		       common.iterate = false
		    else
		       graph.vehicle = nil
		       graph.track = nil

		       collectgarbage()

		       graphics.hide = true
		       network.block = true
		    end
		 end

derived.frustum = {45, 0.05, 35000}
derived.gee = -9.81

dynamics.stepsize = 0.001
dynamics.ceiling = 0.1
dynamics.timescale = 1
dynamics.surfacelayer = 10e-3
-- dynamics.popvelocity = 1/0
-- dynamics.tolerance = {1e-8, 0.7}
dynamics.collision = {
   -- debug = function (a, b)
   -- 	      print (a, b)
   -- 	   end
}

slipstream = {}
blueprint = {}
layout = {}
session = {}

-- Open the configuration bookmarks.

configurations = network.bookmark {
   path = os.getenv ("HOME") .. "/.slipstream.d/configurations"
}

configurations.stock = {}
configurations.custom = resources.loadcache (os.getenv ("HOME") .. "/.slipstream.d/configurations/tags.lua")

-- Open the sessions bookmarks.

sessions = network.bookmark {
   path = os.getenv ("HOME") .. "/.slipstream.d/sessions"
}

sessions.recorded = resources.loadcache (os.getenv ("HOME") .. "/.slipstream.d/sessions/recorded.lua")
sessions.imported = resources.loadcache (os.getenv ("HOME") .. "/.slipstream.d/sessions/imported.lua")

-- Load per-user configuration.

local rcscript = loadfile (os.getenv ("HOME") .. "/.slipstream")

if rcscript then
   print "Reading local user configuration."

   print ("  " .. os.getenv ("HOME") .. "/.slipstream")
   rcscript ()
end

-- Create the base nodes.

graph.framerate = switches.toggle {
   state = not options.nowidgets and options.framerate,

   on =  widgets.display {
      align = {1, 1},
			  
      gauge = widgets.speedometer {
	 color = {1.0, 0.8, 0.2},
	 opacity = 0.6,
      },
   },
}

graph.console = widgets.console {
   padding = {0.007, 0.007},
   align = {-1, 1},
}

graph.controls = frames.event {
   keypress = function (self, key)
		 if key == 'escape' then
		    graphics.close()
		 elseif key == 'f' then
		    if graphics.oldwindow then
		       graphics.window = graphics.oldwindow
		       graphics.oldwindow = nil
		    else
		       graphics.oldwindow = graphics.window
		       graphics.window = configuration.screen
		    end
		 elseif key == 'p' then
		    if dynamics.oldtimescale then
		       dynamics.timescale = dynamics.oldtimescale
		       dynamics.oldtimescale = nil
		    else
		       dynamics.oldtimescale = dynamics.timescale
		       dynamics.timescale = 0
		    end
		 elseif key == 's' then
		    local image, file
		    local w, h

		    w, h = graphics.window[1], graphics.window[2]
		    
		    image = imaging.flip (graphics.colorbuffer, w, h)
		    image = imaging.writeformat (image, w, h, "PNG")
		    
		    __shot = __shot and __shot + 1 or 1

		    file = io.open(string.format('shot_%d.png',
						 __shot), "w")
		    file:write(image)
		    file:close()
		 elseif key == 'page_up' then
		    dynamics.timescale = dynamics.timescale * 2

		    graph.console.text = string.format ([[<span font="Sans 12" color="red">Time scale set to: %g</span>]], dynamics.timescale)
		 elseif key == 'page_down' then
		    dynamics.timescale = dynamics.timescale / 2

		    graph.console.text = string.format ([[<span font="Sans 12" color="red">Time scale set to: %g</span>]], dynamics.timescale)
		 elseif key == 'space' then
		    graph.vehicle.parent = nil
		    
		    session = {}
		    collectgarbage()

		    graph.vehicle = resources.dofile ("slipstream/" ..
		    				      blueprint.model ..
		    				      "/model.lua")
		 else
		    print ('"' .. key ..'" pressed.')
		 end
	      end,
}

-- And load the rest.

resources.dofile "slipstream/control.lua"
resources.dofile "slipstream/tracks.lua"
resources.dofile "slipstream/models.lua"
resources.dofile "slipstream/pages.lua"

if options.fullscreen then
   graphics.window = configuration.screen
end

if type(options.run) ~= "table" then
   options.run = {options.run}
end

for _, line in pairs(options.run) do
   assert(loadstring(line))()
end

-- Signal inspection.

if type (options.examine) ~= "table" then
   options.examine = {options.examine}
end

for i = 1, #options.examine do
   local logger, file

   logger = frames.node {
      step = function (self)
		if graph.vehicle then
		   if not self.probe then
		      self.probe = assert (loadstring ("return (" .. options.examine[i] .. ")"))
		   end

		   file:write (string.format ("%g %g\n", configuration.time[1], self.probe()))
		end
	     end,

      link = function (self)
		file = io.open ("_" .. string.gsub(options.examine[i], "/", "\\"), "w")
	     end,

      unlink = function (self)
		  file:flush()
		  file:close()
	       end   
   }
   
   logger.parent = graph
end

if options.model and options.track then
   resources.dofile (options.model)
   resources.dofile (options.track)

   graph.vehicle = resources.dofile("slipstream/" .. blueprint.model ..
				    "/model.lua") 
   graph.track = resources.dofile("slipstream/tracks/" ..
				  layout.track .. ".lua")
elseif options.restore then
   blueprint, layout, session = resources.dofile (options.restore)

   graph.vehicle = resources.dofile("slipstream/" .. blueprint.model ..
				    "/model.lua") 
   graph.track = resources.dofile("slipstream/tracks/" ..
				  layout.track .. ".lua")
elseif options.interactive then
   network.block = true
   graphics.hide = true
else
   print ("Nothing to do.")
   common.iterate = false
end

if options.save then
   graph.unlink = function (self)
		     local file

		     file = io.open (options.save, "w")
		     file:write ("return \n" .. 
				 string.serialize (blueprint) .. "," ..
			         string.serialize(layout) .. "," ..
			         string.serialize(session))
		     file:close ()
		  end
end
