-- Copyright (C) 2010-2011 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "dynamics"
require "automotive"
require "shading"
require "units"
require "nature"
require "moremath"
require "transforms"
require "toon"

local bitonal = resources.pipeline {resources.quantize(2), textures.mirrored}
local octonal = resources.pipeline {resources.quantize(8), textures.clamped}
local modulated = resources.pipeline {resources.modulate(0.1, 0.4, 1),
				      textures.clamped}

return nature.atmosphere {
   size = {1024, 512},

   turbidity = 3,

   rayleigh = math.scale({6.95e-06, 1.18e-05, 2.44e-05}, 1),
   mie = 7e-5,

   sun = {5.74, 1.09},

   -- The environment.

   land = (options.toon and nature.earth or nature.land) {
      albedo = 1.1,
      
      scenery = nature.elevation {    
      	 size = {1, 1, 9},
      	 resolution = {92.983924, 93.012422, 3737},
      } {
      	 position = units.meters {-23605, -23325, -835},

      	 corporeal = false,
      	 target = 3000,
	 
      	 link = function(self)
      		   local samples, bounds, imagery
		   
      		   samples, bounds = resources.dofile ("slipstream/imagery/scenery-elevation.lc")
      		   imagery = resources.dofile ("slipstream/imagery/scenery-imagery.lc")
		   
      		   self[0] = {samples, bounds, imagery}

      		   -- No need to load more than once.

      		   self.link = nil
      		end,
      },
   },
   
   sunlight = options.toon and toon.lamp {
      position = math.transform (transforms.euler (math.deg(1.09),
						   0,
						   math.deg(5.74)),
				 {0, 1e5, 0}),

      intensity = {[0] = 0.4, [0.04] = 0.6, [0.15] = 0.8, [0.45] = 1},
      orientation = transforms.euler (180, 0, 0),
      ambience = 0.4,

      islamp = true
   } or shading.light {
      position = math.transform (transforms.euler (math.deg(1.09),
						   0,
						   math.deg(5.74)),
				 {0, 1e5, 0}),

      offset = {1, 2},
      volume = {-1000, 1000, -1000, 1000, 0.1, 1000},
      intensity = math.scale({1, 1, 0.8}, 1.2),
      size = {0, 0},
      
      islight = true
   },

   skylight = not options.toon and shading.ambient {
      orientation = transforms.euler (0, 180, 0),
      intensity = modulated "slipstream/imagery/horizon.lc",
   },
   
   haze = (options.toon and toon.haze or shading.fog) {
      link = function (self)
		self.color = self.parent.intensity
		self.linear = 5 * (self.parent.turbidity - 1) *
		self.parent.mie
	  end
   },

   -- The track.

   paint = (options.toon and toon.flat or shading.lambert) {
      color = options.toon and 
   	 resources.periodic "slipstream/imagery/paint.lc",
      diffuse = not options.toon and
   	 resources.periodic "slipstream/imagery/paint.lc",

      mesh = resources.static "slipstream/meshes/paint.lc" {
	 position = {0, 0, 0.008}
      }
   },

   ground = bodies.environment {
      resources.polyhedron "slipstream/polyhedra/ground.lc" {
	 toon.flat {
	    color = resources.clamped "slipstream/imagery/ground.lc",
	    mesh = resources.static "slipstream/meshes/ground.lc" {}
	 },
      },
   },
 
   tarmac = (options.toon and toon.flat or shading.lambert) {
      color =  options.toon and 
	 resources.periodic "slipstream/imagery/asphalt.lc",
      diffuse = not options.toon and
	 resources.periodic "slipstream/imagery/asphalt.lc",
      
      bodies.environment {
	 automotive.racetrack {
	    tesselation = {units.feet(12), 0.01 / 3, 0.01},
	    scale = {2, 2},

	    istrack = true,

	    -- Turn 1.

	    {units.feet(269), units.feet(15), units.feet(15), 0, 0, 0},
	    {units.feet(150), units.feet(15), units.feet(15), 0, -0.045, -0.033},
	    {units.feet(182), units.feet(15), units.feet(15), 1 / units.feet(516), -0.045, -0.119},
	    {units.feet(239), units.feet(20), units.feet(20), 0, -0.045, -0.028},

	    -- Turn 2.

	    {units.feet(301), units.feet(20), units.feet(20), 0, -0.02, -0.03},
	    {units.feet(100), units.feet(20), units.feet(25), 1 / units.feet (127), -0.02, -0.037},
	    {units.feet(173), units.feet(20), units.feet(30), 1 / units.feet (127), -0.02, -0.044},
	    {units.feet(174), units.feet(20), units.feet(25), 1 / units.feet (127), -0.02, -0.038},

	    -- Turn 3.

	    {units.feet(75), units.feet(20), units.feet(20), 0, 0, -0.005},
	    {units.feet(25), units.feet(20), units.feet(20), 0, 0, 0},
	    {units.feet(75), units.feet(20), units.feet(20), 1 / units.feet (-900), 0, 0.022},
	    {units.feet(325), units.feet(20), units.feet(20), 1 / units.feet (-900), 0, 0.022},
	    {units.feet(50), units.feet(20), units.feet(20), 1 / units.feet (-145), 0, 0.022},
	    {units.feet(195), units.feet(20), units.feet(20), 1 / units.feet (-145), 0, 0.026},
	    {units.feet(50), units.feet(20), units.feet(20), 0, 0, 0.015},
	    {units.feet(70), units.feet(20), units.feet(20), 0, 0, 0},

	    -- Turn 4.

	    {units.feet(470), units.feet(20), units.feet(20), 0, 0, 0},
	    {units.feet(80), units.feet(20), units.feet(20), 0, 0, 0.022},
	    {units.feet(90), units.feet(20), units.feet(20), 1 / units.feet (-200), 0, 0.026},
	    {units.feet(150), units.feet(20), units.feet(20), 1 / units.feet (-200), 0, 0.026},
	    {units.feet(90), units.feet(20), units.feet(20), 0, 0, 0.015},
	    {units.feet(50), units.feet(20), units.feet(20), 0, 0, 0},

	    -- Turn 5.

	    {units.feet(298), units.feet(20), units.feet(20), 0, 0, 0},
	    {units.feet(60), units.feet(20), units.feet(20), 1 / units.feet (-1000), 0.01, 0.02},
	    {units.feet(255), units.feet(20), units.feet(20), 1 / units.feet (-1000), 0.01, 0.02},
	    {units.feet(60), units.feet(20), units.feet(20), 0, 0.01, 0},
	    {units.feet(480), units.feet(20), units.feet(20), 0, 0.01, 0},
	    {units.feet(100), units.feet(20), units.feet(20), 0, 0.01, -0.021},
	    {units.feet(80), units.feet(20), units.feet(20), 1 / units.feet (170), 0.03, -0.085},
	    {units.feet(234), units.feet(20), units.feet(20), 1 / units.feet (170), 0.03, -0.113},
	    {units.feet(80), units.feet(17), units.feet(17), 0, 0.04, -0.085},
	    {units.feet(60), units.feet(15), units.feet(15), 0, 0.04, -0.068},

	    -- Turn 6.
	    
	    {units.feet(285), units.feet(15), units.feet(15), 0, 0.07, -0.068},
	    {units.feet(180), units.feet(15), units.feet(15),
	     1 / units.feet (-9000), 0.07, -0.057},
	    {units.feet(180), units.feet(15), units.feet(15), 0, 0.07, -0.066},
	    {units.feet(78), units.feet(15), units.feet(15), 1 / units.feet (1755), 0.07, -0.068},
	    {units.feet(200), units.feet(15), units.feet(15), 0, 0.07, -0.068},
	    {units.feet(50), units.feet(15), units.feet(15), 1 / units.feet (80), 0.07, -0.068},
	    {units.feet(42), units.feet(15), units.feet(15), 1 / units.feet (80), 0.07, -0.096},
	    {units.feet(60), units.feet(15), units.feet(15), 0, 0.07, -0.038},

	    -- Turn 7.
	    
	    {units.feet(224), units.feet(15), units.feet(15), 0, 0.05, 0},
	    {units.feet(80), units.feet(15), units.feet(15), 1 / units.feet (500), 0.05, -0.005},
	    {units.feet(80), units.feet(15), units.feet(15), 0, 0.05, -0.005},
	    {units.feet(56), units.feet(15), units.feet(15), 0, 0.05, -0.005},
	    {units.feet(90), units.feet(15), units.feet(15), 1 / units.feet (1125), 0.05, -0.01},
	    {units.feet(95), units.feet(15), units.feet(15), 0, 0.05, -0.01},
	    {units.feet(83), units.feet(15), units.feet(15), 0, 0.05, -0.01},
	    {units.feet(120), units.feet(15), units.feet(15), 1 / units.feet (-3200), 0.05, -0.015},
	    {units.feet(112), units.feet(15), units.feet(15), 0, 0.05, -0.015},
	    {units.feet(150), units.feet(15), units.feet(15), 1 / units.feet (1950), 0.05, -0.015},
	    {units.feet(139), units.feet(15), units.feet(15), 0, 0.05, -0.015},
	    {units.feet(60), units.feet(15), units.feet(15), 1 / units.feet (-450), 0.05, 0},
	    {units.feet(122), units.feet(15), units.feet(15), 1 / units.feet (-450), 0.05, 0.016},
	    {units.feet(90), units.feet(15), units.feet(15), 0, 0.04, 0},

	    -- Turn 8.
	    
	    {units.feet(127), units.feet(20), units.feet(20), 0, -0.11, -0.066},
	    {units.feet(35), units.feet(20), units.feet(20), 1 / units.feet (75), -0.11, -0.126},
	    {units.feet(51), units.feet(20), units.feet(20), 1 / units.feet (75), -0.11, -0.126},
	    {units.feet(51), units.feet(20), units.feet(20), 1 / units.feet (75), -0.11, -0.066},
	    {units.feet(35), units.feet(20), units.feet(20), 0, -0.11, 0},
	    {units.feet(30), units.feet(20), units.feet(20), 1 / units.feet (-90), -0.11, 0.058},
	    {units.feet(45), units.feet(20), units.feet(20), 1 / units.feet (-90), -0.11, 0.114},
	    {units.feet(45), units.feet(20), units.feet(20), 1 / units.feet (-90), -0.11, 0.068},
	    {units.feet(30), units.feet(20), units.feet(20), 0, -0.11, 0.059},
	    {units.feet(163), units.feet(20), units.feet(20), 1 / units.feet (-1170), -0.09, 0},
	    {units.feet(80), units.feet(20), units.feet(20), 1 / units.feet (-369), -0.09, -0.048},
	    {units.feet(50), units.feet(20), units.feet(20), 0, -0.09, -0.081},
	    {units.feet(20), units.feet(20), units.feet(20), 1 / units.feet (310), -0.09, -0.098},
	    {units.feet(43), units.feet(20), units.feet(20), 1 / units.feet (310), -0.09, -0.114},
	    {units.feet(20), units.feet(20), units.feet(20), 0, -0.09, -0.12},
	    {units.feet(27), units.feet(20), units.feet(20), 1 / units.feet (-545), -0.09, -0.112},
	    {units.feet(10), units.feet(20), units.feet(20), 0, -0.09, -0.116},

	    -- Turn 9.
	    
	    {units.feet(60), units.feet(20), units.feet(20), 1 / units.feet (265), -0.09, -0.125},
	    {units.feet(205), units.feet(20), units.feet(20), 1 / units.feet (265), -0.09, -0.125},
	    {units.feet(200), units.feet(20), units.feet(20), 1 / units.feet (265), -0.09, -0.057},
	    {units.feet(100), units.feet(20), units.feet(20), 0, -0.09, -0.021},

	    -- Turn 10.
	    
	    {units.feet(356), units.feet(20), units.feet(20), 0, -0.02, 0},
	    {units.feet(100), units.feet(20), units.feet(20), 1 / units.feet (-150), -0.02, 0.06},
	    {units.feet(121), units.feet(20), units.feet(20), 1 / units.feet (-150), -0.02, 0.1},
	    {units.feet(60), units.feet(20), units.feet(20), 0, -0.01, 0.066},
	    {units.feet(120), units.feet(20), units.feet(20), 0, -0.01, 0},

	    -- Turn 11.
	    
	    {units.feet(578), units.feet(20), units.feet(20), 0, -0, 0},
	    {units.feet(40), units.feet(20), units.feet(20), 1 / units.feet (45), 0, 0.02},
	    {units.feet(45), units.feet(25), units.feet(25), 1 / units.feet (45), 0, 0.02},
	    {units.feet(40), units.feet(25), units.feet(25), 0, 0, 0.02},
	    {units.feet(514.455), units.feet(25), units.feet(25), 0, 0, 0}, 
	    {units.feet(300), units.feet(25), units.feet(25), 0, 0.1, 0},
	    {units.feet(200), units.feet(25), units.feet(25), 0, 0.0273, 0},
	    {units.feet(50), units.feet(20), units.feet(20), 1 / units.feet (137.628), 0.02, 0},
	    {units.feet(50.85), units.feet(15), units.feet(15), 0, 0, 0},
	    {19.715, units.feet(15), units.feet(15), 0, 0, 0},
	 },
      }
   },

   timer = frames.timer {
      period = 0.01,

      step = function (self)
		local track, u, p

		track = self.parent.tarmac[1][1]

		p = track[parts.chassis.position]
		u = p and p[1]

		if self.beta and u and u - self.beta < -100 then
		   if not self.laptimes then
		      self.laptimes = widgets.display {
			 size = {graphics.window[1] /
				 graphics.window[2], 1},
			 padding = {0.01, 0.01},
			 align = {1, -1},
      
			 frame = widgets.frame {
			    color = {1.0, 0.8, 0.2},
			    radius = 0.005,

			    columns = widgets.row {
			       padding = {0, 0.01},

			       [1] = widgets.column {
				  padding = {0.01, 0},
			       },

			       [2] = widgets.column {
				  padding = {0.01, 0},
			       },
			    },
			 },
		      }

		      self.tick = function (self)
				     local m, s = math.modf((configuration.time[1] - self.alpha) / 60)
				     local time

				     if m > 0 then
					time = string.format ("%d' %.2f\"", m, s * 60)
				     else
					time = string.format ("%.2f\"", s * 60)
				     end

				     self.laptimes.frame.columns[1][self.lap].text = string.format ("<span font=\"Sans 14\" color=\"White\">Lap %d: </span>", self.lap)
				     self.laptimes.frame.columns[2][self.lap].text = string.format ("<span font=\"Sans 14\" color=\"White\">%s</span>", time)
				  end
		   end

		   self.alpha = configuration.time[1]
		   self.lap = self.lap and self.lap + 1 or 1
		   self.laptimes.frame.columns[1][self.lap] = widgets.layout {
		      align = {-1, 0},
		   }

		   self.laptimes.frame.columns[2][self.lap] = widgets.layout {
		      align = {1, 0},
		   }
		end

		self.beta = u
	     end,
   },
}
