<!--
Copyright (C) 2008 Papavasileiou Dimitris                             
                                                                     
This program is free software: you can redistribute it and/or modify 
it under the terms of the GNU General Public License as published by 
the Free Software Foundation, either version 3 of the License, or    
(at your option) any later version.                                  
                                                                     
This program is distributed in the hope that it will be useful,      
but WITHOUT ANY WARRANTY; without even the implied warranty of       
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
GNU General Public License for more details.                         
                                                                     
You should have received a copy of the GNU General Public License    
along with this program.  If not, see <http://www.gnu.org/licenses/>.
-->

<head>
  <title>Engine</title>
  <link rel="stylesheet" href="/stylesheet" type="text/css">
</head>

<body>
  <?lua return fragments.logo ?>
  <?lua return fragments.motorcyclenavigation ?>

  <?lua return fragments.processconfiguration {model = "motorcycle",
					       prefix = "/motorcycle",
					       redirect = "/motorcycle"}
  ?>

  <p>This is the engine section which includes the power plant and
  transmission.  Use the plots to the left and the configuration
  options to tune the engine output.</p>

  <label for="list">Select plot:</label>
  <select id="list" onChange="javascript: document.getElementById('graph').src = '/motorcycle/' + this.options[this.selectedIndex].value">
     <option value="manifold">Manifold Absolute Pressure</option>
     <option value="efficiency">Efficiency</option>
     <option value="torque">Torque</option>
     <option value="power" selected="selected">Power</option>
  </select>

  <br/>

  <img width="100%" type="image/svg+xml"
       id="graph" src="/motorcycle/power" ></img>

  <?lua
    local w
    local speeds, samples = {}, {}

    local induced, friction, gasses, net = {}, {}, {}, {}
    local volumetric, thermal = {}, {}
    local map = {}

    -- Make a test run.

    parts.engine.throttle = 1

    for w = units.hertz(0), units.hertz(16000), units.hertz(10) do
       parts.engine.state = {0, w}
       table.insert(speeds, w)
       table.insert(samples, parts.engine.state)
    end

    -- Collect the results.

    for i = 1, #speeds do
       map[i] = {speeds[i] * 30 / math.pi, samples[i][3] / 100e3}

       volumetric[i] = {speeds[i] * 30 / math.pi, samples[i][4]}
       thermal[i] = {speeds[i] * 30 / math.pi, samples[i][5]}

       induced[i] = {speeds[i] * 30 / math.pi, samples[i][6]}
       gasses[i] = {speeds[i] * 30 / math.pi, samples[i][7]}
       friction[i] = {speeds[i] * 30 / math.pi, samples[i][8]}
       net[i] = {speeds[i] * 30 / math.pi, samples[i][9]}
    end

    -- MAP.

    network.mime['/motorcycle/manifold'] = "image/svg+xml"
    network.pages['/motorcycle/manifold'] =  plotting.cartesian {
       size = {600, 600},
       window = {0, 16000, 0, 1},
       background = {1, 1, 1},
       foreground = {0, 0, 0},

       grid = true,
       box = true,
       ticks = true,
       
       labels = {"Engine speed (rpm)",
		 "MAP (bar)",
		 ""},

       fontscale = 3,
       
       plotting.line {
	  color = plotting.colors[1],
	  style = plotting.styles[1],
	  unpack(map)
       },
    }

    -- Volumetric and thermal efficiency.

    network.mime['/motorcycle/efficiency'] = "image/svg+xml"
    network.pages['/motorcycle/efficiency'] =  plotting.cartesian {
       size = {600, 600},
       window = {0, 16000, 0, 1},
       background = {1, 1, 1},
       foreground = {0, 0, 0},

       grid = true,
       box = true,
       ticks = true,
       
       labels = {"Engine speed (rpm)",
		 "Efficiency",
		 ""},

       fontscale = 3,
       
       plotting.line {
	  color = plotting.colors[1],
	  style = plotting.styles[1],
	  unpack(volumetric)
       },
       
       plotting.line {
	  color = plotting.colors[2],
	  style = plotting.styles[2],
	  unpack(thermal)
       },
    }

    -- Torque

    network.mime['/motorcycle/torque'] = "image/svg+xml"
    network.pages['/motorcycle/torque'] =  plotting.cartesian {
       size = {600, 600},
       window = {0, 16000, 0, 150},
       background = {1, 1, 1},
       foreground = {0, 0, 0},

       grid = true,
       box = true,
       ticks = true,
       
       labels = {"Engine speed (rpm)",
		 "Torque (Nm)",
		 ""},

       fontscale = 3,
       
       plotting.line {
	  color = plotting.colors[1],
	  style = plotting.styles[1],
	  unpack(induced)
       },
       
       plotting.line {
	  color = plotting.colors[2],
	  style = plotting.styles[2],
	  unpack(net)
       },
       
       plotting.line {
	  color = plotting.colors[3],
	  style = plotting.styles[3],
	  unpack(friction)
       },
       
       plotting.line {
	  color = plotting.colors[4],
	  style = plotting.styles[4],
	  unpack(gasses)
       },
    }

    -- Horsepower (converted from torque readings).

    for i = 1, #speeds do
       induced[i][2] = induced[i][2] * speeds[i] / units.mechanicalhorsepower(1)
       gasses[i][2] = gasses[i][2] * speeds[i] / units.mechanicalhorsepower(1)
       friction[i][2] = friction[i][2] * speeds[i] / units.mechanicalhorsepower(1)
       net[i][2] = net[i][2] * speeds[i] / units.mechanicalhorsepower(1)
    end

    network.mime['/motorcycle/power'] = "image/svg+xml"
    network.pages['/motorcycle/power'] =  plotting.cartesian {
       size = {600, 600},
       window = {0, 16000, 0, 200},
       background = {1, 1, 1},
       foreground = {0, 0, 0},

       grid = true,
       box = true,
       ticks = true,
       
       labels = {"Engine speed (rpm)",
		 "Power (BHP)",
		 ""},

       fontscale = 3,
       
       plotting.line {
	  color = plotting.colors[1],
	  style = plotting.styles[1],
	  unpack(induced)
       },
       
       plotting.line {
	  color = plotting.colors[2],
	  style = plotting.styles[2],
	  unpack(net)
       },
       
       plotting.line {
	  color = plotting.colors[3],
	  style = plotting.styles[3],
	  unpack(friction)
       },
       
       plotting.line {
	  color = plotting.colors[4],
	  style = plotting.styles[4],
	  unpack(gasses)
       },
    }
   ?>

  <?lua
    return fragments.beginconfiguration {onapply = "/motorcycle/engine",
					 onsave = "/motorcycle",
					 chunk = "engine"}
  ?>
  
  <table class="configuration">
    <tr><th colspan="2">Engine</th></tr>
    <tr><td>Number of cylinders</td>
        <td><input type="text"
                   name="blueprint.cylinders"
	           value="<?lua return string.format ("%d", blueprint.cylinders) ?>" /></td>
    </tr>
    <tr><td>Cylinder displacement</td>
        <td><input pre="$ * 1e6"
                   post="$ / 1e6"
                   type="text"
                   name="blueprint.displacement"
	           value="<?lua return string.format ("%g", blueprint.displacement) ?>" /> cm<sup>3</sup></td>
    </tr>
    <tr><td>Flywheel inertia</td>
        <td><input type="text"
                   name="blueprint.inertia"
	           value="<?lua return string.format ("%g", blueprint.inertia) ?>" /> kg m<sup>2</sup></td>
    </tr>
    <tr><td>Throttle plate diameter</td>
        <td><input pre="$ * 1e3"
                   post="$ / 1e3"
                   type="text"
                   name="blueprint.intake"
	           value="<?lua return string.format ("%g", blueprint.intake[1]) ?>" /> mm</td>
    </tr>
    <tr><td>Throttle plate discharge coefficient</td>
        <td><input type="text"
                   name="blueprint.intake"
	           value="<?lua return string.format ("%g", blueprint.intake[2]) ?>" /></td>
    </tr>
    <tr><td>Volumetric efficiency</td>
        <td><?lua return string.format ([[
            <input type="text" name="blueprint.volumetric" value="%g" /> +
            <input type="text" name="blueprint.volumetric" value="%g" /> &middot &#969 +
            <input type="text" name="blueprint.volumetric" value="%g" /> &middot &#969<sup>2</sup> +
            <input type="text" name="blueprint.volumetric" value="%g" /> &middot p<sub>m</sub>
]], unpack(blueprint.volumetric)) ?>
        </td>
    </tr>
    <tr><td>Thermal efficiency</td>
        <td><?lua return string.format ([[
            <input type="text" name="blueprint.thermal" value="%g" /> +
            <input type="text" name="blueprint.thermal" value="%g" /> &middot &#969 +
            <input type="text" name="blueprint.thermal" value="%g" /> &middot &#969<sup>2</sup>
]], unpack(blueprint.thermal)) ?>
        </td>
    </tr>
    <tr><td>Gass exchange losses (mep)</td>
        <td>
<?lua return string.format ([[
	    <input type="text" name="blueprint.exchange" value="%g" /> 
            &middot (1 -
	    <input type="text" name="blueprint.exchange" value="%g" />
            &middot
            <table class="fraction">
	      <tr><td>p<sub>m</sub></td></tr>
	      <tr><td>p<sub>0</sub></td></tr>
            </table>) Pa
]], unpack(blueprint.exchange)) ?>
        </td>
    </tr>
    <tr><td>Mechanical friction losses (mep)</td>
        <td><?lua return string.format ([[
            <input type="text" name="blueprint.friction" value="%g" /> +
            <input type="text" name="blueprint.friction" value="%g" /> &middot &#969 +
            <input type="text" name="blueprint.friction" value="%g" /> &middot &#969<sup>2</sup> Pa
]], unpack(blueprint.friction)) ?></td>
    </tr>
    <tr><td/><td>
     In the formulas above &#969 is the engine speed in
     <table class="fraction">
       <tr><td>rad</td></tr>
       <tr><td>s</td></tr>
     </table>, p<sub>m</sub> is the manifold absolute pressure and
     p<sub>0</sub> the ambient pressure, both given in Pa.
    </td>
  </table>

  <table class="configuration">
    <tr><th colspan="2">Transmission</th></tr>
    <tr><td>Primary reduction</td>
        <td><input type="text"
                   name="blueprint.primary"
	           value="<?lua return string.format ("%g", blueprint.primary) ?>" /> </td>
    </tr>
    <?lua
       local html = ""
       local suffixes = {"st", "nd", "rd"}

       for i = 1, #blueprint.gearbox do
	  html = html .. string.format([[
    <tr><td>%d%s gear</td><td><input type="text"
                                     name="blueprint.gearbox"
	                             value="%g" /></td></tr>
]], i, i < 4 and suffixes[i] or "th", blueprint.gearbox[i])
       end

       return html
     ?>
  </table>
  <hr/>
  <?lua return fragments.endconfiguration {apply = true, save = true} ?>
  <hr/>
</body>
