/*
 *   $Id: PatternRow.cpp,v 1.14 2006/04/14 15:58:25 rhizome Exp $
 * 
 *      Copyright (C) 2004, 2005, 2006 Alex Marandon
 *
 *  This file is part of slag, a pattern-based audio sequencer.
 *
 *  slag is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  slag is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with slag; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <qlayout.h>
#include <qvbuttongroup.h> 
#include <qlabel.h> 
#include <qtooltip.h> 

#include "../Slag.h"
#include "../constants.h"
#include "../audio_engine/Pattern.h"
#include "../audio_engine/Pad.h"
#include "../audio_engine/Part.h"
#include "../audio_engine/Pattern.h"
#include "../audio_engine/Channel.h"
#include "../audio_engine/Song.h"
#include "PartMenu.h"
#include "PatternRow.h"
#include "RowContext.h"
#include "ChannelMenu.h"
#include "NamedButton.h"
#include "PadButton.h"

PatternRow::PatternRow(QWidget* parent, Song* song, RowContext* row_context)
    : QWidget(parent),
      row_context(row_context),
      channel(row_context->channel), 
      pattern(row_context->pattern), 
      current_part_label(new QLabel(this)),
      layout(new QHBoxLayout(this, 0)),
      new_part_button(new QPushButton(tr("N"), this)),
      part_menu(new PartMenu(this, song, row_context)),
      part_but(new QPushButton(tr("E"), this)),
      song(song)
{
    QToolTip::add(new_part_button, tr("Create a new part"));
    QToolTip::add(part_but, tr("Edit this part"));
    new_part_button->setMaximumSize(20, padButtonHeight);
    
    if (channel->audioFile() == NULL) {
        new_part_button->setEnabled(false);
        part_but->setEnabled(false);
    }
    part_but->setFixedSize(20, padButtonHeight);

    connect(new_part_button, SIGNAL(clicked()),
            part_menu,       SLOT(createBlank()));

    connect(part_menu, SIGNAL(partChanged(Part*)), 
            this,      SLOT(handlePartChanged(Part*)));

    connect(part_but, SIGNAL(clicked()), 
            this,     SLOT(popup_part_menu()));

    connect(channel, SIGNAL(fileChanged()),
            this,    SLOT(handleFileChanged()));

    Part* part;
    
    if ((part = pattern->channels_parts[channel]) != NULL) {
        connect(part, SIGNAL(renamed(Part*)), 
                this, SLOT(handlePartRenamed(Part*)));
        connect(part, SIGNAL(deleted(Part*)), 
                this, SLOT(deactivate(Part*)));
        current_part_label->setText(pattern->channels_parts[channel]->name());
    } else {
        current_part_label->setText(QString("no part"));
    }

    for (uint pad_i = 1; pad_i < song->nbPads() + 1; ++pad_i) {
        QPushButton * button = new PadButton(this);
        buttons.append(button);
        //button->setMinimumSize(10, padButtonWidth);
        button->setMaximumSize(padButtonWidth, padButtonHeight);
        button->setEnabled(0);
        layout->addSpacing(buttonGap);
        layout->addWidget(button);
        if ((pad_i % 4) == 0) layout->addSpacing(stepGap);
    }
    layout->addWidget(new_part_button);
    layout->addWidget(part_but);
    layout->addWidget(current_part_label);
    layout->addStretch();
}

PatternRow::~PatternRow() { 
    qDebug("Destruction PatternRow");
}

void PatternRow::popup_part_menu() {
    part_menu->exec(part_but->mapToGlobal(QPoint(3,3)));
}

void PatternRow::clearConnections() {
    Part* part = pattern->channels_parts[channel];
    disconnect(part, SIGNAL(renamed(Part*)), this, SLOT(handlePartRenamed(Part*)));
    disconnect(part, SIGNAL(deleted(Part*)), this, SLOT(deactivate(Part*)));
    if (part == NULL) return;
    PadIterator pad_it(part->pads().begin());
    QPtrList<QPushButton>::iterator buttons_end = buttons.end();
    for (QPtrList<QPushButton>::iterator butt_it = buttons.begin();
            butt_it != buttons_end; ++butt_it) {
        (*butt_it)->disconnect(SIGNAL(toggled(bool)));
        (*butt_it)->disconnect(SIGNAL(volumeChanged(int)));
        disconnect(*pad_it, SIGNAL(activeStateChanged(bool)), 
                   *butt_it, SLOT(setOn(bool)));
        disconnect(*pad_it,  SIGNAL(volumeChanged(int)), 
                   *butt_it, SLOT(setVolume(int)));
        ++pad_it;
    }
}

void PatternRow::handlePartChanged(Part* new_part) {
    Part* old_part = pattern->channels_parts[channel];
    if (old_part != NULL) clearConnections();
    pattern->setPart(channel, new_part);
    if (new_part != NULL) activate();
}

void PatternRow::handleFileChanged() {
    part_but->setEnabled(true);
    new_part_button->setEnabled(true);
}

void PatternRow::handlePartRenamed(Part* part) {
    current_part_label->setText(part->name());
}

void PatternRow::activate() {
    Part* part = pattern->channels_parts[channel];
    connect(part, SIGNAL(renamed(Part*)), this, SLOT(handlePartRenamed(Part*)));
    connect(part, SIGNAL(deleted(Part*)), this, SLOT(deactivate(Part*)));

    PadIterator pad_it(pattern->channels_parts[channel]->pads().begin());
    QPtrList<QPushButton>::iterator buttons_end = buttons.end();
    for (QPtrList<QPushButton>::iterator butt_it = buttons.begin();
            butt_it != buttons_end; ++butt_it) {
        // on active galement les pads
        (*butt_it)->setToggleButton(true);
        (*butt_it)->setEnabled(true);
        if ( (*pad_it)->isActive())
            (*butt_it)->setOn(1); 
        else 
            (*butt_it)->setOn(0); 

        ((PadButton*) (*butt_it))->setVolume((*pad_it)->getVolumePercent());

        /// Le click sur le bouton modifie l'tat du pad
        connect(*butt_it, SIGNAL(toggled(bool)), 
                *pad_it,  SLOT(setActive(bool)));
        connect(*butt_it,  SIGNAL(volumeChanged(int)), 
                *pad_it, SLOT(setVolume(int)));

        /// Le changement d'tat du Pad est notifi  tous les
        /// boutons associs
        connect(*pad_it,  SIGNAL(activeStateChanged(bool)), 
                *butt_it, SLOT(setOn(bool)));
        connect(*pad_it,  SIGNAL(volumeChanged(int)), 
                *butt_it, SLOT(setVolume(int)));

        ++pad_it;
    }

    current_part_label->setText(pattern->channels_parts[channel]->name());
}

void PatternRow::deactivate(Part*) {
    QPtrList<QPushButton>::iterator buttons_end = buttons.end();
    for (QPtrList<QPushButton>::iterator butt_it = buttons.begin();
            butt_it != buttons_end; ++butt_it) {

        (*butt_it)->setEnabled(false);
        (*butt_it)->setOn(false); 
    }

    current_part_label->setText(tr("None"));
}

//EOF
