/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 *
 */

/*
 * For a detailed description see iso15693(3)
 */

#ifndef _ISO15693_H_
#define _ISO15693_H_

#include <db.h>
#include <rfid_reader.h>

/*
 * Implementation of ISO/IEC 15693-3:2001(E) & ISO/IEC 15693-2
 * for use as a host <-> reader communication protocol.
 *
 * As found at
 * http://www.iso.org/iso/en/CatalogueDetailPage.CatalogueDetail?CSNUMBER=31428&ICS1=35&ICS2=240&ICS3=15
 *
 * I (Loic Dachary <loic@gnu.org>) bought a copy of the document.  I
 * don't need it any more and I'm willing to give it to anyone willing
 * to work on a Free Software implementation of the standard. Although
 * the standard itself is free, the document describing the standard
 * is not under a license that allows distribution. This is why it is
 * not included in the distribution of this software. To comply with
 * the licensing terms of the document, I'll destroy all digital
 * copies of the document I have after giving it to someone.
 * 
 * Alternatively, anyone can get the final draft of the standard at
 * http://www.wg8.de/sd1.html. It is not the same document but it does
 * not differ much.
 *
 * ISO-15693 is NOT a standard covering host <-> reader protocol but
 * reader <-> transponder protocol. However it is sometime used as a
 * host <-> reader protocol by some reader manufacturers (such as
 * Texas Instruments). Most commands can be used as is, with the
 * notable exception of the INVENTORY command that is, by design,
 * unfit to host <-> reader protocol. The INVENTORY command is
 * therefore modified to fit the needs of a host <-> reader protocol
 * in a fashion that depends on the RFID reader implementation. In
 * general, the response to an INVENTORY command is customized to hold
 * a series of DSFID/UID instead of a single one.
 *
 * This ISO-15693 implementation closely follows the published
 * standard with the exception of the INVENTORY command. The actual
 * RFID reader used (driver field) will probably claim to implement
 * ISO-15693 for host <-> reader communications but really implement a
 * protocol that is significantly different from it (although using
 * similar names or frames layout). In this case, it is the
 * responsibility of the driver to hide these differences (an
 * implementation for the Texas Instrument 6500 series should do that,
 * for instance). If the implementation of ISO-15693 proposed by the
 * RFID reader is so different that hiding the differences requires a
 * complex mechanism, it is best to call it a proprietary protocol and
 * not attempt to pretend it's ISO-15693.
 */

typedef struct iso15693 {
  rfid_reader_t* reader;	  /* RFID reader handler, the actual device. */
  int error;			  /* Errors set by iso15693 functions. */
  rfid_io_t* io;		  /* I/O physical device. */
  int verbose;			  /* Control verbosity level. */
  DB* system_information;         /* GET_SYSTEM_INFORMATION support. */
  DBC* system_information_cursor; /* GET_SYSTEM_INFORMATION support. */
} iso15693_t;

/*
 * Abbreviations:
 *
 * AFI Application Family Identifier
 *
 *     AFI represents the type of application targeted by the VCD and
 *     is used to extract from all the VICCs present only the VICCs
 *     meeting the required application criteria. It may be programmed
 *     and locked by the respective commands. AFI is coded on one
 *     byte, which constitutes 2 nibbles of 4 bits each. The most
 *     significant nibble of AFI is used to code one specific or all
 *     application families. The least significant nibble of AFI is
 *     used to code one specific or all application
 *     sub-families. Sub-family codes different from 0 are
 *     proprietary. The support of AFI by the VICC is optional. If AFI
 *     is not supported by the VICC and if the AFI flag is set, the
 *     VICC shall not answer whatever the AFI value is in the
 *     request. If AFI is supported by the VICC, it shall answer
 *     according to the following matching rules:
 *
 *  +------------+-------------+-----------------------+----------------+
 *  |AFI most    | AFI least   |  Meaning              |  Notes		|
 *  |significant | significant |  VICCs respond from   |		|
 *  |nibble      | nibble      |		       |		|
 *  +------------+-------------+-----------------------+----------------+
 *  |    0       |    0        |  All families and sub |  No applicative|
 *  |            |             |  families             |  preselection	|
 *  +------------+-------------+-----------------------+----------------+
 *  |    X       |    0        |  All sub-families of  |  Wide 		|
 *  |            |             |  family X             |  applicative	|
 *  |            |             |                       |  preselection	|
 *  +------------+-------------+-----------------------+----------------+
 *  |    X       |    Y        |  Only the Yth sub     |		|
 *  |            |             |  family of family X   |		|
 *  +------------+-------------+-----------------------+----------------+
 *  |    0       |    Y        |  Proprietary sub      |		|
 *  |            |             |  family Y only	       |		|
 *  +------------+-------------+-----------------------+----------------+
 *
 *  if(inventory request received) {
 *    if(! AFII flag set)
 *       return inventory;
 *    if(! AFI supported by VICC)
 *       return NO inventory
 *    if(AFI value == 0)
 *       return inventory
 *    if(AFI value == VICC's AFI)
 *       return inventory
 *    else
 *       return NO inventory
 *  }
 *
 * CRC Cyclic Redundancy Check
 *
 *     The CRC shall be calculated in accordance with ISO/IEC
 *     13239. The initial register content shall be all ones:
 *     'FFFF'. The two bytes CRC are appended to each request and each
 *     response, within each frame, before the EOF. The CRC is
 *     calculated on all the bytes after the SOF up to but not
 *     including the CRC field. Upon reception of a request from the
 *     VCD, the VICC shall verify that the CRC value is valid. If it
 *     is invalid, it shall discard the frame and shall not answer
 *     (modulate). Upon reception of a response from the VICC, it is
 *     recommended that the VCD verify that the CRC value is valid. If
 *     it is invalid, actions to be performed are left to the
 *     responsibility of the VCD designer. The CRC is transmitted
 *     least significant byte first. Each byte is transmitted least
 *     significant bit first.
 *
 * DSFID Data Storage Format IDentifier
 *
 *       The Data storage format identifier indicates how the data is
 *       structured in the VICC memory. It may be programmed and
 *       locked by the respective commands. It is coded on one
 *       byte. It allows for instant knowledge on the logical
 *       organisation of the data. If its programming is not supported
 *       by the VICC, the VICC shall respond with the value zero
 *       ('00').
 *
 * RFU Reserved for Future Use
 *
 * UID Unique identifier
 *
 *     The VICCs are uniquely identified by a 64 bits unique
 *     identifier (UID). This is used for addressing each VICC
 *     uniquely and individually, during the anticollision loop and
 *     for one-to-one exchange between a VCD and a VICC. The UID shall
 *     be set permanently by the IC manufacturer.
 *
 *     MSB                                                   LSB
 *     +--------+--------------+-------------------------------+
 *     | 64  57 | 56        49 | 48                          1 |
 *     +--------+--------------+-------------------------------+
 *     |   E0   |  IC Mfg code | IC manufacturer serial number |
 *     +--------+--------------+-------------------------------+
 *
 * VCD Vicinity Coupling Device (or RFID reader)
 *
 * VICC Vicinity Integrated Circuit Card (or RFID transponder or tag)
 */

/*
 * Size, in bytes, of all command/response fields.
 */
#define ISO15693_FLAGS_SIZE			1
#define ISO15693_INFO_FLAGS_SIZE		1
#define ISO15693_COMMAND_SIZE			1
#define ISO15693_CRC_SIZE			2
#define ISO15693_AFI_SIZE			1
#define ISO15693_MASK_LENGTH_SIZE		1
#define ISO15693_MASK_VALUE_SIZE		8
#define ISO15693_UID_SIZE			8
#define ISO15693_DSFID_SIZE			1
#define ISO15693_BLOCK_NUMBER_SIZE		1
#define ISO15693_BLOCK_SECURITY_STATUS_SIZE	1
#define ISO15693_VICC_MEMORY_SIZE_SIZE		2
#define ISO15693_IC_REFERENCE_SIZE		1
/* Not in ISO15693, added for INVENTORY command */
#define NOT_ISO15693_VALID_DATA_FLAGS_SIZE	2
#define NOT_ISO15693_COLLISION_FLAGS_SIZE	2

/*
 * Header, trailer and overhead fixed sizes for command and response.
 */
#define ISO15693_COMMAND_HEADER_SIZE		(ISO15693_FLAGS_SIZE + ISO15693_COMMAND_SIZE)
#define ISO15693_COMMAND_TRAILER_SIZE		(ISO15693_CRC_SIZE)
#define ISO15693_COMMAND_OVERHEAD		(ISO15693_COMMAND_HEADER_SIZE + ISO15693_COMMAND_TRAILER_SIZE)

#define ISO15693_RESPONSE_HEADER_SIZE		(ISO15693_FLAGS_SIZE)
#define ISO15693_RESPONSE_TRAILER_SIZE		(ISO15693_CRC_SIZE)
#define ISO15693_RESPONSE_OVERHEAD		(ISO15693_RESPONSE_HEADER_SIZE + ISO15693_RESPONSE_TRAILER_SIZE)

/*
 * Index of the fields that have a fixed position in the ISO-15693 frames.
 */
#define ISO15693_FLAGS_INDEX			0
#define ISO15693_COMMAND_INDEX			1
#define ISO15693_ERROR_INDEX			1
#define ISO15693_COMMAND_DATA_INDEX		2
#define ISO15693_RESPONSE_DATA_INDEX		1
#define ISO15693_INFO_FLAGS_INDEX		1
#define NOT_ISO15693_VALID_DATA_FLAGS_LSB_INDEX	1
#define NOT_ISO15693_VALID_DATA_FLAGS_MSB_INDEX	2
#define NOT_ISO15693_COLLISION_FLAGS_LSB_INDEX	3
#define NOT_ISO15693_COLLISION_FLAGS_MSB_INDEX	4

/*
 * VICC memory organization. The commands specified in ISO-15693-3
 * assume that the physical memory is organized in blocks (or pages)
 * of fixed size. Up to 256 blocks can be addressed. Block size
 * can be of up to 256 bits. This leads to a maximum memory capacity
 * of up to 8 kBytes (64 kBits) (see ISO_IEC_15693-3 page 6).
 *
 * NOTE: The structure allows for future extension of the maximum
 * memory capacity. The commands described in this standard allow the
 * access (read and write) by block(s). There is no implicit or
 * explicit restriction regarding other access method (e.g. by byte or
 * by logical object in future revision(s) of the standard or in
 * custom commands).
 *
 */
#define ISO15693_BLOCK_COUNT_MAX	255
#define ISO15693_BLOCK_SIZE_MAX		32   /* bytes */
#define ISO15693_MEMORY_MAX		(ISO15693_BLOCK_COUNT_MAX * ISO15693_BLOCK_SIZE_MAX)
/*
 * Maximum size, in bytes, of a ISO-15693 command or answer.
 * The extreme case of reference is a read_multiple_blocks with 
 * a transponder that contains 255 blocks of 32 bytes.
 * 
 */
#define ISO15693_BUFFER_MAX		(ISO15693_MEMORY_MAX * 2)

/*
 * Bit description of the GET_SYSTEM_INFORMATION flags field
 * and the info_flags field of the iso15693_transponder_t
 * structure.
 */
#define ISO15693_INFO_FLAGS_DSFID		0x01
#define ISO15693_INFO_FLAGS_AFI			0x02
#define ISO15693_INFO_FLAGS_VICC_MEMORY_SIZE	0x04
#define ISO15693_INFO_FLAGS_IC_REFERENCE	0x08

/*
 * Meta information about the RFID transponder. (ro) means
 * readonly, (rw) means read-write.
 */
typedef struct {
  u_int8_t info_flags;          /* Meaning is the same as in
				   GET_SYSTEM_INFORMATION command. */
  u_int8_t uid[ISO15693_UID_SIZE]; /* Unique identifier (ro). */
  u_int8_t dsfid;		/* Data storage format identifier (rw). */
  u_int8_t afi;			/* Application family identifier (rw). */
  u_int16_t vicc_memory;	/* Blocks layout information (ro). */
  u_int8_t ic_ref;		/* Proprietary information (ro). */
} iso15693_transponder_t;

/*
 * The block security status is sent back by the VICC as a parameter
 * in the response to a VCD request as specified in clause 10
 * (e.g. Read single block). It is coded on one byte. It is an element
 * of the protocol. There is no implicit or explicit assumption that
 * the 8 bits are actually implemented in the physical memory
 * structure of the VICC.
 */
#define ISO15693_BLOCK_LOCKED		0x01

/*
 * AFI stands for Application family identifier.
 *  page 4, table 1 AFI coding
 */
#define ISO15693_AFI_NONE			0x00
#define ISO15693_AFI_TRANSPORT			0x10
#define ISO15693_AFI_FINANCIAL			0x20
#define ISO15693_AFI_IDENTIFICATION		0x30
#define ISO15693_AFI_TELECOMMUNICATION		0x40
#define ISO15693_AFI_MEDICAL			0x50
#define ISO15693_AFI_MULTIMEDIA			0x60
#define ISO15693_AFI_GAMING			0x70
#define ISO15693_AFI_DATA_STORAGE		0x80
#define ISO15693_AFI_ITEM_MANAGEMENT		0x90
#define ISO15693_AFI_EXPRESS_PARCELS		0xA0
#define ISO15693_AFI_POSTAL_SERVICES		0xB0
#define ISO15693_AFI_AIRLINE_BAGS		0xC0

/*
 * General command format
 * +------------------------------------------------------------+
 * | SOF | FLAGS | COMMAND CODE | PARAMETERS | DATA | CRC | EOF |
 * +-----+-------+--------------+------------+------+-----+-----+
 * |  0  |   1   |      1       |     n      |   n  |  2  |  0  |
 * +------------------------------------------------------------+
 * The second line shows the number of bytes.
 */

/*
 * Request flags. In a request, the field "flags" specifies the
 * actions to be performed by the VICC and whether corresponding
 * fields are present or not. It consists of eight bits.
 */

/*
 * Unset: A single sub-carrier frequency shall be used by the VICC.
 * Set: Two sub-carriers shall be used by the VICC.
 */
#define ISO15693_COMMAND_FLAGS_SUB_CARRIER		0x01
/*
 * Unset: Low data rate shall be used.
 * Set: High data rate shall be used.
 */
#define ISO15693_COMMAND_FLAGS_DATA_RATE		0x02
/*
 * Unset: Flags bits 5 and 6 meaning is SELECT, ADDRESS.
 * Set: Flags bits 5 and 6 meaning is INVENTORY_AFI, NB_SLOTS.
 */
#define ISO15693_COMMAND_FLAGS_INVENTORY		0x04
/*
 * Unset: No protocol format extension.
 * Set: Protocol format is extended. Reserved for future use.
 */
#define ISO15693_COMMAND_FLAGS_PROTOCOL_EXTENSION	0x08
/*
 * (if ISO15693_COMMAND_FLAGS_INVENTORY is NOT set)
 * Unset: Request shall be executed by any VICC according to
 *        the setting of ADDRESS flag.
 * Set: Request shall be executed only by VICC in selected state.
 *      The ADDRESS flag shall be set to 0 and the UID field shall
 *      not be included in the request.
 */
#define ISO15693_COMMAND_FLAGS_SELECT			0x10
/*
 * (if ISO15693_COMMAND_FLAGS_INVENTORY is NOT set)
 * Unset: Request is not addressed. UID field is not included.
 *        it shall be executed by any VICC.
 * Set: Request is addressed. UID field is included. It shall
 *      be executed only by the VICC whose UID matches the UID
 *      specified in the request.
 */
#define ISO15693_COMMAND_FLAGS_ADDRESS			0x20
/*
 * (if ISO15693_COMMAND_FLAGS_INVENTORY is set)
 * Unset: AFI field is not present.
 * Set: AFI field is present.
 */
#define ISO15693_COMMAND_FLAGS_AFI			0x10
/*
 * (if ISO15693_COMMAND_FLAGS_INVENTORY is set)
 * Unset: 16 slots.
 * Set: 1 slot.
 */
#define ISO15693_COMMAND_FLAGS_NB_SLOTS			0x20
/*
 * Unset: Meaning is defined by the command description it shall.
 *        be set to 0 if not otherwise defined by the command.
 * Set: Meaning is defined by the command description.
 */
#define ISO15693_COMMAND_FLAGS_OPTION			0x40
/*
 * Reserved for future use (RFU)
 */
#define ISO15693_COMMAND_FLAGS_RESERVED			0x80

/*
 * Error response format
 * +---------------------------------------------+
 * | SOF | FLAGS |    ERROR CODE     | CRC | EOF |
 * +-----+-------+-------------------+-----+-----+
 * |  0  |   1   |         1         |  2  |  0  |
 * +---------------------------------------------+
 * The second line shows the number of bytes.
 */

#define ISO15693_ERROR_RESPONSE_SIZE	1
#define ISO15693_ERROR_FRAME_SIZE	(ISO15693_FLAGS_SIZE + ISO15693_ERROR_RESPONSE_SIZE + ISO15693_CRC_SIZE)

/*
 * Unset: No error.
 * Set: Error detected. Error code is in the "Error" field.
 */
#define ISO15693_RESPONSE_FLAG_ERROR			0x01
/*
 * Unset: No protocol format extension.
 * Set: Protocol format is extended. Reserved for future use.
 */
#define ISO15693_RESPONSE_FLAG_EXTENSION		0x08

/*
 * Response error code. when the ERROR flag is set by the VICC, the
 * error code field shall be included and provides information about
 * the error that occurred. Error codes are defined below. If the VICC
 * does not support specific error code(s) listed here, it shall
 * answer with the error code '0F' ("Error with no information
 * given").
 */
#define ISO15693_SUCCESS				0x00
/*
 * The command is not supported, i.e. the request code is not recognized.
 */
#define ISO15693_ERROR_NOT_SUPPORTED			0x01
/*
 * The command is not recognized, for example: a format error occured.
 */
#define ISO15693_ERROR_NOT_RECOGNIZED			0x02
/*
 * The command option is not supported.
 */
#define ISO15693_ERROR_OPTION_NOT_SUPPORTED		0x03
/*
 * Error with no information given or a specific error code is not supported.
 */
#define ISO15693_ERROR_NO_INFORMATION			0x0F
/*
 * The specified block is not available (doesn't exist)
 */
#define ISO15693_ERROR_BLOCK_NOT_AVAILABLE		0x10
/*
 * The specified block is already locked and thus cannot be locked again.
 */
#define ISO15693_ERROR_BLOCK_CANNOT_RELOCK		0x11
/*
 * The specified block is locked and its content cannot be changed.
 */
#define ISO15693_ERROR_BLOCK_CANNOT_WRITE		0x12
/*
 * The specified block was not successfully programmed.
 */
#define ISO15693_ERROR_BLOCK_CANNOT_PROGRAM		0x13
/*
 * The specified block was not successfully locked.
 */
#define ISO15693_ERROR_BLOCK_CANNOT_LOCK		0x14

/*
 * Not part of ISO-15693. 
 */
/*
 * The requested transponder (addressed mode) was not found.
 */
#define NOT_ISO15693_ERROR_TRANSPONDER_NOT_FOUND	0x15
/*
 * The flag combination is incorrect.
 */
#define NOT_ISO15693_ERROR_INVALID_FLAGS		0x16
/*
 * A collision occured after an INVENTORY request.
 */
#define NOT_ISO15693_ERROR_COLLISION			0x17

/*
 * VICC states. A VICC can be in one of the 4 following states:
 * Power-off, Ready, Quiet, Selected. The transition between these
 * states is specified below. The support of power-off, ready
 * and quiet states is mandatory. The support of selected state is
 * optional.
 *
 * Power-off state: The VICC is in the power-off state
 * when it cannot be activated by the VCD.
 *
 * Ready state: The VICC is in the Ready state when it is activated by
 * the VCD. It shall process any request where the select flag is not
 * set.
 *
 * Quiet state: When in the quiet state, the VICC shall process any
 * request where the inventory flag is not set and where the address
 * flag is set.
 *
 * Selected state: Only a VICC in the selected state shall process
 * requests having the select flag set.
 *

   			 +---------+
  +--------------------> |  Power  |<------------------------+
  |   Out of field     	 |   off   |   	    Out of field     |
  |    		       	 +---------+ 	      		     |
  |    	   .	       	    ^ 	|    	      		     |
  |    		       	    |  	|    	      		     |
  |    		      Out   |	| In 	      		     |
  |    		      of    |	| field	      		     |
  |    	       	      field |	|      	      		     |
  | 		       	    |	v    -------  		     |
  |            	    +--------------+/  	    \ Any	     |
  |                 |              |   	    / command	     |
  |            	    |    Ready     |<-------  with   	     |
  |	 +--------->|              |---\    not select flat  |
  |	 |     	    +--------------+   	\      		     |
  |    	 |     	      |          ^     	 \     		     |
  |   Reset    	      |          \    	  \    	 Select uid  |
  |   to |     	      /         Reset  	   ---------+  	     |
  |   Ready    	   Stay	       	to ready       	    |  	     |
  |	 |	 quiet (UID)	with select flag    |  	     |
  |	 |	   /		OR select(!UID)	    |	     |
  |      |        /                   ------\       |        |
  |      |       v                           \      v        |
  |   +---------+       select (UID)       +-----------+     |
  +---|         |------------------------->|           |-----+
      |	 Quiet 	|      	       	       	   |  Selected |
      |         |<-------------------------|           |
      +---------+     stay quiet (uid)     +-----------+
        /    ^          	       	       /    ^
       /      \         	  	      /	     \
       \      /         	  	      \	     /
        \    /          	  	       \    /
         ----          	       	       	        ----
     Any other command where 	  	     Any other command
     address flag is set
     and inventory flag is not set
 *
 *
 */

/*
 * General command format
 * +------------------------------------------------------------+
 * | SOF | FLAGS | COMMAND CODE | PARAMETERS | DATA | CRC | EOF |
 * +-----+-------+--------------+------------+------+-----+-----+
 * |  0  |   1   |      1       |     n      |   n  |  2  |  0  |
 * +------------------------------------------------------------+
 * The second line shows the number of bytes.
 */

/*
 * Command codes
 */

/*
 * INVENTORY:
 *
 * When issuing the inventory command, the VCD shall set the
 * nb_slots flag to the desired setting and add after the command
 * field the mask length and the mask value. The mask length indicates
 * the number of significant bits of the mask value. It can have any
 * value between 0 and 60 when 16 slots are used and any value between
 * 0 and 64 when 1 slot is used. The mask value is contained in an
 * integer number of bytes. If the mask length is not a multiple of 8
 * (bits), the mask value MSB shall be padded with the required number
 * of null (set to 0) bits so that the mask value is contained in an
 * integer number of bytes. The next field starts on the next byte
 * boundary.
 *
 * The inventory flag shall be set.
 *
 * Command:
 * --+---------------+-------------+--------------------------+--
 * ..| Opptional AFI | Mask length | Mask value               |..
 * --+---------------+-------------+--------------------------+--
 * ..|     8 bits    |   8 bits    |   0-64 bits              |..
 * --+---------------+-------------+--------------------------+--
 *
 * Response:
 * --+--------------+----------------------------------------+--
 * ..|  DSFID       |        UID                             |..
 * --+--------------+----------------------------------------+--
 * ..|  8 bits      |        64 bits                         |..
 * --+--------------+----------------------------------------+--
 *
 * The drivers implementing access (s6350, m2xxh...) are required
 * to implement the INVENTORY logic (as described by ISO/IEC 15693-3:2001(E)
 * page 21,22,23) and and to return a cooked response :
 * 
 * Response:
 * --+-------+---------+--------+----------------------------+--
 * ..| Valid | Collide | DSFID  |         UID                |..
 * --+-------+---------+--------+----------------------------+--
 * ..|16 bits| 16 bits | 8 bits |      64 bits               |..
 * --+-------+---------+--------+----------------------------+--
 *                     \- repeat number of 1 bits in Valid --/
 *
 * Each bit set to one in the Valid field counts for 1 set of
 * of DSFID/UID. If Valid == 0xFF, there are 16 DSFID/UID fields.
 *
 * Valid: This 16-bit field corresponds to whether valid data was
 * received in the 16 possible Time Slots. Bits 0 to 7 of the LSB
 * respectively correspond to Time Slots 1 to 8, while bits 0 to 7 of
 * the MSB correspond to Time Slots 9 to 16 respectively. A set bit
 * corresponds to valid data being received in that particular Time
 * slot.
 *
 * Collide: This 16-bit field corresponding to whether a collision
 * occurred in the 16 possible Time Slots. Bits 0 to 7 of the LSB
 * respectively correspond to Time Slots 1 to 8, while bits 0 to 7 of
 * the MSB correspond to Time Slots 9 to 16 respectively. A set bit
 * corresponds to a collision being detected in that particular Time
 * Slot.
 *
 */
#define ISO15693_COMMAND_INVENTORY				0x01
#define ISO15693_COMMAND_INVENTORY_RESPONSE_SIZE		(ISO15693_DSFID_SIZE + ISO15693_UID_SIZE)

/*
 * Number of slots explored by the INVENTORY command.
 */
#define ISO15693_NB_SLOTS_DEFAULT		16
#define ISO15693_NB_SLOTS_DEFAULT_BITS		4
#define ISO15693_NB_SLOTS_ONE			1
#define ISO15693_NB_SLOTS_ONE_BITS		1

/*
 * Input parameter of the inventory command.
 */
typedef struct {
  u_int8_t one_slot;				 /* If set NB_SLOTS is 1 instead
						    of 16 (default) */
  u_int8_t mask_length;				 /* Number of valid BITS in 
						    mask_value. If 0, it is ignored. 
						 */
  u_int8_t mask_value[ISO15693_MASK_VALUE_SIZE]; /* Mask used for inventory pattern
						    matching algorithm. See ISO-15693
						    for a description of the matching
						    algorithm. */
  u_int8_t afi;					 /* Mask used for the inventory 
						    pattern matching algorithm. See
						    AFI Application family identifier
						    above for a table explaining
						    how it is used. If 0, it is
						    ignored. */
} iso15693_inventory_t;

/*
 * Output result of the inventory command.
 */
typedef struct {
  u_int16_t valid;			/* Validity of the data array. */
  int nvalid;			        /* Total number of valid slots. */
  u_int16_t collide;			/* Collision information. */
  struct {
    u_int8_t dsfid;			/* Data Storage Format Identifier. */
    u_int8_t uid[ISO15693_UID_SIZE];	/* Unique identifier. */
  } data[ISO15693_NB_SLOTS_DEFAULT];
} iso15693_inventory_result_t;

/*
 * The INVENTORY command is sent using the <inventory> parameters. If
 * no transponder is found *resultsp and *results_lengthp are set to 0.
 * If at least one transponder is found, an array of iso15693_inventory_result_t
 * structures is malloc'ed and returned in *resultsp. The number of transponders
 * is set to *results_lengthp. It is the responsibility of the caller
 * to free the *resultsp array.
 *
 * Since ISO-15693 is NOT a host <-> reader protocol but a reader <->
 * transponder protocol, the INVENTORY command cannot be used as-is by
 * readers that use ISO-15693 as a host <-> reader protocol. An INVENTORY
 * command in a host <-> reader protocol needs to return many transponder
 * descriptions, matching the transponders currently detected by the
 * reader.
 */
int iso15693_inventory(iso15693_t* iso15693, iso15693_inventory_t* inventory, iso15693_inventory_result_t* result);

/*
 * Full inventory of the RFID transponders currently in the field
 * of the RFID reader. Contrary to the iso15693_inventory function,
 * it resolves collisions.
 */
int iso15693_inventory_logic(struct rfid_reader* reader, iso15693_inventory_t inventory, rfid_transponders_t* transponders);

/*
 * STAY_QUIET:
 *
 * When receiving the Stay quiet command, the VICC shall enter the
 * quiet state and shall NOT send back a response. There is NO
 * response to the Stay quiet command. When in quiet state: a) the VICC
 * shall not process any request where inventory flag is set, b) the
 * VICC shall process any addressed request. The VICC shall exit the
 * quiet state when: a) reset (power off), b) receiving a Select
 * request (it shall then go to the selected state if supported or
 * return an error if not supported), c) receiving a Reset to ready
 * request (it shall then go to the Ready state).
 *
 * Command:
 * --+------------------------------------------------------+--
 * ..|                       UID                            |..
 * --+------------------------------------------------------+--
 * ..|                      64 bits                         |..
 * --+------------------------------------------------------+--

 */
#define ISO15693_COMMAND_STAY_QUIET				0x02

/*
 * The STAY_QUIET command is sent to the transponder with 
 * UID equal to <uid> (ISO15693_UID_SIZE bytes).
 * 
 */
int iso15693_stay_quiet(iso15693_t* iso15693, u_int8_t* uid);

/*
 * READ_SINGLE_BLOCK:
 *
 * When receiving the Read single block command, the VICC shall read
 * the requested block and send back its value in the response. If the
 * option flag is set in the request, the VICC shall return the block
 * security status, followed by the block value. If it is not set, the
 * VICC shall return only the block value.
 *
 * Command:
 * --+-------------------+-----------------------------------+--
 * ..|(address flg) UID  |       Block number                |..
 * --+-------------------+-----------------------------------+--
 * ..|        64 bits    |          8 bits                   |..
 * --+-------------------+-----------------------------------+--
 *
 * Response:
 * --+--------------------------+----------------------------+--
 * ..| Block security status    |         Data               |..
 * --+--------------------------+----------------------------+--
 * ..|          8 bits          |   Block length             |..
 * --+--------------------------+----------------------------+--
 */
#define ISO15693_COMMAND_READ_SINGLE_BLOCK			0x20

/*
 * The READ_SINGLE_BLOCK is sent to the first available transponder if
 * the UID is not set in <transponder> (iso15693_transponder_null) or
 * to a designated transponder. Meta information of <transponder> must
 * be set (using GET_SYSTEM_INFORMATION for instance) before calling
 * this function. The requested block number is <block->block_number>. If
 * <block->security_status> is not 0, the block security status will
 * also be requested. On success, the <block->data> area is filled
 * with the content of block number <block->block_number>. If the block
 * security status was requested, the <block->security_status> is set
 * with its value. If it was not requested, the contend of
 * <block->security_status> is undefined.  It is the responsibility of
 * the caller to set <block->data> to a pointer with enough room to
 * store the content of a block.
 */
int iso15693_read_single_block(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* block);

/*
 * WRITE_SINGLE_BLOCK:
 *
 * When receiving the Write single block command, the VICC shall write
 * the requested block with the data contained in the request and
 * report the success of the operation in the response. If the option
 * flag is not set, the VICC shall return its response when it has
 * completed the write operation. If it is set, the VICC shall wait
 * for the reception of an EOF from the VCD and upon such reception
 * shall return its response.
 *
 * Command:
 * --+------------------+--------------+----------------------+--
 * ..|(address flg) UID | Block number |         Data         |..
 * --+------------------+--------------+----------------------+--
 * ..|     64 bits      |    8 bits    |    Block length      |..
 * --+------------------+--------------+----------------------+--
 *
 * Response:
 * --+------------------------------------------------------+--
 * ..|                   (void)                             |..
 * --+------------------------------------------------------+--
 * ..|                                                      |..
 * --+------------------------------------------------------+--
 */
#define ISO15693_COMMAND_WRITE_SINGLE_BLOCK			0x21

/*
 * The WRITE_SINGLE_BLOCK is sent to the first available transponder
 * if the UID is not set in <transponder> (iso15693_transponder_null)
 * or to a designated transponder. Meta information of <transponder>
 * must be set (using GET_SYSTEM_INFORMATION for instance) before
 * calling this function. On success, <transponder->bytes_per_block> bytes
 * are copied from <block->data> to the <block->block_number> block of
 * the transponder.
 */
int iso15693_write_single_block(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* block);

/*
 * LOCK_BLOCK:
 *
 * When receiving the Lock block command, the VICC shall lock
 * permanently the requested block. If the option flag is not set, the
 * VICC shall return its response when it has completed the lock
 * operation. If it is set, the VICC shall wait for the
 * reception of an EOF from the VCD and upon such reception shall
 * return its response.
 *
 * Command:
 * --+-------------------+-----------------------------------+--
 * ..|(address flg) UID  |       Block number                |..
 * --+-------------------+-----------------------------------+--
 * ..|        64 bits    |          8 bits                   |..
 * --+-------------------+-----------------------------------+--
 * 
 * Response:
 * --+------------------------------------------------------+--
 * ..|                   (void)                             |..
 * --+------------------------------------------------------+--
 * ..|                                                      |..
 * --+------------------------------------------------------+--
 */
#define ISO15693_COMMAND_LOCK_BLOCK				0x22

/*
 * The LOCK_BLOCK is sent to the first available transponder if the
 * UID is not set in <transponder> (iso15693_transponder_null) or to a
 * designated transponder. On success the <block->block_number> block
 * is locked and becomes readonly.
 */
int iso15693_lock_block(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* block);

/*
 * READ_MULTIPLE_BLOCKS:
 *
 * When receiving the Read multiple block command, the VICC shall read
 * the requested block(s) and send back their value in the
 * response. If the option flag is set in the request, the VICC shall
 * return the block security status, followed by the block value
 * sequentially block by block. If the option flag is not set in the
 * request, the VICC shall return only the block value. The blocks are
 * numbered from '00' to 'FF' (0 to 255). The number of blocks in the
 * request is one less than the number of blocks that the VICC shall
 * return in its response. EXAMPLE A value of '06' in the "Number of
 * blocks" field requests to read 7 blocks. A value of '00' requests
 * to read a single block.
 *
 * Command:
 * --+-------------------+-----------------+------------------+--
 * ..|(address flg) UID  | First block num | Number of blocks |..
 * --+-------------------+-----------------+------------------+--
 * ..|        64 bits    |     8 bits      |      8 bits      |..
 * --+-------------------+-----------------+------------------+--
 *
 * Response:
 * --+------------------------+------------------------------+--
 * ..| Block security status  |             Data             |..
 * --+------------------------+------------------------------+--
 * ..|       8 bits           |      Block length            |..
 * --+------------------------+------------------------------+--
 *   \------------ repeat Number of blocks times ------------/
 */
#define ISO15693_COMMAND_READ_MULTIPLE_BLOCKS			0x23

/*
 * The READ_MULTIPLE_BLOCK is sent to the first available transponder
 * if the UID is not set in <transponder> (iso15693_transponder_null)
 * or to a designated transponder. Meta information of <transponder>
 * must be set (using GET_SYSTEM_INFORMATION for instance) before
 * calling this function. On success, the <blocks[i]->data> area of
 * each <blocks> is filled with the content of the consecutive blocks
 * starting at block number <blocks[0]->block_number> and up to
 * <blocks_length> blocks.  The <block->security_status> is set with
 * the security status of each block. It is the responsibility of the
 * caller to set <block->data> to a pointer with enough room to store
 * the content of a block.
 */
int iso15693_read_multiple_blocks(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* blocks, int blocks_length);

/*
 * WRITE_MULTIPLE_BLOCKS:
 *
 * When receiving the Write multiple block command, the VICC shall
 * write the requested block(s) with the data contained in the request
 * and report the success of the operation in the response. If the
 * option flag is not set, the VICC shall return its response when it
 * has completed the write operation. If it is set, the VICC shall
 * wait for the reception of an EOF from the VCD and upon such
 * reception shall return its response. The blocks are numbered from
 * '00' to 'FF' (0 to 255). The number of blocks in the request is one
 * less than the number of blocks that the VICC shall write. EXAMPLE A
 * value of '06' in the "Number of blocks" field requests to write 7
 * blocks. The "Data" field shall contain 7 blocks. A value of '00' in
 * the "Number of blocks" field requests to write 1 block. The "Data"
 * field shall contain 1 block.
 *
 *
 * Command:
 * --+-------------------+-------------+----------+------------+--
 * ..|(address flg) UID  | First block |Num blocks|   Data     |..
 * --+-------------------+-------------+----------+------------+--
 * ..|        64 bits    |    8 bits   |   8 bits |Block length|..
 * --+-------------------+-------------+----------+------------+--
 *                                                 \- Num ----/
 *                                                    blocks 
 *                                                    time
 *
 * Response:
 * --+------------------------------------------------------+--
 * ..|                   (void)                             |..
 * --+------------------------------------------------------+--
 * ..|                                                      |..
 * --+------------------------------------------------------+--
 */
#define ISO15693_COMMAND_WRITE_MULTIPLE_BLOCKS			0x24

/*
 * The WRITE_MULTIPLE_BLOCK is sent to the first available transponder
 * if the UID is not set in <transponder> (iso15693_transponder_null)
 * or to a designated transponder. Meta information of <transponder>
 * must be set (using GET_SYSTEM_INFORMATION for instance) before
 * calling this function. On success, <transponder->bytes_per_block> bytes
 * are copied from each <block[i]->data> to the corresponding block on
 * the transponder. This command writes consecutive blocks only. The first
 * block is <blocks[0]->block_number>. The next block written will be 
 * <blocks[0]->block_number> + 1 and so on up to <blocks[0]->block_number> + 
 * <blocks_length>. The <blocks[i > 0]->block_number> fields are ignored.
 */
int iso15693_write_multiple_blocks(iso15693_t* iso15693, const rfid_transponder_t* transponder, rfid_block_t* blocks, int blocks_length);

/*
 * SELECT:
 *
 * When receiving the Select command:
 *
 *  if the UID is equal to its own UID, the VICC shall enter the
 *   selected state and shall send a response. 
 *
 * . if it is different, the VICC shall return to the Ready state and
 *   shall not send a response. The Select command shall always be
 *   executed in addressed mode. (The select flag is set to 0. The
 *   address flag is set to 1.)
 *
 * Command:
 * --+-------------------------------------------------------+--
 * ..|                      UID                              |..
 * --+-------------------------------------------------------+--
 * ..|                    64 bits                            |..
 * --+-------------------------------------------------------+--
 *
 * Response:
 * --+------------------------------------------------------+--
 * ..|                   (void)                             |..
 * --+------------------------------------------------------+--
 * ..|                                                      |..
 * --+------------------------------------------------------+--
 *
 */
#define ISO15693_COMMAND_SELECT					0x25

/*
 * The SELECT command is sent to the transponder with 
 * UID equal to <uid> (ISO15693_UID_SIZE bytes).
 */
int iso15693_select(iso15693_t* iso15693, const u_int8_t* uid);

/*
 * RESET_TO_READY:
 *
 * When receiving a Reset to ready command, the VICC shall return to
 * the Ready state.
 *
 * Command:
 * --+-------------------------------------------------------+--
 * ..|        (address flg) UID                              |..
 * --+-------------------------------------------------------+--
 * ..|                    64 bits                            |..
 * --+-------------------------------------------------------+--
 *
 * Response:
 * --+------------------------------------------------------+--
 * ..|                   (void)                             |..
 * --+------------------------------------------------------+--
 * ..|                                                      |..
 * --+------------------------------------------------------+--
 *
 */
#define ISO15693_COMMAND_RESET_TO_READY				0x26

/*
 * The RESET_TO_READY command is sent to the transponder with 
 * UID equal to <uid> (ISO15693_UID_SIZE bytes). If <uid> is all
 * 0, the RESET_TO_READY command is sent to whatever transponder
 * is available.
 */
int iso15693_reset_to_ready(iso15693_t* iso15693, const u_int8_t* uid);

/*
 * WRITE_AFI:
 *
 * When receiving the Write AFI request, the VICC shall write the AFI
 * value into its memory. If the option flag is not set, the VICC
 * shall return its response when it has completed the write
 * operation. If it is set, the VICC shall wait for the reception of
 * an EOF from the VCD and upon such reception shall return its
 * response.
 *
 * Command:
 * --+-------------------+-----------------------------------+--
 * ..|(select flag) UID  |           AFI                     |..
 * --+-------------------+-----------------------------------+--
 * ..|        64 bits    |          8 bits                   |..
 * --+-------------------+-----------------------------------+--
 *
 * Response:
 * --+------------------------------------------------------+--
 * ..|                   (void)                             |..
 * --+------------------------------------------------------+--
 * ..|                                                      |..
 * --+------------------------------------------------------+--
 *
 */
#define ISO15693_COMMAND_WRITE_AFI				0x27

/*
 * The WRITE_AFI is sent to the first available transponder
 * if the UID is not set in <transponder> (iso15693_transponder_null)
 * or to a designated transponder. On success the AFI byte of
 * <transponder> is written to the transponder. The function
 * iso15693_transponder_afi_set function must be used by the caller
 * to set the AFI byte in <transponder>.
 */
int iso15693_write_afi(iso15693_t* iso15693, const rfid_transponder_t* transponder);

/*
 * LOCK_AFI:
 *
 * When receiving the Lock AFI request, the VICC shall lock the AFI
 * value permanently into its memory. If the option flag is not set,
 * the VICC shall return its response when it has completed the lock
 * operation. If it is set, the VICC shall wait for the
 * reception of an EOF from the VCD and upon such reception shall
 * return its response.
 *
 * Command:
 * --+-------------------------------------------------------+--
 * ..|        (address flg) UID                              |..
 * --+-------------------------------------------------------+--
 * ..|                    64 bits                            |..
 * --+-------------------------------------------------------+--
 *
 * Response:
 * --+------------------------------------------------------+--
 * ..|                   (void)                             |..
 * --+------------------------------------------------------+--
 * ..|                                                      |..
 * --+------------------------------------------------------+--
 */
#define ISO15693_COMMAND_LOCK_AFI				0x28

/*
 * The LOCK_AFI command is sent to the transponder with 
 * UID equal to <uid> (ISO15693_UID_SIZE bytes). If <uid> is all
 * 0, the LOCK_AFI command is sent to whatever transponder
 * is available. On success the AFI byte becomes readonly.
 */
int iso15693_lock_afi(iso15693_t* iso15693, const u_int8_t* uid);

/*
 * WRITE_DSFID:
 *
 * When receiving the Write DSFID request, the VICC shall write the
 * DSFID value into its memory. If the Option_flag is not set, the
 * VICC shall return its response when it has completed the write
 * operation. If it is set, the VICC shall wait for the reception of
 * an EOF from the VCD and upon such reception shall return its
 * response.
 *
 * Command:
 * --+-------------------+-----------------------------------+--
 * ..|(select flag) UID  |          DSFID                    |..
 * --+-------------------+-----------------------------------+--
 * ..|        64 bits    |          8 bits                   |..
 * --+-------------------+-----------------------------------+--
 *
 * Response:
 * --+------------------------------------------------------+--
 * ..|                   (void)                             |..
 * --+------------------------------------------------------+--
 * ..|                                                      |..
 * --+------------------------------------------------------+--
 *
 */
#define ISO15693_COMMAND_WRITE_DSFID				0x29

/*
 * The WRITE_DSFID is sent to the first available transponder
 * if the UID is not set in <transponder> (iso15693_transponder_null)
 * or to a designated transponder. On success the DSFID byte of
 * <transponder> is written to the transponder. The function
 * iso15693_transponder_dsfid_set function must be used by the caller
 * to set the DSFID byte in <transponder>.
 */
int iso15693_write_dsfid(iso15693_t* iso15693, const rfid_transponder_t* transponder);

/*
 * LOCK_DSFID:
 *
 * When receiving the Lock DSFID request, the VICC shall lock the DSFID
 * value permanently into its memory. If the option flag is not set,
 * the VICC shall return its response when it has completed the lock
 * operation. If it is set, the VICC shall wait for the
 * reception of an EOF from the VCD and upon such reception shall
 * return its response.
 *
 * Command:
 * --+-------------------------------------------------------+--
 * ..|        (address flg) UID                              |..
 * --+-------------------------------------------------------+--
 * ..|                    64 bits                            |..
 * --+-------------------------------------------------------+--
 *
 * Response:
 * --+------------------------------------------------------+--
 * ..|                   (void)                             |..
 * --+------------------------------------------------------+--
 * ..|                                                      |..
 * --+------------------------------------------------------+--
 */
#define ISO15693_COMMAND_LOCK_DSFID				0x2A

/*
 * The LOCK_DSFID command is sent to the transponder with 
 * UID equal to <uid> (ISO15693_UID_SIZE bytes). If <uid> is all
 * 0, the LOCK_DSFID command is sent to whatever transponder
 * is available. On success the DSFID byte becomes readonly.
 */
int iso15693_lock_dsfid(iso15693_t* iso15693, const u_int8_t* uid);

/*
 * GET_SYSTEM_INFORMATION:
 *
 * This command allows for retrieving the system information value
 * from the VICC.
 *
 * Command:
 * --+-------------------------------------------------------+--
 * ..|        (address flg) UID                              |..
 * --+-------------------------------------------------------+--
 * ..|                    64 bits                            |..
 * --+-------------------------------------------------------+--
 *
 * Response:
 * --+-----------+--------+-------+--------+-------------+-------+--
 * ..|Info flags |  UID   | DSFID |  AFI   | VICC memory |IC ref |..
 * --+-----------+--------+-------+--------+-------------+-------+--
 * ..|  8 bits   |64 bits | 8 bits| 8 bits |  16 bits    |8 bits |..
 * --+-----------+--------+-------+--------+-------------+-------+--
 *
 * DSFID, AFI, VICC memory size, IC reference are all optional
 * and are ordered according to their corresponding flag in the
 * Info flags field, as defined below:
 *
 * Info flags bits:
 * 0 : if set, DSFID field present
 * 1 : if set, AFI field present
 * 2 : if set, VICC memory size field present
 * 3 : if set, IC reference field present
 * [4-7] : reserved for future use
 *
 * o VICC memory size information bits:
 *   [0-7] : (8 bits) Number of blocks
 *   [8-12] : (5 bits) Block size in bytes
 *   [13-15] : (3 bits) Reserved for future use
 *
 *   Block size is expressed in number of bytes on 5 bits, allowing to
 *   specify up to 32 bytes i.e. 256 bits. It is one less than the
 *   actual number of bytes. EXAMPLE: A value of '1F' indicates 32
 *   bytes, a value of '00' indicates 1 byte. Number of blocks is on 8
 *   bits, allowing to specify up to 256 blocks. It is one less than the
 *   actual number of blocks. EXAMPLE: A value of 'FF' indicates 256
 *   blocks, a value of '00' indicates 1 block. The three most
 *   significant bits are reserved for future use and shall be set to
 *   zero. 
 *
 * o The IC reference is on 8 bits and its meaning is defined by
 *   the IC manufacturer. 
 */
#define ISO15693_COMMAND_GET_SYSTEM_INFORMATION			0x2B
#define ISO15693_COMMAND_GET_SYSTEM_INFORMATION_RESPONSE_SIZE	(ISO15693_INFO_FLAGS_SIZE + ISO15693_UID_SIZE + ISO15693_DSFID_SIZE + ISO15693_AFI_SIZE + ISO15693_VICC_MEMORY_SIZE_SIZE + ISO15693_IC_REFERENCE_SIZE)

/*
 * The GET_SYSTEM_INFORMATION is sent to the first available transponder
 * if the UID is not set in <transponder> (iso15693_transponder_null)
 * or to a designated transponder. On success the meta information of
 * the transponder is written in <transponder>. The meta information
 * fields must be accessed using the 
 * iso15693_transponder_{flags,afi,dsfid,vicc,ic_ref}* functions declared
 * in this header. The meaning of the flags must be interpreted according to
 * the ISO15693_INFO_FLAGS_* macros defined in this header: they define
 * which field is set and which field is undefined.
 */
int iso15693_get_system_information(iso15693_t* iso15693, rfid_transponder_t* transponder);

/*
 * GET_MULTIPLE_BLOCK_SECURITY_STATUS:
 *
 * When receiving the Get multiple block security status command, the
 * VICC shall send back the block security status. The blocks are
 * numbered from '00 to 'FF' (0 to 255). The number of blocks in the
 * request is one less than the number of block security status that
 * the VICC shall return in its response. EXAMPLE: A value of '06' in
 * the "Number of blocks" field requests to return 7 Block security
 * status. A value of '00' in the "Number of blocks" field requests to
 * return a single Block security status.
 *
 * Command:
 * --+-------------------+-----------------+------------------+--
 * ..|(address flg) UID  | First block num | Number of blocks |..
 * --+-------------------+-----------------+------------------+--
 * ..|        64 bits    |     8 bits      |      8 bits      |..
 * --+-------------------+-----------------+------------------+--
 *
 * Response:
 * --+-------------------------------------------------------+--
 * ..|                  Block security status                |..
 * --+-------------------------------------------------------+--
 * ..|                      8 bits                           |..
 * --+-------------------------------------------------------+--
 *   \------------ repeat Number of blocks times ------------/
 */
#define ISO15693_COMMAND_GET_MULTIPLE_BLOCK_SECURITY_STATUS		0x2C
/*
 * Fill the security_status field of each <blocks> with information
 * from the transponder, starting at <blocks>[0].block_number and
 * successive blocks up to <blocks_length>.  The command is sent to
 * the transponder with UID equal to <uid> (ISO15693_UID_SIZE
 * bytes). If <uid> is all 0, the command is sent to whatever
 * transponder is available.
 */
int iso15693_get_multiple_block_security_status(iso15693_t* iso15693, const u_int8_t* uid, rfid_block_t* blocks, int blocks_length);

/*
 * Get access to iso15693 object.
 */
#define iso15693_from_reader(reader) ((iso15693_t*)reader->private)

/*
 * Return the size, in bytes, of the response to a request. The
 * transponder argument is only necessary when the response size depends
 * on the block size of the transponder.
 */
int iso15693_response_size(rfid_reader_t* reader, char* request, int request_size, int* response_size, const rfid_transponder_t* transponder);

/*
 * Set CRC16 at the end of a ISO-15693 frame (last two bytes).
 */
void iso15693_crc16_set(u_int8_t* request, int request_length);
/*
 * Return true if the CRC16 at the end of a ISO-15693 frame is
 * correct. Return false otherwise.
 */
int iso15693_crc16_check(u_int8_t* request, int request_length);

/*
 * Copy the info flags (see ISO15693_INFO_FLAGS_* macros) from v (u_int8_t) to
 * transponder t.
 */
#define iso15693_transponder_info_flags_set(t,v) \
        (((iso15693_transponder_t*)(t)->private)->info_flags = (v))
/*
 * Copy the info flags (see ISO15693_INFO_FLAGS_* macros) from the transponder t
 * to v (u_int8_t).
 */
#define iso15693_transponder_info_flags_get(t,v) \
	((v) = ((iso15693_transponder_t*)(t)->private)->info_flags)

/*
 * Copy the UID from the v memory area (ISO15693_UID_SIZE bytes) to transponder t.
 */
#define iso15693_transponder_uid_set(t,v) \
	memcpy(((iso15693_transponder_t*)(t)->private)->uid, (v), ISO15693_UID_SIZE)
/*
 * Copy the UID of transponder t to the v memory area (ISO15693_UID_SIZE bytes).
 */
#define iso15693_transponder_uid_get(t,v) \
	memcpy((v), ((iso15693_transponder_t*)(t)->private)->uid, ISO15693_UID_SIZE)

/*
 * Copy the DSFID from the v memory area (ISO15693_DSFID_SIZE bytes) to transponder t.
 */
#define iso15693_transponder_dsfid_set(t,v) \
        (((iso15693_transponder_t*)(t)->private)->dsfid = (v))
/*
 * Copy the DSFID of transponder t to the v memory area (ISO15693_DSFID_SIZE bytes).
 */
#define iso15693_transponder_dsfid_get(t,v) \
	((v) = ((iso15693_transponder_t*)(t)->private)->dsfid)

/*
 * Copy the AFI from the v memory area (ISO15693_AFI_SIZE bytes) to transponder t.
 */
#define iso15693_transponder_afi_set(t,v) \
        (((iso15693_transponder_t*)(t)->private)->afi = (v))
/*
 * Copy the AFI of transponder t to the v memory area (ISO15693_AFI_SIZE bytes).
 */
#define iso15693_transponder_afi_get(t,v) \
	((v) = ((iso15693_transponder_t*)(t)->private)->afi)

/*
 * VICC memory size information bits:
 *   [0-7] : (8 bits) Number of blocks
 *   [8-12] : (5 bits) Block size in bytes
 *   [13-15] : (3 bits) Reserved for future use
 */
/*
 * Copy the VICC from the v variable (u_int16_t) to the transponder t.
 */
#define iso15693_transponder_vicc_memory_set(t,v) \
        (((iso15693_transponder_t*)(t)->private)->vicc_memory = (v))
/*
 * Copy the VICC of transponder t to the v variable (u_int16_t).
 */
#define iso15693_transponder_vicc_memory_get(t,v) \
	((v) = ((iso15693_transponder_t*)(t)->private)->vicc_memory)

/*
 * Set to 0 the blocks part of the VICC of the transponder t.
 */
#define iso15693_transponder_vicc_blocks_clear(t) \
        (((iso15693_transponder_t*)(t)->private)->vicc_memory &= 0xFF00)
/*
 * Copy the v variable (u_int8_t) to the transponder
 * blocks part of the VICC of the transponder t.
 */
#define iso15693_transponder_vicc_blocks_set(t,v) \
        (iso15693_transponder_vicc_blocks_clear(t), (((iso15693_transponder_t*)(t)->private)->vicc_memory |= ((v - 1) & 0xFF)))
/*
 * Copy the blocks part of the VICC of the transponder to the v variable (u_int8_t).
 */
#define iso15693_transponder_vicc_blocks_get(t,v) \
	((v) = (((iso15693_transponder_t*)(t)->private)->vicc_memory & 0xFF) + 1)

/*
 * Set to 0 the bytes_per_block part of the VICC of the transponder t.
 */
#define iso15693_transponder_vicc_bytes_per_block_clear(t) \
        (((iso15693_transponder_t*)(t)->private)->vicc_memory &= 0xE0FF)
/*
 * Copy the v variable (u_int8_t) to the transponder
 * bytes_per_block part of the VICC of the transponder t.
 */
#define iso15693_transponder_vicc_bytes_per_block_set(t,v) \
        (iso15693_transponder_vicc_bytes_per_block_clear(t), (((iso15693_transponder_t*)(t)->private)->vicc_memory |= (((v - 1) & 0x1F) << 8)))
/*
 * Copy the bytes_per_block part of the VICC of the transponder
 * to the v variable (u_int8_t).
 */
#define iso15693_transponder_vicc_bytes_per_block_get(t,v) \
	((v) = ((((iso15693_transponder_t*)(t)->private)->vicc_memory >> 8) & 0x1F) + 1)

/*
 * Copy the IC REFERENCE from the v memory area
 * (ISO15693_IC_REFERENCE_SIZE bytes) to transponder t.
 */
#define iso15693_transponder_ic_ref_set(t,v) \
        (((iso15693_transponder_t*)(t)->private)->ic_ref = (v))
/*
 * Copy the IC REFERENCE of transponder t to the v memory area
 * (ISO15693_IC_REFERENCE_SIZE bytes).
 */
#define iso15693_transponder_ic_ref_get(t,v) \
	((v) = ((iso15693_transponder_t*)(t)->private)->ic_ref)

#endif /* _ISO15693_H_ */
