/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

/*
 * For a detailed description see rfid_reader(3)
 */

#ifndef _RFID_READER_H_
#define _RFID_READER_H_

#include <sys/types.h>

#include <rfid_io.h>

#if defined(__cplusplus)
extern "C" {
#endif

/*
 * Assume that a RFID transponder can hold a maximum of 8192 bytes,
 * according to ISO-15693 definition (256 blocks of 256 bits = 64Kbits).
 */
#define RFID_TRANSPONDER_DATA_SIZE 8192
/*
 * Size of a string holding hexadecimal representation of 
 * a RFID transponder identity, with separators and trailing 
 * NULL.
 */
#define RFID_TRANSPONDER_HEX_ID_SIZE 64

/*
 * RFID transponder identity and data.
 */
typedef struct {
  u_int8_t blocks;			/* Total number of blocks. */
  u_int8_t bytes_per_block;		/* Size, in bytes, of each block. */
  u_int8_t data[RFID_TRANSPONDER_DATA_SIZE + 1]; /* + 1 for trailing NULL */
  int data_length;			/* Length of data in the
					   data field. */
  void* opaque;			/* Opaque reader information. */
} rfid_transponder_t;

/*
 * Chunk of data used to read/write data from/to the RFID transponder.
 */
typedef struct {
  u_int8_t* data;		/* Data buffer whose size must be
				   rfid_transponder_t.bytes_per_block. */
  u_int8_t security_status;	/* If 1 block is locked, if 0 block is
				   not locked. */
  u_int8_t block_number;	/* Position of the data block in the
				   transponder, in the range
				   [0,rfid_transponder_t.blocks[.*/
} rfid_block_t;

/*
 * Array of rfid_transponder_t objects.
 */
typedef struct {
  rfid_transponder_t** list;	/* The transponders. */
  int length;			/* Number of rfid_transponder_t
				   objects in list. */
  int size;			/* Size of list. */
  int alloc;			/* True if list can be free(3), false
				   if is must not be free(3). */
} rfid_transponders_t;

#define RFID_DRIVERS_MAX	32

typedef struct rfid_reader {
  /* I/O handle. */
  rfid_io_t* io;
  /* RFID reader descriptive string. */
  char* id_string;
  /* RFID reader internal id. */
  int id;
  /* Drivers path. */
  char drivers[RFID_DRIVERS_MAX];
  /* Maximum size of the output buffer. */
  int buffer_out_max;
  /* Maximum size of the input buffer. */
  int buffer_in_max;
  /* Opaque reader information. */
  void* opaque;
  /*
   * If the reader was created by iso15693(3), points
   * to the iso15693(3) reader (the parent).
   */
  struct rfid_reader* iso15693_reader;

  /* Allocate *reader and set the objects to null (must be first). */
  int (*alloc_f)(struct rfid_reader** reader);
  /* Free all objects related to reader and reader itself (must be last). */
  int (*free_f)(struct rfid_reader* reader);

  /* Figure out if the object is able to manage the RFID reader. */
  int (*probe_f)(struct rfid_reader* reader, const char* drivers);
  /* Initiate connection with the reader device (after alloc_f). */
  int (*init_f)(struct rfid_reader* reader, const char* drivers);
  /* Terminate connection with the reader device (before free_f). */
  void (*end_f)(struct rfid_reader* reader);

  /* Return a malloc'ed string representation of the current error. */
  char* (*strerror_f)(struct rfid_reader* reader);
  /* Return the current error number. */
  int (*error_f)(struct rfid_reader* reader);

  /* Return the version of the reader. */
  const char* (*version_f)(struct rfid_reader* reader);
  /* Set the verbosity level. */
  void (*verbose_f)(struct rfid_reader* reader, int verbosity);

  /* Read number_of_blocks starting at block_start and store in transponder->data. */
  int (*read_f)(struct rfid_reader* reader, int block_start, int number_of_blocks, rfid_transponder_t* transponder);
  /* Write number_of_blocks starting at block_start from transponder->data. */
  int (*write_f)(struct rfid_reader* reader, int block_start, int number_of_blocks, rfid_transponder_t* transponder);
  /* List transponders in the range of the reader. */
  int (*inventory_f)(struct rfid_reader* reader, rfid_transponders_t* transponders);
  /* Fill meta information about transponder or fail with ENOENT.  */
  int (*transponder_present_f)(struct rfid_reader* reader, rfid_transponder_t* transponder);
  /* Return a malloc'ed ascii representation of the transponder unique identifier. */
  int (*transponder_id_get_f)(struct rfid_reader* reader, const rfid_transponder_t* transponder, char** idp);
  /* Set the transponder unique identifier from a null terminated ascii representation. */
  int (*transponder_id_set_f)(struct rfid_reader* reader, rfid_transponder_t* transponder, char* id);

  /* Send and rework a ISO-15693-3 conformant request to the reader. */
  int (*iso15693_command_f)(struct rfid_reader* reader, u_int8_t* data, int data_length, const rfid_transponder_t* transponder);
  /* Get a ISO-15693-3 answer from reader, rework it for conformance and return it. */
  int (*iso15693_response_f)(struct rfid_reader* reader, u_int8_t** datap, int* data_lengthp, const rfid_transponder_t* transponder);

  /* Return a malloc'ed human readable string describing the transponder. */
  char* (*transponder_describe_f)(struct rfid_reader* reader, const rfid_transponder_t* transponder);
  /* Return 0 if a and b are equal, -1 if a < b, 1 if a > b. */
  int (*transponder_cmp_f)(struct rfid_reader* reader, rfid_transponder_t* transponder_a, rfid_transponder_t* transponder_b);
  /* Return 0 if transponder is null (meaning we don't know which transponder it is). */
  int (*transponder_null_f)(struct rfid_reader* reader, rfid_transponder_t* transponder);
  /* Allocate a transponder, including opaque data. */
  int (*transponder_alloc_f)(struct rfid_reader* reader, rfid_transponder_t** transponder);
  /* Free a transponder, including opaque data. */
  int (*transponder_free_f)(struct rfid_reader* reader, rfid_transponder_t* transponder);
  /* Copy transponder "from" to transponder "to", including opaque data. */
  int (*transponder_copy_f)(struct rfid_reader* reader, rfid_transponder_t* to, rfid_transponder_t* from);
  /* Set transponder to 0, including opaque data. */
  int (*transponder_clear_f)(struct rfid_reader* reader, rfid_transponder_t* transponder);
} rfid_reader_t;

/*
 * Dynamically load the module implementing the first driver
 * of the drivers list (s6350, iso15693/s6350, m2xxh, iso15693/m2xxh...)
 * and return a driver instance allocated with its
 * alloc_f method.
 */
int rfid_alloc(const char* driver_stack, rfid_reader_t** reader, int verbose);
/*
 * Free a driver instance previously allocated by rfid_alloc.
 */
int rfid_free(rfid_reader_t* reader);

/*
 * Write an hexadecimal ascii representation of frame
 * on stderr.
 */
void rfid_dump_frame(u_int8_t* frame, int frame_length);

#if defined(__cplusplus)
}
#endif

#endif /* _RFID_READER_H_ */
