/*
 * Copyright (C) 2003 _INRIA_
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <memory.h>
#include <errno.h>

#include "rfid.h"

static void report_error(rfid_t* rfid, const char* message)
{
  char* str = rfid_strerror(rfid);
  fprintf(stderr, "%s: %s\n", message, str);
  free(str);
}

static rfid_t rfid;

static RETSIGTYPE timeout_init(int sig)
{
  fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to establish communication with RFID reader at %s.\n\
FAILURE: Make sure a RFID reader is plugged in and turned on.\n\
\n\
", rfid.device);
  exit(-1);
}

static RETSIGTYPE timeout_read(int sig)
{
  fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to read information from a transponder.\n\
FAILURE: Make sure a transponder is placed on the RFID reader.\n\
\n\
");
  exit(-1);
}

static RETSIGTYPE timeout_write(int sig)
{
  fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to write information to a transponder.\n\
FAILURE: Make sure a transponder is placed on the RFID reader.\n\
\n\
");
  exit(-1);
}

static char* test_string = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

static int callback(rfid_t* rfid, const rfid_transponder_t* transponder, void* data)
{
  int* count = (int*)data;
  (*count)--;
  
  const s6350_transponder_t* info = &transponder->info;
  fprintf(stderr, "\
========================================\n\
transponder_id = 0x%02x%02x%02x%02x\n\
manufacturer = 0x%02x\n\
version_number = 0x%04x\n\
blocks = %d\n\
bytes_per_block = %d\n\
",
	  info->transponder_id[3],
	  info->transponder_id[2],
	  info->transponder_id[1],
	  info->transponder_id[0],
	  info->manufacturer,
	  info->version_number,
	  info->blocks,
	  info->bytes_per_block
	  );


  {
    int i;
    for(i = 0; i < transponder->data_length; i++)
      fprintf(stderr, "%c(0x%02x)/", transponder->data[i], transponder->data[i]);
    fprintf(stderr, "\n");
  }

  if(transponder->data_length < strlen(test_string) ||
     memcmp(transponder->data, test_string, strlen(test_string))) {
    fprintf(stderr, "FAILURE: expected %s\n", test_string);
    return RFID_CALLBACK_ERROR;
  }
  
  fprintf(stderr, "=> %d more read attempt to go\n", *count);

  return *count > 0 ? RFID_CALLBACK_OK : RFID_CALLBACK_END;
}

int main(int argc, char** argv)
{

  if(argc > 1)
    rfid.device = argv[1];
  else
    rfid.device = "/dev/ttyS0";

  rfid_options(&rfid, RFID_OPTION_POLL);

  if(signal(SIGALRM, timeout_init) < 0) {
    perror("signal(SIGALRM)");
    return -1;
  }

  alarm(5);
  if(rfid_init(&rfid) < 0) {
    report_error(&rfid, "rfid_init");
    return -1;
  }
  alarm(0);

  if(signal(SIGALRM, timeout_write) < 0) {
    perror("signal(SIGALRM)");
    return -1;
  }

  alarm(5);
  {
    rfid_transponder_t transponder;
    const s6350_transponder_t* info = &transponder.info;
    memset(&transponder, '\0', sizeof(rfid_transponder_t));
    memcpy(transponder.data, test_string, strlen(test_string));
    transponder.data_length = strlen(test_string);
    if(rfid_write(&rfid, &transponder) < 0) {
      report_error(&rfid, "rfid_write");
      return -1;
    }

    fprintf(stderr, "\
========================================\n\
Wrote string %s to transponder:\n\
transponder_id = 0x%02x%02x%02x%02x\n\
manufacturer = 0x%02x\n\
version_number = 0x%04x\n\
blocks = %d\n\
bytes_per_block = %d\n\
",
	    test_string,
	    info->transponder_id[3],
	    info->transponder_id[2],
	    info->transponder_id[1],
	    info->transponder_id[0],
	    info->manufacturer,
	    info->version_number,
	    info->blocks,
	    info->bytes_per_block
	    );
  }
  alarm(5);
  {
    rfid_transponder_t transponder;
    s6350_transponder_t* info = &transponder.info;
    memset(&transponder, '\0', sizeof(rfid_transponder_t));
    /* Fake manufacturer, unlikely to match an existing one to trigger failure. */
    info->manufacturer = 1;
    if(rfid_write(&rfid, &transponder) >= 0 ||
       rfid.error != EXDEV) {
      report_error(&rfid, "rfid_write (EXDEV was expected)");
      return -1;
    }
  }
  alarm(0);
  
  if(signal(SIGALRM, timeout_read) < 0) {
    perror("signal(SIGALRM)");
    return -1;
  }

  alarm(5);
  {
    int count = 3;
    if(rfid_read(&rfid, callback, &count) < 0) {
      report_error(&rfid, "rfid_read");
      return -1;
    }
  }
  alarm(0);
  
  rfid_end(&rfid);
  
  return 0;
}
