/*
 * Copyright (C) 2003 _INRIA_
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>

#include "rfid.h"

int rfid_init(rfid_t* rfid)
{
  rfid->error = 0;
  rfid->s6350.device = rfid->device;
  memset(&rfid->last, '\0', sizeof(rfid_transponder_t));
  if(s6350_init(&(rfid->s6350)) < 0)
    return -1;

  return 0;
}

void rfid_end(rfid_t* rfid)
{
  s6350_end(&(rfid->s6350));
}

char* rfid_strerror(rfid_t* rfid)
{
  s6350_t* s6350 = &rfid->s6350;
  int message_size = 32 + strlen(s6350->device) + S6350_ERROR_LONGEST_MESSAGE_SIZE + strlen(strerror(s6350->error));
  char* message = (char*)malloc(message_size);
  
  sprintf(message, "Device %s: ", s6350->device);

  if(rfid->error) {
      sprintf(message + strlen(message), "0x%02x: %s", rfid->error, strerror(rfid->error));
  } else {
    if(s6350->error == EIO) {
      sprintf(message + strlen(message), "S6350 Error: %s", s6350_error(s6350));
    } else {
      sprintf(message + strlen(message), "0x%02x: %s", s6350->error, strerror(s6350->error));
    }
  }

  return message;
}

int rfid_error(rfid_t* rfid)
{
  s6350_t* s6350 = &rfid->s6350;

  if(rfid->error) {
    return rfid->error;
  } else {
    if(s6350->error == EIO) {
      return -(s6350->s6350_error);
    } else {
      return s6350->error;
    }
  }
}

/*
 * Return 0 if a and b are equal, return non 0 otherwise.
 */
static int transponder_cmp(rfid_transponder_t* a, rfid_transponder_t* b)
{
  return memcmp(a->info.transponder_id, b->info.transponder_id, sizeof(u_int8_t) * S6350_TRANSPONDER_DETAILS_TRANSPONDER_ID_SIZE) ||
    a->info.manufacturer != b->info.manufacturer ||
    a->info.version_number != b->info.version_number;
}

int rfid_read(rfid_t* rfid, rfid_read_callback_t* callback, void* data)
{
  int retval = 0;
  int done = 0;
  /* 5 milliseconds */
  struct timespec zzz = {
    0,
    5000000,
  };
  rfid_transponder_t transponder;
  s6350_t* s6350 = &rfid->s6350;

  rfid->error = 0;

  while(!done) {
    if((retval = rfid_poll(rfid, &transponder)) < 0) {

      if(rfid->error == ENOENT) {
	/*
	 * Poll did not find any transponder.
	 */
	if(rfid->verbose == 1) fprintf(stderr, ".");
	if(!(rfid->options & RFID_OPTION_POLL))
	  memset(&rfid->last, '\0', sizeof(rfid_transponder_t));

      } else if(s6350->error == EAGAIN) {
	/*
	 * Recoverable error, try again.
	 */
	if(rfid->verbose == 1) fprintf(stderr, "!");

      } else {
	/*
	 * Unrecoverable error, abort.
	 */
	done = 1;
      }
    } else {
      int call;
      call = (rfid->options & RFID_OPTION_POLL) || transponder_cmp(&transponder, &rfid->last);

      if(call) {
	if(rfid->verbose == 1) fprintf(stderr, "R");
	if((retval = (*callback)(rfid, &transponder, data)) != 0) {
	  done = 1;
	}
	rfid->last = transponder;
      } else {
	if(rfid->verbose == 1) fprintf(stderr, "=");
      }
    }
    /*
     * Short pause to avoid using all the CPU. More a safety
     * measure than really usefull since the delay imposed
     * by the serial line dialog are more than likely to introduce
     * enough delay anyway.
     */
    if(!done)
      nanosleep(&zzz, 0);
  }

  return retval;
}

int rfid_poll(rfid_t* rfid, rfid_transponder_t* transponder)
{
  int i;
  s6350_block_t* blocks;
  s6350_t* s6350 = &rfid->s6350;
  int block_map = 0;

  rfid->error = 0;

  if(s6350_transponder_details(s6350, &transponder->info) < 0) {
    if(s6350->error == EIO && s6350->s6350_error == S6350_ERROR_TRANSPONDER_NOT_FOUND)
      rfid->error = ENOENT;
    return -1;
  }

  transponder->data_length = transponder->info.blocks * transponder->info.bytes_per_block;
  if(transponder->data_length > RFID_TRANSPONDER_DATA_SIZE) {
    transponder->data_length = 0;
    rfid->error = EFBIG;
    return -1;
  }
  memset(transponder->data, '\0', RFID_TRANSPONDER_DATA_SIZE + 1); /* + 1 for trailing NULL */

  blocks = (s6350_block_t*)malloc(sizeof(s6350_block_t) * transponder->info.blocks);

  block_map = ((1 << transponder->info.blocks) - 1) & 0xFF;
  
  for(i = 0; i < transponder->info.blocks; i++)
    blocks[i].data = &transponder->data[i * transponder->info.bytes_per_block];
  
  if(s6350_special_read_block(s6350, &transponder->info, block_map, blocks) < 0) {
    free(blocks);
    if(s6350->error == EIO && s6350->s6350_error == S6350_ERROR_TRANSPONDER_NOT_FOUND)
      rfid->error = ENOENT;
    return -1;
  }

  free(blocks);

  return 0;
}

int rfid_write(rfid_t* rfid, rfid_transponder_t* transponder)
{
  s6350_t* s6350 = &rfid->s6350;

  /* 5 milliseconds */
  struct timespec zzz = {
    0,
    5000000,
  };

  rfid->error = 0;

  while(1) {
    if(s6350_transponder_details(s6350, &transponder->info) < 0) {

      if(s6350->error == EIO && s6350->s6350_error == S6350_ERROR_TRANSPONDER_NOT_FOUND) {
	/*
	 * No RFID transponder, try again.
	 */
	if(rfid->verbose == 1) fprintf(stderr, ".");

      } else if(s6350->error == EAGAIN) {
	/*
	 * Recoverable error, try again.
	 */
	if(rfid->verbose == 1) fprintf(stderr, "!");

      } else {
	/*
	 * Unrecoverable error, abort.
	 */
	return -1;
      }
    } else {
      /*
       * The RFID transponder is available, proceed.
       */
      break;
    }
    nanosleep(&zzz, 0);
  }

  if(transponder->data_length > transponder->info.bytes_per_block * transponder->info.blocks) {
    rfid->error = EFBIG;
    return -1;
  }

  {
    int block_number;
    int retval;
    int offset;
    u_int8_t* block_data = (u_int8_t*)malloc(transponder->info.bytes_per_block * transponder->info.blocks);
  
    memset(block_data, '\0', transponder->info.bytes_per_block * transponder->info.blocks);
    memcpy(block_data, transponder->data, transponder->data_length);
      
    for(block_number = 0, offset = 0; block_number < transponder->info.blocks; block_number++, offset += transponder->info.bytes_per_block) {
      s6350_block_t block;

      block.data = block_data + offset;
      block.block_number = block_number;
      block.lock_status = 0;

      while((retval = s6350_write_block(s6350, &transponder->info, &block)) < 0 &&
	    s6350->error == EAGAIN) {
	if(rfid->verbose == 1) fprintf(stderr, "!");
      }
      if(retval < 0) {
	free(block_data);
	return -1;
      }
      if(rfid->verbose == 1) fprintf(stderr, "W");
    }
      
    free(block_data);
    if(rfid->verbose == 1) fprintf(stderr, "\n");
  }
  
  return 0;
}

const char* rfid_version(void)
{
  return RFID_VERSION;
}
