-------------------------------------------------------------------
--           RAPID - RAPID ADA PORTABLE INTERFACE DESIGNER
--           TCL PEER FOR THE MCC TKI (ToolKit Interface) library
--           Copyright (C) 1999 Martin C. Carlisle.
--
-- RAPID is free software;  you can  redistribute it  and/or modify
-- it under terms of the  GNU General Public License as published
-- by the Free Software  Foundation;  either version 2,  or (at your
-- option) any later version.  RAPID is distributed in the hope that
-- it will be useful, but WITHOUT ANY WARRANTY;  without even the
-- implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
-- PURPOSE.  See the GNU General Public License for more details.
-- You should have  received  a copy of the GNU General Public License
-- distributed with RAPID; see file COPYING.  If not, write to the
-- Free Software Foundation,  59 Temple Place - Suite 330,  Boston,
-- MA 02111-1307, USA.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.  This exception does not apply to executables which
-- are GUI design tools, or that could act as a replacement
-- for RAPID.
------------------------------------------------------------------------------
with Tcl.Tk;
with My_Expanding_Array;
with CArgv;
with Interfaces.C;
with Cargv_Helpers;

package body mcc.tki.Container.Window is
   Close_Command  : Tcl.Tcl_Command;
   Resize_Command : Tcl.Tcl_Command;

   Have_Close_Command  : Boolean := False;
   Have_Resize_Command : Boolean := False;

   ------------
   -- Create --
   ------------

   procedure Create
     (Obj    : in out Main_Window;
      X      : in Natural;
      Y      : in Natural;
      Width  : in Natural;
      Height : in Natural)
   is
   begin
      Obj.My_Peer.Name := new String'(".");
      peer.Eval
        (". configure -width " & mcc.Img (Width) &
         " -height " & mcc.Img (Height) & ASCII.LF &
         "update");
      peer.Eval
        ("wm geometry . =" & mcc.Img (Width) & "x" & mcc.Img (Height));
   end Create;

   ------------
   -- Create --
   ------------

   procedure Create
     (Obj    : in out Subwindow;
      X      : in Natural;
      Y      : in Natural;
      Width  : in Natural;
      Height : in Natural)
   is
      Width_String  : constant String := mcc.Img (Width);
      Height_String : constant String := mcc.Img (Height);
   begin
      Obj.My_Peer := peer.Create_Peer;
      peer.Eval ("toplevel " & Obj.My_Peer.Name.all);
      peer.Eval
        (Obj.My_Peer.Name.all & " configure" &
         " -width " & Width_String &
         " -height " & Height_String & ASCII.LF &
         "update");
      peer.Eval ("focus -force " & Obj.My_Peer.Name.all);
      peer.Eval
        ("wm geometry " & Obj.My_Peer.Name.all &
         " =" & Width_String & "x" & Height_String);
   end Create;

   ----------------
   -- Event_Loop --
   ----------------

   procedure Event_Loop is
   begin
      Tcl.Tk.Tk_MainLoop;
   end Event_Loop;

   ----------
   -- Hide --
   ----------

   procedure Hide (Obj : in Window) is
   begin
      peer.Eval ("wm withdraw " & Obj.My_Peer.Name.all);
   end Hide;

   function Is_Open (Obj : in Window) return Boolean is
      use type peer.String_Pointer;
   begin
      if Obj.My_Peer.Name = null then
         return False;
      end if;
      peer.Eval ("winfo width " & Obj.My_Peer.Name.all);
      return True;
   exception
      when others =>
         return False;
   end Is_Open;

   --------------------------
   -- Tcl callback for
   -- Close events
   --------------------------
   function Close_Command_Function
     (ClientData : in Integer;
      Interp     : in Tcl.Tcl_Interp;
      Argc       : in Interfaces.C.int;
      Argv       : in CArgv.Chars_Ptr_Ptr)
      return       Interfaces.C.int;
   pragma Convention (C, Close_Command_Function);

   -- protocol for arguments will be
   -- 1st argument : lookup into Expanding_Array
   function Close_Command_Function
     (ClientData : in Integer;
      Interp     : in Tcl.Tcl_Interp;
      Argc       : in Interfaces.C.int;
      Argv       : in CArgv.Chars_Ptr_Ptr)
      return       Interfaces.C.int
   is
      Obj : Window_Pointer;
   begin
      Obj :=
        Window_Pointer (My_Expanding_Array.Retrieve
                           (Table    => My_Expanding_Array.Table,
                            Location => Cargv_Helpers.Argument (Argv, 1)));
      Obj.Close_Callback (Obj.all);
      return Tcl.TCL_OK;
   end Close_Command_Function;

   -----------------------
   -- Set_Close_Handler --
   -----------------------

   procedure Set_Close_Handler
     (Obj     : in Window_Pointer;
      Handler : in Close_Handler)
   is
   begin
      if Handler /= null then
         if Obj.My_Peer.Lookup = 0 then
            My_Expanding_Array.Insert
              (Table    => My_Expanding_Array.Table,
               Element  => Sized_Object_Pointer (Obj),
               Location => Obj.My_Peer.Lookup);
         end if;

         if not Have_Close_Command then
            Close_Command      :=
               peer.CreateCommands.Tcl_CreateCommand
                 (peer.Get_Interp,
                  "closecommand",
                  Close_Command_Function'Access,
                  0,
                  null);
            Have_Close_Command := True;
         end if;

         Obj.Close_Callback := Handler;

         peer.Eval
           ("wm protocol " & Obj.My_Peer.Name.all & " WM_DELETE_WINDOW " &
            "{closecommand " & mcc.Img (Obj.My_Peer.Lookup) & "}");
      else
         peer.Eval
           ("wm protocol " & Obj.My_Peer.Name.all & " WM_DELETE_WINDOW " &
            "{destroy " & Obj.My_Peer.Name.all & "}");
      end if;
   end Set_Close_Handler;

   --------------------------
   -- Tcl callback for
   -- Resize events
   --------------------------
   function Resize_Command_Function
     (ClientData : in Integer;
      Interp     : in Tcl.Tcl_Interp;
      Argc       : in Interfaces.C.int;
      Argv       : in CArgv.Chars_Ptr_Ptr)
      return       Interfaces.C.int;
   pragma Convention (C, Resize_Command_Function);

   -- protocol for arguments will be
   -- 1st argument : lookup into Expanding_Array
   -- 2nd argument : width
   -- 3rd argument : height
   function Resize_Command_Function
     (ClientData : in Integer;
      Interp     : in Tcl.Tcl_Interp;
      Argc       : in Interfaces.C.int;
      Argv       : in CArgv.Chars_Ptr_Ptr)
      return       Interfaces.C.int
   is
      Obj : Window_Pointer;
   begin
      Obj :=
        Window_Pointer (My_Expanding_Array.Retrieve
                           (Table    => My_Expanding_Array.Table,
                            Location => Cargv_Helpers.Argument (Argv, 1)));
      -- Unfortunately, tcl binds this event to the window and all
      -- of its descendants.  This verifies the event came from the
      -- window instead of a descendant
      if Obj.My_Peer.Name.all = Cargv_Helpers.Argument (Argv, 2) then
         Obj.Resize_Callback
           (Obj.all,
            Width  => Cargv_Helpers.Argument (Argv, 3),
            Height => Cargv_Helpers.Argument (Argv, 4));
      end if;
      return Tcl.TCL_OK;
   end Resize_Command_Function;

   ------------------------
   -- Set_Resize_Handler --
   ------------------------

   procedure Set_Resize_Handler
     (Obj     : in Window_Pointer;
      Handler : in Resize_Handler)
   is
   begin
      if Obj.My_Peer.Lookup = 0 then
         My_Expanding_Array.Insert
           (Table    => My_Expanding_Array.Table,
            Element  => Sized_Object_Pointer (Obj),
            Location => Obj.My_Peer.Lookup);
      end if;

      if not Have_Resize_Command then
         Resize_Command      :=
            peer.CreateCommands.Tcl_CreateCommand
              (peer.Get_Interp,
               "resizecommand",
               Resize_Command_Function'Access,
               0,
               null);
         Have_Resize_Command := True;
      end if;
      peer.Eval ("update");
      peer.Eval
        ("bind " & Obj.My_Peer.Name.all & " <Configure> " &
         "{resizecommand " & mcc.Img (Obj.My_Peer.Lookup) & " %W %w %h}");
      Obj.Resize_Callback := Handler;
   end Set_Resize_Handler;

   ---------------
   -- Set_Title --
   ---------------

   procedure Set_Title (Obj : in out Window; Title : in String) is
   begin
      peer.Eval
        ("wm title " & Obj.My_Peer.Name.all &
         " """ & peer.Fix_Quotes (Title) & '"');
   end Set_Title;

   ----------
   -- Show --
   ----------

   procedure Show (Obj : in Window) is
   begin
      peer.Eval ("wm deiconify " & Obj.My_Peer.Name.all);
   end Show;

   ------------------------------------------------
   -- procedure To_Top
   --
   -- causes a window to be displayed on top
   ------------------------------------------------
   procedure To_Top (Obj : in Window) is
   begin
      peer.Eval ("raise " & Obj.My_Peer.Name.all);
      peer.Eval ("focus -force " & Obj.My_Peer.Name.all);
   end To_Top;

   -----------------------
   -- Get_Close_Handler --
   -----------------------

   function Get_Close_Handler (Obj : in Window'Class) return Close_Handler is
   begin
      return Obj.Close_Callback;
   end Get_Close_Handler;

   ------------------------
   -- Get_Resize_Handler --
   ------------------------

   function Get_Resize_Handler
     (Obj  : in Window'Class)
      return Resize_Handler
   is
   begin
      return Obj.Resize_Callback;
   end Get_Resize_Handler;

end mcc.tki.Container.Window;
