---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WINDOW.ADS
--  Description : Root of GUI Window Hierarchy
--
--  Copyright (C) 2002, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
-- A Window consists of:
--   a filename (of the .GUI file)
--   a window name (used when code is generated)
--   a width and height
--   a list of widgets
--   a menubar
--   a title (added using suggestion by Dmitri Anisimkov 2/22/99)
--
-- A window knows how to:
--   Create itself (set fields of record)
--   Resize itself (both on display and in record)
--   Display itself
--   Read its widgets, menus, etc. from a file
--   Write itself to a file
--   Generate code for itself (including widgets, menu)
--   Change its filename
---------------------------------------------------------------
with gui.Menu;
with gui.Widget;
with mcc.tki.Container.Frame;
with mcc.tki.Menu;
with Gui_Enum;

package gui.Window is

   type GUI_Window is tagged limited record
      Filename      : String_Pointer;
      Window_Name   : String_Pointer;
      Title         : String_Pointer;
      Width, Height : Integer;
      Widget_List   : gui.Widget.Widget_List;
      Menu          : gui.Menu.Menu_Pointer;
      Menu_Area     : mcc.tki.Container.Frame.Frame;
      Display_Area  : aliased mcc.tki.Container.Frame.Frame;
      Resize_Frame  : aliased mcc.tki.Container.Frame.Frame;
      Menu_Display  : mcc.tki.Menu.Window_Menu;
      Novice_Mode   : Boolean        := False;
      Kind          : Gui_Enum.Window_Kind_T := Gui_Enum.Sub_Window;
      Parent_Frame  : String_Pointer := null;
      Accessibility : Gui_Enum.Accessibility_T := Gui_Enum.Read_Write;
      Snap          : Gui_Enum.Snaptogrid_T := Gui_Enum.S_Off;
   end record;
   type Window_Pointer is access all GUI_Window;

   --------------------------------------------------------
   -- procedure Create_Window
   --
   -- Initialize a GUI_Window
   --
   -- Fill in the fields of the Window
   --------------------------------------------------------
   procedure Create_Window
     (Window        : in out GUI_Window;
      Filename      : in String;
      Width         : in Integer;
      Height        : in Integer;
      Title         : in String;
      Window_Name   : in String;
      Novice_Mode   : in Boolean;
      Window_Kind   : in Gui_Enum.Window_Kind_T := Gui_Enum.Main_Window;
      Parent_Frame  : in String_Pointer := null;
      Accessibility : in Gui_Enum.Accessibility_T := Gui_Enum.Read_Write;
      Snap          : in Gui_Enum.Snaptogrid_T := Gui_Enum.S_Off);

   --------------------------------------------------------
   -- procedure Resize_Window
   --
   -- resize the GUI_Window
   --
   -- update Window record and move window
   --------------------------------------------------------
   procedure Resize_Window
     (Window : in out GUI_Window;
      Width  : in Integer;
      Height : in Integer);

   --------------------------------------------------------
   -- display the GUI Window
   --
   -- Display Window, its menu and widgets
   --------------------------------------------------------
   procedure Display_Window (Window : in out GUI_Window);

   --------------------------------------------------------
   -- procedure Resize_Window
   --
   -- dynamic resize using handle
   --
   -- when Window is resized using the handle, call this
   -- instead-- it just computes width and height and
   -- calls the other
   --------------------------------------------------------
   procedure Resize_Window
     (Window : in out GUI_Window;
      startx : in Integer;
      starty : in Integer;
      endx   : in Integer;
      endy   : in Integer);

   --------------------------------------------------------
   -- procedure Undisplay_Window
   --
   -- destroy the GUI Window
   --------------------------------------------------------
   procedure Undisplay_Window (Window : in GUI_Window);

   --------------------------------------------------------
   -- procedure Read_Window
   --
   -- reads information from a (.GUI) file into GUI_Window
   -- raises Bad_File if GUI file is ill-formed
   --------------------------------------------------------
   Bad_File : exception;
   procedure Read_Window (Window : in out GUI_Window; Filename : in String);

   --------------------------------------------------------
   -- procedure Write_Window
   --
   -- Writes information to a (.GUI) file from GUI_Window
   -- filename is stored in Window
   -- also writes the widgets and menu
   --------------------------------------------------------
   procedure Write_Window (Window : in GUI_Window);

   --------------------------------------------------------
   -- procedure Generate_Window
   --
   -- Generate Ada code for this window.
   -- also generates code for Widgets and menu
   --------------------------------------------------------
   procedure Generate_Window (Window : in GUI_Window;
                              Output_Directory : String := ".");

   --------------------------------------------------------
   -- procedure Change_Filename
   --
   -- changes the filename for the Gui_Window
   -- also updates display
   --------------------------------------------------------
   procedure Change_Filename
     (Window   : in out GUI_Window;
      Filename : in String);

end Gui.Window;
