---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WINDOW.ADB
--  Description : Root of GUI Window Hierarchy
--
--  Copyright (C) 2002, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Ada.Tags;
with Ada.Strings.Fixed;
with Ada.Characters.Handling;
with Ada.Unchecked_Deallocation;
with Ada.Text_IO;
with Gui_Enum;
with gui.Widget;
with gui.Menu;
with File_Helpers;
with Generate_Helpers;
with Rapid_Helpers;
with mcc.tki.Widget.Label;
with mcc.tki.Container.Frame;
with mcc.tki.Container.Window;
with mcc.tki.Colors;
with mcc.Msg;
with Mcc.Directory_Operations;
with Menu_IO, Widget_IO, Menu_Generate;
with main_window;
with state;
with Subwindow_Actions;
with Novice_Mode;

package body gui.Window is

   use type Ada.Tags.Tag;
   use type Gui_Enum.Keyword;
   use type gui.Menu.Menu_Pointer;
   use type gui.Widget.Widget_Pointer;

   Dir_Sep : constant String := "/";
   --          (1 => Mcc.Directory_Operations.Directory_Separator);

   -- size/location of subframes
   Menu_Start_Y   : constant := 85;
   Canvas_Start_Y : constant := 110;
   Menu_Height    : constant Integer := Canvas_Start_Y - Menu_Start_Y;
   Resizer_Size   : constant := 10;
   -- extra past size of window
   Extra_X : constant := 14;
   Extra_Y : constant := 14;

   package Listpkg renames gui.Widget.Widget_List_Package;

   --------------------------------------------------------------
   -- Initialize a GUI_Window
   --------------------------------------------------------------
   procedure Create_Window
     (Window        : in out GUI_Window;
      Filename      : in String;
      Width         : in Integer;
      Height        : in Integer;
      Title         : in String;
      Window_Name   : in String;
      Novice_Mode   : in Boolean;
      Window_Kind   : in Gui_Enum.Window_Kind_T := Gui_Enum.Main_Window;
      Parent_Frame  : in String_Pointer := null;
      Accessibility : in Gui_Enum.Accessibility_T := Gui_Enum.Read_Write;
      Snap          : in Gui_Enum.Snaptogrid_T := Gui_Enum.S_Off)
   is
   begin
      Window.Filename := new String'(Filename);

      if Window_Name'Length = 0 then
         Window.Window_Name := new String'("main");
      else
         Window.Window_Name := new String'(Window_Name);
      end if;

      if Title'Length = 0 then
         Window.Title := null;
      else
         Window.Title := new String'(Title);
      end if;

      Window.Width         := Width;
      Window.Height        := Height;
      Window.Novice_Mode   := Novice_Mode;
      Window.Accessibility := Accessibility;
      Window.Kind          := Window_Kind;
      Window.Parent_Frame  := Parent_Frame;
      Window.Snap          := Snap;
      Listpkg.Initialize (Window.Widget_List);
      gui.Menu.Menu_List_Package.Initialize (Window.Menu);
   end Create_Window;

   --------------------------------------------------------------
   procedure Resize_Window
     (Window : in out GUI_Window;
      Width  : in Integer;
      Height : in Integer)
   is
   begin
      Window.Width  := Width;
      Window.Height := Height;
      mcc.tki.Resize
        (Obj    => mcc.tki.Sized_Object (Window.Display_Area),
         Width  => Window.Width,
         Height => Window.Height);
      mcc.tki.Resize
        (Obj    => mcc.tki.Sized_Object (Window.Menu_Area),
         Width  => Window.Width,
         Height => Menu_Height);
      mcc.tki.Move
        (Obj => mcc.tki.Sized_Object (Window.Resize_Frame),
         X   => Window.Width,
         Y   => Window.Height + Canvas_Start_Y);
      mcc.tki.Container.Window.Resize
        (Obj    => main_window.main_window,
         Width  =>
            Integer'Max
              (Window.Width + Extra_X,
               Limits.Min_Rapid_Window_Width),
         Height =>
            Integer'Max
              (Window.Height + Canvas_Start_Y + Extra_Y,
               Limits.Min_Rapid_Window_Height));
   end Resize_Window;

   --------------------------------------------------------------
   -- reads information from file into GUI_Window
   --------------------------------------------------------------
   procedure Read_Window (Window : in out GUI_Window; Filename : in String) is
      Keyword     : Gui_Enum.Keyword;
      Word        : Word_Type;
      Last        : Natural;
      Menu_Count  : Integer;
      Token_Index : Positive;
   begin -- Read_Window
      Window.Filename := new String'(Filename);

      if Ada.Strings.Fixed.Index (Filename, Dir_Sep) = 0 then
         -- Open file in directory saved in 'state'.
         -- We need to prefix the directory here because in non-interactive
         -- mode, the filename given on the command line was stripped of its
         -- directory. For example, for an invocation
         --   rapid -ni /some/path/myfile.gui
         -- the Filename passed in here is just "myfile.gui" (and the directory
         -- "/some/path" was saved in package State.)
         -- Thus if the current working directory from which we invoked rapid
         -- was not "/some/path" then opening the file would fail.
         Ada.Text_IO.Open
           (File => File_Helpers.Input_File,
            Name => state.Get_Directory & Dir_Sep & Window.Filename.all,
            Mode => Ada.Text_IO.In_File);
      else
         Ada.Text_IO.Open
           (File => File_Helpers.Input_File,
            Name => Window.Filename.all,
            Mode => Ada.Text_IO.In_File);
      end if;

      File_Helpers.Get_Line;
      if File_Helpers.N_Tokens = 0 then
         raise Bad_File;
      end if;

      Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (1).all);
      if Keyword /= Gui_Enum.Window then
         raise Bad_File;
      end if;

      File_Helpers.Get_String (2, Word, Last);
      Window.Window_Name := new String'(Word (Word'First .. Last));

      Token_Index := 3;
      if File_Helpers.N_Tokens < Token_Index then
         File_Helpers.Get_Line;
         Token_Index := 1;
      end if;

      -- get window title if present
      File_Helpers.Get_String (Token_Index, Word, Last);
      if Last >= Word'First then
         Window.Title := new String'(Word (Word'First .. Last));
         if Token_Index = File_Helpers.N_Tokens then
            File_Helpers.Get_Line;
            Token_Index := 1;
         else
            Token_Index := Token_Index + 1;
         end if;
      else
         Window.Title := null;
      end if;

      -- allows backward compatibility of GUI files with
      -- a boolean for novice mode
      begin
         Window.Novice_Mode :=
            Boolean'Value (File_Helpers.Token (Token_Index).all);
         if Token_Index = File_Helpers.N_Tokens then
            File_Helpers.Get_Line;
            Token_Index := 1;
         else
            Token_Index := Token_Index + 1;
         end if;
      exception
         when others =>
            null;
      end;

      Window.Width  := Natural'Value (File_Helpers.Token (Token_Index).all);
      Token_Index   := Token_Index + 1;
      Window.Height := Natural'Value (File_Helpers.Token (Token_Index).all);
      Token_Index   := Token_Index + 1;

      -- For back compatibility, we look inside Window.Title to determine
      -- whether we have a main window. However, this will be overridden
      -- below if we find a window_kind token.
      if Window.Title /= null and then
        Ada.Strings.Fixed.Index (Window.Title.all, "main") > 0 then
         Window.Kind := Gui_Enum.Main_Window;
      end if;

      if Token_Index <= File_Helpers.N_Tokens then
         -- In RAPID 3.2, we had enum values NO_FRAME/FRAME here.
         -- In RAPID 3.3, we have the Gui_Enum.Window_Kind_T values.
         -- In order to support older GUI files, we look at the value.
         declare
            Value : constant String := File_Helpers.Token (Token_Index).all;
         begin
            if Value = "FRAME" then
               Window.Kind := Gui_Enum.Frame_Child;
            elsif Value /= "NO_FRAME" then
               Window.Kind := Gui_Enum.Window_Kind_T'Value (Value);
            end if;
            Token_Index := Token_Index + 1;
         exception
            when others =>
               mcc.Msg.Error
                 ("Gui.Window.Read_Window: invalid window_kind " & Value);
         end;
      end if;

      if Token_Index <= File_Helpers.N_Tokens then
         File_Helpers.Get_String (Token_Index, Word, Last);
         if Last > 0 then
            Window.Parent_Frame := new String' (Word (1 .. Last));
         else
            Window.Parent_Frame := null;
         end if;
         Token_Index := Token_Index + 1;
      end if;

      if Token_Index <= File_Helpers.N_Tokens then
         begin
            Window.Accessibility :=
               Gui_Enum.Accessibility_T'Value
                 (File_Helpers.Token (Token_Index).all);
            Token_Index := Token_Index + 1;
         exception
            when others =>
               Window.Accessibility := Gui_Enum.Read_Write;
         end;
      end if;

      if Token_Index <= File_Helpers.N_Tokens then
         begin
            Window.Snap :=
               Gui_Enum.Snaptogrid_T'Value
                 (File_Helpers.Token (Token_Index).all);
            Token_Index := Token_Index + 1;
         exception
            when others =>
               Window.Snap := Gui_Enum.S_Off;
         end;
      end if;

      File_Helpers.Get_Line;
      Token_Index := 1;

      Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (1).all);
      if Keyword = Gui_Enum.Menubar then
         Menu_Count := 1;
         Menu_IO.Read_Menubar (Window.Menu, Menu_Count);
         File_Helpers.Get_Line;
         Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (1).all);
      else
         gui.Menu.Menu_List_Package.Initialize (Window.Menu);
      end if;

      Listpkg.Initialize (Window.Widget_List);
      if Keyword = Gui_Enum.Widgets then
         Widget_IO.Read_Widgets (Window.Window_Name.all, Window.Widget_List);
         File_Helpers.Get_Line;
         Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (1).all);
      end if;

      -- read keyword in one of previous if statements
      if Keyword /= Gui_Enum.EndOf then
         raise Bad_File;
      end if;

      Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (2).all);
      if Keyword /= Gui_Enum.Window then
         raise Bad_File;
      end if;

      Ada.Text_IO.Close (File_Helpers.Input_File);
   exception
      when E : others =>
         mcc.Msg.Error (E, "Gui.Window.Read_Window");
         raise Bad_File;
   end Read_Window;

   --------------------------------------------------------------
   -- Writes information to file from GUI_Window
   --------------------------------------------------------------
   procedure Write_Window (Window : in GUI_Window) is
      use File_Helpers;
   begin -- Write_Window
      Ada.Text_IO.Create
        (File => Output_File,
         Name => Ada.Characters.Handling.To_Lower (Window.Filename.all),
         Mode => Ada.Text_IO.Out_File);

      Set_Indent (0);
      Put (Gui_Enum.Img (Gui_Enum.Window) & " """);
      Put (Window.Window_Name.all);
      Put ("""");

      Put_String (Window.Title);

      -- mcc: 12/18/02 added novice mode flag
      Put (' ' & Boolean'Image (Window.Novice_Mode));

      Put (Natural'Image (Window.Width));
      Put (Natural'Image (Window.Height));

      Put (" " & Gui_Enum.Window_Kind_T'Image (Window.Kind));

      Put_String (Window.Parent_Frame);

      Put (" " & Gui_Enum.Accessibility_T'Image (Window.Accessibility));

      Put (" " & Gui_Enum.Snaptogrid_T'Image (Window.Snap));
      NL;
      NL;
      if not gui.Menu.Menu_List_Package.IsEmpty (Window.Menu) then
         Menu_IO.Write_Menubar (Window.Menu);
         NL;
      end if;
      if not Listpkg.IsEmpty (Window.Widget_List) then
         Widget_IO.Write_Widgets (Widgets => Window.Widget_List);
         NL;
      end if;

      Put (Gui_Enum.Img (Gui_Enum.EndOf) & " ");
      Put (Gui_Enum.Img (Gui_Enum.Window));
      NL;

      Ada.Text_IO.Close (Output_File);
   end Write_Window;

   Start_X, Start_Y : Integer;
   procedure Mouse_Resizer
     (Obj   : in out mcc.tki.Sized_Object'Class;
      Event : in mcc.tki.Mouse_Event)
   is
      use mcc.tki;
      Window : constant Window_Pointer := state.Get_Current_Window;
   begin
      if Event.Button = Left then
         if Event.Action = Press then
            Start_X := Event.X;
            Start_Y := Event.Y;
         elsif Event.Action = Move then
            Resize_Window (Window.all, Start_X, Start_Y, Event.X, Event.Y);
            state.Set_Changed (True);
         end if;
      end if;
   end Mouse_Resizer;

   Draw_Frame         : mcc.tki.Container.Frame.Frame;
   Draw_Frame_Created : Boolean := False;
   procedure Display_Area_Mouse
     (Obj   : in out mcc.tki.Sized_Object'Class;
      Event : in mcc.tki.Mouse_Event)
   is
      Snap_X, Snap_Y : Integer;
      use type mcc.tki.Mouse_Button;
   begin
      if Event.Button /= mcc.tki.Left then
         return;
      end if;
      Snap_X := Rapid_Helpers.Snap (Event.X);
      Snap_Y := Rapid_Helpers.Snap (Event.Y);
      case Event.Action is
         when mcc.tki.Press =>
            Start_X := Snap_X;
            Start_Y := Snap_Y;
            mcc.tki.Container.Frame.Create
              (Obj    => Draw_Frame,
               X      => Start_X,
               Y      => Start_Y,
               Width  => 0,
               Height => 0,
               Parent => mcc.tki.Container.Container'Class (Obj));
            Draw_Frame_Created := True;
         when mcc.tki.Move =>
            if Draw_Frame_Created then
               mcc.tki.Container.Frame.Move
                 (Obj => Draw_Frame,
                  X   => Integer'Min (Snap_X, Start_X),
                  Y   => Integer'Min (Snap_Y, Start_Y));
               mcc.tki.Container.Frame.Resize
                 (Obj    => Draw_Frame,
                  Width  => abs (Snap_X - Start_X),
                  Height => abs (Snap_Y - Start_Y));
            end if;
         when mcc.tki.Release =>
            if Draw_Frame_Created then
               mcc.tki.Container.Frame.Destroy (Draw_Frame);
               Draw_Frame_Created := False;
               Subwindow_Actions.Add_Widget
                 (Start_X, Start_Y, Snap_X, Snap_Y);
            end if;
         when others =>
            null;
      end case;
   exception
      when others =>
         null;
   end Display_Area_Mouse;

   --------------------------------------------------------------
   -- display the GUI Window
   --------------------------------------------------------------
   procedure Display_Window (Window : in out GUI_Window) is

   begin
      if Window.Title /= null then
         mcc.tki.Widget.Label.Set_Text
           (Obj  => main_window.name,
            Text => Window.Title.all);
      else
         mcc.tki.Widget.Label.Set_Text
           (Obj  => main_window.name,
            Text => Window.Window_Name.all);
      end if;
      mcc.tki.Container.Frame.Create
        (Obj    => Window.Menu_Area,
         Parent => main_window.main_window,
         X      => 0,
         Y      => Menu_Start_Y,
         Width  => Window.Width,
         Height => Menu_Height);
      mcc.tki.Container.Frame.Create
        (Obj    => Window.Display_Area,
         Parent => main_window.main_window,
         X      => 0,
         Y      => Canvas_Start_Y,
         Width  => Window.Width,
         Height => Window.Height);

      mcc.tki.Set_Mouse_Listener
        (Obj      => Window.Display_Area'Unchecked_Access,
         Listener => Display_Area_Mouse'Access);

      begin
         Menu_Generate.Display_Menu_Code
           (Menubar => Window.Menu,
            Window  => Window);
      exception
         when e : others =>
            mcc.Msg.Error (E, "Invalid menus");
      end;

      if not Listpkg.IsEmpty (Window.Widget_List) then
         begin
            Widget_IO.Display_Widgets
              (Window  => Window,
               Widgets => Window.Widget_List);
         exception
            when e : others =>
            mcc.Msg.Error (E, "Invalid widgets");
         end;
      end if;

      mcc.tki.Container.Frame.Create
        (Obj    => Window.Resize_Frame,
         Parent => main_window.main_window,
         X      => Window.Width,
         Y      => Window.Height + Canvas_Start_Y,
         Width  => Resizer_Size,
         Height => Resizer_Size);

      -- mcc.tki.Set_Cursor
      --   (Obj    => mcc.tki.Sized_Object (Window.Resize_Frame),
      --    Cursor => mcc.tki.Resize_SE);

      mcc.tki.Set_Background_Color
        (Obj       => mcc.tki.Sized_Object (Window.Resize_Frame),
         New_Color => mcc.tki.Colors.Named_Color (mcc.tki.Colors.Black));

      mcc.tki.Set_Mouse_Listener
        (Obj      => Window.Resize_Frame'Unchecked_Access,
         Listener => Mouse_Resizer'Access);

      mcc.tki.Container.Window.Resize
        (Obj    => main_window.main_window,
         Width  =>
            Integer'Max
              (Window.Width + Extra_X,
               Limits.Min_Rapid_Window_Width),
         Height =>
            Integer'Max
              (Window.Height + Canvas_Start_Y + Extra_Y,
               Limits.Min_Rapid_Window_Height));

      -- mcc.tki.Set_Cursor
      --   (Obj    => mcc.tki.Sized_Object (Window.Resize_Frame),
      --    Cursor => mcc.tki.Default_Cursor);

      mcc.tki.Container.Window.Set_Title
        (main_window.main_window,
         Window.Filename.all);
   exception
      when e : others =>
         mcc.Msg.Error (E, "Invalid window");
   end Display_Window;

   --------------------------------------------------------------
   --------------------------------------------------------------
   procedure Resize_Window
     (Window : in out GUI_Window;
      startx : in Integer;
      starty : in Integer;
      endx   : in Integer;
      endy   : in Integer)
   is
   begin
      Resize_Window
        (Window => Window,
         Width  => Window.Width + (endx - startx),
         Height => Window.Height + (endy - starty));
   end Resize_Window;

   --------------------------------------------------------------
   -- destroy the GUI Window and its resize button
   -- change title of Window
   --------------------------------------------------------------
   procedure Undisplay_Window (Window : in GUI_Window) is
   begin
      mcc.tki.Destroy (mcc.tki.Object (Window.Display_Area));
      mcc.tki.Destroy (mcc.tki.Object (Window.Menu_Area));
      mcc.tki.Destroy (mcc.tki.Object (Window.Resize_Frame));
      mcc.tki.Widget.Label.Set_Text
        (Obj  => main_window.name,
         Text => "No Window Open");
      mcc.tki.Container.Window.Set_Title (main_window.main_window, "RAPID");
   end Undisplay_Window;

   --------------------------------------------------------------
   -- Generate Ada code for this window.
   --------------------------------------------------------------
   procedure Generate_Window (Window : in GUI_Window;
                              Output_Directory : String := ".") is
      -- wbw 06/06/99
      -- mcc 05/23/00 added enumeration type for radios
      -- Generate spec for the window that looks like:
      -- -- Auto generated...
      -- -- By: Martin C. Carlisle and C2C W. Blair Watkinson
      -- package filename is
      --    procedure Generate_Window;
      -- end filename;
      -- with statements;
      -- package filename is
      --    newButton : mcc.tki.Widget.Button.Picture_Button.Picture_Button;
      --    openButton :
      --    saveButton :
      --    cutButton:
      --    compileButton:
      --    ...
      --
      --    Window : mcc.tki.Container.Window.Subwindow;
      --
      --    procedure Generate_Window;
      --
      -- end filename;
      Rapid_Comment : constant String :=
         "-- Auto generated by RAPID " & mcc.RAPID_Version &
         " (http://savannah.nongnu.org/projects/rapid/)";

      Filename : constant String :=
         File_Helpers.Convert_Window_Name (Window.Window_Name.all) & "_window";

      Filename_With_Path : constant String :=
         Output_Directory & Dir_Sep & Filename;

      Widget_List : constant gui.Widget.Widget_List := Window.Widget_List;

      Already_Context_Widgets : gui.Widget.Widget_List;
      -- check to see if a widget of this type is
      -- already in the Already_Context_Widgets list.
      -- return true if so, false otherwise.
      function Already_Context
        (Widget : in gui.Widget.Widget_Access)
         return   Boolean
      is
         Current_Widget_Position : Listpkg.Position;
         Found                   : Boolean := False;
      begin
         Current_Widget_Position :=
            Listpkg.First (L => Already_Context_Widgets);
         while not Listpkg.IsPastEnd
                     (Already_Context_Widgets,
                      P => Current_Widget_Position)
         loop
            if Listpkg.Retrieve
                  (L => Already_Context_Widgets,
                   P => Current_Widget_Position).all'Tag =
               Widget'Tag
            then
               Found := True;
               exit;
            end if;
            Listpkg.GoAhead
              (L => Already_Context_Widgets,
               P => Current_Widget_Position);
         end loop;
         return Found;
      end Already_Context;

      function Anything_To_Fill_Or_Read
        (Widget_List : in gui.Widget.Widget_List)
         return        Boolean
      is
         Current_Pos    : gui.Widget.Widget_Pointer;
         Current_Widget : gui.Widget.Widget_Access;
      begin
         Current_Pos := Listpkg.First (Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Pos) loop
            Current_Widget := Listpkg.Retrieve (Widget_List, Current_Pos);
            if gui.Widget.Has_Anything_To_Fill_Or_Read (Current_Widget.all)
            then
               return True;
            end if;
            Listpkg.GoAhead (Widget_List, Current_Pos);
         end loop;
         return False;
      end Anything_To_Fill_Or_Read;

      procedure Generate_Spec
      is
         procedure PL (Text : String) is
         begin
            File_Helpers.P (Text, Indent => False);
         end PL;

         use File_Helpers;

         procedure Declare_Proc (Name : String; Comment : String := "") is
         begin
            if Comment /= "" then
               P ("-- " & Comment);
            end if;
            P ("procedure " & Name & ";");
            P ("-- same but usable as a button callback");
            P ("procedure " & Name &
               " (Obj : in out mcc.tki.Widget.Button.Button'Class);");
            NL;
         end Declare_Proc;

         -- Window : constant gui.Window.Window_Pointer
         --        := state.Get_Current_Window;
         Current_Widget_Position : gui.Widget.Widget_Pointer;
         Current_Widget : gui.Widget.Widget_Access;
         NoMenu      : Boolean;
         Window_Name : constant String
                     := Filename (Filename'First .. Filename'Last - 7);
      begin
         NoMenu := gui.Menu.Menu_List_Package.IsEmpty (Window.Menu);
         Ada.Text_IO.Create
           (File => File_Helpers.Output_File,
            Mode => Ada.Text_IO.Out_File,
            Name => Filename_With_Path & ".ads");
         Set_Indent (0);
         P (Rapid_Comment);
         NL;
         Generate_Helpers.Generate_With ("mcc.tki.Container.Window");
         if Window.Parent_Frame /= null then
           Generate_Helpers.Generate_With ("mcc.tki.Container.Frame");
         end if;
         if not NoMenu then
            Generate_Helpers.Generate_With ("mcc.tki.Menu");
         end if;
         Generate_Helpers.Generate_With ("mcc.tki.Widget.Button");
         NL;
         -- for each widget, add it to the already_context_widgets
         -- list if it is the first of its type
         Current_Widget_Position := Listpkg.First (L => Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Widget_Position)
         loop
            Current_Widget :=
               Listpkg.Retrieve
                 (L => Widget_List,
                  P => Current_Widget_Position);
            if not Already_Context (Current_Widget) then
               Listpkg.AddToFront
                 (L => Already_Context_Widgets,
                  X => Current_Widget);
            end if;
            Listpkg.GoAhead (L => Widget_List, P => Current_Widget_Position);
         end loop;

         -- generate the with's for each widget's mcc.tki type
         -- using the already_context_widgets list
         -- (this is for the spec, so don't need withs for actions
         --  yet)
         Current_Widget_Position :=
            Listpkg.First (L => Already_Context_Widgets);
         while not Listpkg.IsPastEnd
                     (L => Already_Context_Widgets,
                      P => Current_Widget_Position)
         loop
            gui.Widget.Generate_Widget_Context_Clause
              (Widget =>
                 Listpkg.Retrieve
                    (L => Already_Context_Widgets,
                     P => Current_Widget_Position).all);
            Listpkg.GoAhead
              (L => Already_Context_Widgets,
               P => Current_Widget_Position);
         end loop;
         NL;
         P ("package " & Filename & " is");
         NL;
         Set_Indent (1);

         -- declarations (menus)
         if not NoMenu then
            P (Window_Name & "_Menu : aliased mcc.tki.Menu.Window_Menu;");
            gui.Menu.Generate_Menu_Declaration (Menu => Window.Menu);
         end if;

         -- declarations (widgets)
         Current_Widget_Position := Listpkg.First (L => Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Widget_Position)
         loop
            gui.Widget.Generate_Widget_Declaration
              (Widget =>
                 Listpkg.Retrieve
                    (L => Widget_List,
                     P => Current_Widget_Position).all);
            Listpkg.GoAhead (L => Widget_List, P => Current_Widget_Position);
         end loop;
         Widget_IO.Generate_Radio_Groups (Widgets => Widget_List);
         NL;
         P (Filename & " : ", Newline => False);
         case Window.Kind is
            when Gui_Enum.Main_Window =>
               PL ("aliased mcc.tki.Container.Window.Main_Window;");
               NL;
               P ("function Window_Open return Boolean;");
            when Gui_Enum.Sub_Window =>
               PL ("aliased mcc.tki.Container.Window.Subwindow;");
               NL;
               P ("function Window_Open return Boolean;");
            when Gui_Enum.Frame_Child =>
               PL ("mcc.tki.Container.Frame.Frame;");
               NL;
               P ("Window_Open : Boolean := False;");
         end case;
         NL;
         Declare_Proc ("Generate_Window", "create the window");
         if Anything_To_Fill_Or_Read (Widget_List) then
            Declare_Proc
               ("Fill_Window", "fill in values as specified in RAPID");
            Declare_Proc
              ("Generate_and_Fill_Window",
               "do both Generate_Window and Fill_Window");
            if Gui_Enum."=" (Window.Accessibility, Gui_Enum.Read_Write) then
               P ("-- read out values as specified in RAPID");
               P ("-- if a constraint_error occurs (bad range)");
               P ("-- success will be false, and the offending");
               P ("-- entry will be highlighted if Highlight_Error is true");
               P ("-- additionally, a beep will occur if Beep_On_Error is true");
               P ("procedure Read_Window");
               P ("  (Success         :    out Boolean;");
               P ("   Highlight_Error : in     Boolean := True;");
               P ("   Beep_On_Error   : in     Boolean := True);");
               P ("procedure Read_Window" &
                  " (Obj : in out mcc.tki.Widget.Button.Button'Class);");
               NL;
               P ("procedure Ok;");
               P ("procedure Ok" &
                  " (Obj : in out mcc.tki.Widget.Button.Button'Class);");
               NL;
            end if;
         end if;
         Declare_Proc ("Close_Window", "close the window");
         Set_Indent (0);
         P ("end " & Filename & ";");
         Ada.Text_IO.Close (File_Helpers.Output_File);
      end Generate_Spec;

      procedure Declare_Menu_Action_Callbacks
                  (Menulist : Gui.Menu.Menu_Pointer) is
         package P_List renames Gui.Menu.Menu_List_Package;
         Current_Position: Gui.Menu.Menu_Position := P_List.First (Menulist);
         M : Gui.Menu.Menu_Access;
         use File_Helpers;
      begin
         while not P_List.IsPastEnd (Menulist, Current_Position) loop
            M := P_List.Retrieve (Menulist, Current_Position);
            if M.all in GUI.Menu.Submenu'Class then
               Declare_Menu_Action_Callbacks (GUI.Menu.Submenu (M.all).Items);
            elsif M.all in GUI.Menu.Menu_Item'Class
              and then M.Action /= null
              and then Ada.Strings.Fixed.Index (M.Action.all, ".") = 0
            then
               declare
                  LC_Text : constant String  :=
                     Ada.Characters.Handling.To_Lower (M.Action.all);
               begin
                  if LC_Text /= "fill_window"
                    and then LC_Text /= "generate_window"
                    and then LC_Text /= "generate_and_fill_window"
                    and then LC_Text /= "read_window"
                    and then LC_Text /= "close_window"
                  then
                     Put ("   procedure ");
                     Generate_Helpers.Generate_Subprogram_Name (M.Action.all);
                     P (" is separate;", Indent => False);
                     NL;
                  end if;
               end;
            end if;
            P_List.GoAhead (Menulist, Current_Position);
         end loop;
      end Declare_Menu_Action_Callbacks;


      procedure Generate_Body
      is
         use File_Helpers;

         procedure Declare_Button_Callback (Name : String) is
         begin
            NL;
            P ("procedure " & Name &
               " (Obj : in out mcc.tki.Widget.Button.Button'Class) is");
            P ("begin");
            P ("   " & Name & ";");
            P ("end " & Name & ";");
            NL;
         end Declare_Button_Callback;

         Current_Widget_Position : gui.Widget.Widget_Pointer;
         NoMenu                  : Boolean;
         Window_Name : constant String
                     := Filename (Filename'First .. Filename'Last - 7);
         use type Gui_Enum.Window_Kind_T;
      begin
         NoMenu := gui.Menu.Menu_List_Package.IsEmpty (Window.Menu);
         Ada.Text_IO.Create
           (File => File_Helpers.Output_File,
            Name => Filename_With_Path & ".adb",
            Mode => Ada.Text_IO.Out_File);
         Set_Indent (0);
         P (Rapid_Comment);
         NL;
         Generate_Helpers.Generate_With ("mcc.Msg");
         Generate_Helpers.Generate_With ("mcc.tki.Fonts");
         if Window.Parent_Frame /= null then
           Generate_Helpers.Generate_With (Window.Parent_Frame.all);
         end if;
         -- context clauses for widget actions
         Current_Widget_Position := Listpkg.First (Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Widget_Position)
         loop
            declare
               Widget : constant Gui.Widget.Widget_Access :=
                  Listpkg.Retrieve (Widget_List, Current_Widget_Position);
            begin
               gui.Widget.Generate_Action_Context_Clause (Widget.all);
            end;
            Listpkg.GoAhead (L => Widget_List, P => Current_Widget_Position);
         end loop;
         NL;
         -- context clauses for menu actions
         gui.Menu.Generate_Action_Context_Clause (Menu => Window.Menu);
         NL;
         P ("package body " & Filename & " is");
         Set_Indent (1);
         NL;
         -- set up action callbacks for menu widgets
         if not Window.Novice_Mode then
            Declare_Menu_Action_Callbacks (Window.Menu);
         end if;
         -- set up action callbacks for button widgets
         Current_Widget_Position := Listpkg.First (Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Widget_Position)
         loop
            declare
               Widget : constant Gui.Widget.Widget_Access :=
                  Listpkg.Retrieve (Widget_List, Current_Widget_Position);
            begin
               gui.Widget.Generate_Callback_Action (Widget.all);
            end;
            Listpkg.GoAhead (L => Widget_List, P => Current_Widget_Position);
         end loop;
         NL;
         if Window.Kind /= Gui_Enum.Frame_Child then
            P ("function Window_Open return Boolean is");
            P ("begin");
            P ("   return mcc.tki.Container.Window.Is_Open");
            P ("     (mcc.tki.Container.Window.Window (" & Filename & "));");
            P ("end Window_Open;");
            NL;
         end if;
         P ("procedure Generate_Window is");
         P ("begin");
         Set_Indent (2);
         P ("if Window_Open then");
         P ("   return;");
         P ("end if;");
         NL;
         if Window.Kind = Gui_Enum.Frame_Child then
            P (Window.Parent_Frame.all & ".Clear;");
            P (FileName & " := " & Window.Parent_Frame.all & ".Frame;");
         else
            P ("mcc.tki.Container.Window.Create");
            P ("  (Obj    => " & Filename & ",");
            P ("   X      => " & "0" & ",");
            P ("   Y      => " & "0" & ",");
            P ("   Width  => " & mcc.Img (Window.Width) & ",");
            P ("   Height => " & mcc.Img (Window.Height) & ");");
            NL;
            P ("mcc.tki.Container.Window.Set_Title");
            P ("  (Obj   => " & Filename & ",");
            if Window.Title /= null then
               P ("   Title => """ & Window.Title.all & """);");
            else
               P ("   Title => """ & Window_Name & """);");
            end if;
         end if;
         NL;
         if not NoMenu then
            P ("mcc.tki.Menu.Create");
            P ("  (Obj    => " & Window_Name & "_Menu,");
            P ("   Window => " & Filename & ");");
            NL;
            gui.Menu.Generate_Menu_Creation
              (Menu   => Window.Menu,
               Parent => Window_Name & "_Menu");
         end if;
         NL;
         Current_Widget_Position := Listpkg.First (Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Widget_Position)
         loop
            gui.Widget.Generate_Widget_Creation
              (Widget      =>
                 Listpkg.Retrieve
                    (L => Widget_List,
                     P => Current_Widget_Position).all,
               Window_Name => Filename);
            gui.Widget.Generate_Widget_Font
              (Widget =>
                 Listpkg.Retrieve
                    (L => Widget_List,
                     P => Current_Widget_Position).all);
            Listpkg.GoAhead (L => Widget_List, P => Current_Widget_Position);
         end loop;
         NL;
         if Window.Kind = Gui_Enum.Frame_Child then
            P ("Window_Open := True;");
         end if;
         Set_Indent (1);
         P ("end Generate_Window;");
         Declare_Button_Callback ("Generate_Window");
         Widget_IO.Generate_Radio_Groups_Body (Widgets => Widget_List);
         NL;
         if Anything_To_Fill_Or_Read (Widget_List) then
            P ("procedure Fill_Window is");
            P ("begin");
            Set_Indent (2);
            P ("if not Window_Open then");
            P ("   return;");
            P ("end if;");
            NL;
            -- fill actions
            Current_Widget_Position := Listpkg.First (Widget_List);
            while not Listpkg.IsPastEnd
                        (L => Widget_List,
                         P => Current_Widget_Position)
            loop
               gui.Widget.Generate_Fill_Action
                 (Widget =>
                    Listpkg.Retrieve
                       (L => Widget_List,
                        P => Current_Widget_Position).all);
               Listpkg.GoAhead
                 (L => Widget_List,
                  P => Current_Widget_Position);
            end loop;
            NL;
            Set_Indent (1);
            P ("end Fill_Window;");
            Declare_Button_Callback ("Fill_Window");
            P ("procedure Generate_and_Fill_Window is");
            P ("begin");
            P ("    Generate_Window;");
            P ("    Fill_Window;");
            P ("end Generate_and_Fill_Window;");
            Declare_Button_Callback ("Generate_and_Fill_Window");
            if Gui_Enum."=" (Window.Accessibility, Gui_Enum.Read_Write) then
               P ("procedure Read_Window");
               P ("  (Success         :    out Boolean;");
               P ("   Highlight_Error : in     Boolean := True;");
               P ("   Beep_On_Error   : in     Boolean := True) is");
               P ("begin");
               Set_Indent (2);
               P ("Success := True;");
               -- fill actions
               Current_Widget_Position := Listpkg.First (Widget_List);
               while not Listpkg.IsPastEnd
                           (L => Widget_List,
                            P => Current_Widget_Position)
               loop
                  gui.Widget.Generate_Read_Action
                    (Widget =>
                       Listpkg.Retrieve
                          (L => Widget_List,
                           P => Current_Widget_Position).all);
                  Listpkg.GoAhead
                    (L => Widget_List,
                     P => Current_Widget_Position);
               end loop;
               Set_Indent (1);
               P ("exception when others =>");
               P ("   Success := False;");
               P ("   if Beep_On_Error then");
               P ("      mcc.tki.Bell;");
               P ("   end if;");
               P ("end Read_Window;");
               NL;
               P ("procedure Read_Window" &
                  " (Obj : in out mcc.tki.Widget.Button.Button'Class) is");
               P ("   Success : Boolean;");
               P ("begin");
               P ("   Read_Window (Success);");
               P ("end Read_Window;");
               NL;
               P ("procedure Ok is");
               P ("   Success : Boolean;");
               P ("begin");
               P ("   Read_Window (Success);");
               P ("   if Success then");
               P ("      Close_Window;");
               P ("   end if;");
               P ("end Ok;");
               Declare_Button_Callback ("Ok");
               NL;
            end if;
         end if;
         P ("procedure Close_Window is");
         P ("begin");
         if Window.Parent_Frame /= null then
            P ("   Window_Open := False;");
         end if;
         P ("   mcc.tki.Destroy (mcc.tki.Object (" & Filename & "));");
         P ("end Close_Window;");
         Declare_Button_Callback ("Close_Window");
         P ("end " & Filename & ";", Indent => False);
         NL;
         Ada.Text_IO.Close (File_Helpers.Output_File);
      end Generate_Body;

   begin -- Generate_Window

      Generate_Spec;
      Generate_Body;
      if Window.Novice_Mode then
         Novice_Mode.Generate_Files
           (Window_Name => Window.Window_Name.all,
            Window      => Window);
      end if;
      Generate_Helpers.Reset_Withlist;
   end Generate_Window;

   --------------------------------------------------------------
   -- changes the filename for the Gui_Window
   --
   -- 1) Free the filename string
   -- 2) allocate new string
   -- 3) change title of window using filename w/o directory
   --------------------------------------------------------------
   procedure Change_Filename
     (Window   : in out GUI_Window;
      Filename : in String)
   is
      procedure Free is new Ada.Unchecked_Deallocation (
         String,
         String_Pointer);
   begin
      Free (Window.Filename);
      Window.Filename := new String'(Filename);
      mcc.tki.Container.Window.Set_Title
        (Obj   => main_window.main_window,
         Title => Window.Filename.all);
   end Change_Filename;

end Gui.Window;
