---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET.ADS
--  Description : Root of GUI Widget Hierarchy
--
--  Copyright (C) 2000, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
-- Change log:
-- 05/23/00 (mcc) : added Generate_Fill_Action, Generate_Read_Action
-- 08/20/98 (mcc) : incorporate workaround for GNAT 3.10p bug
--                  workaround contributed by Frederic Abiven
---------------------------------------------------------------
-- This starts the Gui-Widget hierarchy.  Most widgets
-- will be a direct descendant of GUI_Widget
--
-- Most other packages will use Widget_Access, which can
-- point to any widget in the hierarchy.  Also the widget
-- list packages are used in Gui-Window to keep track of
-- all the widgets in a window.
--
-- ATTRIBUTES:
-- All widgets must have a location, size and name (see below)
--
-- METHODS:
-- All widgets know how to:
-- 1) Set their values by reading from a file
-- 2) Write themselves to a file
-- 3) Display and undisplay themselves
-- 4) Generate code for themselves
-- 5) Highlight and unhighlight themselves
-- 6) Move themselves (with highlight)
-- 7) Display a properties dialog and
--    change their values based on the result
---------------------------------------------------------------
with Lists_Generic_2Keys;
with mcc.tki.Container.Window;
with mcc.tki.Widget.Text_Entry;
with mcc.tki.Widget.Label;
with mcc.tki.Widget.Rectangle;
with mcc.tki.Fonts;

package gui.Widget is

   type GUI_Widget;
   type Widget_Access is access all GUI_Widget'Class;

   -- Note this is abstract, so GUI_Widget can't be
   -- declared directly, it is only holding
   -- values shared by all its children
   type GUI_Widget is abstract tagged record
      x, y, Width, Height : Integer;
      Name                : String_Pointer := null;
      -- consider Font_Family, etc., only if Have_Font is true
      Have_Font   : Boolean                   := False;
      Font_Family : mcc.tki.Fonts.Font_Family := mcc.tki.Fonts.Serif;
      Font_Size   : mcc.tki.Fonts.Font_Size   := 10;
      Font_Style  : mcc.tki.Fonts.Font_Style  := mcc.tki.Fonts.Plain;
      -- used to display the selected font
      Font_Label  : mcc.tki.Widget.Label.Label_Pointer;
      Style_Label : mcc.tki.Widget.Label.Label_Pointer;
      -- used to read common entries
      Name_Entry   : mcc.tki.Widget.Text_Entry.Text_Entry_Pointer;
      X_Entry      : mcc.tki.Widget.Text_Entry.Text_Entry_Pointer;
      Y_Entry      : mcc.tki.Widget.Text_Entry.Text_Entry_Pointer;
      Width_Entry  : mcc.tki.Widget.Text_Entry.Text_Entry_Pointer;
      Height_Entry : mcc.tki.Widget.Text_Entry.Text_Entry_Pointer;
      Properties   : mcc.tki.Container.Window.Window_Pointer;
      The_Widget   : mcc.tki.Widget.Widget_Pointer;
      -- used to track if highlighted
      Is_Selected : Boolean := False;
      -- used if the widget is highlighted
      NW_Handle : aliased mcc.tki.Widget.Rectangle.Rectangle;
      N_Handle  : aliased mcc.tki.Widget.Rectangle.Rectangle;
      NE_Handle : aliased mcc.tki.Widget.Rectangle.Rectangle;
      E_Handle  : aliased mcc.tki.Widget.Rectangle.Rectangle;
      SE_Handle : aliased mcc.tki.Widget.Rectangle.Rectangle;
      S_Handle  : aliased mcc.tki.Widget.Rectangle.Rectangle;
      SW_Handle : aliased mcc.tki.Widget.Rectangle.Rectangle;
      W_Handle  : aliased mcc.tki.Widget.Rectangle.Rectangle;
   end record;

   -------------------------------------------------
   -- Get plain string (i.e. which is not required to be
   -- delimited by quotation marks) from the current
   -- File_Helpers.Token_Index.
   -- If the File_Helpers.Token(File_Helpers.Token_Index) IS
   -- delimited by quotation marks then they are not returned.
   -------------------------------------------------
   function Get_String return String_Pointer;

   -------------------------------------------------
   -- procedure Read_Widget
   --
   -- reads information from a (.GUI) file into
   -- a previously created GUI_Widget,
   -- assumes keyword already read.
   --
   -- For example, if we have:
   --   LABEL label1 0 5 110 20 "Widget name"
   --
   -- then LABEL was read and a LABEL widget created
   -- (this happens in widget_io), then this gets
   -- called to read the rest and fill in the fields
   -- of the LABEL
   -------------------------------------------------
   procedure Read_Widget (Widget : in out GUI_Widget);

   -------------------------------------------------
   -- procedure Write_Widget
   --
   --
   -- Writes information from this Widget to a (.GUI) file
   -- The output file is File_Helpers.Output_File
   -- This includes the entire line including the
   -- type of widget (see Read_Widget above)
   -------------------------------------------------
   procedure Write_Widget (Widget : in GUI_Widget);

   -------------------------------------------------
   -- procedure Generate_Action_Context_Clause
   --
   -- As before, not abstract, but null, and should be overridden for
   -- each widget containing action(s).
   --
   -- Since an action is a fully-qualified Ada procedure
   -- this will create with clause(s) for appropriate
   -- packages referenced in the procedure call
   -- (right now only does 1st)
   --
   -- called from Gui.Window.Generate_Window
   -------------------------------------------------
   -- wbw 6/6/99
   procedure Generate_Action_Context_Clause
     (Widget : in GUI_Widget);

   -- SAG 02.Dec.1999
   procedure Generate_Callback_Context_Clause
     (Widget     : in GUI_Widget;
      Windowname : in String);
   -- SAG 05.Sep.2000
   procedure Generate_Callback_Action
     (Widget : in GUI_Widget);

   -------------------------------------------------
   -- procedure Generate_Widget_Context_Clause
   --
   -- Abstract, requiring each widget to create this
   -- procedure
   --
   -- Since a widget is a fully-qualified Ada object,
   -- and includes methods and attributes,
   -- this will create the with clause for the
   -- appropriate packages referenced by the Ada
   -- object
   --
   -- called from Gui.Window.Generate_Window
   -------------------------------------------------
   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause
     (Widget : in GUI_Widget) is abstract;

   -------------------------------------------------
   -- procedure Generate_Widget_Declaration
   --
   -- Since a widget is a variable which must be
   -- declared as an Ada object or type, this
   -- will create the appropriate declarations for
   -- each widget
   --
   -- called from Gui.Window.Generate_Window
   -------------------------------------------------
   -- wbw 6/6/99
   procedure Generate_Widget_Declaration
     (Widget : in GUI_Widget) is abstract;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in GUI_Widget;
      Window_Name : in String) is abstract;

   -------------------------------------------------
   -- procedure Generate_Widget_Font
   --
   -- if widget has a font, create it and then set
   -- it for the widget
   -------------------------------------------------
   procedure Generate_Widget_Font
     (Widget : in GUI_Widget);

   -------------------------------------------------
   -- function Has_Anything_To_Fill_Or_Read
   --
   -- Returns True if widget has an action for the
   -- Fill_Window/Read_Window procedures.
   -- Default implementation returns False.
   -------------------------------------------------
   function Has_Anything_To_Fill_Or_Read
     (Widget : in GUI_Widget)
      return   Boolean;

   -------------------------------------------------
   -- procedure Generate_Fill_Action
   --
   -- if widget has an action for the Fill_Window
   -- procedure, generate it here
   -------------------------------------------------
   procedure Generate_Fill_Action
     (Widget : in GUI_Widget);

   -------------------------------------------------
   -- procedure Generate_Read_Action
   --
   -- if widget has an action for the Read_Window
   -- procedure, generate it here
   -------------------------------------------------
   procedure Generate_Read_Action
     (Widget : in GUI_Widget);

   -------------------------------------------------
   -- procedure Display_Widget
   --
   -- display the widget to user's view of window
   -- being edited
   -------------------------------------------------
   procedure Display_Widget
     (Widget    : in out GUI_Widget;
      Container : in out mcc.tki.Container.Container'Class);

   -------------------------------------------------
   -- procedure Undisplay_Widget
   --
   -- remove the widget from user's view of window
   -- being edited
   -------------------------------------------------
   procedure Undisplay_Widget (Widget : in GUI_Widget);

   -------------------------------------------------
   -- procedure Highlight
   --
   -- highlight a selected widget, that is,
   -- display the sizer buttons around it
   --
   -- see Subwindow_Actions.Select_widget for a usage
   -------------------------------------------------
   procedure Highlight (Widget : in out GUI_Widget);

   -------------------------------------------------
   -- procedure Move_Highlight
   --
   -- if user moves the widget, move the displayed
   -- resizing buttons
   -------------------------------------------------
   procedure Move_Highlight (Widget : in out GUI_Widget);

   -------------------------------------------------
   -- procedure Unhighlight
   --
   -- destroy resizing buttons (useful if user
   -- selects a different widget, for example)
   -------------------------------------------------
   procedure Unhighlight (Widget : in GUI_Widget);

   -------------------------------------------------
   -- procedure Set_Properties
   --
   -- run properties dialog with properties from widget
   -- (when user right clicks on widget)
   -- called from Subwindow_Actions.Modify_Widget
   -------------------------------------------------
   procedure Set_Properties (Widget : in out GUI_Widget);

   -------------------------------------------------
   -- procedure Apply_Properties
   --
   -- called when OK pushed in properties dialog
   -------------------------------------------------
   procedure Apply_Properties (Widget : in out GUI_Widget);

   -------------------------------------------------
   -- procedure Check_Properties
   --
   -- looks at entries in dialog, and sees if they are
   -- ok.  If not, highlights the incorrect entry.
   -------------------------------------------------
   procedure Check_Properties
     (Widget : in out GUI_Widget;
      Ok     : out Boolean);

   -------------------------------------------------
   -- procedure Close_Properties
   --
   -- destroys dialog.  Called after Check_Properties
   -- if user pushes "OK", or directly on "Cancel"
   -------------------------------------------------
   procedure Close_Properties (Widget : in GUI_Widget);

   -------------------------------------------------
   -- function Get_Name
   --
   -- simple selector that returns name stored in Widget.Name
   -- GUI_Widget should probably be private, so this
   -- would enable us to find out the name w/o knowing
   -- the representation.
   -------------------------------------------------
   function Get_Name (Widget : in Widget_Access) return String;

   -------------------------------------------------
   -- procedure Set_Location
   --
   -- set x,y,width and height for a widget
   -- would assist in making private (still too many refs
   -- in subwindow_actions to do this)
   -------------------------------------------------
   procedure Set_Location
     (Widget              : in Widget_Access;
      x, y, Width, Height : in Integer);

   --   package Widget_List_Package is new Lists_Generic(Widget_Access);
   --   package Widget_Key_List_Package is new Widget_List_Package.Key(
   --      GetKey => Get_Name);
   -- GNAT 3.10p doesn't correctly manage nested scope
   -- workaround by Frederic Abiven

   -------------------------------------------------
   -- function Get_The_Widget
   -------------------------------------------------
   function Get_The_Widget
     (Widget : in Widget_Access)
      return   mcc.tki.Widget.Widget_Pointer;

   package WK is new Lists_Generic_2Keys (
      Element => Widget_Access,
      GetKey => Get_Name,
      Key2type => mcc.tki.Widget.Widget_Pointer,
      GetKey2 => Get_The_Widget);

   package Widget_List_Package renames WK.List_Package;
   package Widget_Key_List_Package renames WK.List_Package_Key;
   package Widget_Key2_List_Package renames WK.List_Package_Key2;

   -- simple renamings to make things a bit shorter
   -- Don't confuse Widget_Pointer (a place in a widget list)
   -- with Widget_Access (a pointer to a widget)
   subtype Widget_Pointer is Widget_List_Package.Position;
   subtype Widget_List is Widget_List_Package.List;

end Gui.Widget;
