---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-TEXT.ADB
--  Description : GUI Widget Text entry
--
--  Copyright (C) 2001, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Ada.Text_IO;
with Ada.Strings.Fixed;
with Gui_Enum;
with File_Helpers;
with Generate_Helpers;
with textentry_dialog_window;
with mcc.Msg;
with Generate_Helpers;
with mcc.tki.Widget;          use type mcc.tki.Widget.Widget_Pointer;
with mcc.text_entry_types;
with state;

package body gui.Widget.Text is
   -- reads information from file into Text_Entry,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Text_Entry) is
   begin
      Read_Widget (GUI_Widget (Widget));
      if File_Helpers.Token_Index > File_Helpers.N_Tokens then
         return;
      end if;
      Widget.Data_Item         := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      Widget.Data_Type         := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      Widget.Base_Type         := mcc.text_entry_types.Default_Base_Type;
      -- test for EOL for backward compatibility
      if File_Helpers.Token_Index > File_Helpers.N_Tokens then
         return;
      end if;
      declare
         Base_Type : constant String :=
            File_Helpers.Token (File_Helpers.Token_Index).all;
      begin
         Widget.Base_Type := mcc.text_entry_types.Base_Type'Value (Base_Type);
      exception
         when others =>
            Ada.Text_IO.Put_Line
              ("Syntax error at base type: " &
               Base_Type &
               ", using default (" &
               mcc.text_entry_types.Base_Type'Image
                  (mcc.text_entry_types.Default_Base_Type) &
               ")");
      end;
   end Read_Widget;

   -- Writes information to file from Text_Entry
   procedure Write_Widget (Widget : in Text_Entry) is
   begin
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.TextEntry) & " ");
      Write_Widget (GUI_Widget (Widget));
      if Widget.Data_Item /= null and Widget.Data_Type /= null then
         File_Helpers.Put_String (Widget.Data_Item);
         File_Helpers.Put_String (Widget.Data_Type);
         File_Helpers.Put
           (" " & mcc.text_entry_types.Base_Type'Image (Widget.Base_Type));
      end if;
      File_Helpers.NL;
   end Write_Widget;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Text_Entry) is
   begin
      Generate_Helpers.Generate_With ("mcc.tki.Widget.Text_Entry");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Text_Entry) is
   begin
      File_Helpers.P
        (Widget.Name.all &
         " : aliased mcc.tki.Widget.Text_Entry.Text_Entry;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Text_Entry;
      Window_Name : in String)
   is
      use File_Helpers;
   begin
      P ("mcc.tki.Widget.Text_Entry.Create");
      P ("  (Obj    => " & Widget.Name.all & ",");
      P ("   Parent => " & Window_Name & ",");
      P ("   X      => " & mcc.Img (Widget.x) & ",");
      P ("   Y      => " & mcc.Img (Widget.y) & ",");
      P ("   Width  => " & mcc.Img (Widget.Width) & ",");
      P ("   Height => " & mcc.Img (Widget.Height) & ");");
   end Generate_Widget_Creation;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Text_Entry;
      Container : in out mcc.tki.Container.Container'Class)
   is
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.tki.Widget.Text_Entry.Text_Entry;
      end if;

      mcc.tki.Widget.Text_Entry.Create
        (Obj    =>
           mcc.tki.Widget.Text_Entry.Text_Entry (Widget.The_Widget.all),
         Parent => Container,
         X      => Widget.x,
         Y      => Widget.y,
         Width  => Widget.Width,
         Height => Widget.Height);
      Display_Widget (GUI_Widget (Widget), Container);
   exception
      when E : others =>
         mcc.Msg.Error (E, "Can't display: " & Widget.Name.all);
   end Display_Widget;

   procedure Set_Properties (Widget : in out Text_Entry) is
   begin
      -- fill in by copying to global
      mcc.text_entry_types.Default_Base_Type := Widget.Base_Type;

      textentry_dialog_window.Generate_and_Fill_Window;

      -- where are the entries?
      Widget.Properties         :=
        textentry_dialog_window.textentry_dialog_window'Access;
      Widget.Name_Entry         := textentry_dialog_window.entry1'Access;
      Widget.X_Entry            := textentry_dialog_window.entry2'Access;
      Widget.Y_Entry            := textentry_dialog_window.entry3'Access;
      Widget.Width_Entry        := textentry_dialog_window.entry4'Access;
      Widget.Height_Entry       := textentry_dialog_window.entry5'Access;
      Widget.Data_Item_Entry    := textentry_dialog_window.entry6'Access;
      Widget.Base_Type_Dropdown :=
        textentry_dialog_window.base_dropdown'Access;
      Widget.Data_Type_Entry    := textentry_dialog_window.entry7'Access;
      Set_Properties (GUI_Widget (Widget));

      if state.Get_Current_Window.Novice_Mode then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Item_Entry.all,
            Text => "disabled for novice");
         mcc.tki.Widget.Text_Entry.Disable (Widget.Data_Item_Entry.all);
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Type_Entry.all,
            Text => "disabled for novice");
         mcc.tki.Widget.Text_Entry.Disable (Widget.Data_Type_Entry.all);
      else
         if Widget.Data_Item /= null then
            mcc.tki.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Data_Item_Entry.all,
               Text => Widget.Data_Item.all);
         end if;

         if Widget.Data_Type /= null then
            mcc.tki.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Data_Type_Entry.all,
               Text => Widget.Data_Type.all);
         end if;
      end if;
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Text_Entry) is
      Ok : Boolean;
   begin
      Apply_Properties (GUI_Widget (Widget));
      declare
         Data_Item : String :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Data_Item_Entry.all);
      begin
         if Data_Item /= "disabled for novice" then
            Widget.Data_Item := new String'(Data_Item);
         end if;
      end;
      declare
         Data_Type     : String :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Data_Type_Entry.all);
         Base_Type_Pos : Integer;
      begin
         if Data_Type /= "disabled for novice" then
            Widget.Data_Type := new String'(Data_Type);
            Base_Type_Pos    :=
               mcc.tki.Widget.Dropdown.Get_Selected
                 (Widget.Base_Type_Dropdown.all);
            if Base_Type_Pos > 0 then
               Widget.Base_Type :=
                  mcc.text_entry_types.Base_Type'Val (Base_Type_Pos - 1);
            else
               Widget.Base_Type := mcc.text_entry_types.Default_Base_Type;
            end if;
         end if;
      end;
   end Apply_Properties;

   procedure Check_Properties
     (Widget : in out Text_Entry;
      Ok     : out Boolean)
   is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);

      if Ok then
         textentry_dialog_window.Read_Window
           (Success         => Ok,
            Highlight_Error => True,
            Beep_On_Error   => True);
      else
         mcc.tki.Bell;
      end if;
   end Check_Properties;

   procedure Generate_Action_Context_Clause (Widget : in Text_Entry) is
      use type mcc.text_entry_types.Base_Type;
   begin
      gui.Widget.Typed_Object.Generate_Action_Context_Clause
        (gui.Widget.Typed_Object.Object (Widget));

      if Widget.Base_Type in mcc.text_entry_types.Float_Types then
         Generate_Helpers.Generate_With ("Mcc.Text_Entry_Types");
      elsif Widget.Base_Type = mcc.text_entry_types.Unsigned then
         Generate_Helpers.Generate_With ("Interfaces");
      elsif Widget.Data_Item /= null
        and then Widget.Data_Item.all /= ""
        and then Widget.Base_Type = mcc.text_entry_types.Unbounded_String
        and then Widget.Data_Type = null then
         Generate_Helpers.Generate_With ("Ada.Strings.Unbounded");
      end if;
   end Generate_Action_Context_Clause;

   function Has_Anything_To_Fill_Or_Read
     (Widget : in Text_Entry)
      return   Boolean
   is
   begin
      return Widget.Data_Item /= null and then Widget.Data_Item.all /= "";
   end Has_Anything_To_Fill_Or_Read;

   function Normalize (Name : String) return String
      renames Generate_Helpers.Undash_Name;

   procedure Generate_Fill_Action (Widget : in Text_Entry)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;

      use type mcc.Text_Entry_Types.Base_Type;
   begin
      -- don't bother if there's no data item
      if Widget.Data_Item = null or else Widget.Data_Item.all = "" then
         return;
      end if;

      P ("mcc.tki.Widget.Text_Entry.Set_Text");
      P ("  (Obj    => " & Widget.Name.all & ",");

      case Widget.Base_Type is
         when mcc.text_entry_types.String_Subtype =>
            P ("   Text   => " & Normalize (Widget.Data_Item.all) & ");");
         when mcc.text_entry_types.Unbounded_String
            | mcc.text_entry_types.Bounded_String =>
            if Widget.Data_Type /= null then
               declare
                  Last_Dot : Natural := Ada.Strings.Fixed.Index
                    (Widget.Data_Type.all, ".", Ada.Strings.Backward);
               begin
                  P ("   Text   => " &
                     Normalize (Widget.Data_Type (1 .. Last_Dot)) &
                     "To_String (" & Normalize (Widget.Data_Item.all) & "));");
               end;
            elsif Widget.Base_Type = mcc.text_entry_types.Bounded_String then
               mcc.Msg.Error
                 ("Gui.Widget.Text.Generate_Fill_Action (" & Widget.Name.all &
                  ") : user data type required for BOUNDED_STRING");
            else
               P ("   Text   => Ada.Strings.Unbounded.To_String (" &
                  Normalize (Widget.Data_Item.all) & "));");
            end if;
         when mcc.text_entry_types.Integer =>
            P ("   Text   => Integer (" & Normalize (Widget.Data_Item.all) & "));");
         when mcc.text_entry_types.Unsigned =>
            P ("   Text   => Interfaces.Unsigned_32 (" &
               Normalize (Widget.Data_Item.all) & "));");
         when mcc.text_entry_types.Float_1 |
              mcc.text_entry_types.Float_2 |
              mcc.text_entry_types.Float_3 |
              mcc.text_entry_types.Float_E =>
            P ("   Text   => Mcc.Text_Entry_Types.Image (Float (" &
               Normalize (Widget.Data_Item.all) &
               "), Mcc.Text_Entry_Types." &
               mcc.text_entry_types.Base_Type'Image (Widget.Base_Type) &
               "));");
         when mcc.text_entry_types.Enumeration =>
            if Widget.Data_Type = null or else Widget.Data_Type.all = "" then
               mcc.Msg.Error
                 ("Gui.Widget.Text.Generate_Fill_Action (" & Widget.Name.all &
                  ") : user data type required for ENUMERATION");
            else
               P ("   Text   => " & Normalize (Widget.Data_Type.all) &
                  "'Image (" & Normalize (Widget.Data_Item.all) & "));");
            end if;
      end case;
   end Generate_Fill_Action;

   procedure Generate_Read_Action (Widget : in Text_Entry) is
      use type mcc.text_entry_types.Base_Type;
      use File_Helpers;

      procedure Print_Declare_Block (Data_Type : String := "String") is
      begin
         P ("declare");
         P ("   X : " & Data_Type & " := mcc.tki.Widget.Text_Entry.Get_Text");
         P ("                    (" & Widget.Name.all & ");");
         P ("begin");
         P ("   " & Normalize (Widget.Data_Item.all) & " := ", Newline => False);
      end Print_Declare_Block;

   begin
      -- don't bother if there's no data item
      if Widget.Data_Item = null or else Widget.Data_Item.all = "" then
         return;
      end if;

      case Widget.Base_Type is
         when mcc.text_entry_types.String_Subtype =>
            Print_Declare_Block;
            P ("mcc.Pad (X, " & Normalize (Widget.Data_Item.all) & "'Length);");
         when mcc.text_entry_types.Unbounded_String =>
            Print_Declare_Block;
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= "" then
               declare
                  Last_Dot : constant Natural :=
                     Ada.Strings.Fixed.Index (Widget.Data_Type.all, ".",
                                              Going => Ada.Strings.Backward);
               begin
                  P (Normalize (Widget.Data_Type (1 .. Last_Dot)) &
                     "To_Unbounded_String (X);", Indent => False);
               end;
            else
               P ("Ada.Strings.Unbounded.To_Unbounded_String (X);",
                  Indent => False);
            end if;
         when mcc.text_entry_types.Bounded_String =>
            Print_Declare_Block;
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= "" then
               declare
                  Last_Dot : constant Natural :=
                     Ada.Strings.Fixed.Index (Widget.Data_Type.all, ".",
                                              Going => Ada.Strings.Backward);
               begin
                  P (Normalize (Widget.Data_Type (1 .. Last_Dot)) &
                     "To_Bounded_String (X);", Indent => False);
               end;
            else
               mcc.Msg.Error
                 ("Gui.Widget.Text.Generate_Read_Action(BOUNDED_STRING): " &
                  "Data type is required " &
                  "(instantiation of Ada.Strings.Bounded.Generic_Bounded_Length)");
            end if;
         when mcc.text_entry_types.Integer =>
            Print_Declare_Block ("Integer");
            -- only add a cast if we need to
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= ""
            then
               P (Normalize (Widget.Data_Type.all) & " (X);",
                  Indent => False);
            else
               P ("X;", Indent => False);
            end if;
         when mcc.text_entry_types.Unsigned =>
            Print_Declare_Block ("Interfaces.Unsigned_32");
            -- only add a cast if we need to
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= ""
            then
               P (Normalize (Widget.Data_Type.all) & " (X);",
                  Indent => False);
            else
               P ("X;", Indent => False);
            end if;
         when mcc.text_entry_types.Float_1 |
              mcc.text_entry_types.Float_2 |
              mcc.text_entry_types.Float_3 |
              mcc.text_entry_types.Float_E =>
            Print_Declare_Block;
            -- if data type is blank, use standard.float
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= ""
            then
               P (Normalize (Widget.Data_Type.all) & "'Value (X);",
                  Indent => False);
            else
               P ("Float'Value (X);", Indent => False);
            end if;
         when mcc.text_entry_types.Enumeration =>
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= ""
            then
               Print_Declare_Block;
               P (Normalize (Widget.Data_Type.all) & "'Value (X);",
                  Indent => False);
            end if;
      end case;
      P ("exception when Constraint_Error =>");
      P ("   if Highlight_Error then");
      P ("      mcc.tki.Widget.Text_Entry.Highlight (" &
         Widget.Name.all & ");");
      P ("   end if;");
      P ("   raise;");
      P ("end;");
   end Generate_Read_Action;

end Gui.Widget.Text;
