---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  RAPID.ADB
--  Description : Main procedure
--
--  Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.  This exception does not apply to executables which
-- are GUI design tools, or that could act as a replacement
-- for RAPID.
---------------------------------------------------------------
with Ada.Command_Line;
with Ada.Strings.Unbounded;
with Ada.Strings.Fixed;
with Ada.Text_IO;

with main_window;
with Toolbar;
with mcc.tki.Container.Window;
with Rapid_Helpers;
with mcc.Directory_Operations;
with state;
with gui.Window;
with mcc.Msg;
with File_Menu;

procedure Rapid is

   Dir_Sep : constant String := "/";
   --          (1 => Mcc.Directory_Operations.Directory_Separator);

   procedure Set_State (Arg : String) is
      End_Dir : constant Natural :=
         Ada.Strings.Fixed.Index (Arg, Dir_Sep, Ada.Strings.Backward);
   begin
      if End_Dir > 0 then
         State.Set_Directory (Arg (Arg'First .. End_Dir - 1));
         State.Set_Filename (Arg (End_Dir + 1 .. Arg'Last));
      else
         State.Set_Directory (Mcc.Directory_Operations.Get_Current_Dir);
         State.Set_Filename (Arg);
      end if;
   end Set_State;

   use Ada.Strings.Unbounded;
   Filename, Output_Dir : Unbounded_String;
   Expecting_Output_Dir : Boolean := False;

   Argc : constant Natural := Ada.Command_Line.Argument_Count;

begin
   Mcc.Directory_Operations.Determine_Rapid_Root_Dir;
   Output_Dir := To_Unbounded_String (".");
   -- decode switches
   for I in 1 .. Argc loop
      declare
         Arg : constant String := Ada.Command_Line.Argument (I);
      begin
         if Arg (1) = '-' then
            if Arg = "-ni" or else Arg = "--noninteractive" then
               if Argc < 2 then
                  Ada.Text_IO.Put_Line
                    ("non-interactive mode requires filename");
                  return;
               end if;
               mcc.Interactive := False;
            elsif Arg = "-od" or else Arg = "--output-directory" then
               Expecting_Output_Dir := True;
            elsif Arg = "-v" or else Arg = "--verbose" then
               mcc.Verbose := True;
            else
               Ada.Text_IO.Put_Line ("ignoring unknown switch " & Arg);
            end if;
         elsif not mcc.Interactive then
            if Expecting_Output_Dir then
               Output_Dir := To_Unbounded_String (Arg);
               Expecting_Output_Dir := False;
            else
               Set_State (Arg);
               state.Set_Current_Window (state.Allocate_Window);
               gui.Window.Read_Window
                 (Window   => state.Get_Current_Window.all,
                  Filename => state.Get_Filename);
               gui.Window.Generate_Window (state.Get_Current_Window.all,
                                           To_String (Output_Dir));
            end if;
         else
            Filename := To_Unbounded_String (Arg);
            if I < Argc then
               Ada.Text_IO.Put_Line
                 ("Interactive mode permits only one file on the command line.");
               Ada.Text_IO.Put_Line
                 ("Using " & Arg & " as input GUI file.");
               Ada.Text_IO.Put_Line
                 ("Ignoring further arguments.");
            end if;
            exit;
         end if;
      end;
   end loop;

   if not mcc.Interactive then
      return;
   end if;
   if mcc.Verbose then
      Ada.Text_IO.Put_Line
        ("Rapid Ada Portable Interface Designer version " &
         mcc.RAPID_Version);
   end if;

   if Filename /= Null_Unbounded_String then
      declare
         Arg : constant String := To_String (Filename);
         End_Dir : constant Natural :=
            Ada.Strings.Fixed.Index (Arg, Dir_Sep, Ada.Strings.Backward);
      begin
         if End_Dir > 0 then
            Filename := To_Unbounded_String (Arg (End_Dir + 1 .. Arg'Last));
            state.Set_Filename (To_String (Filename));
            mcc.Directory_Operations.Change_Dir
              (Arg (Arg'First .. End_Dir - 1));
         else
            state.Set_Filename (Arg);
         end if;
      end;
   end if;
   State.Set_Directory (Mcc.Directory_Operations.Get_Current_Dir);

   main_window.Generate_Window;
   Toolbar.Initialize_Toolbar;
   -- set minimum window size
   mcc.tki.Container.Window.Set_Resize_Handler
     (Obj     => main_window.main_window'Access,
      Handler => Rapid_Helpers.Rapid_Resize_Handler'Access);

   -- adapted from submission by Dmitri Anisimkov
   -- allows Windows users to double click a .GUI file
   -- and have it be opened in RAPID
   if Filename /= Null_Unbounded_String then
      begin
         state.Set_Current_Window (state.Allocate_Window);

         gui.Window.Read_Window
           (Window   => state.Get_Current_Window.all,
            Filename => state.Get_Filename);
         gui.Window.Display_Window (Window => state.Get_Current_Window.all);
         state.Set_Changed (False);
      exception
         when E : others =>
            mcc.Msg.Error
              (E, "Unable to open " & To_String (Filename));
      end;
   end if;
   mcc.tki.Container.Window.Set_Close_Handler
     (Obj     => main_window.main_window'Access,
      Handler => File_Menu.Exit_Choice'Access);
   mcc.tki.Set_Key_Listener
     (Obj      => main_window.main_window'Access,
      Listener => state.Main_Window_Key_Listener'Access);
   mcc.tki.Container.Window.Event_Loop;
exception
   when E : others =>
      declare
         Window   : constant gui.Window.Window_Pointer := state.Get_Current_Window;
         Filename : aliased String            := "emergency.gui";
         use type gui.Window.Window_Pointer;
      begin
         if Window /= null then
            Window.Filename := Filename'Unchecked_Access;
            gui.Window.Write_Window (Window.all);
         end if;
      end;
      mcc.Msg.Error
        (E, "Open window (if any) dumped to emergency.gui" & ASCII.LF &
            "Uncaught exception -- contact author:");
      raise;
end Rapid;
