---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WINDOW.ADB
--  Description : Root of GUI Window Hierarchy
--
--  Copyright (C) 2002, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Ada.Exceptions;
with Ada.Tags;
with Ada.Characters.Handling;
with Ada.Unchecked_Deallocation;
with Ada.Text_IO;
with Gui_Enum;
with gui.Widget;
with gui.Menu;
with File_Helpers;
with Generate_Helpers;
with mcc.Gui.Widget.Label;
with mcc.Gui.Container.Frame;
with mcc.Gui.Container.Window;
with mcc.Gui.Colors;
with mcc.Common_Dialogs;
with Menu_IO, Widget_IO, Menu_Generate;
with main_window;
with state;
with Subwindow_Actions;
with Novice_Mode;

package body gui.Window is

   use type Ada.Tags.Tag;
   use type Gui_Enum.Keyword;
   use type gui.Menu.Menu_Pointer;
   use type gui.Widget.Widget_Pointer;

   -- size/location of subframes
   Menu_Start_Y   : constant := 85;
   Canvas_Start_Y : constant := 110;
   Menu_Height    : Integer   := Canvas_Start_Y - Menu_Start_Y;
   Resizer_Size   : constant := 10;
   -- extra past size of window
   Extra_X : constant := 14;
   Extra_Y : constant := 14;

   package Listpkg renames gui.Widget.Widget_List_Package;

   --------------------------------------------------------------
   -- Initialize a GUI_Window
   --------------------------------------------------------------
   procedure Create_Window
     (Window      : in out GUI_Window;
      Filename    : in String;
      Width       : in Integer;
      Height      : in Integer;
      Title       : in String;
      Window_Name : in String;
      Novice_Mode : in Boolean)
   is
   begin
      Window.Filename := new String'(Filename);

      if Window_Name'Length = 0 then
         Window.Window_Name := new String'("main");
      else
         Window.Window_Name := new String'(Window_Name);
      end if;

      if Title'Length = 0 then
         Window.Title := null;
      else
         Window.Title := new String'(Title);
      end if;

      Window.Width       := Width;
      Window.Height      := Height;
      Window.Novice_Mode := Novice_Mode;
      Listpkg.Initialize (Window.Widget_List);
      gui.Menu.Menu_List_Package.Initialize (Window.Menu);
   end Create_Window;

   --------------------------------------------------------------
   procedure Resize_Window
     (Window : in out GUI_Window;
      Width  : in Integer;
      Height : in Integer)
   is
   begin
      Window.Width  := Width;
      Window.Height := Height;
      mcc.Gui.Resize
        (Obj    => mcc.Gui.Sized_Object (Window.Display_Area),
         Width  => Window.Width,
         Height => Window.Height);
      mcc.Gui.Resize
        (Obj    => mcc.Gui.Sized_Object (Window.Menu_Area),
         Width  => Window.Width,
         Height => Menu_Height);
      mcc.Gui.Move
        (Obj => mcc.Gui.Sized_Object (Window.Resize_Frame),
         X   => Window.Width,
         Y   => Window.Height + Canvas_Start_Y);
      mcc.Gui.Container.Window.Resize
        (Obj    => main_window.main_window,
         Width  =>
            Integer'Max
              (Window.Width + Extra_X,
               Limits.Min_Rapid_Window_Width),
         Height =>
            Integer'Max
              (Window.Height + Canvas_Start_Y + Extra_Y,
               Limits.Min_Rapid_Window_Height));
   end Resize_Window;

   --------------------------------------------------------------
   -- reads information from file into GUI_Window
   --------------------------------------------------------------
   procedure Read_Window (Window : in out GUI_Window; Filename : in String) is
      Keyword     : Gui_Enum.Keyword;
      Word        : Word_Type;
      Last        : Natural;
      File        : Ada.Text_IO.File_Type;
      Menu_Count  : Integer;
      Token_Index : Positive;
   begin -- Read_Window
      Window.Filename := new String'(Filename);

      Ada.Text_IO.Open
        (File => File_Helpers.Input_File,
         Name => Window.Filename.all,
         Mode => Ada.Text_IO.In_File);

      File_Helpers.Get_Line;
      if File_Helpers.N_Tokens = 0 then
         raise Bad_File;
      end if;

      Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (1).all);
      if Keyword /= Gui_Enum.Window then
         raise Bad_File;
      end if;

      File_Helpers.Get_String (2, Word, Last);
      Window.Window_Name := new String'(Word (Word'First .. Last));

      Token_Index := 3;
      if File_Helpers.N_Tokens < Token_Index then
         File_Helpers.Get_Line;
         Token_Index := 1;
      end if;

      -- get window title if present
      File_Helpers.Get_String (Token_Index, Word, Last);
      if Last >= Word'First then
         Window.Title := new String'(Word (Word'First .. Last));
         if Token_Index = File_Helpers.N_Tokens then
            File_Helpers.Get_Line;
            Token_Index := 1;
         else
            Token_Index := Token_Index + 1;
         end if;
      else
         Window.Title := null;
      end if;

   -- allows backward compatibility of GUI files with
   -- a boolean for novice mode
      begin
         Window.Novice_Mode :=
            Boolean'Value (File_Helpers.Token (Token_Index).all);
         if Token_Index = File_Helpers.N_Tokens then
            File_Helpers.Get_Line;
            Token_Index := 1;
         else
            Token_Index := Token_Index + 1;
         end if;
      exception
         when others =>
            null;
      end;

      Window.Width  := Natural'Value (File_Helpers.Token (Token_Index).all);
      Token_Index   := Token_Index + 1;
      Window.Height := Natural'Value (File_Helpers.Token (Token_Index).all);
      Token_Index   := Token_Index + 1;

      if Token_Index <= File_Helpers.N_Tokens then
         begin
            Window.Frame :=
               Gui_Enum.Frame_T'Value
                 (File_Helpers.Token (Token_Index).all);
            Token_Index  := Token_Index + 1;
         exception
            when others =>
               Window.Frame := Gui_Enum.No_Frame;
         end;
      end if;

      File_Helpers.Get_Line;
      Token_Index := 1;

      Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (1).all);
      if Keyword = Gui_Enum.Menubar then
         Menu_Count := 1;
         Menu_IO.Read_Menubar (Window.Menu, Menu_Count);
         File_Helpers.Get_Line;
         Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (1).all);
      else
         gui.Menu.Menu_List_Package.Initialize (Window.Menu);
      end if;

      Listpkg.Initialize (Window.Widget_List);
      if Keyword = Gui_Enum.Widgets then
         Widget_IO.Read_Widgets (Window.Window_Name.all, Window.Widget_List);
         File_Helpers.Get_Line;
         Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (1).all);
      end if;

      -- read keyword in one of previous if statements
      if Keyword /= Gui_Enum.EndOf then
         raise Bad_File;
      end if;

      Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (2).all);
      if Keyword /= Gui_Enum.Window then
         raise Bad_File;
      end if;

      Ada.Text_IO.Close (File_Helpers.Input_File);
   exception
      when others =>
         raise Bad_File;
   end Read_Window;

   --------------------------------------------------------------
   -- Writes information to file from GUI_Window
   --------------------------------------------------------------
   procedure Write_Window (Window : in GUI_Window) is
      use File_Helpers;
   begin -- Write_Window
      Ada.Text_IO.Create
        (File => Output_File,
         Name => Ada.Characters.Handling.To_Lower (Window.Filename.all),
         Mode => Ada.Text_IO.Out_File);

      Set_Indent (0);
      Put (Gui_Enum.Img (Gui_Enum.Window) & " """);
      Put (Window.Window_Name.all);
      Put ("""");

      -- for backward compatibility, we keep title on
      -- same line as name.  This way, Get_String will
      -- return nothing on older .GUI files in the
      -- read.
      if Window.Title /= null then
         Put (" """);
         Put_String (Window.Title.all);
         Put ("""");
      end if;

      P;

      -- mcc: 12/18/02 added novice mode flag
      P (Boolean'Image (Window.Novice_Mode), Indent => False);

      Put (Natural'Image (Window.Width));
      Put (Natural'Image (Window.Height));

      Put (" " & Gui_Enum.Frame_T'Image (Window.Frame));

      if Window.Parent /= null then
         Put (" """);
         Put_String (Window.Parent.all);
         Put ("""");
      end if;
      P;
      P;
      if not gui.Menu.Menu_List_Package.IsEmpty (Window.Menu) then
         Menu_IO.Write_Menubar (Window.Menu);
         P;
      end if;
      if not Listpkg.IsEmpty (Window.Widget_List) then
         Widget_IO.Write_Widgets (Widgets => Window.Widget_List);
         P;
      end if;

      Put (Gui_Enum.Img (Gui_Enum.EndOf) & " ");
      Put (Gui_Enum.Img (Gui_Enum.Window));
      P;

      Ada.Text_IO.Close (Output_File);
   end Write_Window;

   Start_X, Start_Y : Integer;
   procedure Mouse_Resizer
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
      use mcc.Gui;
      Window : Window_Pointer := state.Get_Current_Window;
   begin
      if Event.Button = Left then
         if Event.Action = Press then
            Start_X := Event.X;
            Start_Y := Event.Y;
         elsif Event.Action = Move then
            Resize_Window (Window.all, Start_X, Start_Y, Event.X, Event.Y);
            state.Set_Changed (True);
         end if;
      end if;
   end Mouse_Resizer;

   Draw_Frame         : mcc.Gui.Container.Frame.Frame;
   Draw_Frame_Created : Boolean := False;
   procedure Display_Area_Mouse
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
      use mcc.Gui;
   begin
      if Event.Button = Left then
         if Event.Action = Press then
            Start_X := Event.X;
            Start_Y := Event.Y;
            mcc.Gui.Container.Frame.Create
              (Obj    => Draw_Frame,
               X      => Start_X,
               Y      => Start_Y,
               Width  => 0,
               Height => 0,
               Parent => mcc.Gui.Container.Container'Class (Obj));
            Draw_Frame_Created := True;
         elsif Event.Action = Move then
            if Draw_Frame_Created then
               mcc.Gui.Container.Frame.Move
                 (Obj => Draw_Frame,
                  X   => Integer'Min (Event.X, Start_X),
                  Y   => Integer'Min (Event.Y, Start_Y));
               mcc.Gui.Container.Frame.Resize
                 (Obj    => Draw_Frame,
                  Width  => abs (Event.X - Start_X),
                  Height => abs (Event.Y - Start_Y));
            end if;
         elsif Event.Action = Release then
            if Draw_Frame_Created then
               mcc.Gui.Container.Frame.Destroy (Draw_Frame);
               Draw_Frame_Created := False;
               Subwindow_Actions.Add_Widget
                 (Start_X,
                  Start_Y,
                  Event.X,
                  Event.Y);
            end if;
         end if;
      end if;
   exception
      when others =>
         null;
   end Display_Area_Mouse;

   --------------------------------------------------------------
   -- display the GUI Window
   --------------------------------------------------------------
   procedure Display_Window (Window : in out GUI_Window) is

   begin
      if Window.Title /= null then
         mcc.Gui.Widget.Label.Set_Text
           (Obj  => main_window.name,
            Text => Window.Title.all);
      else
         mcc.Gui.Widget.Label.Set_Text
           (Obj  => main_window.name,
            Text => Window.Window_Name.all);
      end if;
      mcc.Gui.Container.Frame.Create
        (Obj    => Window.Menu_Area,
         Parent => main_window.main_window,
         X      => 0,
         Y      => Menu_Start_Y,
         Width  => Window.Width,
         Height => Menu_Height);
      mcc.Gui.Container.Frame.Create
        (Obj    => Window.Display_Area,
         Parent => main_window.main_window,
         X      => 0,
         Y      => Canvas_Start_Y,
         Width  => Window.Width,
         Height => Window.Height);

      mcc.Gui.Set_Mouse_Listener
        (Obj      => Window.Display_Area'Unchecked_Access,
         Listener => Display_Area_Mouse'Access);

      begin
         Menu_Generate.Display_Menu_Code
           (Menubar => Window.Menu,
            Window  => Window);
      exception
         when e : others =>
            mcc.Common_Dialogs.Ok_Box
              ("Invalid menus: " & Ada.Exceptions.Exception_Information (e));
      end;

      if not Listpkg.IsEmpty (Window.Widget_List) then
         begin
            Widget_IO.Display_Widgets
              (Window  => Window,
               Widgets => Window.Widget_List);
         exception
            when e : others =>
               mcc.Common_Dialogs.Ok_Box
                 ("Invalid widgets: " &
                  Ada.Exceptions.Exception_Information (e));
         end;
      end if;

      mcc.Gui.Container.Frame.Create
        (Obj    => Window.Resize_Frame,
         Parent => main_window.main_window,
         X      => Window.Width,
         Y      => Window.Height + Canvas_Start_Y,
         Width  => Resizer_Size,
         Height => Resizer_Size);

      mcc.Gui.Set_Cursor
        (Obj    => mcc.Gui.Sized_Object (Window.Resize_Frame),
         Cursor => mcc.Gui.Resize_SE);

      mcc.Gui.Set_Background_Color
        (Obj       => mcc.Gui.Sized_Object (Window.Resize_Frame),
         New_Color => mcc.Gui.Colors.Named_Color (mcc.Gui.Colors.Black));

      mcc.Gui.Set_Mouse_Listener
        (Obj      => Window.Resize_Frame'Unchecked_Access,
         Listener => Mouse_Resizer'Access);

      mcc.Gui.Container.Window.Resize
        (Obj    => main_window.main_window,
         Width  =>
            Integer'Max
              (Window.Width + Extra_X,
               Limits.Min_Rapid_Window_Width),
         Height =>
            Integer'Max
              (Window.Height + Canvas_Start_Y + Extra_Y,
               Limits.Min_Rapid_Window_Height));

      mcc.Gui.Container.Window.Set_Title
        (main_window.main_window,
         Window.Filename.all);
   exception
      when e : others =>
         mcc.Common_Dialogs.Ok_Box
           ("Invalid window:" & Ada.Exceptions.Exception_Information (e));
   end Display_Window;

   --------------------------------------------------------------
   --------------------------------------------------------------
   procedure Resize_Window
     (Window : in out GUI_Window;
      startx : in Integer;
      starty : in Integer;
      endx   : in Integer;
      endy   : in Integer)
   is
   begin
      Resize_Window
        (Window => Window,
         Width  => Window.Width + (endx - startx),
         Height => Window.Height + (endy - starty));
   end Resize_Window;

   --------------------------------------------------------------
   -- destroy the GUI Window and its resize button
   -- change title of Window
   --------------------------------------------------------------
   procedure Undisplay_Window (Window : in GUI_Window) is
   begin
      mcc.Gui.Destroy (mcc.Gui.Object (Window.Display_Area));
      mcc.Gui.Destroy (mcc.Gui.Object (Window.Menu_Area));
      mcc.Gui.Destroy (mcc.Gui.Object (Window.Resize_Frame));
      mcc.Gui.Widget.Label.Set_Text
        (Obj  => main_window.name,
         Text => "No Window Open");
      mcc.Gui.Container.Window.Set_Title (main_window.main_window, "RAPID");
   end Undisplay_Window;

   --------------------------------------------------------------
   -- Generate Ada code for this window.
   --------------------------------------------------------------
   procedure Generate_Window (Window : in GUI_Window) is
      -- wbw 06/06/99
      -- mcc 05/23/00 added enumeration type for radios
      -- Generate spec for the window that looks like:
      -- -- Auto generated...
      -- -- By: Martin C. Carlisle and C2C W. Blair Watkinson
      -- package filename is
      --    procedure Generate_Window;
      -- end filename;
      -- with statements;
      -- package filename is
      --    newButton : Mcc.Gui.Widget.Button.Picture_Button.Picture_Button;
      --    openButton :
      --    saveButton :
      --    cutButton:
      --    compileButton:
      --    ...
      --
      --    Window : Mcc.Gui.Container.Window.Subwindow;
      --
      --    procedure Generate_Window;
      --
      -- end filename;
      Rapid_Comment : constant String :=
         "-- Auto generated by RAPID " &
         "(http://savannah.nongnu.org/projects/rapid/)";

      Already_Context_Widgets : gui.Widget.Widget_List;
      -- check to see if a widget of this type is
      -- already in the Already_Context_Widgets list.
      -- return true if so, false otherwise.
      function Already_Context
        (Widget : in gui.Widget.Widget_Access)
         return   Boolean
      is
         Current_Widget_Position : Listpkg.Position;
         Found                   : Boolean := False;
      begin
         Current_Widget_Position :=
            Listpkg.First (L => Already_Context_Widgets);
         while not Listpkg.IsPastEnd
                     (Already_Context_Widgets,
                      P => Current_Widget_Position)
         loop
            if Listpkg.Retrieve
                  (L => Already_Context_Widgets,
                   P => Current_Widget_Position).all'Tag =
               Widget'Tag
            then
               Found := True;
               exit;
            end if;
            Listpkg.GoAhead
              (L => Already_Context_Widgets,
               P => Current_Widget_Position);
         end loop;
         return Found;
      end Already_Context;

      function Anything_To_Fill_Or_Read
        (Widget_List : in gui.Widget.Widget_List)
         return        Boolean
      is
         Current_Pos    : gui.Widget.Widget_Pointer;
         Current_Widget : gui.Widget.Widget_Access;
      begin
         Current_Pos := Listpkg.First (Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Pos) loop
            Current_Widget := Listpkg.Retrieve (Widget_List, Current_Pos);
            if gui.Widget.Has_Anything_To_Fill_Or_Read
                 (Current_Widget.all)
            then
               return True;
            end if;
            Listpkg.GoAhead (Widget_List, Current_Pos);
         end loop;
         return False;
      end Anything_To_Fill_Or_Read;

      procedure Generate_Spec
        (Filename    : in String;
         Widget_List : in gui.Widget.Widget_List)
      is
         Current_Widget_Position : gui.Widget.Widget_Pointer;
         Current_Widget          : gui.Widget.Widget_Access;
         NoMenu                  : Boolean;
         Window_Name             : constant String :=
                             Filename (Filename'First .. Filename'Last - 7);
         use File_Helpers;
      begin
         NoMenu := gui.Menu.Menu_List_Package.IsEmpty (Window.Menu);
         Ada.Text_IO.Create
           (File => File_Helpers.Output_File,
            Mode => Ada.Text_IO.Out_File,
            Name => Filename & ".ads");
         Set_Indent (0);
         P (Rapid_Comment);
         P;
         Generate_Helpers.Generate_With ("Mcc.Gui.Container.Window");
         if not NoMenu then
            Generate_Helpers.Generate_With ("Mcc.Gui.Menu");
         end if;
         P;
         -- for each widget, add it to the already_context_widgets
         -- list if it is the first of its type
         Current_Widget_Position := Listpkg.First (L => Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Widget_Position)
         loop
            Current_Widget :=
               Listpkg.Retrieve
                 (L => Widget_List,
                  P => Current_Widget_Position);
            if not Already_Context (Current_Widget) then
               Listpkg.AddToFront
                 (L => Already_Context_Widgets,
                  X => Current_Widget);
            end if;
            Listpkg.GoAhead (L => Widget_List, P => Current_Widget_Position);
         end loop;

         -- generate the with's for each widget's mcc.gui type
         -- using the already_context_widgets list
         -- (this is for the spec, so don't need withs for actions
         --  yet)
         Current_Widget_Position :=
            Listpkg.First (L => Already_Context_Widgets);
         while not Listpkg.IsPastEnd
                     (L => Already_Context_Widgets,
                      P => Current_Widget_Position)
         loop
            gui.Widget.Generate_Widget_Context_Clause
              (Widget =>
                 Listpkg.Retrieve
                    (L => Already_Context_Widgets,
                     P => Current_Widget_Position).all);
            Listpkg.GoAhead
              (L => Already_Context_Widgets,
               P => Current_Widget_Position);
         end loop;
         P;
         P ("package " & Filename & " is");
         P;
         Set_Indent (1);

         -- declarations (menus)
         if not NoMenu then
            P (Window_Name & "_Menu : aliased Mcc.Gui.Menu.Window_Menu;");
            gui.Menu.Generate_Menu_Declaration (Menu => Window.Menu);
         end if;

         -- declarations (widgets)
         Current_Widget_Position := Listpkg.First (L => Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Widget_Position)
         loop
            gui.Widget.Generate_Widget_Declaration
              (Widget =>
                 Listpkg.Retrieve
                    (L => Widget_List,
                     P => Current_Widget_Position).all);
            Listpkg.GoAhead (L => Widget_List, P => Current_Widget_Position);
         end loop;
         Widget_IO.Generate_Radio_Groups (Widgets => Widget_List);
         P;
         P (Filename & " : aliased Mcc.Gui.Container.Window.",
            Newline => False);
         if Window_Name = "main" then
            P ("Main_Window;", Indent => False);
         else
            P ("SubWindow;", Indent => False);
         end if;
         P;
         P ("Already_Open : exception;");
         P ("-- already open is raised if generate_window is called");
         P ("-- when the window " & Filename & " is already open");
         P ("-- create the window");
         P ("procedure Generate_Window;");
         P;
         if Anything_To_Fill_Or_Read (Widget_List) then
            P ("-- fill in values as specified in RAPID");
            P ("procedure Fill_Window;");
            P;
            P ("-- do both Generate_Window and Fill_Window");
            P ("procedure Generate_and_Fill_Window;");
            P;
            P ("-- read out values as specified in RAPID");
            P ("-- if a constraint_error occurs (bad range)");
            P ("-- success will be false, and the offending");
            P ("-- entry will be highlighted if Highlight_Error is true");
            P ("-- additionally, a beep will occur if Beep_On_Error is true");
            P ("procedure Read_Window");
            P ("  (Success         :    out Boolean;");
            P ("   Highlight_Error : in     Boolean := True;");
            P ("   Beep_On_Error   : in     Boolean := True);");
            P;
         end if;
         P ("-- close the window");
         P ("procedure Close_Window;");
         P;
         Set_Indent (0);
         P ("end " & Filename & ";");
         Ada.Text_IO.Close (File_Helpers.Output_File);
      end Generate_Spec;

      procedure Generate_Body
        (Filename    : in String;
         Widget_List : in gui.Widget.Widget_List)
      is
         Current_Widget_Position : gui.Widget.Widget_Pointer;
         NoMenu                  : Boolean;
         use type Gui_Enum.Frame_T;
         use File_Helpers;
      begin
         NoMenu := gui.Menu.Menu_List_Package.IsEmpty (Window.Menu);
         Ada.Text_IO.Create
           (File => File_Helpers.Output_File,
            Name => Filename & ".adb",
            Mode => Ada.Text_IO.Out_File);
         Set_Indent (0);
         P (Rapid_Comment);
         P;
         Generate_Helpers.Generate_With ("Ada.Exceptions");
         P;
         Generate_Helpers.Generate_With ("Mcc.Gui.Fonts");
         if Window.Frame = Gui_Enum.Frame then
           Generate_Helpers.Generate_With ("Mcc.Gui.Container.Frame");
           Generate_Helpers.Generate_With (Window.Parent.all);
         end if;
         -- context clauses for widget actions
         Current_Widget_Position := Listpkg.First (Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Widget_Position)
         loop
            declare
               Widget : constant Gui.Widget.Widget_Access :=
                  Listpkg.Retrieve (Widget_List, Current_Widget_Position);
            begin
               gui.Widget.Generate_Action_Context_Clause (Widget.all);
            end;
            Listpkg.GoAhead (L => Widget_List, P => Current_Widget_Position);
         end loop;
         P;
         -- context clauses for menu actions
         gui.Menu.Generate_Action_Context_Clause
           (Menu => Window.Menu);
         P;
         P ("package body " & Filename & " is");
         Set_Indent (1);
         P;
         P ("procedure Generate_Window is");
         if Window.Frame = Gui_Enum.Frame then
            P ("   " & Filename & " : Mcc.Gui.Container.Frame.Frame;");
         end if;
         P ("begin");
         Set_Indent (2);
         P;
         P ("if Mcc.Gui.Container.Window.Is_Open");
         P ("     (Mcc.Gui.Container.Window.Window (" & Filename & "))");
         P ("then");
         P ("   raise Already_Open;");
         P ("end if;");
         P;
         if Window.Frame = Gui_Enum.Frame then
            P (Window.Parent.all & ".Clear;");
            P (FileName & " := " & Window.Parent.all & ".Frame;");
         else
            P ("Mcc.Gui.Container.Window.Create");
            P ("  (Obj    => " & Filename & ",");
            P ("   X      => " & "0" & ",");
            P ("   Y      => " & "0" & ",");
            P ("   Width  => " & mcc.Img (Window.Width) & ",");
            P ("   Height => " & mcc.Img (Window.Height) & ");");
            P;
            P ("Mcc.Gui.Container.Window.Set_Title");
            P ("  (Obj   => " & Filename & ",");
            if Window.Title /= null then
               P ("   Title => " & """" & Window.Title.all & """" & ");");
            else
               P ("   Title => " & """" &
                  Filename (Filename'First .. Filename'Last - 7) &
                  """" & ");");
            end if;
         end if;
         P;
         if not NoMenu then
            P ("Mcc.Gui.Menu.Create");
            P ("  (Obj    => " &
               Filename (Filename'First .. Filename'Last - 7) & "_Menu,");
            P ("   Window => " & Filename & ");");
            P;
            gui.Menu.Generate_Menu_Creation
              (Menu   => Window.Menu,
               Parent => Filename (Filename'First .. Filename'Last - 7) &
                         "_Menu");
         end if;
         P;
         Current_Widget_Position := Listpkg.First (Widget_List);
         while not Listpkg.IsPastEnd (Widget_List, Current_Widget_Position)
         loop
            gui.Widget.Generate_Widget_Creation
              (Widget      =>
                 Listpkg.Retrieve
                    (L => Widget_List,
                     P => Current_Widget_Position).all,
               Window_Name => Filename);
            gui.Widget.Generate_Widget_Font
              (Widget =>
                 Listpkg.Retrieve
                    (L => Widget_List,
                     P => Current_Widget_Position).all);
            Listpkg.GoAhead (L => Widget_List, P => Current_Widget_Position);
         end loop;
         P;
         Set_Indent (1);
         P ("end Generate_Window;");
         P;
         Widget_IO.Generate_Radio_Groups_Body (Widgets => Widget_List);
         P;
         if Anything_To_Fill_Or_Read (Widget_List) then
            P ("procedure Fill_Window is");
            P ("begin");
            Set_Indent (2);
            -- fill actions
            Current_Widget_Position := Listpkg.First (Widget_List);
            while not Listpkg.IsPastEnd
                        (L => Widget_List,
                         P => Current_Widget_Position)
            loop
               gui.Widget.Generate_Fill_Action
                 (Widget =>
                    Listpkg.Retrieve
                       (L => Widget_List,
                        P => Current_Widget_Position).all);
               Listpkg.GoAhead
                 (L => Widget_List,
                  P => Current_Widget_Position);
            end loop;
            P;
            Set_Indent (1);
            P ("end Fill_Window;");
            P;
            P ("procedure Generate_and_Fill_Window is");
            P ("begin");
            P ("    Generate_Window;");
            P ("    Fill_Window;");
            P ("end Generate_and_Fill_Window;");
            P;
            P ("procedure Read_Window");
            P ("  (Success         :    out Boolean;");
            P ("   Highlight_Error : in     Boolean := True;");
            P ("   Beep_On_Error   : in     Boolean := True) is");
            P ("begin");
            Set_Indent (2);
            P ("Success := True;");
            -- fill actions
            Current_Widget_Position := Listpkg.First (Widget_List);
            while not Listpkg.IsPastEnd
                        (L => Widget_List,
                         P => Current_Widget_Position)
            loop
               gui.Widget.Generate_Read_Action
                 (Widget =>
                    Listpkg.Retrieve
                       (L => Widget_List,
                        P => Current_Widget_Position).all);
               Listpkg.GoAhead
                 (L => Widget_List,
                  P => Current_Widget_Position);
            end loop;
            Set_Indent (1);
            P ("exception when others =>");
            P ("   Success := False;");
            P ("   if Beep_On_Error then");
            P ("      Mcc.Gui.Bell;");
            P ("   end if;");
            P ("end Read_Window;");
            P;
         end if;
         P ("procedure Close_Window is");
         P ("begin");
         P ("   Mcc.Gui.Destroy(Mcc.Gui.Object(" & Filename & "));");
         P ("end Close_Window;");
         P;
         P ("end " & Filename & ";", Indent => False);
         P;
         Ada.Text_IO.Close (File_Helpers.Output_File);
      end Generate_Body;

      Filename : constant String :=
         File_Helpers.Convert_Window_Name (Window.Window_Name.all) & "_window";
   begin -- Generate_Window

      Generate_Spec (Filename, Window.Widget_List);
      Generate_Body (Filename, Window.Widget_List);
      if Window.Novice_Mode then
         Novice_Mode.Generate_Files
           (Window_Name => Window.Window_Name.all,
            Window      => Window);
      end if;
      Generate_Helpers.Reset_Withlist;
   end Generate_Window;

   --------------------------------------------------------------
   -- changes the filename for the Gui_Window
   --
   -- 1) Free the filename string
   -- 2) allocate new string
   -- 3) change title of window using filename w/o directory
   --------------------------------------------------------------
   procedure Change_Filename
     (Window   : in out GUI_Window;
      Filename : in String)
   is
      procedure Free is new Ada.Unchecked_Deallocation (
         String,
         String_Pointer);
   begin
      Free (Window.Filename);
      Window.Filename := new String'(Filename);
      mcc.Gui.Container.Window.Set_Title
        (Obj   => main_window.main_window,
         Title => Window.Filename.all);
   end Change_Filename;

end Gui.Window;
