---------------------------------------------------------------
--
--  FILE_HELPERS.ADB
--  Description : IO helpers
--
--  Copyright (C) 2000, Martin C. Carlisle <carlislem@acm.org>
--
-- FILE_HELPERS is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- FILE_HELPERS is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
-- Contains two procedures Get_String and Put_String, that
-- allow to read and write all characters inside of " as a single
-- item (may not have ").  If " appears inside, it is doubled.
---------------------------------------------------------------
with Unchecked_Deallocation;
with Ada.Characters.Handling;
with Ada.Strings.Maps;
with Ada.Strings.Fixed;

package body File_Helpers is

   function Convert_Window_Name (Window_Name : in String) return String is
      Result : String := Window_Name;
      Start  : Integer;
   begin
      if Window_Name = "." then
         return "main";
      else
         for I in Result'Range loop
            if Result (I) = '.' or else Result (I) = ' ' then
               Result (I) := '_';
            end if;
         end loop;
      end if;

      Start := Result'First;
      while Result (Start) = '_' loop
         Start := Start + 1;
      end loop;

      return Ada.Characters.Handling.To_Lower (Result (Start .. Result'Last));
   end Convert_Window_Name;

   ----------------------------------------------------
   -- PROCEDURE Get_String
   ----------------------------------------------------

   procedure Get_String
     (Token_Index            : in Token_Range;
      Item                   : out String;
      Last                   : out Natural;
      Expect_Quotation_Marks : in Boolean := True)
   is
   begin -- Get_String
      Last := 0;
      if Token_Index > N_Tokens
        or else Token (Token_Index).all'Length < 1
        or else (Expect_Quotation_Marks and Token (Token_Index) (1) /= '"')
      then
         return;
      end if;

      Last := Item'First - 1;
      declare
         Text      : constant String := Token (Token_Index).all;
         Tok_First : Positive        := 1;
         Tok_Last  : Positive        := Text'Last;
      begin
         if Token (Token_Index) (Tok_First) = '"' then
            Tok_First := 2;
         end if;
         if Token (Token_Index) (Tok_Last) = '"' then
            Tok_Last := Tok_Last - 1;
         end if;
         for I in Tok_First .. Tok_Last loop
            if Text (I) = '"' then
               if Text (I - 1) /= '"' then
                  Last        := Last + 1;
                  Item (Last) := Text (I);
               end if;
            else
               Last        := Last + 1;
               Item (Last) := Text (I);
            end if;
         end loop;
      end;
   end Get_String;

   ----------------------------------------------------
   -- PROCEDURE Put_String
   ----------------------------------------------------
   procedure Put_String (Item : in String) is
   begin
      for i in Item'Range loop
         if Item (i) = '"' then
            Ada.Text_IO.Put (Output_File, """""");
         -- string of two quotation marks
         else
            Ada.Text_IO.Put (Output_File, Item (i));
         end if;
      end loop;
   end Put_String;

   ----------------------------------------------------
   -- PROCEDURE Get_Line
   ----------------------------------------------------
   Linenum : Natural := 0;

   Whitespace : constant Ada.Strings.Maps.Character_Set :=
      Ada.Strings.Maps.To_Set (ASCII.HT & ASCII.VT & ASCII.FF & ' ');

   function Index_Non_Blank (S : String) return Natural is
   begin
      return Ada.Strings.Fixed.Index (S, Whitespace, Ada.Strings.Outside);
   end Index_Non_Blank;

   function Find_Closing_Quote (S : String) return Natural is
      Pos : Natural := S'First;
   begin
      while Pos <= S'Last loop
         if S (Pos) = '"' then
            if Pos = S'Last or else S (Pos + 1) /= '"' then
               return Pos;
            end if;
         end if;
         Pos := Pos + 1;
      end loop;
      return 0;   --  Error: no closing quote found
   end Find_Closing_Quote;

   procedure Tokenize (S : String) is
      Pos     : Natural := S'First;
      End_Pos : Natural;
   begin
      while Pos <= S'Last loop
         if Pos = S'Last then
            End_Pos := Pos;
         elsif S (Pos) = '"' then
            End_Pos := Find_Closing_Quote (S (Pos + 1 .. S'Last));
            if End_Pos = 0 then
               Ada.Text_IO.Put_Line
                 ("File_Helpers.Tokenize: ignoring malformed string at line" &
                  Natural'Image (Linenum));
               exit;
            end if;
         else
            End_Pos := Pos;
            while End_Pos < S'Last
              and then not Ada.Strings.Maps.Is_In
                             (S (End_Pos + 1),
                              Whitespace)
            loop
               End_Pos := End_Pos + 1;
            end loop;
         end if;
         N_Tokens             := N_Tokens + 1;
         Token (N_Tokens)     := new String (1 .. End_Pos - Pos + 1);
         Token (N_Tokens).all := S (Pos .. End_Pos);
         exit when End_Pos = S'Last;
         Pos := Index_Non_Blank (S (End_Pos + 1 .. S'Last));
         exit when Pos = 0;
      end loop;
   end Tokenize;

   procedure Reset_Tokens is
      procedure Free is new Unchecked_Deallocation (String, String_Access);
   begin
      if N_Tokens > 0 then
         for I in 1 .. N_Tokens loop
            if Token (I) /= null then
               Free (Token (I));
            end if;
         end loop;
      end if;
      N_Tokens := 0;
   end Reset_Tokens;

   procedure Get_Line is
      Line : String (1 .. 512);
      Last : Natural := 0;
      Pos  : Natural;
   begin
      Reset_Tokens;
      while not Ada.Text_IO.End_Of_File (Input_File) loop
         Ada.Text_IO.Get_Line (Input_File, Line, Last);
         Linenum := Linenum + 1;
         if Last > 0 then
            if Line (Last) = ASCII.CR then
               Last := Last - 1;  -- ignore DOS line end
            end if;
         end if;
         if Last > 0 then
            Pos := Index_Non_Blank (Line (1 .. Last));
            if Pos > 0 then
               Tokenize (Line (Pos .. Last));
               return;
            end if;
         end if;
      end loop;
   end Get_Line;

   Indent_Level : Natural := 0;

   function Spaces return String is
   begin
      return Ada.Strings.Fixed."*" (3 * Indent_Level, ' ');
   end Spaces;

   procedure P
     (Item : in String := "";
      Indent : Boolean := True;
      Newline : Boolean := True) is
   begin
      if Indent then
         Ada.Text_IO.Put (Output_File, Spaces);
      end if;
      Ada.Text_IO.Put (Output_File, Item);
      if Newline then
         Ada.Text_IO.New_Line (Output_File);
      end if;
   end P;

   procedure Put (Item : in String) is
   begin
      P (Item, False, False);
   end Put;

   procedure Set_Indent (Level : Natural) is
   begin
      Indent_Level := Level;
   end Set_Indent;

end File_Helpers;

