---------------------------------------------------------------
--
--  MCC-COMMON_DIALOGS.ADS
--  Description : Description of Common Dialogs
--
--  Copyright (C) 2001 Martin C. Carlisle <carlislem@acm.org>
--
-- This is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- This is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.  This exception does not apply to executables which
-- are GUI design tools, or that could act as a replacement
-- for RAPID.
---------------------------------------------------------------
with Glib;
with Gtk.Box;
with Gtk.Button;
with Gtk.Dialog;
with Gtk.Label;
with Gtk.Main;
with Gtk.Widget;
with mcc.Gtk_Signals;
with Gtkada.Dialogs;           use type Gtkada.Dialogs.Message_Dialog_Buttons;
with Gtkada.File_Selection;
with Ada.Strings.Unbounded;    use Ada.Strings.Unbounded;
with mcc.Directory_Operations;
with Ada.Strings.Fixed;        use Ada.Strings.Fixed;
package body mcc.Common_Dialogs is

   ------------
   -- Ok_Box --
   ------------

   procedure Ok_Box (Message : in String) is
      Result : Gtkada.Dialogs.Message_Dialog_Buttons;
   begin
      Result :=
         Gtkada.Dialogs.Message_Dialog
           (Msg            => Message,
            Dialog_Type    => Gtkada.Dialogs.Information,
            Buttons        => Gtkada.Dialogs.Button_OK,
            Default_Button => Gtkada.Dialogs.Button_Yes,
            Title          => "");
   end Ok_Box;

   procedure Ok_Box
     (Message : in String;
      Parent  : in mcc.Gui.Container.Window.Window'Class)
   is
   begin
      Ok_Box (Message);
   end Ok_Box;

   -----------------
   -- Open_Dialog --
   -----------------

   procedure Open_Dialog
     (File_Types        : in String;
      Filename          : out String;
      File_Last         : out Natural;
      Title             : in String  := "Open";
      Initial_Directory : in String  := ".";
      Default_Extension : in String  := "";
      Change_Directory  : in Boolean := True)
   is
      Starting_Directory : Unbounded_String := Null_Unbounded_String;
   begin
      if Initial_Directory /= "" then
         Starting_Directory := To_Unbounded_String (Initial_Directory);
         if Initial_Directory (Initial_Directory'Last) /=
            mcc.Directory_Operations.Directory_Separator
         then
            Starting_Directory := Starting_Directory &
                                  mcc.Directory_Operations.Directory_Separator;
         end if;
      end if;
      declare
         Answer           : String          :=
            Gtkada.File_Selection.File_Selection_Dialog
              (Title       => Title,
               Default_Dir => To_String (Starting_Directory),
               Dir_Only    => False,
               Must_Exist  => False);
         Directory_Last   : Natural;
         Separator_String : String (1 .. 1) :=
           (others => mcc.Directory_Operations.Directory_Separator);
      begin
         File_Last                              := Filename'First +
                                                   Answer'Length -
                                                   1;
         Filename (Filename'First .. File_Last) := Answer;
         if Answer /= "" and Change_Directory then
            Directory_Last :=
               Index
                 (Source  => Answer,
                  Pattern => Separator_String,
                  Going   => Ada.Strings.Backward);
            mcc.Directory_Operations.Change_Dir
              (Answer (Answer'First .. Directory_Last));
         end if;
      end;

   end Open_Dialog;

   -----------------
   -- Open_Dialog --
   -----------------

   procedure Open_Dialog
     (File_Types        : in String;
      Filename          : out String;
      File_Last         : out Natural;
      Directory         : out String;
      Dir_Last          : out Natural;
      Title             : in String  := "Open";
      Initial_Directory : in String  := ".";
      Default_Extension : in String  := "";
      Change_Directory  : in Boolean := True)
   is
      Directory_Last   : Natural;
      Separator_String : String (1 .. 1) :=
        (others => mcc.Directory_Operations.Directory_Separator);
   begin
      Open_Dialog
        (File_Types        => File_Types,
         Filename          => Filename,
         File_Last         => File_Last,
         Title             => Title,
         Initial_Directory => Initial_Directory,
         Default_Extension => Default_Extension,
         Change_Directory  => Change_Directory);
      if File_Last > Filename'First then
         Directory_Last                          :=
            Index
              (Source  => Filename (Filename'First .. File_Last),
               Pattern => Separator_String,
               Going   => Ada.Strings.Backward);
         Dir_Last                                := Directory'First +
                                                    Directory_Last -
                                                    Filename'First;
         Directory (Directory'First .. Dir_Last) :=
           Filename (Filename'First .. Directory_Last);
      end if;
   end Open_Dialog;

   -----------------
   -- Quit_Dialog --
   -----------------

   procedure Quit_Dialog (Verify : Boolean := True) is
      Really_Quit : Boolean := True;
   begin
      if Verify then
         Really_Quit :=
            Yesno_Dialog
              (Message => "Are you sure you want to quit?",
               Title   => "Exit now?");
      end if;

      if Really_Quit then
         Gtk.Main.Main_Quit;
      end if;
   end Quit_Dialog;

   -----------------
   -- Save_Dialog --
   -----------------

   procedure Save_Dialog
     (File_Types        : in String;
      Filename          : out String;
      File_Last         : out Natural;
      Title             : in String  := "Save As";
      Initial_Directory : in String  := ".";
      Default_Extension : in String  := "";
      Change_Directory  : in Boolean := True)
   is
      Starting_Directory : Unbounded_String := Null_Unbounded_String;
   begin
      if Initial_Directory /= "" then
         Starting_Directory := To_Unbounded_String (Initial_Directory);
         if Initial_Directory (Initial_Directory'Last) /=
            mcc.Directory_Operations.Directory_Separator
         then
            Starting_Directory := Starting_Directory &
                                  mcc.Directory_Operations.Directory_Separator;
         end if;
      end if;
      declare
         Answer           : String          :=
            Gtkada.File_Selection.File_Selection_Dialog
              (Title       => Title,
               Default_Dir => To_String (Starting_Directory),
               Dir_Only    => False,
               Must_Exist  => False);
         Directory_Last   : Natural;
         Separator_String : String (1 .. 1) :=
           (others => mcc.Directory_Operations.Directory_Separator);
      begin
         File_Last                              := Filename'First +
                                                   Answer'Length -
                                                   1;
         Filename (Filename'First .. File_Last) := Answer;
         if Answer /= "" and Change_Directory then
            Directory_Last :=
               Index
                 (Source  => Answer,
                  Pattern => Separator_String,
                  Going   => Ada.Strings.Backward);
            mcc.Directory_Operations.Change_Dir
              (Answer (Answer'First .. Directory_Last));
         end if;
      end;

   end Save_Dialog;

   -----------------
   -- Save_Dialog --
   -----------------

   procedure Save_Dialog
     (File_Types        : in String;
      Filename          : out String;
      File_Last         : out Natural;
      Directory         : out String;
      Dir_Last          : out Natural;
      Title             : in String  := "Save As";
      Initial_Directory : in String  := ".";
      Default_Extension : in String  := "";
      Change_Directory  : in Boolean := True)
   is
      Directory_Last   : Natural;
      Separator_String : String (1 .. 1) :=
        (others => mcc.Directory_Operations.Directory_Separator);
   begin
      Save_Dialog
        (File_Types        => File_Types,
         Filename          => Filename,
         File_Last         => File_Last,
         Title             => Title,
         Initial_Directory => Initial_Directory,
         Default_Extension => Default_Extension,
         Change_Directory  => Change_Directory);
      if File_Last > Filename'First then
         Directory_Last                          :=
            Index
              (Source  => Filename (Filename'First .. File_Last),
               Pattern => Separator_String,
               Going   => Ada.Strings.Backward);
         Dir_Last                                := Directory'First +
                                                    Directory_Last -
                                                    Filename'First;
         Directory (Directory'First .. Dir_Last) :=
           Filename (Filename'First .. Directory_Last);
      end if;
   end Save_Dialog;

   ------------------
   -- Yesno_Dialog --
   ------------------

   function Yesno_Dialog
     (Message : in String;
      Title   : in String)
      return    Boolean
   is
      Result : Gtkada.Dialogs.Message_Dialog_Buttons;
   begin
      Result :=
         Gtkada.Dialogs.Message_Dialog
           (Msg            => Message,
            Dialog_Type    => Gtkada.Dialogs.Confirmation,
            Buttons        => Gtkada.Dialogs.Button_Yes or
                              Gtkada.Dialogs.Button_No,
            Default_Button => Gtkada.Dialogs.Button_Yes,
            Title          => Title);
      return Result = Gtkada.Dialogs.Button_Yes;
   end Yesno_Dialog;

   function Yesno_Dialog
     (Message : in String;
      Title   : in String;
      Parent  : in mcc.Gui.Container.Window.Window'Class)
      return    Boolean
   is
   begin
      return Yesno_Dialog (Message, Title);
   end Yesno_Dialog;

   function Yesno_Cancel_Dialog
     (Message : in String;
      Title   : in String)
      return    Yesno_Cancel
   is
      Result : Gtkada.Dialogs.Message_Dialog_Buttons;
   begin
      Result :=
         Gtkada.Dialogs.Message_Dialog
           (Msg            => Message,
            Dialog_Type    => Gtkada.Dialogs.Confirmation,
            Buttons        => Gtkada.Dialogs.Button_Yes or
                              Gtkada.Dialogs.Button_No or
                              Gtkada.Dialogs.Button_Cancel,
            Default_Button => Gtkada.Dialogs.Button_Yes,
            Title          => Title);
      case Result is
         when Gtkada.Dialogs.Button_Yes =>
            return Yes;
         when Gtkada.Dialogs.Button_No =>
            return No;
         when others =>
            return Cancel;
      end case;
   end Yesno_Cancel_Dialog;

   function Yesno_Cancel_Dialog
     (Message : in String;
      Title   : in String;
      Parent  : in mcc.Gui.Container.Window.Window'Class)
      return    Yesno_Cancel
   is
   begin
      return Yesno_Cancel_Dialog (Message, Title);
   end Yesno_Cancel_Dialog;

end Mcc.Common_Dialogs;
