---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  MENU_EDIT.ADB
--  Description : Used to visually edit the menus
--
--  Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with menuedit_window;
with gui, gui.Menu, gui.Window;                     use type 
  gui.String_Pointer;
with state;
with Ada.Unchecked_Deallocation;
with submenu_dialog_Window, menuitem_dialog_window;
with Menu_Generate;
with mcc.Gui.Widget.Text_Entry;
with mcc.Gui.Widget.Listbox;
with mcc.Gui.Widget.Button.Radio;
use type mcc.Gui.Widget.Button.Radio.Radio_Pointer;
with mcc.Common_Dialogs; -- for debugging

package body Menu_Edit is
   Indent_Width  : constant := 20;
   Item_Height   : constant := 15;
   Canvas_Width  : constant := 269;
   Insert_Inside : Boolean := False;

   Number_Entries : Integer;
   Selected_Item  : Integer;
   Selected_Menu  : gui.Menu.Menu_Access;
   type Item_Type is (Menu, Item, Separator);
   Am_Inserting : Boolean := False;
   Active_Type  : Item_Type;
   -- we cannot allow two subdialogs to run simultaneously
   Subdialog_Running : Boolean := False;

   type Operation_Type is (Insert, Delete, Edit);

   procedure Free is new Ada.Unchecked_Deallocation (
      String,
      gui.String_Pointer);
   -----------------------------------------------------------------
   -- procedure Find_Selected_Item
   --
   -- Look for the appropriate number in the menubar
   -- return if it is found, its position, and perform an operation
   -- (either insert, delete, or edit)
   --
   -- Walk through the list until you find the number
   -- On finding a submenu, recursively call this procedure
   --
   -- If we find the number, delete it, or set the
   -- package variable Selected_Menu (for edit),
   -- or insert something in the list.  For insert, check if
   -- highlighted is a submenu.  If so, determine where to insert
   -- based on that submenus Insert_Inside setting.
   -----------------------------------------------------------------
   procedure Find_Selected_Item
     (Seeking   : in Integer;
      Menubar   : in out gui.Menu.Menu_Pointer;
      Position  : out gui.Menu.Menu_Position;
      Operation : in Operation_Type;
      Found     : out Boolean)
   is
      Current_Position : gui.Menu.Menu_Position :=
         gui.Menu.Menu_List_Package.First (Menubar);
      Current_Menu     : gui.Menu.Menu_Access;
   begin
      while not gui.Menu.Menu_List_Package.IsPastEnd
                  (Menubar,
                   Current_Position)
      loop
         Current_Menu :=
            gui.Menu.Menu_List_Package.Retrieve (Menubar, Current_Position);
         if Current_Menu.Number = Seeking then
            Position := Current_Position;
            Found    := True;
            case Operation is
               when Insert =>
                  declare
                     Before_Menu : gui.Menu.Menu_Access;
                  begin
                     Before_Menu :=
                        gui.Menu.Menu_List_Package.Retrieve
                          (Menubar,
                           Position);
                     if Before_Menu.all in gui.Menu.Submenu'Class
                       and then Insert_Inside
                     then
                        gui.Menu.Menu_List_Package.AddToFront
                          (gui.Menu.Submenu (Before_Menu.all).Items,
                           Selected_Menu);
                     else
                        gui.Menu.Menu_List_Package.Insert
                          (Menubar,
                           Selected_Menu,
                           Position);
                     end if;
                  end;
               when Delete =>
                  gui.Menu.Menu_List_Package.Delete (Menubar, Position);
               when Edit =>
                  Selected_Menu :=
                     gui.Menu.Menu_List_Package.Retrieve (Menubar, Position);
            end case;
            return;
         elsif Current_Menu.all in gui.Menu.Submenu'Class then
            Find_Selected_Item
              (Seeking   => Seeking,
               Menubar   => gui.Menu.Submenu (Current_Menu.all).Items,
               Operation => Operation,
               Position  => Position,
               Found     => Found);
            if Found then
               return;
            end if;
         end if;
         gui.Menu.Menu_List_Package.GoAhead (Menubar, Current_Position);
      end loop;
      Found := False;
   end Find_Selected_Item;

   -----------------------------------------------------------------
   -- display submenu or menu item
   --
   -----------------------------------------------------------------
   procedure Display_Item
     (Menu         : in gui.Menu.Menu_Access;
      Indent_Level : in Integer;
      Number       : in out Integer;
      Cascade      : in Boolean)
   is
      Postfix : Character;
   begin
      if Cascade then
         Postfix := '*';
      else
         Postfix := ' ';
      end if;

      Menu.Number       := Number;
      Menu.Indent_Level := Indent_Level;

      declare
         Indent : String (1 .. Indent_Level * 3) := (others => ' ');
      begin
         mcc.Gui.Widget.Listbox.Add_Entry
           (Obj      => menuedit_window.menu_display,
            Location => Number,
            Text     => Indent & Menu.Name.all & Postfix);
      end;

      if Menu.Number > Number_Entries then
         Number_Entries := Menu.Number;
      end if;

      Number := Number + 1;
   end Display_Item;

   -----------------------------------------------------------------
   -- Display menus in edit window
   --
   -- Traverse list calling Display_Item
   -- recursively call this adding 1 to indent level
   -- for submenus
   -----------------------------------------------------------------
   procedure Display_Menus
     (Menubar      : in gui.Menu.Menu_Pointer;
      Indent_Level : in Integer := 1;
      Number       : in out Integer)
   is

      Position : gui.Menu.Menu_Position :=
         gui.Menu.Menu_List_Package.First (Menubar);
      Traverse : gui.Menu.Menu_Access;
   begin
      if Indent_Level = 1 then
         mcc.Gui.Widget.Listbox.Add_Entry
           (Obj      => menuedit_window.menu_display,
            Location => 0,
            Text     => "MENUBAR");
      end if;

      while not gui.Menu.Menu_List_Package.IsPastEnd (Menubar, Position) loop
         Traverse := gui.Menu.Menu_List_Package.Retrieve (Menubar, Position);
         if Traverse.all in gui.Menu.Menu_Item'Class
           or else Traverse.all in gui.Menu.Separator'Class
         then
            Display_Item
              (Menu         => Traverse,
               Indent_Level => Indent_Level,
               Number       => Number,
               Cascade      => False);
         elsif Traverse.all in gui.Menu.Submenu'Class then
            Display_Item
              (Menu         => Traverse,
               Indent_Level => Indent_Level,
               Number       => Number,
               Cascade      => True);
            Display_Menus
              (Menubar      => gui.Menu.Submenu (Traverse.all).Items,
               Indent_Level => Indent_Level + 1,
               Number       => Number);
         else
            raise Constraint_Error;
         end if;
         gui.Menu.Menu_List_Package.GoAhead (Menubar, Position);
      end loop;
   end Display_Menus;

   procedure Menu_Display_Mouse_Listener
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
      use mcc.Gui;
   begin
      if (Event.Button = Left and Event.Action = Double_Click) or
         (Event.Button = Right and Event.Action = Press)
      then
         Edit_Item;
      end if;
   end Menu_Display_Mouse_Listener;
   -----------------------------------------------------------------
   -- pops up a window that Edits the menu
   --
   -- 1) Generate window
   -- 2) Select a radio
   -- 3) bind close and mouse
   -----------------------------------------------------------------
   procedure Edit
     (Menubar : in gui.Menu.Menu_Pointer;
      Window  : in mcc.Gui.Container.Container'Class)
   is
      Number : Integer;
   begin -- Edit
      state.Set_Dialog_Running (True);
      menuedit_window.Generate_Window;

      mcc.Gui.Container.Window.Set_Close_Handler
        (Obj     => menuedit_window.menuedit_window'Access,
         Handler => Done_Edit'Access);

      Number_Entries := 0;
      Selected_Item  := 0;
      Number         := 1;

      Display_Menus (Menubar => Menubar, Number => Number);

      -- select a position to insert
      mcc.Gui.Widget.Button.Radio.Select_Radio (menuedit_window.inside);

      -- bind double click in Menu_Display to edit_Item?
      mcc.Gui.Set_Mouse_Listener
        (Obj      => menuedit_window.menu_display'Access,
         Listener => Menu_Display_Mouse_Listener'Access);
   end Edit;

   -----------------------------------------------------------------
   -- procedure Insert_Menu
   --
   -- get selected item
   -- set insert_inside
   -- set am_inserting
   -- set active_type
   -- set subdialog_running
   -- run submenu dialog
   -- give focus to entry 1
   -- set window close to cancel
   -----------------------------------------------------------------
   procedure Insert_Menu is
   begin
      if Subdialog_Running then
         return;
      end if;
      -- offset for MENUBAR
      Selected_Item :=
        mcc.Gui.Widget.Listbox.Get_Selected (menuedit_window.menu_display) -
        1;

      Insert_Inside :=
        (mcc.Gui.Widget.Button.Radio.Get_Selected
            (menuedit_window.insert_group) =
         menuedit_window.inside'Access);

      Am_Inserting      := True;
      Active_Type       := Menu;
      Subdialog_Running := True;
      submenu_dialog_Window.Generate_Window;
      mcc.Gui.Widget.Text_Entry.Highlight
        (Obj   => submenu_dialog_Window.entry1,
         Start => 0,
         Stop  => 0);
      if state.Get_Current_Window.Novice_Mode then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => submenu_dialog_Window.entry3,
            Text => "disabled for novice");
         mcc.Gui.Widget.Text_Entry.Disable
           (Obj => submenu_dialog_Window.entry3);
      end if;
      mcc.Gui.Container.Window.Set_Close_Handler
        (Obj     => submenu_dialog_Window.submenu_dialog_Window'Access,
         Handler => Cancel_Insert_Dialog'Access);
   end Insert_Menu;
   procedure Insert_Menu (Obj : in out mcc.Gui.Widget.Button.Button'Class) is
   begin
      Insert_Menu;
   end Insert_Menu;

   -----------------------------------------------------------------
   -- procedure Insert_Menu
   --
   -- get selected item
   -- set insert_inside
   -- set am_inserting
   -- set active_type
   -- set subdialog_running
   -- run menuitem dialog
   -- give focus to entry 1
   -- set window close to cancel
   -----------------------------------------------------------------
   procedure Insert_Choice is
   begin
      if Subdialog_Running then
         return;
      end if;
      -- offset for MENUBAR
      Selected_Item :=
        mcc.Gui.Widget.Listbox.Get_Selected (menuedit_window.menu_display) -
        1;

      Insert_Inside :=
        (mcc.Gui.Widget.Button.Radio.Get_Selected
            (menuedit_window.insert_group) =
         menuedit_window.inside'Access);

      Am_Inserting      := True;
      Active_Type       := Item;
      Subdialog_Running := True;
      menuitem_dialog_window.Generate_Window;
      mcc.Gui.Widget.Text_Entry.Highlight
        (Obj   => menuitem_dialog_window.entry1,
         Start => 0,
         Stop  => 0);
      if state.Get_Current_Window.Novice_Mode then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => menuitem_dialog_window.entry3,
            Text => "disabled for novice");
         mcc.Gui.Widget.Text_Entry.Disable
           (Obj => menuitem_dialog_window.entry3);
      end if;
      mcc.Gui.Container.Window.Set_Close_Handler
        (Obj     => menuitem_dialog_window.menuitem_dialog_window'Access,
         Handler => Cancel_Insert_Dialog'Access);
   end Insert_Choice;
   procedure Insert_Choice
     (Obj : in out mcc.Gui.Widget.Button.Button'Class)
   is
   begin
      Insert_Choice;
   end Insert_Choice;

   procedure Insert_Separator
     (Obj : in out mcc.Gui.Widget.Button.Button'Class)
   is
   begin
      if Subdialog_Running then
         return;
      end if;
      -- offset for MENUBAR
      Selected_Item :=
        mcc.Gui.Widget.Listbox.Get_Selected (menuedit_window.menu_display) -
        1;

      Insert_Inside :=
        (mcc.Gui.Widget.Button.Radio.Get_Selected
            (menuedit_window.insert_group) =
         menuedit_window.inside'Access);

      Am_Inserting := True;
      Active_Type  := Separator;
      Done_Insert_Dialog;
   end Insert_Separator;
   -----------------------------------------------------------------
   -- procedure Redraw_Display
   --
   -- delete everything from canvas and redraw them all
   -- not exactly efficient, but it works
   -----------------------------------------------------------------
   procedure Redraw_Display is
      Window : gui.Window.Window_Pointer := state.Get_Current_Window;
      Number : Integer;
   begin
      mcc.Gui.Widget.Listbox.Clear (menuedit_window.menu_display);

      Number := 1;

      Display_Menus (Menubar => Window.Menu, Number => Number);
      if Selected_Item > Number_Entries then
         Selected_Item := Number_Entries;
      end if;
      -- offset for MENUBAR
      mcc.Gui.Widget.Listbox.Select_Item
        (Obj    => menuedit_window.menu_display,
         Number => Selected_Item + 1);
   end Redraw_Display;

   -----------------------------------------------------------------
   -- procedure Delete
   --
   -- call Find_Selected_Item to do work for you
   -- Redraw display if something was found
   -----------------------------------------------------------------
   procedure Delete is
      Window : gui.Window.Window_Pointer := state.Get_Current_Window;
      Item   : gui.Menu.Menu_Position;
      Found  : Boolean;
   begin
      if Subdialog_Running then
         return;
      end if;

      Selected_Item :=
         mcc.Gui.Widget.Listbox.Get_Selected (menuedit_window.menu_display);

      if Selected_Item = 0 then
         return;
      else
         -- offset for MENUBAR
         Selected_Item := Selected_Item - 1;
      end if;

      Find_Selected_Item
        (Seeking   => Selected_Item,
         Menubar   => Window.Menu,
         Position  => Item,
         Operation => Delete,
         Found     => Found);

      if Found then
         Redraw_Display;
      end if;

      state.Set_Changed (True);
   end Delete;
   procedure Delete (Obj : in out mcc.Gui.Widget.Button.Button'Class) is
   begin
      Delete;
   end Delete;

   -----------------------------------------------------------------
   -- procedure Edit_Item
   --
   -- run Find_Selected_Item
   -- then performs code similar to insert for appropriate
   -- type of menu (submenu or item), also using Common_Entries
   -- to set name, underline and action text entries.
   -----------------------------------------------------------------
   procedure Edit_Item is
      Window    : gui.Window.Window_Pointer := state.Get_Current_Window;
      This_Item : gui.Menu.Menu_Position;
      Found     : Boolean;
   begin
      if Subdialog_Running then
         return;
      end if;

      Selected_Item :=
         mcc.Gui.Widget.Listbox.Get_Selected (menuedit_window.menu_display);

      if Selected_Item = 0 then
         return;
      else
         -- offset for MENUBAR
         Selected_Item := Selected_Item - 1;
      end if;

      Find_Selected_Item
        (Seeking   => Selected_Item,
         Menubar   => Window.Menu,
         Position  => This_Item,
         Operation => Edit,
         Found     => Found);

      Am_Inserting := False;

      if Found then
         if Selected_Menu.all in gui.Menu.Submenu'Class then
            Active_Type       := Menu;
            Subdialog_Running := True;
            submenu_dialog_Window.Generate_Window;
            mcc.Gui.Widget.Text_Entry.Highlight
              (Obj   => submenu_dialog_Window.entry1,
               Start => 0,
               Stop  => 0);
            mcc.Gui.Container.Window.Set_Close_Handler
              (Obj     => submenu_dialog_Window.submenu_dialog_Window'Access,
               Handler => Cancel_Insert_Dialog'Access);
            mcc.Gui.Widget.Text_Entry.Set_Text
              (Obj  => submenu_dialog_Window.entry1,
               Text => Selected_Menu.Name.all);
            mcc.Gui.Widget.Text_Entry.Set_Text
              (Obj  => submenu_dialog_Window.entry2,
               Text => Integer'Image (Selected_Menu.Underline_Position));
            if state.Get_Current_Window.Novice_Mode then
               mcc.Gui.Widget.Text_Entry.Set_Text
                 (Obj  => submenu_dialog_Window.entry3,
                  Text => "disabled for novice");
               mcc.Gui.Widget.Text_Entry.Disable
                 (Obj => submenu_dialog_Window.entry3);
            else
               if Selected_Menu.Action /= null then
                  mcc.Gui.Widget.Text_Entry.Set_Text
                    (Obj  => submenu_dialog_Window.entry3,
                     Text => Selected_Menu.Action.all);
               end if;
            end if;
         elsif Selected_Menu.all in gui.Menu.Menu_Item'Class then
            Active_Type       := Item;
            Subdialog_Running := True;
            menuitem_dialog_window.Generate_Window;
            mcc.Gui.Container.Window.Set_Close_Handler
              (Obj     =>
                 menuitem_dialog_window.menuitem_dialog_window'Access,
               Handler => Cancel_Insert_Dialog'Access);
            mcc.Gui.Widget.Text_Entry.Set_Text
              (Obj  => menuitem_dialog_window.entry1,
               Text => Selected_Menu.Name.all);
            mcc.Gui.Widget.Text_Entry.Set_Text
              (Obj  => menuitem_dialog_window.entry2,
               Text => Integer'Image (Selected_Menu.Underline_Position));
            if state.Get_Current_Window.Novice_Mode then
               mcc.Gui.Widget.Text_Entry.Set_Text
                 (Obj  => menuitem_dialog_window.entry3,
                  Text => "disabled for novice");
               mcc.Gui.Widget.Text_Entry.Disable
                 (Obj => menuitem_dialog_window.entry3);
            elsif Selected_Menu.Action /= null then
               mcc.Gui.Widget.Text_Entry.Set_Text
                 (Obj  => menuitem_dialog_window.entry3,
                  Text => Selected_Menu.Action.all);
            end if;
            if gui.Menu.Menu_Item (Selected_Menu.all).Accelerator /=
               null
            then
               mcc.Gui.Widget.Text_Entry.Set_Text
                 (Obj  => menuitem_dialog_window.entry4,
                  Text =>
                    gui.Menu.Menu_Item (Selected_Menu.all).Accelerator.all);
            end if;
         elsif Selected_Menu.all in gui.Menu.Separator'Class then
            null; -- can't edit a separator
         else
            raise Constraint_Error;
         end if;
      end if;
   end Edit_Item;
   procedure Edit_Item (Obj : in out mcc.Gui.Widget.Button.Button'Class) is
   begin
      Edit_Item;
   end Edit_Item;

   -----------------------------------------------------------------
   -- procedure Done_Insert_Dialog
   --
   -- keeps track of whether inserting vs editing (same dialog is used)
   -----------------------------------------------------------------
   procedure Done_Insert_Dialog is
      Number    : Integer;
      Cascade   : Boolean;
      Found     : Boolean;
      Window    : gui.Window.Window_Pointer := state.Get_Current_Window;
      This_Item : gui.Menu.Menu_Position;
   begin
      if Am_Inserting then
         if Active_Type = Menu then
            Selected_Menu := new gui.Menu.Submenu;
         elsif Active_Type = Item then
            Selected_Menu := new gui.Menu.Menu_Item;
         elsif Active_Type = Separator then
            Selected_Menu := new gui.Menu.Separator;
         end if;

         if Selected_Item > 0 then
            Find_Selected_Item
              (Seeking   => Selected_Item,
               Menubar   => Window.Menu,
               Position  => This_Item,
               Operation => Insert,
               Found     => Found);
         else
            Found := True;
            gui.Menu.Menu_List_Package.AddToFront
              (Window.Menu,
               Selected_Menu);
         end if;

         if not Found then
            return;
         end if;

      end if;

      Free (Selected_Menu.Name);
      Free (Selected_Menu.Action);
      if Selected_Menu.all in gui.Menu.Submenu'Class then
         Cascade := True;
         declare
            Name : String :=
               mcc.Gui.Widget.Text_Entry.Get_Text
                 (Obj => submenu_dialog_Window.entry1);
         begin
            if Name'Length > 0 then
               Selected_Menu.Name := new String'(Name);
            else
               Selected_Menu.Name := new String'("Untitled");
            end if;
         end;
         begin
            Selected_Menu.Underline_Position :=
               mcc.Gui.Widget.Text_Entry.Get_Text
                 (Obj => submenu_dialog_Window.entry2);
         exception
            when others =>
               Selected_Menu.Underline_Position := 0;
         end;
         declare
            Action : String :=
               mcc.Gui.Widget.Text_Entry.Get_Text
                 (Obj => submenu_dialog_Window.entry3);
         begin
            if state.Get_Current_Window.Novice_Mode then
               Selected_Menu.Action :=
                 new String'(state.Get_Current_Window.Window_Name.all &
                             "_Actions." &
                             gui.Menu.Convert_Menu_Name
                                (Selected_Menu.Name.all) &
                             "_Selected");
            else
               if Action'Length > 0 then
                  Selected_Menu.Action := new String'(Action);
               else
                  Selected_Menu.Action := new String'("null");
               end if;
            end if;
         end;
         Cancel_Insert_Dialog;
      elsif Selected_Menu.all in gui.Menu.Menu_Item'Class then
         Cascade := False;
         Free (gui.Menu.Menu_Item (Selected_Menu.all).Accelerator);
         declare
            Name : String :=
               mcc.Gui.Widget.Text_Entry.Get_Text
                 (Obj => menuitem_dialog_window.entry1);
         begin
            if Name'Length > 0 then
               Selected_Menu.Name := new String'(Name);
            else
               Selected_Menu.Name := new String'("Untitled");
            end if;
         end;
         begin
            Selected_Menu.Underline_Position :=
               mcc.Gui.Widget.Text_Entry.Get_Text
                 (Obj => menuitem_dialog_window.entry2);
         exception
            when others =>
               Selected_Menu.Underline_Position := 0;
         end;
         declare
            Action : String :=
               mcc.Gui.Widget.Text_Entry.Get_Text
                 (Obj => menuitem_dialog_window.entry3);
         begin
            if state.Get_Current_Window.Novice_Mode then
               Selected_Menu.Action :=
                 new String'(state.Get_Current_Window.Window_Name.all &
                             "_Actions." &
                             gui.Menu.Convert_Menu_Name
                                (Selected_Menu.Name.all) &
                             "_Selected");
            else
               if Action'Length > 0 then
                  Selected_Menu.Action := new String'(Action);
               else
                  Selected_Menu.Action := new String'("null");
               end if;
            end if;
         end;
         declare
            Accelerator : String :=
               mcc.Gui.Widget.Text_Entry.Get_Text
                 (Obj => menuitem_dialog_window.entry4);
         begin
            gui.Menu.Menu_Item (Selected_Menu.all).Accelerator :=
              new String'(Accelerator);
         end;
         Cancel_Insert_Dialog;
      elsif Selected_Menu.all in gui.Menu.Separator'Class then
         Selected_Menu.Name := Separator_Name'Access;
      else
         raise Constraint_Error;
      end if;

      if not Am_Inserting then
         mcc.Gui.Widget.Listbox.Delete_Entry
           (Obj   => menuedit_window.menu_display,
            Start => Selected_Menu.Number + 1,
            Stop  => Selected_Menu.Number + 1);

         -- since Number is in out, use a copy
         Number := Selected_Menu.Number;
         Display_Item
           (Menu         => Selected_Menu,
            Indent_Level => Selected_Menu.Indent_Level,
            Number       => Number,
            Cascade      => Cascade);
         mcc.Gui.Widget.Listbox.Select_Item
           (Obj    => menuedit_window.menu_display,
            Number => Number);
      else
         Redraw_Display;
         -- move down 1 (see Redraw_Display)
         mcc.Gui.Widget.Listbox.Select_Item
           (Obj    => menuedit_window.menu_display,
            Number => Selected_Item + 2);
      end if;

      state.Set_Changed (True);
   end Done_Insert_Dialog;

   -----------------------------------------------------------------
   -- procedure Cancel_Insert_Dialog
   --
   -- destroy the dialog
   -----------------------------------------------------------------
   procedure Cancel_Insert_Dialog is
   begin
      if Active_Type = Menu then
         mcc.Gui.Container.Window.Destroy
           (submenu_dialog_Window.submenu_dialog_Window);
      else
         mcc.Gui.Container.Window.Destroy
           (menuitem_dialog_window.menuitem_dialog_window);
      end if;
      Subdialog_Running := False;
   end Cancel_Insert_Dialog;
   procedure Cancel_Insert_Dialog
     (Obj : in out mcc.Gui.Container.Window.Window'Class)
   is
   begin
      Cancel_Insert_Dialog;
   end Cancel_Insert_Dialog;

   procedure Cancel_Insert_Dialog
     (Obj : in out mcc.Gui.Widget.Button.Button'Class)
   is
   begin
      Cancel_Insert_Dialog;
   end Cancel_Insert_Dialog;

   procedure Done_Edit is
      Window      : gui.Window.Window_Pointer := state.Get_Current_Window;
      Was_Changed : Boolean;
   begin
      if Subdialog_Running then
         return;
      end if;

      mcc.Gui.Container.Window.Destroy (menuedit_window.menuedit_window);
      state.Set_Dialog_Running (False);
      -- Ugh!  Doing this resizes the window, so
      -- I remember the window size above, and
      -- reinstate it after redisplaying the menu
      -- even worse, this resize changes the state to true
      Was_Changed := state.Get_Changed;
      Menu_Generate.Display_Menu_Code
        (Menubar   => Window.Menu,
         Window    => Window.all,
         Redisplay => True);

      state.Set_Changed (Was_Changed);
   end Done_Edit;

   procedure Done_Edit (Obj : in out mcc.Gui.Container.Window.Window'Class) is
   begin
      Done_Edit;
   end Done_Edit;

   procedure Done_Edit (Obj : in out mcc.Gui.Widget.Button.Button'Class) is
   begin
      Done_Edit;
   end Done_Edit;

   procedure Done_Insert_Dialog
     (Obj : in out mcc.Gui.Widget.Button.Button'Class)
   is
   begin
      Done_Insert_Dialog;
   end Done_Insert_Dialog;

end Menu_Edit;
