// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/xception.h"
#include "../libpdrx/conversions.h"
#include "../libpdrx/build.h"
#include "config_impl.h"
#include "db.h"
#include "out_ftree.h"
#include "diagram.h"

#include <cstdarg>

const int HbA1c_interval = 90; // days

//=== FImpl (abstract base class) ==========================================
FImpl::FImpl (const type_info* pReturnType, const string& name, size_t n, ...)
	: m_returntype(pReturnType)
	, m_name(name)
	, m_paramtypes()
{
	va_list ap;
	va_start(ap, n);
	for (size_t i = 0; i < n; i++)
		m_paramtypes.push_back(va_arg(ap, const type_info*));
	va_end(ap);
}

FImpl::~FImpl ()
{
}

//=== helper classes =======================================================
//
// these classes do nothing, we need their type_info objects to represent
// something in function prototypes, i.e. an empty return value or an
// open parameter list
//
class nothing {};
class ellipse {};

class estring: public string {
	public:
	estring (const string& s)
		: string(s)
	{
	}
};

//=== helper macros ========================================================
//
// each of these macros defines a new concrete function implementation by
// deriving a new class from FImpl, the macros differ in the number of
// parameters
//
#define DEFINE_FIMPL_0(classname, returntype, funcname) \
class classname: public FImpl {\
	public:\
	classname () : FImpl(&returntype, funcname, 0) {}\
	virtual	any Evaluate (FContext& context, const Params& params) const throw (Xception);\
};\
any classname::Evaluate (FContext& context, const Params& params) const throw (Xception)

#define DEFINE_FIMPL_1(classname, returntype, funcname, paramtype1) \
class classname: public FImpl {\
	public:\
	classname () : FImpl(&returntype, funcname, 1, &paramtype1) {}\
	virtual	any Evaluate (FContext& context, const Params& params) const throw (Xception);\
};\
any classname::Evaluate (FContext& context, const Params& params) const throw (Xception)

#define DEFINE_FIMPL_2(classname, returntype, funcname, paramtype1, paramtype2) \
class classname: public FImpl {\
	public:\
	classname () : FImpl(&returntype, funcname, 2, &paramtype1, &paramtype2) {}\
	virtual	any Evaluate (FContext& context, const Params& params) const throw (Xception);\
};\
any classname::Evaluate (FContext& context, const Params& params) const throw (Xception)

#define DEFINE_FIMPL_3(classname, returntype, funcname, paramtype1, paramtype2, paramtype3) \
class classname: public FImpl {\
	public:\
	classname () : FImpl(&returntype, funcname, 3, &paramtype1, &paramtype2, &paramtype3) {}\
	virtual	any Evaluate (FContext& context, const Params& params) const throw (Xception);\
};\
any classname::Evaluate (FContext& context, const Params& params) const throw (Xception)

#define DEFINE_FIMPL_4(classname, returntype, funcname, paramtype1, paramtype2, paramtype3, paramtype4) \
class classname: public FImpl {\
	public:\
	classname () : FImpl(&returntype, funcname, 4, &paramtype1, &paramtype2, &paramtype3, &paramtype4) {}\
	virtual	any Evaluate (FContext& context, const Params& params) const throw (Xception);\
};\
any classname::Evaluate (FContext& context, const Params& params) const throw (Xception)

#define DEFINE_FIMPL_5(classname, returntype, funcname, paramtype1, paramtype2, paramtype3, paramtype4, paramtype5) \
class classname: public FImpl {\
	public:\
	classname () : FImpl(&returntype, funcname, 5, &paramtype1, &paramtype2, &paramtype3, &paramtype4, &paramtype5) {}\
	virtual	any Evaluate (FContext& context, const Params& params) const throw (Xception);\
};\
any classname::Evaluate (FContext& context, const Params& params) const throw (Xception)

#define DEFINE_FIMPL_6(classname, returntype, funcname, paramtype1, paramtype2, paramtype3, paramtype4, paramtype5, paramtype6) \
class classname: public FImpl {\
	public:\
	classname () : FImpl(&returntype, funcname, 6, &paramtype1, &paramtype2, &paramtype3, &paramtype4, &paramtype5, &paramtype6) {}\
	virtual	any Evaluate (FContext& context, const Params& params) const throw (Xception);\
};\
any classname::Evaluate (FContext& context, const Params& params) const throw (Xception)

#define DEFINE_FIMPL_7(classname, returntype, funcname, paramtype1, paramtype2, paramtype3, paramtype4, paramtype5, paramtype6, paramtype7) \
class classname: public FImpl {\
	public:\
	classname () : FImpl(&returntype, funcname, 7, &paramtype1, &paramtype2, &paramtype3, &paramtype4, &paramtype5, &paramtype6, &paramtype7) {}\
	virtual	any Evaluate (FContext& context, const Params& params) const throw (Xception);\
};\
any classname::Evaluate (FContext& context, const Params& params) const throw (Xception)

//=== helper functions =====================================================
static char value_type (const Selection& selection)
{
	try
	{
		if (selection.empty())
			return 'e';

		if (selection[0][1].type() == typeid(double))
			return 'd';

		if (selection[0][1].type() == typeid(Ratio))
			return 'r';

		if (selection[0][1].type() == typeid(string))
			return 's';

		return '\0';
	}
	CATCH_RETHROW("")
}

typedef pair<any, any> Result;

template<class T> static void over_all (const Selection& selection, Selection& result)
{
	try
	{
		const size_t* pDimensions = selection.shape();
		result.resize(extents[1][2]);
		const Result& r = T()(selection, 0, pDimensions[0] - 1);
		result[0][0] = (r.first.empty()) ? any(string()) : r.first;
		result[0][1] = r.second;
	}
	CATCH_RETHROW("")
}

template<class T> static void over_all (const Selection& selection, Selection& result, const time_duration& from_time, const time_duration& to_time)
{
	try
	{
		const size_t* pDimensions = selection.shape();
		Selection temp;
		temp.resize(extents[pDimensions[0]][2]);
		size_t rows = 0;
		for (size_t row = 0; row < pDimensions[0]; row++)
		{
			ptime t = lexical_cast<ptime>(any_cast<string>(selection[row][0]));
			date d = t.date();
			ptime begin(d, from_time);
			ptime end(d, to_time);
			if (from_time >= to_time)
				end += days(1);
			if (t >= begin && t < end)
			{
				temp[rows][0] = selection[row][0];
				temp[rows][1] = selection[row][1];
				rows++;
			}
		}

		if (rows > 0)
		{
			result.resize(extents[1][2]);
			const Result& r = T()(temp, 0, rows - 1);
			result[0][0] = (r.first.empty()) ? any(string()) : r.first;
			result[0][1] = r.second;
		}
	}
	CATCH_RETHROW("")
}

	static size_t get_group_length (const string& aggr_interval)
	{
		// this function returns the number of characters in the
		// portion of a datetime that is used for grouping values,
		// if we group by day we use 10 characters from the left of
		// a datetime, this ist the date YYYY-MM-DD

		typedef map<string, size_t> Keywords;
		static Keywords keywords;

		if (keywords.empty())
		{
			// YYYY-MM-DD hh:mm:ss
			keywords.insert(Keywords::value_type("year",	4));
			keywords.insert(Keywords::value_type("years",	4));
			keywords.insert(Keywords::value_type("month",	7));
			keywords.insert(Keywords::value_type("months",	7));
//			keywords.insert(Keywords::value_type("week",	?));
//			keywords.insert(Keywords::value_type("weeks",	?));
			keywords.insert(Keywords::value_type("day",	10));
			keywords.insert(Keywords::value_type("days",	10));
			keywords.insert(Keywords::value_type("hour",	13));
			keywords.insert(Keywords::value_type("hours",	13));
			keywords.insert(Keywords::value_type("minute",	16));
			keywords.insert(Keywords::value_type("minutes",	16));
			keywords.insert(Keywords::value_type("second",	19));
			keywords.insert(Keywords::value_type("seconds",	19));
		}

		Keywords::const_iterator I = keywords.find(aggr_interval);
		if (I == keywords.end())
			THROW(format("unknown aggregation interval: %s") % aggr_interval);
		return (*I).second;
	}

	static string get_group_from_timestamp (const string& timestamp, const string& aggr_interval, const time_duration& midnight)
	{
		try
		{
			// we subtract the desired "midnight" from the
			// timestamp, this makes times before "midnight"
			// belong to the day before, times past "midnight"
			// belong to the next day
			ptime t = lexical_cast<ptime>(timestamp) - midnight;

			// then we shorten the resulting timestamp to a
			// significant length
			return string(lexical_cast<string>(t), 0, get_group_length(aggr_interval));
		}
		CATCH_RETHROW("")
	}

template<class T> static void aggregate (const Selection& selection, Selection& result, const string& aggr_interval, const time_duration& midnight)
{
	try
	{
		const size_t* pDimensions = selection.shape();
		size_t rows = 0;
		string group;
		size_t r_lower = 0, r_upper = 0;
		for (size_t row = 0; row < pDimensions[0]; row++)
		{
			const string& g = get_group_from_timestamp(any_cast<string>(selection[row][0]), aggr_interval, midnight);
			if (group.empty())
			{
				group = g;
				r_lower = r_upper = row;
			}
			else
			{
				if (group == g)
					r_upper++;
				else
				{
					result.resize(extents[++rows][2]);
					const Result& r = T()(selection, r_lower, r_upper);
					result[rows - 1][0] = (r.first.empty()) ? group : r.first;
					result[rows - 1][1] = r.second;
					group = g;
					r_lower = r_upper = row;
				}
			}
		}
		if (!group.empty())
		{
			result.resize(extents[++rows][2]);
			const Result& r = T()(selection, r_lower, r_upper);
			result[rows - 1][0] = (r.first.empty()) ? group : r.first;
			result[rows - 1][1] = r.second;
		}
	}
	CATCH_RETHROW("")
}

	static string get_group_from_timestamp (const string& timestamp, const string& aggr_interval, const time_duration& from_time, const time_duration& to_time)
	{
		try
		{
			ptime t = lexical_cast<ptime>(timestamp);

			// if we group by day we make a special check here:
			// the given timestamp must be in the range between
			// from_time and to_time, the problem is that
			// to_time can be after midnight, so we must really
			// compare timestamps, not just times
			//
			// in all cases where to_time is less than
			// from_time and the given timestamp is between 0:00
			// and to_time of the next day we return a value
			// based on the date of the previous day
			size_t len = get_group_length(aggr_interval);
			if (len == 10) // day, days
			{
				date d = t.date();
				ptime begin(d, from_time);
				ptime end(d, to_time);
				if (from_time >= to_time)
					end += days(1);
				if (t < begin || t >= end)
					return string(); // out of range
				t = begin;
			}

			return string(lexical_cast<string>(t), 0, len);
		}
		CATCH_RETHROW("")
	}

template<class T> static void aggregate (const Selection& selection, Selection& result, const string& aggr_interval, const time_duration& from_time, const time_duration& to_time)
{
	try
	{
		const size_t* pDimensions = selection.shape();
		size_t rows = 0;
		string group;
		size_t r_lower = 0, r_upper = 0;
		for (size_t row = 0; row < pDimensions[0]; row++)
		{
			const string& g = get_group_from_timestamp(any_cast<string>(selection[row][0]), aggr_interval, from_time, to_time);
			if (g.empty())
				continue; // out of range, just ignore
			if (group.empty())
			{
				group = g;
				r_lower = r_upper = row;
			}
			else
			{
				if (group == g)
					r_upper++;
				else
				{
					result.resize(extents[++rows][2]);
					const Result& r = T()(selection, r_lower, r_upper);
					result[rows - 1][0] = (r.first.empty()) ? group : r.first;
					result[rows - 1][1] = r.second;
					group = g;
					r_lower = r_upper = row;
				}
			}
		}
		if (!group.empty())
		{
			result.resize(extents[++rows][2]);
			const Result& r = T()(selection, r_lower, r_upper);
			result[rows - 1][0] = (r.first.empty()) ? group : r.first;
			result[rows - 1][1] = r.second;
		}
	}
	CATCH_RETHROW("")
}

	class fo_avg_d;

template<class T> static void over_3months (const Selection& selection, Selection& result)
{
	try
	{
		Selection avgs;
		aggregate<fo_avg_d>(selection, avgs, "day", time_duration(0, 0, 0, 0));

		const size_t* pDimensions = avgs.shape();
		if (pDimensions[0] == 0)
			return; // no data

		size_t rows = 0, r_lower = 0, r_upper = 0;
		while (true)
		{
			date first = lexical_cast<date>(any_cast<string>(avgs[r_lower][0]));
			while (r_upper < pDimensions[0] && lexical_cast<date>(any_cast<string>(avgs[r_upper][0])) < first + days(HbA1c_interval))
				r_upper++;
			if (r_upper == pDimensions[0])
				break; // end of data

			result.resize(extents[++rows][2]);
			const Result& r = T()(avgs, r_lower, r_upper);
			result[rows - 1][0] = r.first; // the last day!
			result[rows - 1][1] = r.second;

			r_lower++;
		}
	}
	CATCH_RETHROW("")
}

//=== Implementations ======================================================
//
// all concrete pdx function implementations are located here
//
// note: these function implementations define C++ classes, that's why they
// have a unique name differing from the pdx function's none-unique name,
// respect the following naming scheme:
//
//   return-type _ function-name [_ param-type1 _ param-type2 _ ...]
//
// we can define multiple function implementations having the same name,
// in this case we must ensure to have significant different parameters

//--- functions defining time durations ------------------------------------
DEFINE_FIMPL_0(time_year, typeid(time_duration), "year")
{
	try
	{
		return any(time_duration(365 * 24, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_year_int, typeid(time_duration), "year", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(365 * 24 * n, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_years_int, typeid(time_duration), "years", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(365 * 24 * n, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_0(time_month, typeid(time_duration), "month")
{
	try
	{
		return any(time_duration(30 * 24, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_month_int, typeid(time_duration), "month", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(30 * 24 * n, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_months_int, typeid(time_duration), "months", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(30 * 24 * n, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_0(time_week, typeid(time_duration), "week")
{
	try
	{
		return any(time_duration(7 * 24, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_week_int, typeid(time_duration), "week", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(7 * 24 * n, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_weeks_int, typeid(time_duration), "weeks", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(7 * 24 * n, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_0(time_day, typeid(time_duration), "day")
{
	try
	{
		return any(time_duration(24, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_day_int, typeid(time_duration), "day", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(24 * n, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_days_int, typeid(time_duration), "days", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(24 * n, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_0(time_hour, typeid(time_duration), "hour")
{
	try
	{
		return any(time_duration(1, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_hour_int, typeid(time_duration), "hour", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(n, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_hours_int, typeid(time_duration), "hours", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(n, 0, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_0(time_minute, typeid(time_duration), "minute")
{
	try
	{
		return any(time_duration(0, 1, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_minute_int, typeid(time_duration), "minute", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(0, n, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_minutes_int, typeid(time_duration), "minutes", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(0, n, 0, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_0(time_second, typeid(time_duration), "second")
{
	try
	{
		return any(time_duration(0, 0, 1, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_second_int, typeid(time_duration), "second", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(0, 0, n, 0));
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_1(time_seconds_int, typeid(time_duration), "seconds", typeid(int))
{
	try
	{
		int n = any_cast<int>(params[0]->Evaluate(context));
		return any(time_duration(0, 0, n, 0));
	}
	CATCH_RETHROW("")
}

//--- timestamp functions --------------------------------------------------
DEFINE_FIMPL_0(timestamp_now, typeid(ptime), "now")
{
	try
	{
		return any(context.m_config.GetNow());
	}
	CATCH_RETHROW("")
}

	static time_duration get_midnight (const FContext& context)
	{
		try
		{
			return (context.m_config.HasOption("midnight"))
				? lexical_cast<time_duration>(context.m_config.GetStringOption("midnight"))
				: time_duration(0, 0, 0, 0);
		}
		CATCH_RETHROW("")
	}

DEFINE_FIMPL_0(timestamp_midnight, typeid(ptime), "midnight")
{
	try
	{
		ptime t = context.m_config.GetNow() + days(1);
		t -= t.time_of_day();
		t += get_midnight(context);
		return any(t);
	}
	CATCH_RETHROW("")	
}

DEFINE_FIMPL_2(timestamp_plus_timestamp_time, typeid(ptime), "+", typeid(ptime), typeid(time_duration))
{
	try
	{
		ptime t = any_cast<ptime>(params[0]->Evaluate(context));
		time_duration d = any_cast<time_duration>(params[1]->Evaluate(context));

		return any(t + d);
	}
	CATCH_RETHROW("")
}

DEFINE_FIMPL_2(timestamp_minus_timestamp_time, typeid(ptime), "-", typeid(ptime), typeid(time_duration))
{
	try
	{
		ptime t = any_cast<ptime>(params[0]->Evaluate(context));
		time_duration d = any_cast<time_duration>(params[1]->Evaluate(context));

		return any(t - d);
	}
	CATCH_RETHROW("")
}

//--- select ---------------------------------------------------------------

// (select "*")
DEFINE_FIMPL_1(selection_select_string, typeid(shared_ptr<Selection>), "select", typeid(string))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		context.m_database.Select(result, collection, not_a_date_time, not_a_date_time);
		return any(result);
	}
	CATCH_RETHROW("")
}

// (select "*" begin)
DEFINE_FIMPL_2(selection_select_string_timestamp, typeid(shared_ptr<Selection>), "select", typeid(string), typeid(ptime))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const ptime& begin = any_cast<ptime>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		context.m_database.Select(result, collection, begin, not_a_date_time);
		return any(result);
	}
	CATCH_RETHROW("")
}

// (select "*" begin end)
DEFINE_FIMPL_3(selection_select_string_timestamp_timestamp, typeid(shared_ptr<Selection>), "select", typeid(string), typeid(ptime), typeid(ptime))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const ptime& begin = any_cast<ptime>(params[1]->Evaluate(context));
		const ptime& end = any_cast<ptime>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		context.m_database.Select(result, collection, begin, end);
		return any(result);
	}
	CATCH_RETHROW("")
}

// (select "*" duration)
DEFINE_FIMPL_2(selection_select_string_time, typeid(shared_ptr<Selection>), "select", typeid(string), typeid(time_duration))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const time_duration& duration = any_cast<time_duration>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		ptime end = context.m_config.GetNow();
		context.m_database.Select(result, collection, end - duration, end);
		return any(result);
	}
	CATCH_RETHROW("")
}

// (select "*" duration end)
DEFINE_FIMPL_3(selection_select_string_time_timestamp, typeid(shared_ptr<Selection>), "select", typeid(string), typeid(time_duration), typeid(ptime))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const time_duration& duration = any_cast<time_duration>(params[1]->Evaluate(context));
		const ptime& end = any_cast<ptime>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		context.m_database.Select(result, collection, end - duration, end);
		return any(result);
	}
	CATCH_RETHROW("")
}

//--- merge/fold -----------------------------------------------------------

	enum Operation {op_avg, op_min, op_max, op_sum, op_first, op_last};

	static Operation get_operation (const string& name, const kstring& keyword)
	{
		if (keyword == "avg")
			return op_avg;

		if (keyword == "min")
			return op_min;

		if (keyword == "max")
			return op_max;

		if (keyword == "sum")
			return op_sum;

		if (keyword == "first")
			return op_first;

		if (keyword == "last")
			return op_last;

		THROW(format("%s: illegal operation: %s") % name % keyword);
	}

	typedef vector<shared_ptr<Selection> > Selections;

	static void get_selections_and_type (const string& name, FContext& context, const FImpl::Params& params, Selections& selections, char& type)
	{
		// here we find out the selections to be merged and the
		// type of the resulting selection, this type is equal to
		// the type of all selections to be merged except the case
		// of empty selections - we allow empty selections to be
		// merged with any other selection

		try
		{
			foreach (FTree* pParam, params)
			{
				const any& param = pParam->Evaluate(context);
				if (param.type() == typeid(kstring))
					continue;
				if (param.type() == typeid(shared_ptr<Selection>))
				{
					shared_ptr<Selection> s = any_cast<shared_ptr<Selection> >(param);
					char t = value_type(*s);
					switch (type)
					{
						case '\0':
						case 'e':
						{
							type = t;
							break;
						}
						default:
						{
							if (t != 'e' && type != t)
								THROW(format("%s: selections have different types") % name);
							break;
						}
					}
					selections.push_back(s);
				}
				else
					THROW(format("%s: illegal parameter value type") % name);
			}
		}
		CATCH_RETHROW("")
	}

	static void merge_value (const string& name, Selection& result, size_t r_row, Selection& source, size_t s_row, char type, Operation op)
	{
		try
		{
			if (result[r_row][1].empty())
				result[r_row][1] = source[s_row][1];
			else
			{
				switch (type)
				{
					case 'n':
					{
						double v1 = any_cast<double>(result[r_row][1]);
						double v2 = any_cast<double>(source[s_row][1]);
						switch (op)
						{
							case op_avg:	result[r_row][1] = (v1 + v2) / 2.0; break;
							case op_min:	result[r_row][1] = std::min(v1, v2); break;
							case op_max:	result[r_row][1] = std::max(v1, v2); break;
							case op_sum:	result[r_row][1] = v1 + v2; break;
							case op_first:	result[r_row][1] = v1; break;
							case op_last:	result[r_row][1] = v2; break;
							default:	THROW(format("%s: illegal operation on numeric selection") % name);
						}
						break;
					}
					case 'r':
					{
						Ratio v1 = any_cast<Ratio>(result[r_row][1]);
						Ratio v2 = any_cast<Ratio>(source[s_row][1]);
						switch (op)
						{
							case op_first:	result[r_row][1] = v1; break;
							case op_last:	result[r_row][1] = v2; break;
							default:	THROW(format("%s: illegal operation on ratio selection") % name);
						}
						break;
					}
					case 's':
					{
						string v1 = any_cast<string>(result[r_row][1]);
						string v2 = any_cast<string>(source[s_row][1]);
						switch (op)
						{
							case op_sum:	result[r_row][1] = v1 + "; " + v2; break;
							case op_first:	result[r_row][1] = v1; break;
							case op_last:	result[r_row][1] = v2; break;
							default:	THROW(format("%s: illegal operation on text selection") % name);
						}
						break;
					}
					default:
						break;
				}
			}
		}
		CATCH_RETHROW("")
	}

// (selection merge keyword ...)
DEFINE_FIMPL_2(selection_merge_keyword_ellipse, typeid(shared_ptr<Selection>), "merge", typeid(kstring), typeid(ellipse))
{
	try
	{
		// get the specified operation to be used in the case of
		// equal timestamps, the resulting selection can still not
		// have two values on one timestamp, so we must somehow
		// compute a single value
		Operation op = get_operation(m_name, any_cast<kstring>(params[0]->Evaluate(context)));

		// check the parameters in the ellipse, all these parameters
		// must be selections
		Selections selections;
		char type = '\0';
		get_selections_and_type(m_name, context, params, selections, type);

		// set up the result selection, add a row for each timestamp
		shared_ptr<Selection> result(new Selection());
		{
			typedef set<string> Timestamps;
			Timestamps timestamps;

			foreach (const shared_ptr<Selection>& source, selections)
			{
				const size_t* pDimensions = (*source).shape();
				for (size_t s_row = 0; s_row < pDimensions[0]; s_row++)
				{
					timestamps.insert(any_cast<string>((*source)[s_row][0]));
				}
			}

			(*result).resize(extents[timestamps.size()][2]);
			size_t r_row = 0;
			foreach (const string& t, timestamps)
			{
				(*result)[r_row++][0] = t;
			}
		}

		// fill the data values of the result selection
		foreach (const shared_ptr<Selection>& source, selections)
		{
			size_t r_row = 0;
			const size_t* pDimensions = (*source).shape();
			for (size_t s_row = 0; s_row < pDimensions[0]; s_row++)
			{
				const string& t = any_cast<string>((*source)[s_row][0]);
				while (any_cast<string>((*result)[r_row][0]) != t)
					r_row++;
				merge_value(m_name, *result, r_row, *source, s_row, type, op);
			}
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

	enum FoldInterval {fi_year, fi_month, fi_day, fi_hour, fi_minute};

	static Diagram::FoldInterval get_fold_interval (const string& name, const kstring& keyword)
	{
		if (keyword == "year")
			return Diagram::year;

		if (keyword == "month")
			return Diagram::month;

		if (keyword == "day")
			return Diagram::day;

		if (keyword == "hour")
			return Diagram::hour;

		if (keyword == "minute")
			return Diagram::minute;

		THROW(format("%s: illegal folding interval: %s") % name % keyword);
	}

	static string fold (const string& timestamp, Diagram::FoldInterval fi)
	{
		string result(timestamp);
		switch (fi)
		{
			case Diagram::year:	result.replace(0,  4, "9999"); break;
			case Diagram::month:	result.replace(0,  7, "9999-01"); break;
			case Diagram::day:	result.replace(0, 10, "9999-01-01"); break;
			case Diagram::hour:	result.replace(0, 13, "9999-01-01 00"); break;
			case Diagram::minute:	result.replace(0, 16, "9999-01-01 00:00"); break;
		}
		return result;
	}

// (selection fold keyword keyword selection)
DEFINE_FIMPL_3(selection_fold_keyword_keyword_selection, typeid(shared_ptr<Selection>), "fold", typeid(kstring), typeid(kstring), typeid(shared_ptr<Selection>))
{
	try
	{
		Diagram::FoldInterval fi = get_fold_interval(m_name, any_cast<kstring>(params[0]->Evaluate(context)));

		// get the specified operation to be used in the case of
		// equal timestamps, the resulting selection can still not
		// have two values on one timestamp, so we must somehow
		// compute a single value
		Operation op = get_operation(m_name, any_cast<kstring>(params[1]->Evaluate(context)));

		// check the parameters in the ellipse, all these parameters
		// must be selections
		shared_ptr<Selection> source = any_cast<shared_ptr<Selection> >(params[2]->Evaluate(context));
		const size_t* pDimensions = (*source).shape();
		char type = value_type(*source);

		// set up the result selection, add a row for each timestamp
		shared_ptr<Selection> result(new Selection());
		{
			typedef set<string> Timestamps;
			Timestamps timestamps;

			for (size_t s_row = 0; s_row < pDimensions[0]; s_row++)
			{
				timestamps.insert(fold(any_cast<string>((*source)[s_row][0]), fi));
			}

			(*result).resize(extents[timestamps.size()][2]);
			size_t r_row = 0;
			foreach (const string& t, timestamps)
			{
				(*result)[r_row++][0] = t;
			}
		}

		// fill the data values of the result selection
		for (size_t s_row = 0; s_row < pDimensions[0]; s_row++)
		{
			const string& t = fold(any_cast<string>((*source)[s_row][0]), fi);
			size_t r_row = 0;
			while (any_cast<string>((*result)[r_row][0]) != t)
				r_row++;
			merge_value(m_name, *result, r_row, *source, s_row, type, op);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- avg ------------------------------------------------------------------

	class fo_avg_d {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				double value = 0.0;
				for (size_t row = r_lower; row <= r_upper; row++)
				{
					value += any_cast<double>(selection[row][1]);
				}
				value /= double(r_upper - r_lower + 1);
				return Result(any(), value);
			}
			CATCH_RETHROW("")
		}
	};

	class fo_avg_r {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				Ratio value(0.0, 0.0);
				for (size_t row = r_lower; row <= r_upper; row++)
				{
					const Ratio& R = any_cast<Ratio>(selection[row][1]);
					value.m_numerator += R.m_numerator;
					value.m_denominator += R.m_denominator;
				}
				value.m_numerator /= double(r_upper - r_lower + 1);
				value.m_denominator /= double(r_upper - r_lower + 1);
				return Result(any(), value);
			}
			CATCH_RETHROW("")
		}
	};

// (avg selection)
DEFINE_FIMPL_1(selection_avg_selection, typeid(shared_ptr<Selection>), "avg", typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_all<fo_avg_d>(*selection, *result); break;
			case 'r':	over_all<fo_avg_r>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (avg selection from_time to_time)
DEFINE_FIMPL_3(selection_avg_selection_time_time, typeid(shared_ptr<Selection>), "avg", typeid(shared_ptr<Selection>), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[1]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_all<fo_avg_d>(*selection, *result, from_time, to_time); break;
			case 'r':	over_all<fo_avg_r>(*selection, *result, from_time, to_time); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (avg selection aggr_interval)
DEFINE_FIMPL_2(selection_avg_selection_keyword, typeid(shared_ptr<Selection>), "avg", typeid(shared_ptr<Selection>), typeid(kstring))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));

		const time_duration& midnight = get_midnight(context);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	aggregate<fo_avg_d>(*selection, *result, keyword, midnight); break;
			case 'r':	aggregate<fo_avg_r>(*selection, *result, keyword, midnight); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (avg selection aggr_interval from_time to_time)
DEFINE_FIMPL_4(selection_avg_selection_keyword_time_time, typeid(shared_ptr<Selection>), "avg", typeid(shared_ptr<Selection>), typeid(kstring), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[2]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[3]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	aggregate<fo_avg_d>(*selection, *result, keyword, from_time, to_time); break;
			case 'r':	aggregate<fo_avg_r>(*selection, *result, keyword, from_time, to_time); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (avg selection n_before n_after)
DEFINE_FIMPL_3(selection_avg_selection_int_int, typeid(shared_ptr<Selection>), "avg", typeid(shared_ptr<Selection>), typeid(int), typeid(int))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		int n_before = any_cast<int>(params[1]->Evaluate(context));
		int n_after = any_cast<int>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		result->resize(extents[pDimensions[0]][pDimensions[1]]); // same size

		for (size_t row = 0; row < pDimensions[0]; row++)
		{
			size_t r_lower = (size_t)max(0, int(row - n_before));
			size_t r_upper = min(pDimensions[0] - 1, row + n_after);
			(*result)[row][0] = (*selection)[row][0];
			switch (value_type(*selection))
			{
				case 'd':	(*result)[row][1] = fo_avg_d()(*selection, r_lower, r_upper).second; break;
				case 'r':	(*result)[row][1] = fo_avg_r()(*selection, r_lower, r_upper).second; break;
				default:	THROW(format("%s: illegal value type in selection") % m_name);
			}
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- sdv ------------------------------------------------------------------

	class fo_sdv_d {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				double sum = 0.0;
				double avg = 0.0;
				for (size_t row = r_lower; row <= r_upper; row++)
				{
					double d = any_cast<double>(selection[row][1]);
					sum += d * d;
					avg += d;
				}
				avg /= double(r_upper - r_lower + 1);
				return Result(any(), sqrt(sum / double(r_upper - r_lower + 1) - avg * avg));
			}
			CATCH_RETHROW("")
		}
	};

	class fo_sdv_r {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				Ratio sum(0.0, 0.0);
				Ratio avg(0.0, 0.0);
				for (size_t row = r_lower; row <= r_upper; row++)
				{
					const Ratio& R = any_cast<Ratio>(selection[row][1]);
					sum.m_numerator += R.m_numerator * R.m_numerator;
					sum.m_denominator += R.m_denominator * R.m_denominator;
					avg.m_numerator += R.m_numerator;
					avg.m_denominator += R.m_denominator;
				}
				avg.m_numerator /= double(r_upper - r_lower + 1);
				avg.m_denominator /= double(r_upper - r_lower + 1);
				return Result(
					any(),
					Ratio(
						sqrt(sum.m_numerator / double(r_upper - r_lower + 1) - avg.m_numerator * avg.m_numerator),
						sqrt(sum.m_denominator / double(r_upper - r_lower + 1) - avg.m_denominator * avg.m_denominator)
					)
				);
			}
			CATCH_RETHROW("")
		}
	};

// (sdv selection)
DEFINE_FIMPL_1(selection_sdv_selection, typeid(shared_ptr<Selection>), "sdv", typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_all<fo_sdv_d>(*selection, *result); break;
			case 'r':	over_all<fo_sdv_r>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (sdv selection from_time to_time)
DEFINE_FIMPL_3(selection_sdv_selection_time_time, typeid(shared_ptr<Selection>), "sdv", typeid(shared_ptr<Selection>), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[1]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_all<fo_sdv_d>(*selection, *result, from_time, to_time); break;
			case 'r':	over_all<fo_sdv_r>(*selection, *result, from_time, to_time); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (sdv selection aggr_interval)
DEFINE_FIMPL_2(selection_sdv_selection_keyword, typeid(shared_ptr<Selection>), "sdv", typeid(shared_ptr<Selection>), typeid(kstring))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));

		const time_duration& midnight = get_midnight(context);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	aggregate<fo_sdv_d>(*selection, *result, keyword, midnight); break;
			case 'r':	aggregate<fo_sdv_r>(*selection, *result, keyword, midnight); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (sdv selection aggr_interval from_time to_time)
DEFINE_FIMPL_4(selection_sdv_selection_keyword_time_time, typeid(shared_ptr<Selection>), "sdv", typeid(shared_ptr<Selection>), typeid(kstring), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[2]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[3]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	aggregate<fo_sdv_d>(*selection, *result, keyword, from_time, to_time); break;
			case 'r':	aggregate<fo_sdv_r>(*selection, *result, keyword, from_time, to_time); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (sdv selection n_before n_after)
DEFINE_FIMPL_3(selection_sdv_selection_int_int, typeid(shared_ptr<Selection>), "sdv", typeid(shared_ptr<Selection>), typeid(int), typeid(int))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		int n_before = any_cast<int>(params[1]->Evaluate(context));
		int n_after = any_cast<int>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		result->resize(extents[pDimensions[0]][pDimensions[1]]); // same size

		for (size_t row = 0; row < pDimensions[0]; row++)
		{
			size_t r_lower = (size_t)max(0, int(row - n_before));
			size_t r_upper = min(pDimensions[0] - 1, row + n_after);
			(*result)[row][0] = (*selection)[row][0];
			switch (value_type(*selection))
			{
				case 'd':	(*result)[row][1] = fo_sdv_d()(*selection, r_lower, r_upper).second; break;
				case 'r':	(*result)[row][1] = fo_sdv_r()(*selection, r_lower, r_upper).second; break;
				default:	THROW(format("%s: illegal value type in selection") % m_name);
			}
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- min ------------------------------------------------------------------

	class fo_min_d {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				string timestamp;
				double value = 0.0;
				for (size_t row = r_lower; row <= r_upper; row++)
				{
					double d = any_cast<double>(selection[row][1]);
					if (row == 0 || value > d)
					{
						timestamp = any_cast<string>(selection[row][0]);
						value = d;
					}
				}
				return Result(timestamp, value);
			}
			CATCH_RETHROW("")
		}
	};

// (min selection)
DEFINE_FIMPL_1(selection_min_selection, typeid(shared_ptr<Selection>), "min", typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_all<fo_min_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (min selection from_time to_time)
DEFINE_FIMPL_3(selection_min_selection_time_time, typeid(shared_ptr<Selection>), "min", typeid(shared_ptr<Selection>), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[1]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_all<fo_min_d>(*selection, *result, from_time, to_time); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (min selection aggr_interval)
DEFINE_FIMPL_2(selection_min_selection_keyword, typeid(shared_ptr<Selection>), "min", typeid(shared_ptr<Selection>), typeid(kstring))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));

		const time_duration& midnight = get_midnight(context);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	aggregate<fo_min_d>(*selection, *result, keyword, midnight); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (min selection aggr_interval from_time to_time)
DEFINE_FIMPL_4(selection_min_selection_keyword_time_time, typeid(shared_ptr<Selection>), "min", typeid(shared_ptr<Selection>), typeid(kstring), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[2]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[3]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	aggregate<fo_min_d>(*selection, *result, keyword, from_time, to_time); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- max ------------------------------------------------------------------

	class fo_max_d {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				string timestamp;
				double value = 0.0;
				for (size_t row = r_lower; row <= r_upper; row++)
				{
					double d = any_cast<double>(selection[row][1]);
					if (row == 0 || value < d)
					{
						timestamp = any_cast<string>(selection[row][0]);
						value = d;
					}
				}
				return Result(timestamp, value);
			}
			CATCH_RETHROW("")
		}
	};

// (max selection)
DEFINE_FIMPL_1(selection_max_selection, typeid(shared_ptr<Selection>), "max", typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_all<fo_max_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (max selection from_time to_time)
DEFINE_FIMPL_3(selection_max_selection_time_time, typeid(shared_ptr<Selection>), "max", typeid(shared_ptr<Selection>), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[1]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_all<fo_max_d>(*selection, *result, from_time, to_time); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (max selection aggr_interval)
DEFINE_FIMPL_2(selection_max_selection_keyword, typeid(shared_ptr<Selection>), "max", typeid(shared_ptr<Selection>), typeid(kstring))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));

		const time_duration& midnight = get_midnight(context);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	aggregate<fo_max_d>(*selection, *result, keyword, midnight); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (max selection aggr_interval from_time to_time)
DEFINE_FIMPL_4(selection_max_selection_keyword_time_time, typeid(shared_ptr<Selection>), "max", typeid(shared_ptr<Selection>), typeid(kstring), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[2]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[3]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	aggregate<fo_max_d>(*selection, *result, keyword, from_time, to_time); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- count ----------------------------------------------------------------

	class fo_count {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				return Result(any(), (double)(r_upper - r_lower + 1));
			}
			CATCH_RETHROW("")
		}
	};

// (count selection)
DEFINE_FIMPL_1(selection_count_selection, typeid(shared_ptr<Selection>), "count", typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		over_all<fo_count>(*selection, *result);
		return any(result);
	}
	CATCH_RETHROW("")
}

// (count selection from_time to_time)
DEFINE_FIMPL_3(selection_count_selection_time_time, typeid(shared_ptr<Selection>), "count", typeid(shared_ptr<Selection>), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[1]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		over_all<fo_count>(*selection, *result, from_time, to_time);
		return any(result);
	}
	CATCH_RETHROW("")
}

// (count selection aggr_interval)
DEFINE_FIMPL_2(selection_count_selection_keyword, typeid(shared_ptr<Selection>), "count", typeid(shared_ptr<Selection>), typeid(kstring))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));

		const time_duration& midnight = get_midnight(context);

		shared_ptr<Selection> result(new Selection());
		aggregate<fo_count>(*selection, *result, keyword, midnight);
		return any(result);
	}
	CATCH_RETHROW("")
}

// (count selection aggr_interval from_time to_time)
DEFINE_FIMPL_4(selection_count_selection_keyword_time_time, typeid(shared_ptr<Selection>), "count", typeid(shared_ptr<Selection>), typeid(kstring), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[2]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[3]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		aggregate<fo_count>(*selection, *result, keyword, from_time, to_time);
		return any(result);
	}
	CATCH_RETHROW("")
}

//--- sum ------------------------------------------------------------------

	class fo_sum_d {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				double sum = 0.0;
				for (size_t row = r_lower; row <= r_upper; row ++)
				{
					sum += any_cast<double>(selection[row][1]);
				}
				return Result(any(), sum);
			}
			CATCH_RETHROW("")
		}
	};

	class fo_sum_s {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				string s;
				for (size_t row = r_lower; row <= r_upper; row ++)
				{
					if (!s.empty())
						s += "; ";
					s += any_cast<string>(selection[row][1]);
				}
				return Result(any(), s);
			}
			CATCH_RETHROW("")
		}
	};

// (sum selection)
DEFINE_FIMPL_1(selection_sum_selection, typeid(shared_ptr<Selection>), "sum", typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_all<fo_sum_d>(*selection, *result); break;
			case 's':	over_all<fo_sum_s>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (sum selection from_time to_time)
DEFINE_FIMPL_3(selection_sum_selection_time_time, typeid(shared_ptr<Selection>), "sum", typeid(shared_ptr<Selection>), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[1]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_all<fo_sum_d>(*selection, *result, from_time, to_time); break;
			case 's':	over_all<fo_sum_s>(*selection, *result, from_time, to_time); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (sum selection aggr_interval)
DEFINE_FIMPL_2(selection_sum_selection_keyword, typeid(shared_ptr<Selection>), "sum", typeid(shared_ptr<Selection>), typeid(kstring))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));

		const time_duration& midnight = get_midnight(context);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	aggregate<fo_sum_d>(*selection, *result, keyword, midnight); break;
			case 's':	aggregate<fo_sum_s>(*selection, *result, keyword, midnight); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (sum selection aggr_interval from_time to_time)
DEFINE_FIMPL_4(selection_sum_selection_keyword_time_time, typeid(shared_ptr<Selection>), "sum", typeid(shared_ptr<Selection>), typeid(kstring), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[2]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[3]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	aggregate<fo_sum_d>(*selection, *result, keyword, from_time, to_time); break;
			case 's':	aggregate<fo_sum_s>(*selection, *result, keyword, from_time, to_time); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- first ----------------------------------------------------------------

	class fo_first {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				return Result(selection[r_lower][0], selection[r_lower][1]);
			}
			CATCH_RETHROW("")
		}
	};

// (first selection)
DEFINE_FIMPL_1(selection_first_selection, typeid(shared_ptr<Selection>), "first", typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		over_all<fo_first>(*selection, *result);

		return any(result);
	}
	CATCH_RETHROW("")
}

// (first selection from_time to_time)
DEFINE_FIMPL_3(selection_first_selection_time_time, typeid(shared_ptr<Selection>), "first", typeid(shared_ptr<Selection>), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[1]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		over_all<fo_first>(*selection, *result, from_time, to_time);

		return any(result);
	}
	CATCH_RETHROW("")
}

// (first selection aggr_interval)
DEFINE_FIMPL_2(selection_first_selection_keyword, typeid(shared_ptr<Selection>), "first", typeid(shared_ptr<Selection>), typeid(kstring))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));

		const time_duration& midnight = get_midnight(context);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		aggregate<fo_first>(*selection, *result, keyword, midnight);

		return any(result);
	}
	CATCH_RETHROW("")
}

// (first selection aggr_interval from_time to_time)
DEFINE_FIMPL_4(selection_first_selection_keyword_time_time, typeid(shared_ptr<Selection>), "first", typeid(shared_ptr<Selection>), typeid(kstring), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[2]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[3]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		aggregate<fo_first>(*selection, *result, keyword, from_time, to_time);

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- last -----------------------------------------------------------------

	class fo_last {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				return Result(selection[r_upper][0], selection[r_upper][1]);
			}
			CATCH_RETHROW("")
		}
	};

// (last selection)
DEFINE_FIMPL_1(selection_last_selection, typeid(shared_ptr<Selection>), "last", typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		over_all<fo_last>(*selection, *result);

		return any(result);
	}
	CATCH_RETHROW("")
}

// (last selection from_time to_time)
DEFINE_FIMPL_3(selection_last_selection_time_time, typeid(shared_ptr<Selection>), "last", typeid(shared_ptr<Selection>), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[1]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[2]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		over_all<fo_last>(*selection, *result, from_time, to_time);

		return any(result);
	}
	CATCH_RETHROW("")
}

// (last selection aggr_interval)
DEFINE_FIMPL_2(selection_last_selection_keyword, typeid(shared_ptr<Selection>), "last", typeid(shared_ptr<Selection>), typeid(kstring))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));

		const time_duration& midnight = get_midnight(context);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		aggregate<fo_last>(*selection, *result, keyword, midnight);

		return any(result);
	}
	CATCH_RETHROW("")
}

// (last selection aggr_interval from_time to_time)
DEFINE_FIMPL_4(selection_last_selection_keyword_time_time, typeid(shared_ptr<Selection>), "last", typeid(shared_ptr<Selection>), typeid(kstring), typeid(time_duration), typeid(time_duration))
{
	try
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		const kstring& keyword = any_cast<kstring>(params[1]->Evaluate(context));
		const time_duration& from_time = any_cast<time_duration>(params[2]->Evaluate(context));
		const time_duration& to_time = any_cast<time_duration>(params[3]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		aggregate<fo_last>(*selection, *result, keyword, from_time, to_time);

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- unit -----------------------------------------------------------------
// (unit selection)
DEFINE_FIMPL_1(string_unit_string, typeid(string), "unit", typeid(string))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));

		return any(context.m_database.GetCollectionUnit(collection));
	}
	CATCH_RETHROW("")
}

//--- purpose --------------------------------------------------------------
// (purpose selection)
DEFINE_FIMPL_1(string_purpose_string, typeid(string), "purpose", typeid(string))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));

		return any(context.m_database.GetCollectionPurpose(collection));
	}
	CATCH_RETHROW("")
}

//--- HbA1c-----------------------------------------------------------------

	class fo_HbA1c_d {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				// this selection contains averages per day
				// over 90 days

				double long_term_avg = 0.0;
				for (size_t row = r_lower; row <= r_upper; row++)
				{
					long_term_avg += any_cast<double>(selection[row][1]);
				}
				long_term_avg /= double(r_upper - r_lower + 1);

				double value;
				if (long_term_avg > 20.0)
				{
					// mg/dL
					value = (long_term_avg + 86.0) / 33.0;
				}
				else
				{
					// mmol/L
					value = ((long_term_avg * 18.0182) + 86.0) / 33.0;
				}

				return Result(selection[r_upper][0], value);
			}
			CATCH_RETHROW("")
		}
	};

// (HbA1c string)
DEFINE_FIMPL_1(selection_HbA1c_string, typeid(shared_ptr<Selection>), "HbA1c", typeid(string))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));

		shared_ptr<Selection> selection(new Selection());
		context.m_database.Select(selection, collection, not_a_date_time, not_a_date_time);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_3months<fo_HbA1c_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (HbA1c string begin)
DEFINE_FIMPL_2(selection_HbA1c_string_timestamp, typeid(shared_ptr<Selection>), "HbA1c", typeid(string), typeid(ptime))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const ptime& begin = any_cast<ptime>(params[1]->Evaluate(context));

		shared_ptr<Selection> selection(new Selection());
		context.m_database.Select(selection, collection, begin - days(HbA1c_interval), not_a_date_time);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_3months<fo_HbA1c_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (HbA1c string begin end)
DEFINE_FIMPL_3(selection_HbA1c_string_timestamp_timestamp, typeid(shared_ptr<Selection>), "HbA1c", typeid(string), typeid(ptime), typeid(ptime))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const ptime& begin = any_cast<ptime>(params[1]->Evaluate(context));
		const ptime& end = any_cast<ptime>(params[2]->Evaluate(context));

		shared_ptr<Selection> selection(new Selection());
		context.m_database.Select(selection, collection, begin - days(HbA1c_interval), end);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_3months<fo_HbA1c_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (HbA1c string duration)
DEFINE_FIMPL_2(selection_HbA1c_string_time, typeid(shared_ptr<Selection>), "HbA1c", typeid(string), typeid(time_duration))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const time_duration& duration = any_cast<time_duration>(params[1]->Evaluate(context));

		ptime end = context.m_config.GetNow();
		shared_ptr<Selection> selection(new Selection());
		context.m_database.Select(selection, collection, end - duration - days(HbA1c_interval), end);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_3months<fo_HbA1c_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (HbA1c string duration end)
DEFINE_FIMPL_3(selection_HbA1c_string_time_timestamp, typeid(shared_ptr<Selection>), "HbA1c", typeid(string), typeid(time_duration), typeid(ptime))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const time_duration& duration = any_cast<time_duration>(params[1]->Evaluate(context));
		const ptime& end = any_cast<ptime>(params[2]->Evaluate(context));

		shared_ptr<Selection> selection(new Selection());
		context.m_database.Select(selection, collection, end - duration - days(HbA1c_interval), end);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_3months<fo_HbA1c_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- HbA1c2 ---------------------------------------------------------------

	class fo_HbA1c2_d {
		public:
		Result operator () (const Selection& selection, size_t r_lower, size_t r_upper) const
		{
			try
			{
				// this selection contains averages per day
				// over 90 days - here in HbA1c2 we give
				// each of them a specific weight - so the
				// oldest value is much less important than
				// the newest value

				double long_term_avg = 0.0;
				int n = 0;
				for (size_t row = r_lower, weight = 1; row <= r_upper; row++, weight++)
				{
					long_term_avg += any_cast<double>(selection[row][1]) * double(weight);
					n += weight;
				}
				long_term_avg /= double(n);

				double value;
				if (long_term_avg > 20.0)
				{
					// mg/dL
					value = (long_term_avg + 86.0) / 33.0;
				}
				else
				{
					// mmol/L
					value = ((long_term_avg * 18.0182) + 86.0) / 33.0;
				}

				return Result(selection[r_upper][0], value);
			}
			CATCH_RETHROW("")
		}
	};

// (HbA1c2 string)
DEFINE_FIMPL_1(selection_HbA1c2_string, typeid(shared_ptr<Selection>), "HbA1c2", typeid(string))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));

		shared_ptr<Selection> selection(new Selection());
		context.m_database.Select(selection, collection, not_a_date_time, not_a_date_time);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_3months<fo_HbA1c2_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (HbA1c2 string begin)
DEFINE_FIMPL_2(selection_HbA1c2_string_timestamp, typeid(shared_ptr<Selection>), "HbA1c2", typeid(string), typeid(ptime))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const ptime& begin = any_cast<ptime>(params[1]->Evaluate(context));

		shared_ptr<Selection> selection(new Selection());
		context.m_database.Select(selection, collection, begin - days(HbA1c_interval), not_a_date_time);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_3months<fo_HbA1c2_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (HbA1c2 string begin end)
DEFINE_FIMPL_3(selection_HbA1c2_string_timestamp_timestamp, typeid(shared_ptr<Selection>), "HbA1c2", typeid(string), typeid(ptime), typeid(ptime))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const ptime& begin = any_cast<ptime>(params[1]->Evaluate(context));
		const ptime& end = any_cast<ptime>(params[2]->Evaluate(context));

		shared_ptr<Selection> selection(new Selection());
		context.m_database.Select(selection, collection, begin - days(HbA1c_interval), end);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_3months<fo_HbA1c2_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (HbA1c2 string duration)
DEFINE_FIMPL_2(selection_HbA1c2_string_time, typeid(shared_ptr<Selection>), "HbA1c2", typeid(string), typeid(time_duration))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const time_duration& duration = any_cast<time_duration>(params[1]->Evaluate(context));

		ptime end = context.m_config.GetNow();
		shared_ptr<Selection> selection(new Selection());
		context.m_database.Select(selection, collection, end - duration - days(HbA1c_interval), end);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_3months<fo_HbA1c2_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (HbA1c2 string duration end)
DEFINE_FIMPL_3(selection_HbA1c2_string_time_timestamp, typeid(shared_ptr<Selection>), "HbA1c2", typeid(string), typeid(time_duration), typeid(ptime))
{
	try
	{
		const string& collection = any_cast<string>(params[0]->Evaluate(context));
		const time_duration& duration = any_cast<time_duration>(params[1]->Evaluate(context));
		const ptime& end = any_cast<ptime>(params[2]->Evaluate(context));

		shared_ptr<Selection> selection(new Selection());
		context.m_database.Select(selection, collection, end - duration - days(HbA1c_interval), end);

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = selection->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*selection))
		{
			case 'd':	over_3months<fo_HbA1c2_d>(*selection, *result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- arithmetics ----------------------------------------------------------

// (+ double double)
DEFINE_FIMPL_2(selection_plus_double_double, typeid(shared_ptr<Selection>), "+", typeid(double), typeid(double))
{
	try
	{
		double v1 = any_cast<double>(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		result->resize(extents[1][2]);
		(*result)[0][1] = v1 + v2;

		return any(result);
	}
	CATCH_RETHROW("")
}

	template <class FO> void selection_x_double (const shared_ptr<Selection>& v1, double v2, shared_ptr<Selection>& result)
	{
		try
		{
			const size_t* pDimensions = v1->shape();
			result->resize(extents[pDimensions[0]][2]);
			for (size_t row = 0; row < pDimensions[0]; row++)
			{
				(*result)[row][0] = (*v1)[row][0];
				(*result)[row][1] = FO()(any_cast<double>((*v1)[row][1]), v2);
			}
		}
		CATCH_RETHROW("")
	};

	struct fo_plus
	{
		double operator () (double v1, double v2) const
		{
			return v1 + v2;
		}
	};

// (+ selection double)
DEFINE_FIMPL_2(selection_plus_selection_double, typeid(shared_ptr<Selection>), "+", typeid(shared_ptr<Selection>), typeid(double))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*v1))
		{
			case 'd':	selection_x_double<fo_plus>(v1, v2, result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

	// this function compares two timestamps even if they have
	// different lengths
	static int compare_timestamps (const string& t1, const string& t2)
	{
		size_t len = min(t1.length(), t2.length());
		string s1(t1, 0, len), s2(t2, 0,len);
		if (s1 < s2)
			return -1;
		if (s1 > s2)
			return 1;
		return 0;
	}

	template <class FO> void selection_x_selection (const shared_ptr<Selection>& v1, const shared_ptr<Selection>& v2, shared_ptr<Selection>& result, const string& name)
	{
		try
		{
			// the first line in v2 must be older than the first
			// line in v1, we need an initial value for the
			//  second operand
			if (compare_timestamps(any_cast<string>((*v1)[0][0]), any_cast<string>((*v2)[0][0])) == -1)
				THROW(format("%s: selection 2 has no second operand older than the first line of selection 1") % name);

			const size_t* pDimensions = v1->shape();
			result->resize(extents[pDimensions[0]][2]);

			for (size_t row1 = 0, row2 = 0; row1 < pDimensions[0]; row1++)
			{
				// find the youngest value in v2 younger
				// than the current value in v1
				while (row2 < v2->shape()[0] && compare_timestamps(any_cast<string>((*v1)[row1][0]), any_cast<string>((*v2)[row2][0])) != -1)
					row2++;
				(*result)[row1][0] = (*v1)[row1][0];
				(*result)[row1][1] = FO()(any_cast<double>((*v1)[row1][1]), any_cast<double>((*v2)[row2 - 1][1]));
			}
		}
		CATCH_RETHROW("")
	}

// (+ selection selection)
DEFINE_FIMPL_2(selection_plus_selection_selection, typeid(shared_ptr<Selection>), "+", typeid(shared_ptr<Selection>), typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		shared_ptr<Selection> v2 = any_cast<shared_ptr<Selection> >(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result
		if (v2->shape()[0] == 0)
			return any(result);

		switch (value_type(*v1))
		{
			case 'd':
			{
				switch (value_type(*v2))
				{
					case 'd':	selection_x_selection<fo_plus>(v1, v2, result, m_name); break;
					default:	THROW(format("%s: illegal value type in selection 2") % m_name);
				}
				break;
			}
			default:	THROW(format("%s: illegal value type in selection 1") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (+ string string ...)
DEFINE_FIMPL_3(string_plus_string_string_ellipse, typeid(string), "+", typeid(string), typeid(string), typeid(ellipse))
{
	try
	{
		string result;
		foreach (FTree* pParam, params)
		{
			const any& param = pParam->Evaluate(context);

			if (param.empty())
				continue;

			if (param.type() == typeid(string))
			{
				result += any_cast<string>(param);
				continue;
			}

			if (param.type() == typeid(estring))
			{
				result += any_cast<estring>(param);
				continue;
			}

			if (param.type() == typeid(fstring))
			{
				result += any_cast<fstring>(param);
				continue;
			}

			if (param.type() == typeid(kstring))
			{
				result += any_cast<kstring>(param);
				continue;
			}

			THROW(format("%s: illegal parameter value type") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (- double double)
DEFINE_FIMPL_2(selection_minus_double_double, typeid(shared_ptr<Selection>), "-", typeid(double), typeid(double))
{
	try
	{
		double v1 = any_cast<double>(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		result->resize(extents[1][2]);
		(*result)[0][1] = v1 - v2;

		return any(result);
	}
	CATCH_RETHROW("")
}

	struct fo_minus
	{
		double operator () (double v1, double v2) const
		{
			return v1 - v2;
		}
	};

// (- selection double)
DEFINE_FIMPL_2(selection_minus_selection_double, typeid(shared_ptr<Selection>), "-", typeid(shared_ptr<Selection>), typeid(double))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*v1))
		{
			case 'd':	selection_x_double<fo_minus>(v1, v2, result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (- selection selection)
DEFINE_FIMPL_2(selection_minus_selection_selection, typeid(shared_ptr<Selection>), "-", typeid(shared_ptr<Selection>), typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		shared_ptr<Selection> v2 = any_cast<shared_ptr<Selection> >(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result
		if (v2->shape()[0] == 0)
			return any(result);

		switch (value_type(*v1))
		{
			case 'd':
			{
				switch (value_type(*v2))
				{
					case 'd':	selection_x_selection<fo_minus>(v1, v2, result, m_name); break;
					default:	THROW(format("%s: illegal value type in selection 2") % m_name);
				}
				break;
			}
			default:	THROW(format("%s: illegal value type in selection 1") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (* double double)
DEFINE_FIMPL_2(selection_mult_double_double, typeid(shared_ptr<Selection>), "*", typeid(double), typeid(double))
{
	try
	{
		double v1 = any_cast<double>(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		result->resize(extents[1][2]);
		(*result)[0][1] = v1 * v2;

		return any(result);
	}
	CATCH_RETHROW("")
}

	struct fo_mult
	{
		double operator () (double v1, double v2) const
		{
			return v1 * v2;
		}
	};

// (* selection double)
DEFINE_FIMPL_2(selection_mult_selection_double, typeid(shared_ptr<Selection>), "*", typeid(shared_ptr<Selection>), typeid(double))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*v1))
		{
			case 'd':	selection_x_double<fo_mult>(v1, v2,result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (* selection selection)
DEFINE_FIMPL_2(selection_mult_selection_selection, typeid(shared_ptr<Selection>), "*", typeid(shared_ptr<Selection>), typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		shared_ptr<Selection> v2 = any_cast<shared_ptr<Selection> >(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result
		if (v2->shape()[0] == 0)
			return any(result);

		switch (value_type(*v1))
		{
			case 'd':
			{
				switch (value_type(*v2))
				{
					case 'd':	selection_x_selection<fo_mult>(v1, v2, result, m_name); break;
					default:	THROW(format("%s: illegal value type in selection 2") % m_name);
				}
				break;
			}
			default:	THROW(format("%s: illegal value type in selection 1") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (/ double double)
DEFINE_FIMPL_2(selection_div_double_double, typeid(shared_ptr<Selection>), "/", typeid(double), typeid(double))
{
	try
	{
		double v1 = any_cast<double>(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		result->resize(extents[1][2]);
		(*result)[0][1] = v1 / v2;

		return any(result);
	}
	CATCH_RETHROW("")
}

	struct fo_div
	{
		double operator () (double v1, double v2) const
		{
			return v1 / v2;
		}
	};

// (/ selection double)
DEFINE_FIMPL_2(selection_div_selection_double, typeid(shared_ptr<Selection>), "/", typeid(shared_ptr<Selection>), typeid(double))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*v1))
		{
			case 'd':	selection_x_double<fo_div>(v1, v2, result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (/ selection selection)
DEFINE_FIMPL_2(selection_div_selection_selection, typeid(shared_ptr<Selection>), "/", typeid(shared_ptr<Selection>), typeid(shared_ptr<Selection>))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		shared_ptr<Selection> v2 = any_cast<shared_ptr<Selection> >(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result
		if (v2->shape()[0] == 0)
			return any(result);

		switch (value_type(*v1))
		{
			case 'd':
			{
				switch (value_type(*v2))
				{
					case 'd':	selection_x_selection<fo_div>(v1, v2, result, m_name); break;
					default:	THROW(format("%s: illegal value type in selection 2") % m_name);
				}
				break;
			}
			default:	THROW(format("%s: illegal value type in selection 1") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- comparison -----------------------------------------------------------

	//conditional version of selection_x_double
	template <class FO> void selection_x_double_cond (const shared_ptr<Selection>& v1, double v2, shared_ptr<Selection>& result)
	{
		try
		{
			const size_t* pDimensions = v1->shape();
			for (size_t source_row = 0, result_row = 0; source_row < pDimensions[0]; source_row++)
			{
				if (FO()(any_cast<double>((*v1)[source_row][1]), v2))
				{
					result->resize(extents[result_row + 1][2]);
					(*result)[result_row][0] = (*v1)[source_row][0];
					(*result)[result_row][1] = (*v1)[source_row][1];
					result_row++;
				}
			}
		}
		CATCH_RETHROW("")
	};

	struct fo_eq
	{
		bool operator () (double v1, double v2) const
		{
			return v1 == v2;
		}
	};

// (< selection double)
DEFINE_FIMPL_2(selection_eq_selection_double, typeid(shared_ptr<Selection>), "==", typeid(shared_ptr<Selection>), typeid(double))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*v1))
		{
			case 'd':	selection_x_double_cond<fo_eq>(v1, v2, result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

	struct fo_lt
	{
		bool operator () (double v1, double v2) const
		{
			return v1 < v2;
		}
	};

// (< selection double)
DEFINE_FIMPL_2(selection_lt_selection_double, typeid(shared_ptr<Selection>), "<", typeid(shared_ptr<Selection>), typeid(double))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*v1))
		{
			case 'd':	selection_x_double_cond<fo_lt>(v1, v2, result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

	struct fo_gt
	{
		bool operator () (double v1, double v2) const
		{
			return v1 > v2;
		}
	};

// (> selection double)
DEFINE_FIMPL_2(selection_gt_selection_double, typeid(shared_ptr<Selection>), ">", typeid(shared_ptr<Selection>), typeid(double))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*v1))
		{
			case 'd':	selection_x_double_cond<fo_gt>(v1, v2, result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

	struct fo_le
	{
		bool operator () (double v1, double v2) const
		{
			return v1 <= v2;
		}
	};

// (<= selection double)
DEFINE_FIMPL_2(selection_le_selection_double, typeid(shared_ptr<Selection>), "<=", typeid(shared_ptr<Selection>), typeid(double))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*v1))
		{
			case 'd':	selection_x_double_cond<fo_le>(v1, v2, result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

	struct fo_ge
	{
		bool operator () (double v1, double v2) const
		{
			return v1 >= v2;
		}
	};

// (>= selection double)
DEFINE_FIMPL_2(selection_ge_selection_double, typeid(shared_ptr<Selection>), ">=", typeid(shared_ptr<Selection>), typeid(double))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*v1))
		{
			case 'd':	selection_x_double_cond<fo_ge>(v1, v2, result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

	struct fo_ne
	{
		bool operator () (double v1, double v2) const
		{
			return v1 != v2;
		}
	};

// (!= selection double)
DEFINE_FIMPL_2(selection_ne_selection_double, typeid(shared_ptr<Selection>), "!=", typeid(shared_ptr<Selection>), typeid(double))
{
	try
	{
		shared_ptr<Selection> v1 = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		double v2 = any_cast<double>(params[1]->Evaluate(context));

		shared_ptr<Selection> result(new Selection());
		const size_t* pDimensions = v1->shape();
		if (pDimensions[0] == 0)
			return any(result); // empty input, empty result

		switch (value_type(*v1))
		{
			case 'd':	selection_x_double_cond<fo_ne>(v1, v2, result); break;
			default:	THROW(format("%s: illegal value type in selection") % m_name);
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

//--- format ---------------------------------------------------------------

	struct ColumnInfo {
		string			m_left_gap;
		bool			m_datetime_column;
		shared_ptr<Selection>	m_selection;
		string			m_format_spec;
	};

	static string format_double (const string& format_spec, double value)
	{
		try
		{
			if (!format_spec.empty())
			{
				try
				{
					string f(format_spec);
					f.insert(0, "%");
					if (f.find('.') == string::npos)
					{
						f += ".0f";
						return (format(f) % value).str();
					}
					else
					{
						if (f.find(".0") == f.length() - 2)
						{
							f[f.length() - 1] = '1';
							f += "f";
							string s((format(f) % value).str());
							if (s.find(".0") != string::npos)
							{
								s.erase(s.length() - 2);
								s += "  ";
							}
							return s;
						}
						else
							return (format(f + 'f') % value).str();
					}
				}
				catch (const boost::io::bad_format_string& )
				{
					THROW(format("invalid format specification for numeric value: <%s>") % format_spec);
				}
			}

			return (format("%g") % value).str();
		}
		CATCH_RETHROW("")
	}

	static string format_string (const string& format_spec, const string& value)
	{
		try
		{
			if (!format_spec.empty())
			{
				try
				{
					size_t length = lexical_cast<size_t>(format_spec);
					if (value.length() > length)
						return string(value, 0, length);
					else
						return value + string(length - value.length(), ' ');
				}
				catch (const boost::bad_lexical_cast& )
				{
					THROW(format("invalid format specification for text value: <%s>") % format_spec);
				}
			}
			else
				return value;
		}
		CATCH_RETHROW("")
	}

// (string format ...)
DEFINE_FIMPL_1(string_format_ellipse, typeid(string), "format", typeid(ellipse))
{
	// we assume that the parameters are a list of strings, selections
	// and format specifications
	//
	// the selections will be joined on the basis of their datetime
	// values, the result is a table with multiple lines and columns,
	// the fields in this table can be empty
	//
	// the terminus "gap" means the room between two columns of the
	// table, we fill these rooms with user defined strings to produce
	// the output lines the user wants
	//
	// we return the result as a single, formatted block of text

	try
	{
		vector<ColumnInfo> columns;

		string accu, empty;
		set<string> timestamps;

		// step 1: evaluate the parameters
		foreach (FTree* pParam, params)
		{
			const any& param = pParam->Evaluate(context);

			if (param.empty())
				continue;

			if (param.type() == typeid(string))
			{
				accu += any_cast<string>(param); // accumulate
				continue;
			}

			if (param.type() == typeid(estring))
			{
				empty = any_cast<estring>(param);
				continue;
			}

			if (param.type() == typeid(fstring))
			{
				const fstring& f = any_cast<const fstring&>(param);
				if (columns.empty())
					THROW(format("%s: format specification before selection <%s>") % m_name % f);
				ColumnInfo& ci = columns.back();
				if (!ci.m_format_spec.empty())
					THROW(format("%s: second format specification after selection <%s>") % m_name % f);
				ci.m_format_spec = f;
				continue;
			}

			if (param.type() == typeid(kstring))
			{
				const fstring& keyword = any_cast<kstring>(param);
				if (keyword == "newline")
					accu += "\n";
				else
				{
					if (keyword == "datetime")
					{
						columns.push_back(ColumnInfo());
						ColumnInfo& ci = columns.back();
						ci.m_left_gap = accu;
						accu.clear();
						ci.m_datetime_column = true;
					}
					else
						THROW(format("%s: illegal keyword %s") % m_name % keyword);
				}
				continue;
			}

			if (param.type() == typeid(shared_ptr<Selection>))
			{
				columns.push_back(ColumnInfo());
				ColumnInfo& ci = columns.back();
				ci.m_left_gap = accu;
				accu.clear();
				ci.m_datetime_column = false;
				ci.m_selection = any_cast<shared_ptr<Selection> >(param);

				// we need all the datetime values of the
				// selections collected in a single sorted
				// list
				const size_t* pDimensions = ci.m_selection->shape();
				for (size_t row = 0; row < pDimensions[0]; row++)
					timestamps.insert(any_cast<string>((*ci.m_selection)[row][0]));

				continue;
			}

			THROW(format("%s: illegal parameter value type") % m_name);
		}
		// accu contains here the last gap on the right side of the
		// last value, this is normally a line end or something like
		// this, we will still need it later

		int datetime_column = -1;
		{
			for (size_t index = 0; datetime_column == -1 && index < columns.size(); index++)
			{
				if (columns[index].m_datetime_column)
					datetime_column = (int)index;
			}
		}

		// step 2: make a full left and right outer join with the
		// given selections, this creates a new multicolumn
		// selection
		Selection join;
		{
			// in the case of no datetime column in the output
			// we put this column at index -1, we surely need a
			// datetime column for building the table and
			// getting their lines sorted
			if (datetime_column == -1)
			{
				join.resize(extents[timestamps.size()][columns.size() + 1]);
				vector<int> indexes;
				indexes.push_back(0);
				indexes.push_back(-1);
				join.reindex(indexes);
			}
			else
				join.resize(extents[timestamps.size()][columns.size()]);

			// fill up the (visible or invisible) datetime
			// column
			size_t j_row = 0;
			foreach (const string& timestamp, timestamps)
			{
				join[j_row++][datetime_column] = timestamp;
			}

			// fillup the other columns
			size_t j_col = 0;
			foreach (const ColumnInfo& ci, columns)
			{
				if (!ci.m_datetime_column)
				{
					const size_t* pDimensions = ci.m_selection->shape();
					for (size_t s_row = 0; s_row < pDimensions[0]; s_row++)
					{
						j_row = 0;
						while (any_cast<string>(join[j_row][datetime_column]) < any_cast<string>((*ci.m_selection)[s_row][0]))
							j_row++;
						join[j_row][j_col] = (*ci.m_selection)[s_row][1];
					}
				}
				j_col++;
			}
		}

		// step 3: build the output string containing all the lines
		string result;
		{
			const size_t* pDimensions = join.shape();
			size_t cols = pDimensions[1];
			if (datetime_column == -1)
				cols--;

			for (size_t row = 0; row < pDimensions[0]; row++)
			{
				for (size_t col = 0; col < cols; col++)
				{
					const ColumnInfo& ci = columns[col];

					result += ci.m_left_gap;

					const any& a = join[row][col];
					if (a.empty())
						result += empty;
					else
					{
						if (a.type() == typeid(double))
							result += format_double(ci.m_format_spec, any_cast<double>(a));
						else
						{
							if (a.type() == typeid(Ratio))
							{
								result += format_double(ci.m_format_spec, any_cast<Ratio>(a).m_numerator);
								result += "/";
								result += format_double(ci.m_format_spec, any_cast<Ratio>(a).m_denominator);
							}
							else
							{
								if (a.type() == typeid(string))
									result += format_string(ci.m_format_spec, any_cast<string>(a));
							}
						}
					}
				}
				result += accu;
			}
		}

		return any(result);
	}
	CATCH_RETHROW("")
}

// (estring empty string)
DEFINE_FIMPL_1(estring_empty_string, typeid(estring), "empty", typeid(string))
{
	try
	{
		const string& s = any_cast<string>(params[0]->Evaluate(context));
		return any(estring(s));
	}
	CATCH_RETHROW("")
}

// (version)
DEFINE_FIMPL_0(string_version, typeid(string), "version")
{
	try
	{
		return any(string(VERSION));
	}
	CATCH_RETHROW("")
}

// (build)
DEFINE_FIMPL_0(string_build, typeid(string), "build")
{
	try
	{
		return any(string(BUILD));
	}
	CATCH_RETHROW("")
}

// (database)
DEFINE_FIMPL_0(string_database, typeid(string), "database")
{
	try
	{
		return any(context.m_database.GetVersion());
	}
	CATCH_RETHROW("")
}

//--- diagram -------------------------------------------------------------

	void find_and_complete_missing_parameters_in_axes_function (FContext& context, FImpl::Params& params);

// (nothing diagram width height bg_color ...)
DEFINE_FIMPL_4(nothing_diagram_int_int_color_ellipse, typeid(nothing), "diagram", typeid(int), typeid(int), typeid(size_t), typeid(ellipse))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		size_t width = any_cast<int>(params[0]->Evaluate(context));
		size_t height = any_cast<int>(params[1]->Evaluate(context));
		size_t bg_color = any_cast<size_t>(params[2]->Evaluate(context));

		dc.m_diagram.OpenCanvas(width, height, bg_color);

		// if we have an axes function call with missing parameters
		// we complete parameters according to the selections used
		// in other function calls - we do this before the axes
		// function is evaluated, note the evil const_cast
		find_and_complete_missing_parameters_in_axes_function(context, const_cast<Params&>(params));

		for (size_t i = 3; i < params.size(); i++)
		{
			// we assume that all these params do not return
			// results, they should just draw something onto the
			// diagram
			params[i]->Evaluate(context);
		}

		dc.m_diagram.CloseCanvas();

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing sundays color)
DEFINE_FIMPL_1(nothing_sundays_color, typeid(nothing), "sundays", typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		size_t color = any_cast<size_t>(params[0]->Evaluate(context));

		dc.m_diagram.SetSundaysColor(color);

		return any();
	}
	CATCH_RETHROW("")
}

	static Diagram::XUnit x_unit (const ptime& x_min, const ptime& x_max)
	{
		try
		{
			const time_duration& diff = x_max - x_min;
			if (diff.hours() > 365 * 24 * 2)
				return Diagram::years;
			else
			{
				if (diff.hours() > 30 * 24 * 2)
					return Diagram::months;
				else
				{
					if (diff.hours() > 24 * 2)
						return Diagram::days;
					else
					{
						if (diff.hours() > 2)
							return Diagram::hours;
						else
						{
							if (diff.minutes() > 2)
								return Diagram::minutes;
							else
								return Diagram::seconds;
						}
					}
				}
			}
		}
		CATCH_RETHROW("")
	}

// (nothing axes timestamp timestamp double double double color string)
DEFINE_FIMPL_7(nothing_axes_timestamp_timestamp_double_double_double_color_string, typeid(nothing), "axes", typeid(ptime), typeid(ptime), typeid(double), typeid(double), typeid(double), typeid(size_t), typeid(string))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		const ptime& x_min = any_cast<ptime>(params[0]->Evaluate(context));
		const ptime& x_max = any_cast<ptime>(params[1]->Evaluate(context));
		double y_min = any_cast<double>(params[2]->Evaluate(context));
		double y_max = any_cast<double>(params[3]->Evaluate(context));
		double y_unit = any_cast<double>(params[4]->Evaluate(context));
		size_t color = any_cast<size_t>(params[5]->Evaluate(context));
		const string& text = any_cast<string>(params[6]->Evaluate(context));

		const ConfigImpl::Periods& periods = dynamic_cast<const ConfigImpl&>(context.m_config).GetPeriods();
		dc.m_diagram.DrawAxes(x_min, x_max, x_unit(x_min, x_max), y_min, y_max, y_unit, color, text, periods);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing axes time timestamp double double double color string)
DEFINE_FIMPL_7(nothing_axes_time_timestamp_double_double_double_color_string, typeid(nothing), "axes", typeid(time_duration), typeid(ptime), typeid(double), typeid(double), typeid(double), typeid(size_t), typeid(string))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		const time_duration& duration = any_cast<time_duration>(params[0]->Evaluate(context));
		const ptime& x_max = any_cast<ptime>(params[1]->Evaluate(context));
		double y_min = any_cast<double>(params[2]->Evaluate(context));
		double y_max = any_cast<double>(params[3]->Evaluate(context));
		double y_unit = any_cast<double>(params[4]->Evaluate(context));
		size_t color = any_cast<size_t>(params[5]->Evaluate(context));
		const string& text = any_cast<string>(params[6]->Evaluate(context));

		const ptime& x_min = x_max - duration;
		const ConfigImpl::Periods& periods = dynamic_cast<const ConfigImpl&>(context.m_config).GetPeriods();
		dc.m_diagram.DrawAxes(x_min, x_max, x_unit(x_min, x_max), y_min, y_max, y_unit, color, text, periods);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing axes time double double double color string)
DEFINE_FIMPL_6(nothing_axes_time_double_double_double_color_string, typeid(nothing), "axes", typeid(time_duration), typeid(double), typeid(double), typeid(double), typeid(size_t), typeid(string))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		const time_duration& duration = any_cast<time_duration>(params[0]->Evaluate(context));
		double y_min = any_cast<double>(params[1]->Evaluate(context));
		double y_max = any_cast<double>(params[2]->Evaluate(context));
		double y_unit = any_cast<double>(params[3]->Evaluate(context));
		size_t color = any_cast<size_t>(params[4]->Evaluate(context));
		const string& text = any_cast<string>(params[5]->Evaluate(context));

		const ptime& x_max = context.m_config.GetNow();
		const ptime& x_min = x_max - duration;
		const ConfigImpl::Periods& periods = dynamic_cast<const ConfigImpl&>(context.m_config).GetPeriods();
		dc.m_diagram.DrawAxes(x_min, x_max, x_unit(x_min, x_max), y_min, y_max, y_unit, color, text, periods);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing axes keyword double double double color string)
DEFINE_FIMPL_6(nothing_axes_keyword_double_double_double_color_string, typeid(nothing), "axes", typeid(kstring), typeid(double), typeid(double), typeid(double), typeid(size_t), typeid(string))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		const kstring& keyword = any_cast<kstring>(params[0]->Evaluate(context));
		double y_min = any_cast<double>(params[1]->Evaluate(context));
		double y_max = any_cast<double>(params[2]->Evaluate(context));
		double y_unit = any_cast<double>(params[3]->Evaluate(context));
		size_t color = any_cast<size_t>(params[4]->Evaluate(context));
		const string& text = any_cast<string>(params[5]->Evaluate(context));

		Diagram::FoldInterval fi = get_fold_interval(m_name, keyword);
		dc.m_diagram.DrawAxes(fi, y_min, y_max, y_unit, color, text);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing axes timestamp timestamp double color string)
DEFINE_FIMPL_5(nothing_axes_timestamp_timestamp_double_color_string, typeid(nothing), "axes", typeid(ptime), typeid(ptime), typeid(double), typeid(size_t), typeid(string))
{
	// this implementation is never called
	return any();
}

// (nothing axes time timestamp double color string)
DEFINE_FIMPL_5(nothing_axes_time_timestamp_double_color_string, typeid(nothing), "axes", typeid(time_duration), typeid(ptime), typeid(double), typeid(size_t), typeid(string))
{
	// this implementation is never called
	return any();
}

// (nothing axes time double color string)
DEFINE_FIMPL_4(nothing_axes_time_double_color_string, typeid(nothing), "axes", typeid(time_duration), typeid(double), typeid(size_t), typeid(string))
{
	// this implementation is never called
	return any();
}

// (nothing axes keyword double color string)
DEFINE_FIMPL_4(nothing_axes_keyword_double_color_string, typeid(nothing), "axes", typeid(kstring), typeid(double), typeid(size_t), typeid(string))
{
	// this implementation is never called
	return any();
}

// (nothing axes double double double color string)
DEFINE_FIMPL_5(nothing_axes_double_double_double_color_string, typeid(nothing), "axes", typeid(double), typeid(double), typeid(double), typeid(size_t), typeid(string))
{
	// this implementation is never called
	return any();
}

// (nothing axes double color string)
DEFINE_FIMPL_3(nothing_axes_double_color_string, typeid(nothing), "axes", typeid(double), typeid(size_t), typeid(string))
{
	// this implementation is never called
	return any();
}

/*
	The axis function has the following ten implementations named
	here by the letter at the beginning of the line:

		|----------x----------| |-------y-------|
  a	(axes	{timestamp} {timestamp}	{double} {double}	{double} {color} {string})
  b	(axes	{time} {timestamp}	{double} {double}	{double} {color} {string})
  c	(axes	{time}			{double} {double}	{double} {color} {string})
  d	(axes	keyword			{double} {double}	{double} {color} {string})
  e	(axes	{timestamp} {timestamp}				{double} {color} {string})
  f	(axes	{time} {timestamp}				{double} {color} {string})
  g	(axes	{time}						{double} {color} {string})
  h	(axes	keyword						{double} {color} {string})
  i	(axes				{double} {double}	{double} {color} {string})
  j	(axes							{double} {color} {string})

*/
	void find_and_complete_missing_parameters_in_axes_function (FContext& context, FImpl::Params& params)
	{
		// We get here the parameters of the diagram function. What
		// we try to do here is to complete the parameter list of
		// a possible axes function with missing x or y parameters
		// (implementations e ... j). The result is always an axes
		// function call of type a, the most concrete axes
		// implementation at all.

		// find an axes function call with missing parameters
		FExpr* pAxes = NULL;
		enum {irrelevant, e, f, g, h, i, j} type = irrelevant;
		for (size_t z = 3; type == irrelevant && z < params.size(); z++)
		{
			FExpr* pExpr = dynamic_cast<FExpr*>(params[z]);
			if (!pExpr)
				continue;
			pAxes = pExpr;
			const FImpl* pFImpl = pExpr->GetFImpl();
			if (dynamic_cast<const nothing_axes_timestamp_timestamp_double_color_string*>(pFImpl))
				type = e;
			else
			{
				if (dynamic_cast<const nothing_axes_time_timestamp_double_color_string*>(pFImpl))
					type = f;
				else
				{
					if (dynamic_cast<const nothing_axes_time_double_color_string*>(pFImpl))
						type = g;
					else
					{
						if (dynamic_cast<const nothing_axes_keyword_double_color_string*>(pFImpl))
							type = h;
						else
						{
							if (dynamic_cast<const nothing_axes_double_double_double_color_string*>(pFImpl))
								type = i;
							else
							{
								if (dynamic_cast<const nothing_axes_double_color_string*>(pFImpl))
									type = j;
							}
						}
					}
				}
			}
		}
		if (type == irrelevant)
			return; // do nothing

		// find minimum and maximum x and y values based on the
		// selections used in other function calls
		bool has_x_min = false, has_x_max = false, has_y_min = false, has_y_max = false;
		ptime x_min, x_max;
		double y_min = 0.0, y_max = 0.0;
		for (size_t i = 3; i < params.size(); i++)
		{
			if (FExpr* pExpr = dynamic_cast<FExpr*>(params[i]))
			{
				const string& n = pExpr->GetFunction();
				if (n == "curve" || n == "spline" || n == "bars" || n == "stairs")
				{
					if (FExpr* pParam = dynamic_cast<FExpr*>(pExpr->GetParams()[0])) // {selection}
					{
						any a = pParam->Evaluate(context);
						if (a.type() == typeid(shared_ptr<Selection>))
						{
							shared_ptr<Selection> pSelection = any_cast<shared_ptr<Selection> >(a);
							const size_t* pDimensions = pSelection->shape(); // [0] rows, [1] columns
							for (size_t row = 0; row < pDimensions[0]; row++)
							{
								any t = (*pSelection)[row][0];
								if (t.type() == typeid(string))
								{
									ptime timestamp = Diagram::CompletePtimeFromString(any_cast<string>(t));
									if (!has_x_min || timestamp < x_min)
									{
										x_min = timestamp;
										has_x_min = true;
									}
									if (!has_x_max || x_max < timestamp)
									{
										x_max = timestamp;
										has_x_max = true;
									}
								}
								any v = (*pSelection)[row][1];
								if (v.type() == typeid(double))
								{
									double value = any_cast<double>(v);
									if (!has_y_min || value < y_min)
									{
										y_min = value;
										has_y_min = true;
									}
									if (!has_y_max || y_max < value)
									{
										y_max = value;
										has_y_max = true;
									}
								}
							}
						}
					}
				}
			}
		}
		if (!has_x_min || !has_x_max || !has_y_min || !has_y_max)
			THROW("cannot complete missing parameters in axes statement");

		// adjust the new limits a little bit to avoid drawing on
		// the x-axis
		double diff = y_max - y_min;
		double y;
		y = y_max + diff * 0.05; y_max = (y_max < 0.0) ? ((y < 0.0) ? y : 0.0) : y;
		y = y_min - diff * 0.05; y_min = (y_min > 0.0) ? ((y > 0.0) ? y : 0.0) : y;

		// add missing parameters in pAxes
		FExpr::Params& p = const_cast<FExpr::Params&>(pAxes->GetParams());
		switch (type)
		{
			case e: // (axes {timestamp} {timestamp} y_min y_max {double} {color} {string})
			case f: // (axes {time} {timestamp} y_min y_max {double} {color} {string})
			{
				p.insert(p.begin() + 2, new FDouble(y_min));
				p.insert(p.begin() + 3, new FDouble(y_max));
				break;
			}
			case g: // (axes {time} y_min y_max {double} {color} {string})
			case h: // (axes keyword y_min y_max {double} {color} {string})
			{
				p.insert(p.begin() + 1, new FDouble(y_min));
				p.insert(p.begin() + 2, new FDouble(y_max));
				break;
			}
			case i: // (axes x_min x_max {double} {double} {double} {color} {string})
			{
				p.insert(p.begin() + 0, new FDateTime(x_min));
				p.insert(p.begin() + 1, new FDateTime(x_max));
				break;
			}
			case j: // (axes x_min x_max y_min y_max {double} {color} {string})
			{
				p.insert(p.begin() + 0, new FDateTime(x_min));
				p.insert(p.begin() + 1, new FDateTime(x_max));
				p.insert(p.begin() + 2, new FDouble(y_min));
				p.insert(p.begin() + 3, new FDouble(y_max));
				break;
			}
			default:
				break;
		}

		// reset the m_pFImpl object in pAxes because currently
		// it points still to an axes implementation with missing
		// parameters
		pAxes->ResetImpl();
	}

// (nothing hline double color)
DEFINE_FIMPL_2(nothing_hline_double_color, typeid(nothing), "hline", typeid(double), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		double y = any_cast<double>(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));

		dc.m_diagram.DrawHLine(y, color, 0.0);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing hline double double color)
DEFINE_FIMPL_3(nothing_hline_double_double_color, typeid(nothing), "hline", typeid(double), typeid(double), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		double y = any_cast<double>(params[0]->Evaluate(context));
		double thickness = any_cast<double>(params[1]->Evaluate(context));
		size_t color = any_cast<size_t>(params[2]->Evaluate(context));

		dc.m_diagram.DrawHLine(y, color, thickness);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing hlines double color)
DEFINE_FIMPL_2(nothing_hlines_double_color, typeid(nothing), "hlines", typeid(double), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		double y = any_cast<double>(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));

		dc.m_diagram.DrawHLines(y, color, 0.0);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing hlines double double color)
DEFINE_FIMPL_3(nothing_hlines_double_double_color, typeid(nothing), "hlines", typeid(double), typeid(double), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		double y = any_cast<double>(params[0]->Evaluate(context));
		double thickness = any_cast<double>(params[1]->Evaluate(context));
		size_t color = any_cast<size_t>(params[2]->Evaluate(context));

		dc.m_diagram.DrawHLines(y, color, thickness);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing vline timestamp color)
DEFINE_FIMPL_2(nothing_vline_timestamp_color, typeid(nothing), "vline", typeid(ptime), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		ptime x = any_cast<ptime>(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));

		dc.m_diagram.DrawVLine(x, color, 0.0);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing vline timestamp double color)
DEFINE_FIMPL_3(nothing_vline_timestamp_double_color, typeid(nothing), "vline", typeid(ptime), typeid(double), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		ptime x = any_cast<ptime>(params[0]->Evaluate(context));
		double thickness = any_cast<double>(params[1]->Evaluate(context));
		size_t color = any_cast<size_t>(params[2]->Evaluate(context));

		dc.m_diagram.DrawVLine(x, color, thickness);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing vline time color)
DEFINE_FIMPL_2(nothing_vline_time_color, typeid(nothing), "vline", typeid(time_duration), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		time_duration t = any_cast<time_duration>(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));

		dc.m_diagram.DrawVLine(t, color, 0.0);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing vline time double color)
DEFINE_FIMPL_3(nothing_vline_time_double_color, typeid(nothing), "vline", typeid(time_duration), typeid(double), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		time_duration t = any_cast<time_duration>(params[0]->Evaluate(context));
		double thickness = any_cast<double>(params[1]->Evaluate(context));
		size_t color = any_cast<size_t>(params[2]->Evaluate(context));

		dc.m_diagram.DrawVLine(t, color, thickness);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing curve selection color)
DEFINE_FIMPL_2(nothing_curve_selection_color, typeid(nothing), "curve", typeid(shared_ptr<Selection>), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));

		dc.m_diagram.DrawCurve(*selection, color, Diagram::zigzag, 0.0, 0);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing curve selection color double)
DEFINE_FIMPL_3(nothing_curve_selection_color_double, typeid(nothing), "curve", typeid(shared_ptr<Selection>), typeid(size_t), typeid(double))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));
		double thickness = any_cast<double>(params[2]->Evaluate(context));

		dc.m_diagram.DrawCurve(*selection, color, Diagram::zigzag, thickness, 0);

		return any();
	}
	CATCH_RETHROW("")
}

	static Diagram::LineStyle get_symbol_style (const string& name, const string& s)
	{
		if (s == ".")
			return Diagram::symbol_dot;

		if (s == "+")
			return Diagram::symbol_plus;

		if (s == "|")
			return Diagram::symbol_vbar;

		if (s == "-")
			return Diagram::symbol_hbar;

		if (s == "x")
			return Diagram::symbol_x;

		if (s == "X")
			return Diagram::symbol_X;

		if (s == "°")
			return Diagram::symbol_circle;

		if (s == "#")
			return Diagram::symbol_square;

		THROW(format("%s: unknown diagram line symbol: %s, use one of [.+|-xX°#]") % name % s);
	}

// (nothing curve selection color string)
DEFINE_FIMPL_3(nothing_curve_selection_color_string, typeid(nothing), "curve", typeid(shared_ptr<Selection>), typeid(size_t), typeid(string))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));
		string symbol = any_cast<string>(params[2]->Evaluate(context));

		Diagram::LineStyle style = get_symbol_style(m_name, symbol);

		dc.m_diagram.DrawCurve(*selection, color, style, 0.0, 0);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing curve selection color string double)
DEFINE_FIMPL_4(nothing_curve_selection_color_string_double, typeid(nothing), "curve", typeid(shared_ptr<Selection>), typeid(size_t), typeid(string), typeid(double))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));
		string symbol = any_cast<string>(params[2]->Evaluate(context));
		double thickness = any_cast<double>(params[3]->Evaluate(context));

		Diagram::LineStyle style = get_symbol_style(m_name, symbol);

		dc.m_diagram.DrawCurve(*selection, color, style, thickness, 0);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing spline selection color int)
DEFINE_FIMPL_3(nothing_spline_selection_color_int, typeid(nothing), "spline", typeid(shared_ptr<Selection>), typeid(size_t), typeid(int))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));
		int splineValues = any_cast<int>(params[2]->Evaluate(context));

		dc.m_diagram.DrawCurve(*selection, color, Diagram::zigzag, 0.0, splineValues);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing spline selection color double int)
DEFINE_FIMPL_4(nothing_spline_selection_color_double_int, typeid(nothing), "spline", typeid(shared_ptr<Selection>), typeid(size_t), typeid(double), typeid(int))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));
		double thickness = any_cast<double>(params[2]->Evaluate(context));
		int splineValues = any_cast<int>(params[3]->Evaluate(context));

		dc.m_diagram.DrawCurve(*selection, color, Diagram::zigzag, thickness, splineValues);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing bars selection color)
DEFINE_FIMPL_2(nothing_bars_selection_color, typeid(nothing), "bars", typeid(shared_ptr<Selection>), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));

		const time_duration& midnight = get_midnight(context);

		dc.m_diagram.DrawBars(*selection, color, 0, 0, midnight);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing bars selection color int int)
DEFINE_FIMPL_4(nothing_bars_selection_color_int_int, typeid(nothing), "bars", typeid(shared_ptr<Selection>), typeid(size_t), typeid(int), typeid(int))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));
		int bars1 = any_cast<int>(params[2]->Evaluate(context));
		int bars2 = any_cast<int>(params[3]->Evaluate(context));

		const time_duration& midnight = get_midnight(context);

		dc.m_diagram.DrawBars(*selection, color, bars1, bars2, midnight);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing stairs selection color)
DEFINE_FIMPL_2(nothing_stairs_selection_color, typeid(nothing), "stairs", typeid(shared_ptr<Selection>), typeid(size_t))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));

		dc.m_diagram.DrawStairs(*selection, color, 1.0);

		return any();
	}
	CATCH_RETHROW("")
}

// (nothing stairs selection color double)
DEFINE_FIMPL_3(nothing_stairs_selection_color_double, typeid(nothing), "stairs", typeid(shared_ptr<Selection>), typeid(size_t), typeid(double))
{
	try
	{
		if (typeid(context) != typeid(FDiagramContext))
			THROW(format("%s: function call needs a diagram context, you cannot call this function in interactive mode") % m_name);
		FDiagramContext& dc = dynamic_cast<FDiagramContext&>(context);

		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(params[0]->Evaluate(context));
		size_t color = any_cast<size_t>(params[1]->Evaluate(context));
		double thickness = any_cast<double>(params[2]->Evaluate(context));

		dc.m_diagram.DrawStairs(*selection, color, thickness);

		return any();
	}
	CATCH_RETHROW("")
}

//=== FImplFactory =========================================================
FImplFactory::FImplFactory ()
	: m_impls()
	, m_typenames()
{
	// functions defining time durations
	m_impls.insert(new time_year());
	m_impls.insert(new time_year_int());
	m_impls.insert(new time_years_int());
	m_impls.insert(new time_month());
	m_impls.insert(new time_month_int());
	m_impls.insert(new time_months_int());
	m_impls.insert(new time_week());
	m_impls.insert(new time_week_int());
	m_impls.insert(new time_weeks_int());
	m_impls.insert(new time_day());
	m_impls.insert(new time_day_int());
	m_impls.insert(new time_days_int());
	m_impls.insert(new time_hour());
	m_impls.insert(new time_hour_int());
	m_impls.insert(new time_hours_int());
	m_impls.insert(new time_minute());
	m_impls.insert(new time_minute_int());
	m_impls.insert(new time_minutes_int());
	m_impls.insert(new time_second());
	m_impls.insert(new time_second_int());
	m_impls.insert(new time_seconds_int());

	// timestamp functions
	m_impls.insert(new timestamp_now());
	m_impls.insert(new timestamp_midnight());
	m_impls.insert(new timestamp_plus_timestamp_time());
	m_impls.insert(new timestamp_minus_timestamp_time());

	// select
	m_impls.insert(new selection_select_string());
	m_impls.insert(new selection_select_string_timestamp());
	m_impls.insert(new selection_select_string_timestamp_timestamp());
	m_impls.insert(new selection_select_string_time());
	m_impls.insert(new selection_select_string_time_timestamp());

	// merge/fold
	m_impls.insert(new selection_merge_keyword_ellipse());
	m_impls.insert(new selection_fold_keyword_keyword_selection());

	// statistic functions
	m_impls.insert(new selection_avg_selection());
	m_impls.insert(new selection_avg_selection_time_time());
	m_impls.insert(new selection_avg_selection_keyword());
	m_impls.insert(new selection_avg_selection_keyword_time_time());
	m_impls.insert(new selection_avg_selection_int_int());

	m_impls.insert(new selection_sdv_selection());
	m_impls.insert(new selection_sdv_selection_time_time());
	m_impls.insert(new selection_sdv_selection_keyword());
	m_impls.insert(new selection_sdv_selection_keyword_time_time());
	m_impls.insert(new selection_sdv_selection_int_int());

	m_impls.insert(new selection_min_selection());
	m_impls.insert(new selection_min_selection_time_time());
	m_impls.insert(new selection_min_selection_keyword());
	m_impls.insert(new selection_min_selection_keyword_time_time());

	m_impls.insert(new selection_max_selection());
	m_impls.insert(new selection_max_selection_time_time());
	m_impls.insert(new selection_max_selection_keyword());
	m_impls.insert(new selection_max_selection_keyword_time_time());

	m_impls.insert(new selection_count_selection());
	m_impls.insert(new selection_count_selection_time_time());
	m_impls.insert(new selection_count_selection_keyword());
	m_impls.insert(new selection_count_selection_keyword_time_time());

	m_impls.insert(new selection_sum_selection());
	m_impls.insert(new selection_sum_selection_time_time());
	m_impls.insert(new selection_sum_selection_keyword());
	m_impls.insert(new selection_sum_selection_keyword_time_time());

	m_impls.insert(new selection_first_selection());
	m_impls.insert(new selection_first_selection_time_time());
	m_impls.insert(new selection_first_selection_keyword());
	m_impls.insert(new selection_first_selection_keyword_time_time());

	m_impls.insert(new selection_last_selection());
	m_impls.insert(new selection_last_selection_time_time());
	m_impls.insert(new selection_last_selection_keyword());
	m_impls.insert(new selection_last_selection_keyword_time_time());

	m_impls.insert(new string_unit_string());
	m_impls.insert(new string_purpose_string());

	m_impls.insert(new selection_HbA1c_string());
	m_impls.insert(new selection_HbA1c_string_timestamp());
	m_impls.insert(new selection_HbA1c_string_timestamp_timestamp());
	m_impls.insert(new selection_HbA1c_string_time());
	m_impls.insert(new selection_HbA1c_string_time_timestamp());

	m_impls.insert(new selection_HbA1c2_string());
	m_impls.insert(new selection_HbA1c2_string_timestamp());
	m_impls.insert(new selection_HbA1c2_string_timestamp_timestamp());
	m_impls.insert(new selection_HbA1c2_string_time());
	m_impls.insert(new selection_HbA1c2_string_time_timestamp());

	m_impls.insert(new selection_plus_double_double());
	m_impls.insert(new selection_plus_selection_double());
	m_impls.insert(new selection_plus_selection_selection());
	m_impls.insert(new string_plus_string_string_ellipse());
	m_impls.insert(new selection_minus_double_double());
	m_impls.insert(new selection_minus_selection_double());
	m_impls.insert(new selection_minus_selection_selection());
	m_impls.insert(new selection_mult_double_double());
	m_impls.insert(new selection_mult_selection_double());
	m_impls.insert(new selection_mult_selection_selection());
	m_impls.insert(new selection_div_double_double());
	m_impls.insert(new selection_div_selection_double());
	m_impls.insert(new selection_div_selection_selection());

	// comparison
	m_impls.insert(new selection_eq_selection_double());
	m_impls.insert(new selection_lt_selection_double());
	m_impls.insert(new selection_gt_selection_double());
	m_impls.insert(new selection_le_selection_double());
	m_impls.insert(new selection_ge_selection_double());
	m_impls.insert(new selection_ne_selection_double());

	// report functions
	m_impls.insert(new string_format_ellipse());
	m_impls.insert(new estring_empty_string());
	m_impls.insert(new string_version());
	m_impls.insert(new string_build());
	m_impls.insert(new string_database());

	// diagram functions
	m_impls.insert(new nothing_diagram_int_int_color_ellipse());

	m_impls.insert(new nothing_sundays_color());

	m_impls.insert(new nothing_axes_timestamp_timestamp_double_double_double_color_string());
	m_impls.insert(new nothing_axes_time_timestamp_double_double_double_color_string());
	m_impls.insert(new nothing_axes_time_double_double_double_color_string());
	m_impls.insert(new nothing_axes_keyword_double_double_double_color_string());
	m_impls.insert(new nothing_axes_timestamp_timestamp_double_color_string());
	m_impls.insert(new nothing_axes_time_timestamp_double_color_string());
	m_impls.insert(new nothing_axes_time_double_color_string());
	m_impls.insert(new nothing_axes_keyword_double_color_string());
	m_impls.insert(new nothing_axes_double_double_double_color_string());
	m_impls.insert(new nothing_axes_double_color_string());

	m_impls.insert(new nothing_hline_double_color());
	m_impls.insert(new nothing_hline_double_double_color());
	m_impls.insert(new nothing_hlines_double_color());
	m_impls.insert(new nothing_hlines_double_double_color());

	m_impls.insert(new nothing_vline_timestamp_color());
	m_impls.insert(new nothing_vline_timestamp_double_color());
	m_impls.insert(new nothing_vline_time_color());
	m_impls.insert(new nothing_vline_time_double_color());

	m_impls.insert(new nothing_curve_selection_color());
	m_impls.insert(new nothing_curve_selection_color_double());
	m_impls.insert(new nothing_curve_selection_color_string());
	m_impls.insert(new nothing_curve_selection_color_string_double());

	m_impls.insert(new nothing_spline_selection_color_int());
	m_impls.insert(new nothing_spline_selection_color_double_int());

	m_impls.insert(new nothing_bars_selection_color());
	m_impls.insert(new nothing_bars_selection_color_int_int());

	m_impls.insert(new nothing_stairs_selection_color());
	m_impls.insert(new nothing_stairs_selection_color_double());

	// the type names made readable
	m_typenames.insert(TypeNames::value_type(&typeid(double),		"{double}"));
	m_typenames.insert(TypeNames::value_type(&typeid(Ratio),		"{ratio}"));
	m_typenames.insert(TypeNames::value_type(&typeid(ptime),		"{timestamp}"));
	m_typenames.insert(TypeNames::value_type(&typeid(date),			"{date}"));
	m_typenames.insert(TypeNames::value_type(&typeid(time_duration),	"{time}"));
	m_typenames.insert(TypeNames::value_type(&typeid(string),		"{string}"));
	m_typenames.insert(TypeNames::value_type(&typeid(fstring),		"{format}"));
	m_typenames.insert(TypeNames::value_type(&typeid(kstring),		"keyword"));
	m_typenames.insert(TypeNames::value_type(&typeid(estring),		"{string}"));
	m_typenames.insert(TypeNames::value_type(&typeid(int),			"{int}"));
	m_typenames.insert(TypeNames::value_type(&typeid(size_t),		"{color}"));
	m_typenames.insert(TypeNames::value_type(&typeid(shared_ptr<Selection>),"{selection}"));
	m_typenames.insert(TypeNames::value_type(&typeid(ellipse),		"..."));
	m_typenames.insert(TypeNames::value_type(&typeid(nothing),		"{nothing}"));
}

FImplFactory::~FImplFactory ()
{
	foreach (FImpl* pFImpl, m_impls)
	{
		delete pFImpl;
	}
}

	class search_impl: public FImpl {
		public:
		search_impl (const string& funcname) : FImpl(NULL, funcname, 0) {}
		virtual	any Evaluate (FContext& , const Params& ) const throw (Xception) {return any();}
	};

const FImpl* FImplFactory::Match (const string& name, const FImpl::ParamTypes& paramtypes) const throw (Xception)
{
	try
	{
		// find function implementations by name and match the given
		// parameters to find a concrete function implementation
		search_impl dummy(name);
		Impls::const_iterator L = m_impls.lower_bound(&dummy);
		Impls::const_iterator U = m_impls.upper_bound(&dummy);
		if (L == U)
			THROW(format("unknown function \"%s\"") % name);

		Impls::const_iterator I = L;
		while (I != U)
		{
			FImpl* pFImpl = *I;

			size_t params_needed = 0;
			bool params_ellipse = false;
			if (!pFImpl->m_paramtypes.empty())
			{
				params_needed = pFImpl->m_paramtypes.size();
				if (*pFImpl->m_paramtypes.back() == typeid(ellipse))
				{
					params_needed--;
					params_ellipse = true;
				}
			}

			if ((!params_ellipse && paramtypes.size() == params_needed) || (params_ellipse && paramtypes.size() >= params_needed))
			{
				bool match = true;
				for (size_t i = 0; match && i < params_needed; i++)
				{
					if (*pFImpl->m_paramtypes[i] != *paramtypes[i])
						match = false;
				}
				if (match)
					break;
			}

			I++;
		}

		// (if needed, produce a detailed error message)
		if (I == U)
		{
			stringstream ss;
			ss << "parameter mismatch for function \"" << name << "\", called as:" << endl;
			ss << "  (" << name;
			foreach (const type_info* pTI, paramtypes)
			{
				ss << ' ' << (*m_typenames.find(pTI)).second;
			}
			ss << ')' << endl;
			ss << "available function implementations are:";
			I = L;
			while (I != U)
			{
				FImpl* pFImpl = *I++;
				ss << endl << "  (" << name;
				foreach (const type_info* pTI, pFImpl->m_paramtypes)
				{
					ss << ' ' << (*m_typenames.find(pTI)).second;
				}
				ss << ")  ->  " << (*m_typenames.find(pFImpl->m_returntype)).second;
			}

			THROW(ss.str());
		}

		return *I;
	}
	CATCH_RETHROW("")
}

void FImplFactory::Help (ostream& os, const string& name) const
{
	try
	{
		regex rx(name);
		size_t n = 0;
		foreach (const FImpl* pFImpl, m_impls)
		{
			if (!name.empty() && !regex_match(pFImpl->m_name, rx, boost::match_not_dot_newline))
				continue;

			os << "  (" << pFImpl->m_name;
			foreach (const type_info* pTI, pFImpl->m_paramtypes)
			{
				os << ' ' << (*m_typenames.find(pTI)).second;
			}
			os << ")  ->  " << (*m_typenames.find(pFImpl->m_returntype)).second << endl;
			n++;
		}
		os << "  " << n << " function implementation" << ((n == 1) ? "" : "s") << " found" << endl;
	}
	catch (boost::regex_error)
	{
		THROW("error in regular expression");
	}
	CATCH_RETHROW("")
}
