// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace boost::filesystem;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/xception.h"
#include "config_impl.h"
#include "diagram_impl.h"

//=== Diagram ==============================================================
Diagram::~Diagram ()
{
}

ptime Diagram::CompletePtimeFromString (const string& s)
{
	try
	{
		// s: YYYY[-MM[-DD[ hh[:mm[:ss]]]]]

		static const regex rx("([0-9]+)(-[0-9]+)?(-[0-9]+)?( [0-9]+)?(\\:[0-9]+)?(\\:[0-9]+)?");
		smatch mr;
		regex_match(s, mr, rx);

		string t(s);
		if (mr.length(6) > 0)
			;
		else
		{
			if (mr.length(5) > 0)
				t += ":00"; // add seconds
			else
			{
				if (mr.length(4) > 0)
					t += ":00:00"; // add minutes and seconds
				else
				{
					if (mr.length(3) > 0)
						t += " 00:00:00"; // add complete time
					else
					{
						if (mr.length(2) > 0)
							t += "-01 00:00:00"; // add day and complete time
						else
							t += "-01-01 00:00:00"; // add month, day and complete time
					}
				}
			}
		}

		// t: YYYY-MM-DD hh:mm:ss
		t += ".000";

		return time_from_string(t);
	}
	CATCH_RETHROW("")
}

//=== DiagramFactory =======================================================
Diagram* DiagramFactory::Create (const string& filename, bool antialias) throw (Xception)
{
	try
	{
		const string ext(path(filename).extension().string());

#ifdef USE_BOARD
		if (ext == ".svg" || ext == ".SVG")
			return new DiagramSVG();

		if (ext == ".eps" || ext == ".EPS")
			return new DiagramEPS();

		if (ext == ".fig" || ext == ".FIG")
			return new DiagramXFig();
#endif // USE_BOARD

#ifdef USE_CAIRO
		if (ext == ".png" || ext == ".PNG")
			return new DiagramPNG(antialias);
#endif // USE_CAIRO

		// ...
		THROW(format("unknown diagram extension: %s") % ext);
	}
	CATCH_RETHROW("")
}
