// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace boost::filesystem;

#include "../libpdrx/xception.h"
#include "../libpdrx/conversions.h"
#include "../libpdrx/config.h"
#include "../libpdrx/build.h"
#include "db.h"
#include "in.h"
#include "out.h"

//=== ConfigImpl ===========================================================
class ConfigImpl: public Config {

	public:

	ConfigImpl ();

	virtual void HandleApplicationName ();
	virtual void ModifyCmdLineOptions (parsed_options& po);
};

ConfigImpl::ConfigImpl ()
	: Config()
{
}

void ConfigImpl::HandleApplicationName ()
{
	if (m_app_name == "pdri")
	{
		m_od_visible.add_options()

			// common options
			("help,?",						"show this help screen and quit")
			("version,V",						"display version number and quit")
			("config,c", value<string>(),				"use specific configuration file")

			// special options
			("debug",						"dump call stack in case of exceptions")
		;

		m_od_cmdline.add_options() ("interactive", "");
	}
	else
	{
		m_od_visible.add_options()

			// common options
			("help,?",						"show this help screen and quit")
			("version,V",						"display version number and quit")
			("verbose,v",						"produce verbose output on stdout")
			("config,c", value<string>(),				"use specific configuration file")
			("interactive,i",					"start interactive")

			// database options
			("list-collections,l",					"list all available collections in the database")
			("add-collection,a", value<vector<string> >(),		"add a collection to the database, arg is a string like \"name[, n|r|t[, unit[, purpose]]]\"")
			("delete-collection,d", value<vector<string> >(),	"delete a collection from the database, arg is a string like \"name\"")
			("delete-all-collections,D",				"delete all collections from the database")
			("list-rejections,r",					"list all rejected expressions")
			("delete-all-rejections,R",				"delete all rejected expressions")

			// input options
			("expression,e", value<vector<string> >(),		"use arg as expression for database input")
			("txt,t", value<vector<string> >(),			"use txt file arg for database input")
			("csv,C", value<vector<string> >(),			"use csv file arg for database input")
			("xml,x", value<vector<string> >(),			"use xml file arg for database input")

			// output options
			("export,X", value<string>(),				"create an xml export file")

			// special options
			("create-config",					"create the local configuration files and quit")
#ifndef _WIN32
			("browser", value<string>(),				"specify your browser executable including path (if needed) for Twitter authentication")
#endif
			("debug",						"dump call stack in case of exceptions")
		;

		// in pdr we need a second options description for a hidden
		// parameter containing all the command line arguments
		m_od_cmdline.add_options() ("cmd_line_args", value<string>(), "");
		m_pod.add("cmd_line_args", -1); // unlimited
	}
}

	struct fo_cmd_line_arg: public unary_function<option&, bool> {
		string& m_expr;
		fo_cmd_line_arg (string& expr)
			: m_expr(expr)
		{
		}
		result_type operator () (argument_type a)
		{
			try
			{
				if (a.string_key == "cmd_line_args")
				{
					if (!m_expr.empty())
						m_expr += ' ';
					m_expr += a.value[0];
					return true;
				}
				else
					return false;
			}
			CATCH_RETHROW("")
		}
	};

	struct fo_expression: public unary_function<const option&, bool> {
		result_type operator () (argument_type a) const
		{
			return a.string_key == "expression";
		}
	};

void ConfigImpl::ModifyCmdLineOptions (parsed_options& po)
{
	try
	{
		// in pdr we transform all the given command line arguments
		// (the arguments, not the options) into one single
		// expression, this expression will then be added to the
		// list of normal expressions from -e, the hidden parameters
		// "cmd_line_args" are removed here

		string expr;
		{
			vector<option>::iterator I = remove_if(po.options.begin(), po.options.end(), fo_cmd_line_arg(expr));
			if (I != po.options.end())
				po.options.erase(I, po.options.end());
		}

		if (!expr.empty())
		{
			vector<option>::iterator I = find_if(po.options.begin(), po.options.end(), fo_expression());
			if (I == po.options.end())
			{
				vector<string> values;
				values.push_back(expr);
				po.options.push_back(option("expression", values));
			}
			else
				(*I).value.push_back(expr);
		}

		// check if we are called interactively, then we fake a
		// -i parameter
		if (m_app_name == "pdri")
			po.options.push_back(option("interactive", vector<string>()));
	}
	CATCH_RETHROW("")
}

//=== create_config ========================================================
void create_config (const Config& config)
{
	try
	{
		// ensure the directories exist
		path p;
		p = config.GetStringOption("home");
		p /= ".config";
		if (!is_directory(p) && !create_directory(p))
			THROW("could not create directory ~/.config");

		p /= "pdrx";
		if (!is_directory(p) && !create_directory(p))
			THROW("could not create directory ~/.config/pdrx");

		// create a default pdrxrc
		p /= "pdrxrc";
		ofstream ofs(p.string().c_str(), ios::out);
		if (ofs.good())
		{
			ofs << "# pdrxrc" << endl;
			ofs << "# " << lexical_cast<string>(config.GetNow()).c_str() << endl;
			ofs << endl;
			ofs << "# --- general options ------------------------------------------------------" << endl;
			ofs << endl;
			ofs << "# operate verbose (yes, no)" << endl;
			ofs << "verbose = yes" << endl;
			ofs << endl;
			ofs << "# define your inputs here (input1, input2, input3 ...)" << endl;
			ofs << "inputs = input1, input2" << endl;
			ofs << endl;
			ofs << "# define your outputs here (output1, output2, output3 ...)" << endl;
			ofs << "outputs = output1" << endl;
			ofs << endl;
			ofs << "# specify the encoding of your console" << endl;
			ofs << "# (one of ASCII, ISO-8859-1, ISO-8859-15, UTF-8, UTF-16, Windows-1252, ASCII/CP437, ASCII/CP850)" << endl;
#ifdef _WIN32
			ofs << "encoding = ASCII/CP437" << endl;
#else
			ofs << "encoding = UTF-8" << endl;
#endif
			ofs << endl;
			ofs << "# pdx: say when is midnight (hh:mm:ss)" << endl;
			ofs << "midnight = 2:00" << endl;
			ofs << endl;
			ofs << "# pdx: cache internal results for faster operation (yes, no)" << endl;
			ofs << "fast = yes" << endl;
			ofs << endl;
			ofs << "# --- time periods ---------------------------------------------------------" << endl;
			ofs << endl;
			ofs << "# this section lists periods of time which are displayed in diagrams with" << endl;
			ofs << "# a special background color, each line has the following format:" << endl;
			ofs << "#" << endl;
			ofs << "#     period = <begin> <end> <color>" << endl;
			ofs << "#" << endl;
			ofs << "# where <begin> and <end> are dates in the form YYYY-MM-DD and <color> is" << endl;
			ofs << "# a HTML style hexadecimal color definition, example:" << endl;
			ofs << "#" << endl;
			ofs << "#     period = 2009-03-08 2009-03-15 #F0F0F0" << endl;
			ofs << endl;
			ofs << "# --- database options -----------------------------------------------------" << endl;
			ofs << endl;
			ofs << "# define a SQLite or a MySQL database here" << endl;
			ofs << endl;
			ofs << "database.type = sqlite" << endl;
			ofs << "database.connect = \"<path to SQLite database file>\"" << endl;
			ofs << endl;
			ofs << "#     database.type = mysql" << endl;
			ofs << "#     database.connect = \"user=<MySQL user name>;password=<MySQL user password<;db=<MySQL database name>\"" << endl;
			ofs << endl;
			ofs << "# --- inputs (pdr) ---------------------------------------------------------" << endl;
			ofs << endl;
			ofs << "# input 1: an IMAP server" << endl;
			ofs << "input1.type = imap" << endl;
			ofs << "input1.server = \"<imap server name>\"" << endl;
			ofs << "input1.account = \"<imap server account>\"" << endl;
			ofs << "input1.password = \"<imap server password>\"" << endl;
			ofs << "input1.subject = \"<mail subject>\"" << endl;
			ofs << "input1.folder = \"INBOX\"" << endl;
			ofs << "input1.keep = true" << endl;
			ofs << "input1.expunge = true" << endl;
			ofs << endl;
			ofs << "# input 2: a Twitter account" << endl;
			ofs << "input2.type = twitter/rss" << endl;
			ofs << endl;
			ofs << "# ..." << endl;
			ofs << endl;
			ofs << "# --- outputs (pdx) --------------------------------------------------------" << endl;
			ofs << endl;
			ofs << "# output 1: a report" << endl;
			ofs << "output1.type = report" << endl;
			ofs << "output1.dependencies = " << endl;
			ofs << "output1.comment_begin = \"<!---\"" << endl;
			ofs << "output1.comment_end = \"--->\"" << endl;
			ofs << "output1.input_file = \"<input file name>\"" << endl;
			ofs << "output1.output_file = \"<output file name>\"" << endl;
			ofs << "output1.encoding = ISO-8859-1" << endl;
			ofs << endl;
			ofs << "# output 2 (could be a diagram)" << endl;
			ofs << endl;
			ofs << "# ..." << endl;
			ofs << endl;
		}

		encoded::cout << "file " << p.string() << " created successfully"<< endl;
		encoded::cout << "you should now edit this file to configure pdr and pdx" << endl;

		const string& cfg = config.GetConfigFile();
		if (p != cfg)
		{
			encoded::cout << "=== local configuration file found: " << cfg << endl;
			encoded::cout << "=== this local file overrides the created configuration file" << endl;
		}
	}
	CATCH_RETHROW("")
}

//=== main =================================================================
int main (int argc, char* argv[])
{
	try
	{
		// parse the command line and the config file
		auto_ptr<Config> config(ConfigFactory<ConfigImpl>::Create(argc, argv));

		// handle the very simple cases
		if (config->GetBoolOption("version"))
		{
			if (config->GetApplicationName() == "pdri")
			{
				encoded::cout
					<< "pdri - personal data recorder (interactive), " << VERSION << endl;
			}
			else
			{
				encoded::cout
					<< "pdr - personal data recorder, " << VERSION << endl;
			}
			encoded::cout
				<< "(C) 2010-2012 Torsten Mueller, Bern, Switzerland" << endl
				<< '(' << BUILD << ')' << endl
				<< "This program is free software under the terms of the GNU General Public License v2 or later." << endl;
			return 0;
		}

		if (config->GetBoolOption("help"))
		{
			if (config->GetApplicationName() == "pdri")
			{
				encoded::cout
					<< "pdri - personal data recorder (interactive)" << endl
					<< "usage: pdri [options]" << endl;
			}
			else
			{
				encoded::cout
					<< "pdr - personal data recorder" << endl
					<< "usage: pdr [options] [args]" << endl
					<< "[args] are transformed into one big input expression" << endl;
			}
			config->Help(encoded::cout);
			return 0;
		}

		if (config->GetBoolOption("create-config"))
		{
			create_config(*config);
			return 0;
		}

		// not very simple, work with the database
		{
			auto_ptr<Database> database(DBFactory::Create(*config));

			if (config->GetBoolOption("interactive"))
			{
				auto_ptr<Input>(InputFactory().GetInputInteractive())->Do(*config, *database);
				return 0;
			}

			// list operations
			{
				bool abort = false;
				if (config->GetBoolOption("list-collections"))
				{
					database->ListCollections();
					abort = true;
				}
				if (config->GetBoolOption("list-rejections"))
				{
					database->ListRejections();
					abort = true;
				}
				if (abort)
					return 0;
			}

			// add or delete operations
			{
				bool abort = false;
				if (config->GetBoolOption("delete-all-collections"))
				{
					database->DeleteAllCollections();
					abort = true;
				}
				else
				{
					const vector<string>& collections = config->GetVectorOption("delete-collection");
					foreach (const string& collection, collections)
					{
						try
						{
							database->DeleteCollection(collection);
						}
						catch (const Xception& x)
						{
							encoded::cerr << x.Message(Xception::complete) << endl;
						}
						abort = true;
					}
				}
				{
					const vector<string>& collections = config->GetVectorOption("add-collection");
					foreach (const string& collection, collections)
					{
						try
						{
							database->AddCollection(collection);
						}
						catch (const Xception& x)
						{
							encoded::cerr << x.Message(Xception::complete) << endl;
						}
						abort = true;
					}
				}
				if (config->GetBoolOption("delete-all-rejections"))
				{
					database->DeleteAllRejections();
					abort = true;
				}
				if (abort)
					return 0;
			}

			// check if we still have relevant command line
			// options or expressions, if so we ignore all the
			// configured inputs and outputs and do strictly
			// what's told on the command line
			bool has_action_option = false;
			{
				vector<string> except;
				except.push_back("verbose");
				except.push_back("config");
#ifndef _WIN32
				except.push_back("browser");
#endif
				except.push_back("debug");
				has_action_option = config->HasAnyOption(except);
			}

			// get input
			{
				InputFactory factory;
				const InputFactory::Inputs& inputs = factory.GetInputs(*config, has_action_option);
				foreach (const Input* pInput, inputs)
				{
					pInput->Do(*config, *database);
				}
			}

			// produce output
			{
				OutputFactory factory;
				const OutputFactory::Outputs& outputs = factory.GetOutputs(*config, has_action_option);
				foreach (const Output* pOutput, outputs)
				{
					pOutput->Do(*config, *database);
				}
			}
		}

		return 0;
	}
	catch (const Xception& x)
	{
		encoded::cerr << "*** fatal ***" << endl << x.Message(Xception::complete) << endl;
		return 1;
	}
	catch (const Poco::Exception& x)
	{
		encoded::cerr << "*** fatal ***" << endl << x.displayText() << endl;
		return 1;
	}
	catch (const std::exception& x)
	{
		encoded::cerr << "*** fatal ***" << endl << x.what() << endl;
		return 1;
	}
	catch (...)
	{
		encoded::cerr << "*** fatal ***" << endl << "internal error" << endl;
		return 1;
	}
}
