// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/xception.h"
#include "../libpdrx/conversions.h"
#include "../libpdrx/config.h"
#include "db.h"
#include "in_impl.h"

#include <Poco/SAX/SAXParser.h>
#include <Poco/SAX/SAXException.h>
#include <Poco/SAX/ErrorHandler.h>
#include <Poco/SAX/ContentHandler.h>
#include <Poco/SAX/Attributes.h>
#include <Poco/SAX/InputSource.h>

using namespace Poco;
using namespace Poco::XML;

//=== XMLFile ==============================================================
XMLFile::XMLFile (const string& option_key)
	: FileInputImpl(option_key, "xml")
{
}

namespace XMLFileLocal {

	// error_handler
	class error_handler: public ErrorHandler {
		public:
		error_handler ();
		virtual void fatalError (const SAXException& exc);
		virtual void error (const SAXException& exc);
		virtual void warning (const SAXException& exc);
		protected:
		void handle (const string& s, const SAXException& exc);
	};

	error_handler::error_handler ()
	{
	}

	void error_handler::fatalError (const SAXException& exc)
	{
		handle("XML fatal", exc);
	}

	void error_handler::error (const SAXException& exc)
	{
		handle("XML error", exc);
	}

	void error_handler::warning (const SAXException& exc)
	{
		handle("XML warning", exc);
	}

	void error_handler::handle (const string& s, const SAXException& exc)
	{
		string msg(exc.displayText());
		string::size_type pos = msg.find("Exception: ");
		if (pos != string::npos)
			msg.erase(0, pos + 11);
		THROW(s + ": " + msg);
	}

	// content_handler
	class content_handler: public ContentHandler {
		SAXParser& m_parser;

	public:
		enum State {none, pdr, pdr_collection, pdr_collection_item};
		typedef stack<State> StateStack;
		StateStack m_state;
		string m_current_collection;
		string m_current_collection_type;
		string m_current_collection_purpose;
		string m_current_collection_unit;
		Database& m_database;
		Database::CollectionsItems& m_items;

	public:
		content_handler (SAXParser& parser, Database& database, Database::CollectionsItems& items);
		virtual void characters (const XMLChar ch[], int start, int length) {}
		virtual void startDocument () {}
		virtual void endDocument () {}
		virtual void startElement (const XMLString& uri, const XMLString& localName, const XMLString& qname, const Attributes& attrList);
		virtual void endElement (const XMLString& uri, const XMLString& localName, const XMLString& qname);
		virtual void startPrefixMapping (const XMLString& prefix, const XMLString& uri) {}
		virtual void endPrefixMapping (const XMLString& prefix) {}
		virtual void ignorableWhitespace (const XMLChar ch[], int start, int length) {}
		virtual void processingInstruction (const XMLString& target, const XMLString& data) {}
		virtual void setDocumentLocator (const Locator* loc) {}
		virtual void skippedEntity (const XMLString& name) {}
	};

	content_handler::content_handler (SAXParser& parser, Database& database, Database::CollectionsItems& items)
		: m_parser(parser)
		, m_state()
		, m_current_collection()
		, m_current_collection_type()
		, m_current_collection_purpose()
		, m_current_collection_unit()
		, m_database(database)
		, m_items(items)
	{
		m_state.push(none);
	}

	// the idea is to parse several XMLformats using one parser, so we
	// have to distinguish these formats, we use the name of the first
	// element for that, this element sets the m_format variable

		void none_start_pdr (content_handler& ch, const XMLString& uri, const Attributes& attrList)
		{
			ch.m_state.push(content_handler::pdr);
		}

		void pdr_start_collection (content_handler& ch, const XMLString& uri, const Attributes& attrList)
		{
			ch.m_state.push(content_handler::pdr_collection);

			ch.m_current_collection = attrList.getValue(uri, "name");
			ch.m_current_collection_type = attrList.getValue(uri, "type");
			ch.m_current_collection_purpose = attrList.getValue(uri, "purpose");
			ch.m_current_collection_unit = attrList.getValue(uri, "unit");

			if (ch.m_current_collection != "*" && ch.m_current_collection != "#")
				ch.m_database.AddCollection(ch.m_current_collection + ',' + ch.m_current_collection_type[0] + ',' + ch.m_current_collection_purpose + ',' + ch.m_current_collection_unit);
		}

		void pdr_collection_start_item (content_handler& ch, const XMLString& uri, const Attributes& attrList)
		{
			ch.m_state.push(content_handler::pdr_collection_item);

			ptime timestamp(not_a_date_time);
			{
				string s(attrList.getValue(uri, "datetime"));
				trim(s);
				try
				{
					if (s.find('T') != string::npos)
						timestamp = from_iso_string(s);
					else
						timestamp = lexical_cast<ptime>(s);
				}
				catch (bad_lexical_cast)
				{
					THROW(format("value error, %s is not a datetime") % s);
				}
			}

			any a;
			{
				string s(attrList.getValue(uri, "value"));
				if (ch.m_current_collection_type == "numeric")
				{
					try
					{
						a = lexical_cast<double>(s);
					}
					catch (bad_lexical_cast)
					{
						THROW(format("value error, %s is not a double") % s);
					}
				}
				else
				{
					if (ch.m_current_collection_type == "ratio")
					{
						try
						{
							a = lexical_cast<Ratio>(s);
						}
						catch (bad_lexical_cast)
						{
							THROW(format("value error, %s is not a Ratio") % s);
						}
					}
					else
					{
						if (ch.m_current_collection_type == "text")
						{

		// regardless of our native encoding and the encoding
		// used in the XML file the encoding here is strict UTF-8,
		// so we don't decode here anything

							a = s;
						}
						else
							; // ignore
					}
				}
			}

			ch.m_items.insert(Database::CollectionsItems::value_type(ch.m_current_collection, Database::CollectionItem(timestamp, a)));
		}

	void content_handler::startElement (const XMLString& uri, const XMLString& localName, const XMLString& qname, const Attributes& attrList)
	{
		typedef void (*CallbackProc) (content_handler& , const XMLString& , const Attributes& );

		struct Data {
			State state;
			const char* keyword;
			CallbackProc proc;
		};

		static const Data data[] = {
			{none,			"pdr",		none_start_pdr},
			{pdr,			"collection",	pdr_start_collection},
			{pdr_collection,	"item",		pdr_collection_start_item}
			// ...
		};

		for (size_t i = 0; i < sizeof(data) / sizeof(Data); i++)
		{
			if (m_state.top() == data[i].state && localName == data[i].keyword)
			{
				data[i].proc(*this, uri, attrList);
				break;
			}
		}
	}

		void end_default (content_handler& ch)
		{
			if (ch.m_state.top() != content_handler::none)
				ch.m_state.pop();
		}

		void pdr_collection_end_collection (content_handler& ch)
		{
			end_default(ch);

			ch.m_current_collection.clear();
			ch.m_current_collection_type.clear();
			ch.m_current_collection_purpose.clear();
			ch.m_current_collection_unit.clear();
		}

	void content_handler::endElement (const XMLString& uri, const XMLString& localName, const XMLString& qname)
	{
		typedef void (*CallbackProc) (content_handler& );

		struct Data {
			State state;
			const char* keyword;
			CallbackProc proc;
		};

		static const Data data[] = {
			{pdr,			"pdr",		end_default},
			{pdr_collection,	"collection",	pdr_collection_end_collection},
			{pdr_collection_item,	"item",		end_default}
			// ...
		};

		for (size_t i = 0; i < sizeof(data) / sizeof(Data); i++)
		{
			if (m_state.top() == data[i].state && localName == data[i].keyword)
			{
				data[i].proc(*this);
				break;
			}
		}
	}

} // namespace XMLFileLocal

void XMLFile::ProcessFile (const Config& config, Database& database, ifstream& ifs, Database::CollectionsItems& items) const throw (Xception)
{
	// parse XML
	try
	{
		SAXParser parser;

		XMLFileLocal::error_handler eh;
		parser.setErrorHandler(&eh);

		XMLFileLocal::content_handler ch(parser, database, items);
		parser.setContentHandler(&ch);

		EncodingNames names;
		GetEncodingNames(names);
		foreach (const string& name, names)
		{
			parser.addEncoding(name, &SpecificEncoding(name));
		}

		InputSource isrc(ifs);
		parser.parse(&isrc);
	}
	CATCH_RETHROW("")
	catch (...)
	{
		THROW("xml parse error");
	}
}
