// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace boost::filesystem;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/xception.h"
#include "../libpdrx/conversions.h"
#include "config_impl.h"
#include "db.h"
#include "out_impl.h"
#include "out_ftree.h"

#ifdef USE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

//=== OutputInteractive ====================================================
OutputInteractive::OutputInteractive (const string& option_key)
	: OutputImpl(option_key)
{
}

void OutputInteractive::Do (const Config& config, const Database& database) const throw (Xception)
{
	encoded::cout << "pdx " << VERSION << ", interactive mode (press q to quit)" << endl;

#ifdef USE_READLINE
	// disable auto-complete
	rl_bind_key('\t', rl_abort);

	// build the history file name depending on the config file name
	string history_filename;
	{
		path p(config.GetConfigFile());
		if (p.filename() == "pdrxrc")
			history_filename = (p.parent_path() / "pdx_history").string();
		else
			history_filename = (p.parent_path() / ".pdx_history").string();
	}

	// load the history file if we have one
	clear_history();
	read_history(history_filename.c_str());
#endif

	string line;
	while (true)
	{
#ifdef USE_READLINE
		char* buf = readline(Encode(string("> "), NativeEncoding()).c_str());
		if (!buf)
		{
			encoded::cout << endl;
			break; // Ctrl-D
		}
		line = Decode(string(buf), NativeEncoding());
		free(buf);
#else
		encoded::cout << "> ";
		if (!getline(encoded::cin, line))
			break;
#endif
		trim(line);
		if (line.empty())
			continue;

		if (line == "q" || line == ".q" || line == "quit" || line == "exit" || line == "bye")
			break;

#ifdef USE_READLINE
		add_history(line.c_str());
#endif

		try
		{
			if (line[0] == '?')
			{
				line.erase(0, 1);
				trim(line);
				FExpr::fImplFactory.Help(encoded::cout, line);
				continue;
			}

			bool dot = false;
			if (line[0] == '.')
			{
				dot = true;
				line.erase(0, 1);
			}

			auto_ptr<FTree> ft(Parse(line));
			if (dot)
				encoded::cout << *ft; // dump structure
			else
			{
				FContext context(config, database);
				encoded::cout << ft->Evaluate(context) << endl;
			}
		}
		catch (const Xception& x)
		{
			encoded::cerr << x.Message(Xception::complete) << endl;
		}
	}

#ifdef USE_READLINE
	// store the history file
	write_history(history_filename.c_str());
#endif
}
