// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "../libpdrx/encoding.h"
#include "../libpdrx/build.h"
#include "db.h"
#include "in.h"
#include "out.h"

//=== ConfigImpl ===========================================================
class ConfigImpl: public Config {

	public:

	ConfigImpl ();

	virtual void ModifyCmdLineOptions (parsed_options& po);
};

ConfigImpl::ConfigImpl ()
	: Config()
{
	m_od_visible.add_options()

		// common options
		("help,?",						"show this help screen and quit")
		("version,V",						"display version number and quit")
		("verbose,v",						"produce verbose output on stdout")
		("config,c", value<string>(),				"use specific configuration file")
		("interactive,i",					"start interactive")

		// database options
		("list-collections,l",					"list all available collections in the database")
		("add-collection,a", value<vector<string> >(),		"add a collection to the database, arg is a string like \"name[, n|r|t[, unit[, purpose]]]\"")
		("delete-collection,d", value<vector<string> >(),	"delete a collection from the database, arg is a string like \"name\"")
		("delete-all-collections,D",				"delete all collections from the database")
		("list-rejections,r",					"list all rejected expressions")
		("delete-all-rejections,R",				"delete all rejected expressions")

		// input options
		("expression,e", value<vector<string> >(),		"use arg as expression for database input")
		("txt,t", value<vector<string> >(),			"use txt file arg for database input")
		("csv,C", value<vector<string> >(),			"use csv file arg for database input")
		("xml,x", value<vector<string> >(),			"use xml file arg for database input")
		("none,n",						"don't use any configured input, just the command line")

		// output options
		("export,X", value<string>(),				"create an xml export file")
	;

	// in pdr we need a second options description for a hidden
	// parameter containing all the command line arguments
	m_od_cmdline.add_options() ("cmd_line_args", value<string>(), "");
	m_pod.add("cmd_line_args", -1); // unlimited
}

	struct fo_cmd_line_arg: public unary_function<option&, bool> {
		string& m_expr;
		fo_cmd_line_arg (string& expr)
			: m_expr(expr)
		{
		}
		result_type operator () (argument_type a)
		{
			if (a.string_key == "cmd_line_args")
			{
				if (!m_expr.empty())
					m_expr += ' ';
				m_expr += a.value[0];
				return true;
			}
			else
				return false;
		}
	};

	struct fo_expression: public unary_function<const option&, bool> {
		result_type operator () (argument_type a) const
		{
			return a.string_key == "expression";
		}
	};

void ConfigImpl::ModifyCmdLineOptions (parsed_options& po)
{
	// in pdr we transform all the given command line arguments (the
	// arguments, not the options) into one single expression, this
	// expression will then be added to the list of normal expressions
	// from -e, the hidden parameters "cmd_line_args" are removed here

	string expr;
	{
		vector<option>::iterator I = remove_if(po.options.begin(), po.options.end(), fo_cmd_line_arg(expr));
		if (I != po.options.end())
			po.options.erase(I, po.options.end());
	}

	if (!expr.empty())
	{
		vector<option>::iterator I = find_if(po.options.begin(), po.options.end(), fo_expression());
		if (I == po.options.end())
		{
			vector<string> values;
			values.push_back(expr);
			po.options.push_back(option("expression", values));
		}
		else
			(*I).value.push_back(expr);
	}
}

//=== main =================================================================
int main (int argc, char* argv[])
{
	try
	{
		InitEncodings();

		// parse the command line and the config file
		auto_ptr<Config> config(ConfigFactory<ConfigImpl>::Create(argc, argv));

		// handle the very simple cases
		if (config->GetBoolOption("version"))
		{
			cout	<< "pdr - personal data recorder, " << VERSION << endl
				<< "(C) 2010-2011 Torsten Mueller, Bern, Switzerland" << endl
				<< '(' << BUILD << ')' << endl
				<< "This program is free software under the terms of the GNU General Public License v2 or later." << endl;
			return 0;
		}

		if (config->GetBoolOption("help"))
		{
			cout << "pdr - personal data recorder" << endl;
			cout << "usage: pdr [options] [args]" << endl;
			cout << "args are transformed into one big input expression" << endl;
			config->Help(cout);
			return 0;
		}

		// not very simple, work with the database
		{
			auto_ptr<Database> database(DBFactory::Create(*config));

			if (config->GetBoolOption("interactive"))
			{
				auto_ptr<Input>(InputFactory().GetInputInteractive())->Do(*config, *database);
				return 0;
			}

			// list operations
			{
				bool abort = false;
				if (config->GetBoolOption("list-collections"))
				{
					database->ListCollections();
					abort = true;
				}
				if (config->GetBoolOption("list-rejections"))
				{
					database->ListRejections();
					abort = true;
				}
				if (abort)
					return 0;
			}

			// add or delete operations
			{
				bool abort = false;
				if (config->GetBoolOption("delete-all-collections"))
				{
					database->DeleteAllCollections();
					abort = true;
				}
				else
				{
					const vector<string>& collections = config->GetVectorOption("delete-collection");
					foreach (const string& collection, collections)
					{
						try
						{
							database->DeleteCollection(collection);
						}
						catch (const Xception& x)
						{
							cerr << x.Message() << ", ignoring this" << endl;
						}
						abort = true;
					}
				}
				{
					const vector<string>& collections = config->GetVectorOption("add-collection");
					foreach (const string& collection, collections)
					{
						try
						{
							database->AddCollection(collection);
						}
						catch (const Xception& x)
						{
							cerr << x.Message() << ", ignoring this" << endl;
						}
						abort = true;
					}
				}
				if (config->GetBoolOption("delete-all-rejections"))
				{
					database->DeleteAllRejections();
					abort = true;
				}
				if (abort)
					return 0;
			}

			// get input
			{
				InputFactory factory;
				set<string> msgs;
				const InputFactory::Inputs& inputs = factory.GetInputs(*config);
				foreach (const Input* pInput, inputs)
				{
					try
					{
						pInput->Do(*config, *database);
					}
					catch (const Xarning& w)
					{
						msgs.insert(w.Message());
					}
				}
				if (!msgs.empty())
				{
					string s;
					foreach (const string& msg, msgs)
					{
						if (!s.empty())
							s += '\n';
						s += msg;
					}
					throw Xarning(s);
				}
			}

			// produce output
			{
				OutputFactory factory;
				const OutputFactory::Outputs& outputs = factory.GetOutputs(*config);
				foreach (const Output* pOutput, outputs)
				{
					pOutput->Do(*config, *database);
				}
			}
		}

		return 0;
	}
	catch (const Xarning& e)
	{
		cerr << "*** warning: " << e.Message() << endl;
		return 0;
	}
	catch (const Xception& e)
	{
		cerr << "*** fatal: " << e.Message() << endl;
		return 1;
	}
	catch (...)
	{
		cerr << "*** fatal: internal error" << endl;
		return 1;
	}
}
