// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef __OUT_FUNC_H
#define __OUT_FUNC_H

class FContext;
class FTree;

//=== FImpl (abstract base class) ==========================================
//
// derived classes ...
//   - describe a concrete function implementation with it's parameters
//   - contain the specific code for it's evaluation
//
// defining a pdx function means ...
//   a) derive a new FImpl class (use one of the DEFINE-macros)
//   b) implement it's Evaluate method
//   c) register the new class in the factory (make the function known)
//
class FImpl {

	public:

	typedef vector<const type_info*> ParamTypes;
	typedef	vector<FTree*> Params;

	const type_info*	m_returntype;
	string			m_name;
	ParamTypes		m_paramtypes;

		FImpl (const type_info* pReturnType, const string& name, size_t n, ...);

	virtual	any Evaluate (FContext& context, const Params& params) const throw (Xception) =0;
};

//=== FImplFactory =========================================================
class FImplFactory {

	protected:

	struct less_impl: public binary_function<const FImpl*, const FImpl*, bool> {
		result_type operator () (first_argument_type a, second_argument_type b) const
		{
			return a->m_name < b->m_name;
		}
	};

	typedef multiset<FImpl*, less_impl> Impls;
	typedef map<const type_info*, string> TypeNames;

	Impls m_impls;
	TypeNames m_typenames;

	public:

		FImplFactory ();
	virtual ~FImplFactory ();

	virtual const FImpl* Match (const string& name, const FImpl::ParamTypes& paramtypes) const throw (Xception);
	virtual void Help (ostream& os, const string& name) const;
};

#endif
