// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "common.h"

#include <Poco/ASCIIEncoding.h>
#include <Poco/Latin1Encoding.h>
#include <Poco/Latin9Encoding.h>
#include <Poco/UTF8Encoding.h>
#include <Poco/UTF16Encoding.h>
#include <Poco/Windows1252Encoding.h>
#include <Poco/TextConverter.h>

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "encoding.h"

void InitEncodings ()
{
	Poco::TextEncoding& g = Poco::TextEncoding::global();
	g.add(Poco::SharedPtr<Poco::TextEncoding>(new Poco::ASCIIEncoding()));
	g.add(Poco::SharedPtr<Poco::TextEncoding>(new Poco::Latin1Encoding()));
	g.add(Poco::SharedPtr<Poco::TextEncoding>(new Poco::Latin9Encoding()));
	g.add(Poco::SharedPtr<Poco::TextEncoding>(new Poco::UTF8Encoding()));
	g.add(Poco::SharedPtr<Poco::TextEncoding>(new Poco::UTF16Encoding()));
	g.add(Poco::SharedPtr<Poco::TextEncoding>(new Poco::Windows1252Encoding()));
}

void GetEncodingNames (EncodingNames& names)
{
	static const char* n[] = {
		"ASCII",
		"ISO-8859-1", "Latin1", "Latin-1",
		"ISO-8859-15", "Latin9", "Latin-9",
		"UTF-8", "UTF8",
		"UTF-16", "UTF16",
		"Windows-1252", "CP1252"
	};

	names.clear();
	for (size_t i = 0; i < sizeof(n) / sizeof(const char*); i++)
		names.insert(n[i]);
}

Poco::TextEncoding& GetEncoding ()
{
	return Poco::TextEncoding::global();
}

Poco::TextEncoding& GetEncoding (const string& encoding)
{
	Poco::TextEncoding& g = Poco::TextEncoding::global();
	Poco::TextEncoding::Ptr p = g.find(encoding);
	return (p) ? *p : g;
}

string ConvertFrom (const string& text, Poco::TextEncoding& from)
{
	Poco::TextEncoding& g = Poco::TextEncoding::global();
	if (&from == &g)
		return text;
	else
	{
		string s;
		Poco::TextConverter(from, g).convert(text, s);
		return s;
	}
}

string ConvertTo (const string& text, Poco::TextEncoding& to)
{
	Poco::TextEncoding& g = Poco::TextEncoding::global();
	if (&to == &g)
		return text;
	else
	{
		string s;
		Poco::TextConverter(g,to).convert(text, s);
		return s;
	}
}
