// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifdef USE_SQLITE

#include "../libpdrx/common.h"

#include <Poco/Data/Common.h>
#include <Poco/Data/SQLite/Connector.h>
#include <Poco/Data/SQLite/SQLiteException.h>
#include <Poco/Data/RecordSet.h>

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace Poco::Data;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "db_impl.h"

//=== SQLiteDatabase =======================================================
SQLiteDatabase::SQLiteDatabase (const string& connect, bool verbose)
	: m_connect(connect)
	, m_verbose(verbose)
	, m_pSession(NULL)
	, m_collectionTypes()
{
	SessionFactory::instance().add(SQLite::Connector::KEY, new SQLite::Connector());
}

void SQLiteDatabase::Connect () throw (Xception)
{
	// open session
	try
	{
		if (m_verbose)
			cout << "connecting to database" << endl;

		m_pSession = new Session(SQLite::Connector::KEY, m_connect);
	}
	catch (const SQLite::CantOpenDBFileException& x)
	{
		if (m_pSession)
		{
			delete m_pSession;
			m_pSession = NULL;
		}
		throw Xception(format("cannot open database: %s, %s") % m_connect % x.displayText());
	}

	// check presence of a schema, create schema if neccessary
	bool has_schema = false;
	try
	{
		if (m_verbose)
			cout << "checking schema" << endl;

		(*m_pSession) << "select * from TCollections;", now;
		has_schema = true;
	}
	catch (...)
	{
		// no schema!
	}

	if (!has_schema)
	{
		if (m_verbose)
			cout << "creating new schema" << endl;

		try
		{
			DBTransactor transactor(m_pSession);

			(*m_pSession) << "create table TCollections "
					 "("
						"name varchar(64) not null primary key,"
						"type char(1) not null,"
						"tbl_name varchar(64) not null"
					 ");", now;

			(*m_pSession) << "create table TRejected "
					 "("
						"i integer primary key autoincrement,"
						"t char(19) not null,"
						"expr varchar(1024) not null"
					 ");", now;

			(*m_pSession) << "create table C0 "
					 "("
						"t char(19) not null primary key,"
						"v double not null"
					 ");", now;

			(*m_pSession) << "create table C1 "
					 "("
						"t char(19) not null primary key,"
						"v varchar(1024) not null"
					 ");", now;

			(*m_pSession) << "insert into TCollections values ('*','n','C0');", now;
			(*m_pSession) << "insert into TCollections values ('#','t','C1');", now;

			transactor.Commit();
		}
		catch (...)
		{
			throw Xception("could not create schema");
		}
	}
}

SQLiteDatabase::~SQLiteDatabase ()
{
	if (m_verbose)
		cout << "disconnecting from database" << endl;

	if (m_pSession && m_pSession->isConnected())
		m_pSession->close();
	delete m_pSession;

	SessionFactory::instance().remove(SQLite::Connector::KEY);
}

void SQLiteDatabase::ListCollections () throw (Xception)
{
	if (m_verbose)
		cout << "listing collections" << endl;

	struct fill {
		const string& m_s;
		fill (const string& s)
			: m_s(s)
		{
		}
		string operator () (int idx)
		{
			static const size_t widths[] = {6, 8, 6, 7, 20}; // real column widths
			size_t width = widths[idx];
			string result(m_s);
			while (result.length() < width)
				result += ' ';
			if (result.length() > width)
				result.erase(width - 1);
			return result + ' '; // + 1 space as column delimiter
		}
	};

	try
	{
		Statement select(*m_pSession);
		select << "select name, type,tbl_name from TCollections order by name;";
		select.execute();

		RecordSet rs(select);
		if (rs.moveFirst())
		{
			cout << "  " << fill("name")(0) << fill("type")(1) << fill("table")(2) << fill("recs")(3) << fill("first")(4) << "last" << endl;

			do {
				cout << "  " << fill(rs.value(0).convert<string>())(0);

				switch (rs.value(1).convert<char>())
				{
					case 'n':	cout << fill("numeric")(1); break;
					case 'r':	cout << fill("ratio")(1); break;
					case 't':	cout << fill("text")(1); break;
				}

				cout << fill(rs.value(2).convert<string>())(2);

				int count;
				string min, max;
				(*m_pSession) << "select count(*),min(t),max(t) from " << rs.value(2).convert<string>() << ";", into(count), into(min), into(max), now;
				cout << fill(lexical_cast<string>(count))(3) << fill(min)(4) << max << endl;

			} while (rs.moveNext());
		}
	}
	catch (...)
	{
	}
}

void SQLiteDatabase::AddCollection (const string& name) throw (Xception) // "name[, n|r|t]"
{
	if (m_verbose)
		cout << "adding collection" << endl;

	try
	{
		// split collection name and type
		regex rx("\\s*([^,\\s]+)(?:\\s*,\\s*([nrt])\\s*)?");
		smatch mr;
		if (!regex_match(name, mr, rx))
			throw Xception(format("illegal collection name: %s") % name);
		string n(mr[1]);
		string t(mr[2]);
		if (t.empty())
			t = "n"; // numeric

		// check for built-in collection
		if (n == "*" || n == "#")
			throw Xception(format("cannot create built-in collection: %s") % n);

		// check for existing collection
		int count = 0;
		(*m_pSession) << "select count(*) from TCollections where name=:name;", use(n), into(count), now;
		if (count != 0)
			throw Xception(format("collection already exists: %s") % n);

		// build new table name
		string tbl_name;
		(*m_pSession) << "select max(tbl_name) from TCollections;", into(tbl_name), now;
		tbl_name.erase(0, 1); // C
		int i = lexical_cast<int>(tbl_name);
		tbl_name = (format("C%d") % ++i).str();

		{
			// create collection
			DBTransactor transactor(m_pSession);

			string sql("create table ");
			sql += tbl_name;
			sql += " (t char(19) not null primary key,";
			switch (t[0])
			{
				case 'n':	sql += "v double not null"; break;
				case 'r':	sql += "n double not null,d double not null"; break;
				case 't':	sql += "v varchar(1024) not null"; break;
			}
			sql += ");";
			(*m_pSession) << sql, now;
			(*m_pSession) << "insert into TCollections values (:n,:t,:tbl_name);", use(n), use(t), use(tbl_name), now;

			transactor.Commit();
		}
	}
	catch (...)
	{
		throw Xception(format("could not create collection: %s") % name);
	}
}

void SQLiteDatabase::DeleteCollection (const string& name) throw (Xception)
{
	if (m_verbose)
		cout << "deleting collection" << endl;

	try
	{
		// check for built-in collection
		if (name == "*" || name == "#")
			throw Xception(format("cannot delete built-in collection: %s") % name);

		// check for unknown collection
		int count = 0;
		(*m_pSession) << "select count(*) from TCollections where name=:name;", use(name), into(count), now;
		if (count != 1)
			throw Xception(format("unknown collection: %s") % name);

		{
			// delete collection
			DBTransactor transactor(m_pSession);

			string tbl_name;
			(*m_pSession) << "select tbl_name from TCollections where name=:name;", use(name), into(tbl_name), now;
			(*m_pSession) << "delete from TCollections where name = :name;", use(name), now;
			(*m_pSession) << "drop table " << tbl_name << ";", now;

			transactor.Commit();
		}
	}
	catch (...)
	{
		throw Xception(format("could not delete collection: %s") % name);
	}
}

void SQLiteDatabase::DeleteAllCollections () throw (Xception)
{
	if (m_verbose)
		cout << "deleting all collections" << endl;

	try
	{
		DBTransactor transactor(m_pSession);

		{
			Statement select(*m_pSession);
			select << "select name,tbl_name from TCollections where name not in ('*','#');";
			select.execute();

			RecordSet rs(select);
			if (rs.moveFirst())
			{
				do {
					(*m_pSession) << "drop table " << rs.value(1).convert<string>() << ";", now;
				} while (rs.moveNext());
			}
		}

		(*m_pSession) << "delete from TCollections where name not in ('*','#');", now;
		(*m_pSession) << "delete from C0;", now;
		(*m_pSession) << "delete from C1;", now;

		transactor.Commit();
	}
	catch (...)
	{
		throw Xception("could not delete collections");
	}
}

char SQLiteDatabase::GetCollectionType (const string& name) throw (Xception)
{
	try
	{
		CollectionTypes::const_iterator I = m_collectionTypes.find(name);
		if (I != m_collectionTypes.end())
			return (*I).second;

		string type;
		(*m_pSession) << "select type from TCollections where name=:name;", use(name), into(type), now;
		m_collectionTypes.insert(CollectionTypes::value_type(name, type[0]));

		return type[0];
	}
	catch (...)
	{
		throw Xception(format("unknown collection: %s") % name);
	}
}

void SQLiteDatabase::ListRejections () throw (Xception)
{
	if (m_verbose)
		cout << "listing rejections" << endl;

	struct fill {
		const string& m_s;
		fill (const string& s)
			: m_s(s)
		{
		}
		string operator () (int idx)
		{
			static const size_t widths[] = {20}; // real column widths
			size_t width = widths[idx];
			string result(m_s);
			while (result.length() < width)
				result += ' ';
			if (result.length() > width)
				result.erase(width - 1);
			return result + ' '; // + 1 space as column delimiter
		}
	};

	try
	{

		Statement select(*m_pSession);
		select << "select t,expr from TRejected order by t;";
		select.execute();

		RecordSet rs(select);
		if (rs.moveFirst())
		{
			cout << "  " << fill("timestamp")(0) << "expression" << endl;
			do {
				cout << "  " << fill(rs.value(0).convert<string>())(0) << rs.value(1).convert<string>() << endl;
			} while (rs.moveNext());
		}
	}
	catch (...)
	{
	}
}

void SQLiteDatabase::DeleteAllRejections () throw (Xception)
{
	if (m_verbose)
		cout << "deleting all rejections" << endl;

	try
	{
		(*m_pSession) << "delete from TRejected;", now;
	}
	catch (...)
	{
		throw Xception("could not delete rejections");
	}
}

void SQLiteDatabase::AddCollectionElements (const CollectionElements& elements) throw (Xception)
{
	try
	{
		DBTransactor transactor(m_pSession);

		foreach (const CollectionElement& element, elements)
		{
			// get collection type and table name
			string type, tbl_name;
			(*m_pSession) << "select type,tbl_name from TCollections where name=:name;", use(element.m_name), into(type), into(tbl_name), now;

			// insert or update?
			const string& t = lexical_cast<string, ptime>(element.m_timestamp);
			int count;
			(*m_pSession) << "select count(*) from " << tbl_name << " where t=:t;", use(t), into(count), now;

			// do it
			if (type == "n")
			{
				double data = any_cast<double>(element.m_data);
				if (count == 0)
					(*m_pSession) << "insert into " << tbl_name << " values (:t,:v);", use(t), use(data), now;
				else
					(*m_pSession) << "update " << tbl_name << " set v=:v where t=:t;", use(data), use(t), now;
			}
			else
			{
				if (type == "r")
				{
					Ratio data = any_cast<Ratio>(element.m_data);
					if (count == 0)
						(*m_pSession) << "insert into " << tbl_name << " values (:t,:n,:d);", use(t), use(data.m_numerator), use(data.m_denominator), now;
					else
						(*m_pSession) << "update " << tbl_name << " set n=:n,d=:d where t=:t;", use(data.m_numerator), use(data.m_denominator), use(t), now;
				}
				else
				{
					if (type == "t")
					{
						string data = any_cast<string>(element.m_data);
						if (count == 0)
							(*m_pSession) << "insert into " << tbl_name << " values (:t,:v);", use(t), use(data), now;
						else
							(*m_pSession) << "update " << tbl_name << " set v=:v where t=:t;", use(data), use(t), now;
					}
				}
			}
		}

		transactor.Commit();
	}
	catch (...)
	{
		throw Xception("could not insert or update values");
	}
}

void SQLiteDatabase::AddRejected (const ptime& timestamp, const string& expr) throw (Xception)
{
	try
	{
		const string& t = lexical_cast<string, ptime>(timestamp);
		(*m_pSession) << "insert into TRejected values (null,:timestamp,:expr);", use(t), use(expr), now;
	}
	catch (...)
	{
		throw Xception("could not insert expression");
	}
}

#endif // USE_SQLITE
