// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace boost::filesystem;

#include "datatypes.h"
#include "config.h"

//=== Config ===============================================================
Config::Config ()
	: m_config_file()
	, m_od1(256)
	, m_od2()
	, m_pod()
	, m_vm()
	, m_now(second_clock::local_time())
{
}

Config::~Config ()
{
}

void Config::Init (int argc, char* argv[]) throw (Xception)
{
	try
	{
		// evaluate command line
		command_line_parser cp(argc, argv);
		m_od2.add(m_od1); // a bit tricky
		cp.options(m_od2);
		cp.positional(m_pod);
		parsed_options po(cp.run());

		ModifyCmdLineOptions(po);

		store(po, m_vm);
	}
	catch (...)
	{
		throw Xception("error in command line options");
	}

	try
	{
		// evaluate environment (HOME)
		{
			struct X {
				static string func (const string& var)
				{
					return (var == "HOME") ? "home" : "";
				}
			};
			options_description od;
			od.add_options() ("home", "") ;
			store(parse_environment(od, X::func), m_vm);
		}

		// get configuration filename
		m_config_file = GetFilenameOption("config");
		if (m_config_file.empty())
		{
			// .pdrx in current directory?
			path p(".");
			p /= ".pdrx";
			if (ifstream(p.file_string().c_str()).good())
				m_config_file = p.file_string();
			else
			{
				// .pdrx in HOME-directory?
				p = GetStringOption("home");
				p /= ".pdrx";
				if (ifstream(p.file_string().c_str()).good())
					m_config_file = p.file_string();
			}
		}

		// evaluate configuration file
		ifstream ifs(m_config_file.c_str(), ios::in);
		if (ifs.good())
		{
			options_description od;
			parsed_options po(&od);
			string line;
			while (getline(ifs, line))
			{
				static regex rx_cfgfileline("^\\s*([A-Za-z0-9_.-]+)\\s*=\\s*(.*)\\s*$");
				smatch mr;
				if (regex_match(line, mr, rx_cfgfileline, boost::match_not_dot_newline))
				{
					string option_name(mr[1]);

					string option_value(mr[2]);
					trim(option_value);
					if (!option_value.empty() && option_value[0] == '"')
						option_value.erase(0, 1);
					if (!option_value.empty() && option_value[option_value.length() - 1] == '"')
						option_value.erase(option_value.length() - 1, 1);

					if (od.find_nothrow(option_name, false))
					{
						foreach (option& o, po.options)
						{
							if (o.string_key == option_name)
							{
								o.value.push_back(option_value);
								break;
							}
						}
					}
					else
					{
						od.add_options() (option_name.c_str(), value<string>(), "");
						vector<string> values;
						values.push_back(option_value);
						po.options.push_back(option(option_name, values));
					}
				}
			}
			store(po, m_vm);
		}

		notify(m_vm);

		// check if we have a -n parameter
		const string& n = GetStringOption("now");
		if (!n.empty())
		{                          //  CCYY     -MM      -DD     [-hh         :mm        [:ss]]
			static const regex rx("([0-9]+)(-[0-9]+)(-[0-9]+)(-[0-9]+)?(\\:[0-9]+)?(\\:[0-9]+)?");
			smatch mr;
			regex_match(n, mr, rx);

			string t(n);
			if (mr.length(6) > 0)
				;
			else
			{
				if (mr.length(5) > 0)
					t += ":00"; // add seconds
				else
				{
					if (mr.length(4) > 0)
						t += ":00:00"; // add minutes and seconds
					else
						t += " 00:00:00"; // add complete time
				}
			}
			t += ".000"; // add milliseconds

			m_now = time_from_string(t);
		}
	}
	catch (...)
	{
		throw Xception("error in configuration file");
	}
}

void Config::ModifyCmdLineOptions (parsed_options& po)
{
}

void Config::Help (ostream& os) const
{
	m_od1.print(os);
}

bool Config::HasOption (const string& option) const
{
	try
	{
		return !m_vm[option].empty();
	}
	catch (...)
	{
	}

	return false;
}

bool Config::GetBoolOption (const string& option) const
{
	try
	{
		const variable_value& v = m_vm[option];
		if (!v.empty())
		{
			const any& a = v.value();
			if (a.type() == typeid(string))
			{
				string s(any_cast<string>(a));
				trim(s);
				to_upper(s);
				return s.empty() || s == "YES" || s == "TRUE" || s == "1";
			}
			else
			{
				if (a.type() == typeid(bool))
					return any_cast<bool>(a);
			}
		}
	}
	catch (...)
	{
	}

	return false;
}

string Config::GetStringOption (const string& option) const
{
	try
	{
		const variable_value& v = m_vm[option];
		if (!v.empty())
			return any_cast<string>(v.value());
	}
	catch (...)
	{
	}

	return string();
}

double Config::GetDoubleOption (const string& option) const
{
	try
	{
		const variable_value& v = m_vm[option];
		if (!v.empty())
			return lexical_cast<double>(any_cast<string>(v.value()));
	}
	catch (...)
	{
	}

	return 0.0;
}

vector<string> Config::GetVectorOption (const string& option) const
{
	try
	{
		const variable_value& v = m_vm[option];
		if (!v.empty())
			return any_cast<vector<string> >(v.value());
	}
	catch (...)
	{
	}

	return vector<string>();
}

string Config::GetFilenameOption (const string& option) const
{
	try
	{
		const variable_value& v = m_vm[option];
		if (!v.empty())
		{
			string filename(any_cast<string>(v.value()));
			if (filename.find("~/") == 0)
			{
				filename.erase(0, 2);
				path p(any_cast<string>(m_vm["home"].value()));
				p /= filename;
				filename = p.file_string();
			}
			return filename;
		}
	}
	catch (...)
	{
	}

	return string();
}

ptime Config::GetNow () const
{
	// in fast mode we need constant value for "now" for getting usable
	// keys in the selection cache, if we are not fast, we work with
	// real timestamps

	// m_now is thought to be an option in future

	return (GetBoolOption("fast") || !GetStringOption("now").empty()) ? m_now : second_clock::local_time();
}

const string& Config::GetConfigFile () const
{
	return m_config_file;
}
