// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

#include <Poco/Data/Common.h>
#include <Poco/Data/SQLite/Connector.h>
#include <Poco/Data/SQLite/SQLiteException.h>
#include <Poco/Data/RecordSet.h>

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace Poco::Data;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "db_impl.h"

//=== SQLiteDatabase =======================================================
SQLiteDatabase::SQLiteDatabase (const string& connect, bool verbose, bool fast)
	: m_connect(connect)
	, m_verbose(verbose)
	, m_fast(fast)
	, m_pSession(NULL)
	, m_collectionMetaInfos()
	, m_selections()
{
	SessionFactory::instance().add(SQLite::Connector::KEY, new SQLite::Connector());
}

void SQLiteDatabase::Connect () throw (Xception)
{
	// open session
	try
	{
		if (m_verbose)
			cout << "connecting to database" << endl;

		m_pSession = new Session(SQLite::Connector::KEY, m_connect);
	}
	catch (const SQLite::CantOpenDBFileException& )
	{
		if (m_pSession)
		{
			delete m_pSession;
			m_pSession = NULL;
		}
		throw Xception(format("cannot open database: %s") % m_connect);
	}

	// check presence of a schema, get collection meta information
	try
	{
		if (m_verbose)
			cout << "checking schema" << endl;

		Statement select(*m_pSession);
		select << "select name,type,tbl_name from TCollections;";
		select.execute();

		RecordSet rs(select);
		if (rs.moveFirst())
		{
			do {
				CollectionMetaInfo cmi;
				cmi.m_collection = rs.value(0).convert<string>();
				cmi.m_type = rs.value(1).convert<string>()[0];
				cmi.m_tblname = rs.value(2).convert<string>();
				m_collectionMetaInfos.insert(cmi);
			} while (rs.moveNext());
		}
	}
	catch (const SQLite::InvalidSQLStatementException& )
	{
		throw Xception("no schema in database");
	}
}

SQLiteDatabase::~SQLiteDatabase ()
{
	if (m_verbose)
		cout << "disconnecting from database" << endl;

	if (m_pSession && m_pSession->isConnected())
		m_pSession->close();
	delete m_pSession;

	SessionFactory::instance().remove(SQLite::Connector::KEY);
}

	inline string selection_key (const string& collection, const ptime& begin, const ptime& end)
	{
		return collection + '|' + lexical_cast<string>(begin) + '|' + lexical_cast<string>(end);
	}

void SQLiteDatabase::Select (shared_ptr<Selection>& selection, const string& collection, const ptime& begin, const ptime& end) const throw (Xception)
{
	if (m_fast)
	{
		Selections::const_iterator I = m_selections.find(selection_key(collection, begin, end));
		if (I != m_selections.end())
		{
			selection = (*I).second; // just return what we know
			return;
		}
	}

	selection->resize(extents[0][2]); // clear

	CollectionMetaInfo c;
	c.m_collection = collection;
	CollectionMetaInfos::const_iterator I = m_collectionMetaInfos.find(c);
	if (I == m_collectionMetaInfos.end())
		throw Xception(format("unknown collection: %s") % collection);
	const CollectionMetaInfo& cmi = *I;

	try
	{
		// build the SQL string
		string sql("select t,");
		switch (cmi.m_type)
		{
			case 'r':	sql += "n,d"; break;
			default:	sql += "v"; break;
		}
		sql += " from " + cmi.m_tblname;

		string where;
		{
			if (begin != not_a_date_time)
				where += "t>='" + lexical_cast<string, ptime>(begin) + "'";

			if (end != not_a_date_time)
			{
				if (!where.empty())
					where += " and ";
				where += "t<'" + lexical_cast<string, ptime>(end) + "'";
			}
		}
		if (!where.empty())
			sql += " where " + where;

		sql += " order by 1;";

		// execute the statement
		Statement select(*m_pSession);
		select << sql;
		select.execute();

		// fetch data
		RecordSet rs(select);
		if (rs.moveFirst())
		{
			size_t row = 0;
			do {
				selection->resize(extents[row + 1][2]);

				// we return datetime values as string,
				// this is more handy for later grouping
				// during aggregation
				(*selection)[row][0] = rs.value(0).convert<string>();

				switch (cmi.m_type)
				{
					case 'n':	(*selection)[row][1] = rs.value(1).convert<double>(); break;
					case 'r':	(*selection)[row][1] = Ratio(rs.value(1).convert<double>(), rs.value(2).convert<double>()); break;
					default:	(*selection)[row][1] = rs.value(1).convert<string>(); break;
				}

				row++;

			} while (rs.moveNext());
		}

		if (m_fast)
			m_selections.insert(Selections::value_type(selection_key(collection, begin, end), selection));
	}
	catch (...)
	{
		throw Xception("error selecting data");
	}
}
