// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "../libpdrx/encoding.h"
#include "db.h"
#include "in_impl.h"

#include <Poco/Net/NetException.h>
#include <Poco/Net/POP3ClientSession.h>
#include <Poco/Net/MailMessage.h>

using namespace Poco;
using namespace Poco::Net;

//=== Pop3MailClient =======================================================
Pop3MailClient::Pop3MailClient (const string& option_key)
	: InputImpl(option_key)
{
}

void Pop3MailClient::Do (const Config& config, Database& database) const throw (Xception)
{
	try
	{
		bool verbose = config.GetBoolOption("verbose");
		if (verbose)
			cout << "looking for mail" << endl;

		// get configuration data
		const string& server = config.GetStringOption(m_option_key + ".server");
		if (server.empty())
			throw Xception(format("missing specification in configuration file: %s.server)") % m_option_key);
		const string& account = config.GetStringOption(m_option_key + ".account");
		if (account.empty())
			throw Xception(format("missing specification in configuration file: %s.account)") % m_option_key);
		const string& password = config.GetStringOption(m_option_key + ".password");
		const string& subject = config.GetStringOption(m_option_key + ".subject");
		bool keep = config.GetBoolOption(m_option_key + ".keep");

		// open connection and log in
		POP3ClientSession session(server);
		session.login(account, password);

		// process available mails
		POP3ClientSession::MessageInfoVec messages;
		session.listMessages(messages);

		bool data = false;
		bool rejects = false;
		POP3ClientSession::MessageInfoVec::const_iterator I = messages.begin();
		while (I != messages.end())
		{
			int id = (*I++).id;

			MailMessage message;
			session.retrieveMessage(id, message);

			// check subject
			string s(message["Subject"]);
			{
				trim(s);
				static const regex rx_coded_subject("^=\\?.+\\?Q\\?(.+)\\?=$");
				smatch mr;
				if (regex_match(s, mr, rx_coded_subject, boost::match_not_dot_newline))
					s = mr[1];
			}
			if (s != subject)
				continue;

			data = true;

			// extract timestamp
			ptime timestamp(not_a_date_time);
			{
				string d(message["Date"]);
				trim(d);
				smatch mr;
				static const regex rx_date("^(?:.+, )?([0-9]+) ([A-Za-z]+) ([0-9]+) ([0-9]+:[0-9]+:[0-9]+) ([+-][0-9]{2})([0-9]{2})$");
				if (regex_match(d, mr, rx_date, boost::match_not_dot_newline))
				{
					string s(mr[1]);
					if (s.length() == 1)
						s.insert(0, "0");
					stringstream ss;
					ss << mr[3] << '-' << mr[2] << '-' << s << ' ' << mr[4];
					ss >> timestamp;
				}
			}

			// get message body and decode it correctly (this means make it UTF-8)
			string charset;
			{
				const string& content_type = message.getContentType();
				static const regex rx_charset("charset=([^;]+)");
				smatch mr;
				if (regex_search(content_type, mr, rx_charset, boost::match_not_dot_newline))
					charset = mr[1];
			}
			stringstream ss;
			if (charset.empty())
				ss << message.getContent(); // should be 7bit ASCII
			else
				ss << ConvertFrom(message.getContent(), GetEncoding(charset));

			// parse line by line, every line an expression
			string line;
			while (getline(ss, line))
			{
				try
				{
					Database::CollectionElements elements;
					Parse(line, timestamp, verbose, elements);
					database.AddCollectionElements(elements);
					database.AddInserted(timestamp, line);
				}
				catch (const Xception& )
				{
					database.AddRejected(timestamp, line);
					rejects = true;
				}
			}

			// delete if neccessary
			if (!keep)
				session.deleteMessage(id);
		}
		if (verbose && !data)
			cout << "    no data on server" << endl;

		// close connection
		session.close();

		if (rejects)
			throw Xarning("!!! at least one expression has been rejected, try -r to list rejections !!!");
	}
	catch (const POP3Exception& )
	{
		throw Xception("could not retrieve mail data from pop3 server");
	}
	catch (const NetException& )
	{
		throw Xception("could not retrieve mail data from pop3 server");
	}
}
