/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */




package opale.m2d;
import java.awt.*;
import javax.swing.*;

//Nouveaut PAC
import java.awt.Graphics.*;


/**
* This class is a tool box to draw in a coordinate system.
* The methods are similar to the methods of the class <code>java.awt.Graphics</code> but it works in a 2D-coordinate system.
* @author O.C.
* @since Opale-2d 0.1
*/

public class Pen2D
{

private CoordSystem repere; //le repere2D

  //Modif du Graphics en Graphics2D
private Graphics2D g; //crayon en coordonnes reelles 

private Color couleur;

/**
* Constructs from a <code>Graphics</code> instance and a <code>CoordSystem</code> instance.
* @param Graphics g
* @param CoordSystem repere
*/
public Pen2D(Graphics g,CoordSystem repere){

this.repere=repere;
this.g= (Graphics2D) g;

//Modif du PAc afin de mettre pas defaut l'antialiasing
//this.g.setRenderingHint(RenderingHints.KEY_ANTIALIASING,RenderingHints.VALUE_ANTIALIAS_ON );
//this.g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,RenderingHints.VALUE_TEXT_ANTIALIAS_ON );

//clip();
}

/**
* Gets instance of <code>Graphics</code> using to draw.
* @return Graphics 
* @since Opale-2d 0.11
*/
public Graphics2D getGraphics()
	{
	return g;
	}

/**
* Clips the drawing actions in all the coordinates systems.
*/
public void clip()
{
g.setClip(repere.getXinf(),repere.getYinf(),repere.getWidth()+1,repere.getHeight()+1);
}

/**
* Clips the drawing actions in a box
* @param double x1
* @param double y1
* @param double w
* @param double h
*/
/*public void clip(double x1,double y1,double largeur,double hauteur)
{
g.setClip(repere.X(x1,y1),repere.Y(x1,y1),repere.X(largeur,0),repere.Y(hauteur));
}*/

/**
* Draws a point.
* @param double x, x-coordinate
* @param double y, y-coordinate
*/
public void drawPoint(double x,double y){drawLine(x,y,x,y);}


/**
* Draws a line.
* @param double x1,y1,x2,y2
*/
public void drawLine(double x1,double y1,double x2,double y2)
{
//System.out.println(""+repere.X(x1,y1)+" "+repere.Y(x1,y1));
g.drawLine(repere.X(x1,y1),repere.Y(x1,y1),repere.X(x2,y2),repere.Y(x2,y2));
}

/**
* Draws a circle centered in a specified point.
* @param double x,y, coordinates of the center
* @param double a, size of the circle
*/
public void drawCircle(double x,double y,double r)
{
if ( r >= 0 )
	g.drawOval(repere.X(x-r,y+r),repere.Y(x-r,y+r),Math.abs(repere.X(r*2,0) - repere.X(0,0)),Math.abs(repere.Y(0,r*2) - repere.Y(0,0)));
else {} // lancer une exception!??
}

/**
* Draw a "plus" centered in a specified point.
* @param double x,y, coordinates of the center
* @param int a, size in pixels
*/
public void drawPlus(double x,double y,int taille )
{
g.drawLine(repere.X(x,y)-taille,repere.Y(x,y),repere.X(x,y)+taille,repere.Y(x,y));
g.drawLine(repere.X(x,y),repere.Y(x,y)-taille,repere.X(x,y),repere.Y(x,y)+taille);
}

/**
* Draw a crux centered in a specified point.
* @param double x,y, coordinates of the center
* @param int a, size in pixels
*/
public void drawX(double x,double y,int taille )
{
g.drawLine(repere.X(x,y)-taille,repere.Y(x,y)-taille,repere.X(x,y)+taille,repere.Y(x,y)+taille);
g.drawLine(repere.X(x,y)-taille,repere.Y(x,y)+taille,repere.X(x,y)+taille,repere.Y(x,y)-taille);
}

/**
* Draw a square centered in a specified point.
* @param double x,y, coordinates of the center
* @param int a, size of the square in pixels
*/
public void drawSquare(double x,double y,int taille )
	{
	g.drawRect(repere.X(x,y)-(taille>>1),repere.Y(x,y)-(taille>>1),taille,taille);
	}

/**
* Draws an GObject2D. It performs a OIJ changement if necessary.
* @param GObject2D gob.
* @since Opale-2d 0.11
*/
public void draw(GObject2D gob)
	{
	if (gob instanceof Object2D)
		{
		Object2D ob = (Object2D) gob.clone();
		ob.changeOIJ(repere.getOIJ());
		((GObject2D) ob).draw(this);
		}
	else
		gob.draw(this);
	}
		

/**
* Draw a circle centered in a specified point.
* @param double x,y, coordinates of the center
* @param int a, size of the circle in pixels
*/
public void drawCircle(double x,double y,int taille)
	{
	g.drawOval(repere.X(x,y)-(taille>>1),repere.Y(x,y)-(taille>>1),taille,taille);
	}

/**
* Draw a diamond centered in a specified point.
* @param double x,y, coordinates of the center
* @param int a, size of the diamond in pixels
*/
public void drawDiamond(double x,double y,int taille)
	{
	int xR = repere.X(x,y);
	int yR = repere.Y(x,y);
	
	g.drawLine(xR+taille,yR,xR,yR+taille);
	g.drawLine(xR,yR+taille,xR-taille,yR);
	g.drawLine(xR-taille,yR,xR,yR-taille);
	g.drawLine(xR,yR-taille,xR+taille,yR);
	}
/**
* Draw an equilateral triangle, centered in a specified point and oriented to the top.
* @param double x,y, coordinates of the center
* @param int a, the half length of the triangle in pixels
*/
public void drawTriangleUp(double x,double y,int a)
	{
	int xR = repere.X(x,y);
	int yR = repere.Y(x,y);
	int h = (int) Math.round(0.58 * a);
	g.drawLine(xR,yR-2*h,xR+a,yR+h);
	g.drawLine(xR+a,yR+h,xR-a,yR+h);
	g.drawLine(xR-a,yR+h,xR,yR-2*h);
	}

/**
* Draw an equilateral triangle, centered in a specified point and oriented to the bottom.
* @param double x,y, coordinates of the center
* @param int a, the half length of the triangle in pixels
*/
public void drawTriangleDown(double x,double y,int a)
	{
	int xR = repere.X(x,y);
	int yR = repere.Y(x,y);
	int h = (int) Math.round(0.58 * a);
	g.drawLine(xR,yR+2*h,xR+a,yR-h);
	g.drawLine(xR+a,yR-h,xR-a,yR-h);
	g.drawLine(xR-a,yR-h,xR,yR+2*h);
	}

/**
* Draw an equilateral triangle, centered in a specified point and oriented to the left.
* @param double x,y, coordinates of the center
* @param int a, the half length of the triangle in pixels
*/
public void drawTriangleLeft(double x,double y,int a)
	{
	int xR = repere.X(x,y);
	int yR = repere.Y(x,y);
	int h = (int) Math.round(0.58 * a);
	g.drawLine(xR+h,yR-a,xR+h,yR+a);
	g.drawLine(xR+h,yR+a,xR-2*h,yR);
	g.drawLine(xR-2*h,yR,xR+h,yR-a);
	}

/**
* Draw an equilateral triangle, centered in a specified point and oriented to the right.
* @param double x,y, coordinates of the center
* @param int a, the half length of the triangle in pixels
*/
public void drawTriangleRight(double x,double y,int a)
	{
	int xR = repere.X(x,y);
	int yR = repere.Y(x,y);
	int h = (int) Math.round(0.58 * a);
	g.drawLine(xR-h,yR-a,xR-h,yR+a);
	g.drawLine(xR-h,yR+a,xR+2*h,yR);
	g.drawLine(xR+2*h,yR,xR-h,yR-a);
	}

/**
* Draw a tick parallel to the X-axis.
* @pram double x, x-coordinates
* @param double y, y-coordinates
* @param int, size of the tick in pixels
*/
public void drawXtick(double x,double y, int s)
	{
	int xR = repere.X(x,y);
	int yR = repere.Y(x,y);
	
	g.drawLine(xR-s,yR,xR+s,yR);
	}

/**
* Draw a tick parallel to the Y-axis.
* @pram double x, x-coordinates
* @param double y, y-coordinates
* @param int, size of the tick in pixels
*/
public void drawYtick(double x,double y, int s)
	{
	int xR = repere.X(x,y);
	int yR = repere.Y(x,y);
	
	g.drawLine(xR,-s+yR,xR,s+yR);
	}

/**
* Draw a line between two points.
* @param Point2D a, the origin
* @param Point2D b, the end of the line 
*/
public void drawLine(Point2D a, Point2D b)
	{
	drawLine(a.x,a.y,b.x,b.y);
	}

/**
* Draw a horizontal line.
* @param double y, y-coordinate of the line to be drawn.
*/
public void drawHLine(double y)
	{
	drawLine(repere.getXmin(),y,repere.getXmax(),y);
	}

/**
* Draw a vertical line.
* @param double x, x-coordinate of the line to be drawn.
*/
public void drawVLine(double x)
	{
	drawLine(x,repere.getYmin(),x,repere.getYmax());
	}

/**
* Draw an arrow.
* @param double xo, yo the virtual origin of the arrow
* @param double x,y the location of the arrow.
* @param int taille, the size of the arrow.
*/
public void drawArrow(double xo,double yo,double x, double y, int taille)
	{
		int xR = repere.X(x,y);
		int yR = repere.Y(x,y);
		int xoR = repere.X(xo,yo);
		int yoR = repere.Y(xo,yo);
		double norme = Vector2D.norm(xR-xoR,yR-yoR);

		//		System.out.println("norme =" +norme);
	 	if (norme != 0)
			{
			double nx = -(yR-yoR)/norme;
			double ny = (xR-xoR)/norme;
			//System.out.println("nx =" +nx);
			//System.out.println("ny =" +ny);
				
			double xh = xR - taille * ny;
			double yh = yR + taille * nx;
			g.drawLine((int) (xh+nx*5),(int) (yh+ny*5),xR,yR);
			g.drawLine((int) (xh-nx*5),(int) (yh-ny*5),xR,yR);

			}
		
	}

/**
* Draws the outline of the specified rectangle
* @param double x, the x coordinate of the rectangle to be drawn
* @param double y, the y coordinate of the rectangle to be drawn
* @param double w, the width of the rectangle to be drawn
* @param double h, the height of the rectangle to be drawn
*/
public void drawRect(double x, double y, double w, double h)
	{
	g.drawRect(repere.X(x,y),repere.Y(x,y),repere.X(w,0),repere.Y(0,h));
	}

/**
* Draws a text on a specified location.
* @param String, a text
* @param double x, y, a location in the coordinate system
*/
public void drawString(String text, double x, double y)
	{
	g.drawString(text,repere.X(x,y),repere.Y(x,y));
	}

/**
* Draws a text on a specified location.
* @param String, a text
* @param double x, y, a location in the coordinate system
* @param int dx, dy,  shift in screen pixels from the precedent specififed location
*/
public void drawString(String text, double x, double y, int dx, int dy)
	{
	g.drawString(text,repere.X(x,y)+dx,repere.Y(x,y)+dy);
	}
	

/**
* Sets the current color.
* @param Color couleur, a new color
*/
public void setColor(Color couleur)
	{
	this.couleur=couleur;
	g.setColor(couleur);
	}

/**
* Gets the current color.
* @return Color
*/
public Color getColor()
	{
	return couleur;
	}

/**
* Gets the coordinates system.
* @return CoordSystem .
*/
public final CoordSystem getCoordSyst()
	{
	return repere;
	}

/**
* Dessine tous les objets contenus dans une session Opale.
* @param OpaleSet, une session Opale.
*/
/*public final void draw(OpaleSet o)
	{
	int i;
	//legend.draw(this);
	for (i=0; i<o.size();i++)
		{
		if ( o.getObject(i) instanceof GObject2D )
			{
			GObject2D obj = (GObject2D)  o.getObject(i);
			draw(obj);
			}
		}
	}*/

}
 

