/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */


package opale.ode;

import opale.mathtools.*;
import opale.tools.*;
import java.io.*;

/**
* This abstract class represents a time scheme. One method must be definined in the sub classes : the method <code>void  forward(Equation eq)</code> that defines the time scheme.
* @since Opale-ODE 0.1
* @author O.C.
*/

public abstract class TimeScheme extends ObjectODE
{
private double tinit, tmax, tc,dtmax,dtmin;	
private double dt;
private int Nstep;
private Problem pb;
private StepsizeControl stepcontrol;

//private DVect[] past;
private DVect present;
private DVect past;
//private int npast;
	
private DVect dpresent;
private DVect dpast;


/**
* It takes in argument an equation and it must compute the unknown at the current time (See <code>present()</code>). A l'entre dans cette mthode pour le calcul du temps n, la mthode present() renvoie l'inconnue au temps n juste prcdent.
* @param Equation, a equation.
* @deprecated use now <code>abstract public void forward()</code>
*/
public void forward(Equation eq) { }

/**
* This method must compute the unknown at the current time (See <code>present()</code>). A l'entre dans cette mthode pour le calcul du temps n, la mthode present() renvoie l'inconnue au temps n juste prcdent.
*/
abstract public void advance();



/**
* This method computes an error estimate of order 1 : this estimation corresponds to the explicit Euler method. It's possible to override it to change the estimation in order to improve adaptative step control.
*/
public double errorEstimate()
{
double  eps = 0, tmp;

for (int i=0;i<present().size();i++)
	{
	tmp = Math.abs((dt()/2)*(present().get(i) - past().get(i)));
	if (eps < tmp) eps = tmp;
	}
	
return eps;
}

	
public TimeScheme()
	{
	pb = null;
	tinit = 0;
	tmax = 1;
	Nstep = 11;
	dtmin = 0;
	dtmax=1e9;
	}		

/**
* Renvoie les inconnues des temps prcdents. La mthode npast() qui doit tre implmente par une classe fille renvoie le nombre de pas de temps stocks. Attention, les valeurs sont ranges par temps dcroissants : par exmple si on est au temps n, l'appel past(0) renvoie l'inconnue au temps n-1, past(1) au temps n-2...
ATTENTION : l'instance de DVect renvoye contient en premire composante le temps courant, puis les composantes suivantes (gal  la dimension du problme trait) contiennent les valeurs de l'inconnue. 
* @param int i, l'indice du temps dsir.
* @return DVect, l'inconnue au temps souhait.
*/ 
/*final public DVect past(int i)
	{
	return past[i];
	}
*/

/**
* Get the present unknown.
* <P> WARNING : the returned instance of <code>DVect</code> contains :
* <UL>
* <LI> first composante, the current time.
* <LI> then next composantes the value of the unknown.
* @return DVect, the unknown vector at present.
*/ 
final public DVect present()
	{
	return present;
	}
		
/**
* Gets the  unknown at the past time step.
* @return DVect, the unknown vector at past
*/ 
final public DVect past()
	{
	return past;
	}
/**
* Gets the  derivate of the unknown at the past time step.
* @return DVect, the unknown vector at past
*/ 
final public DVect der_past()
	{
	return dpast;
	}


/**
* Gets the  derivate of the unknown at the present time step.
* @return DVect, the unknown vector at present
*/ 
final public DVect der_present()
	{
	return dpresent;
	}


/**
* Sets the time interval for the study.
* @param double t0, initial time.
* @param double t1, final time.
*/
public final void set(double t0, double t1)
	{
	tinit = Math.min(t0,t1);
	tmax = Math.max(t0,t1);
	}
	
/**
* Sets the number of time step. The minimum is 3 and it is impossible to choose less.
* @param int n, number of time step.
*/
public final void setNstep(int n)
	{
	Nstep = Math.max(3,Math.abs(n));
	}

/**
* Gets the number of time step.
* @return int, number of time step.
*/
public final int getNstep()
	{
	return Nstep;
	}
	
/**
* Associates a problem to this Time scheme. It is not possible to solve a problem without attach it to a time scheme.
* @param Problem, a problem.
*/ 	
public final void setPb(Problem pb)
	{
	this.pb = pb;
	}
	

	
/**
* Gets the problem for this time scheme.
* @return Problem.
*/
public final Problem getPb()
	{
	return pb;
	}

/**
* Returns the equation linked to the problem.
* @return Equation.
* @since Opale-ode 0.15
*/
public final Equation getEquation()
	{
	return pb.getEqn();
	}




/**
* Sets a step control method.
* @param StepsizeControl.
*/
public final void setStepsizeControl(StepsizeControl s)
	{
	stepcontrol = s;
	}

/**
* Returns the initial time.
* @return double, initial time.
*/
public final double tmin()
	{
	return tinit;
	}
/**
* Returns final time.
* @return double, final time.
*/
public final double tmax()
	{
	return tmax;
	}
	
/**
* Get the maximum value for the first step dt.
* @return double, the maximum value.
* @since Opale-ODE 0.11
*/
public final double dtmax()
	{
	return dtmax;
	}
	
/**
* Get the minimum value for the first step dt.
* @return double, the minimum value.
* @since Opale-ODE 0.11
*/
public final double dtmin()
	{
	return dtmin;
	}
/**
* Set the maximum value for the first step dt.
* @param double, the maximum value.
* @since Opale-ODE 0.11
*/
public final void dtmax(double dtm)
	{
	dtmax = dtm;
	}
	
/**
* Set the minimum value for the first step dt.
* @param double, the minimum value.
* @since Opale-ODE 0.11
*/
public final void dtmin(double dtm)
	{
	dtmin= dtm;
	}
	

/**
* Get the current time step.
* @return double, the time step.
*/
public final double dt()
	{
	return dt;
	}
/**
* Set the current time step.
* @param double, the time step.
*/
public final void dt(double dt)
	{
	this.dt = dt;
	}
	
/**
* Returns the current time..
* @return double, the current time.
*/
public final double time()
	{
	return tc;
	}	


/**
* Initializes the resolution (memory allocation, time step computing ...). To call necessary before a resolution.
*/
public void init()
	{
	if (pb == null)
	{
//	System.err.println("Le schma n'est associ  aucun problme.\nArret anormal du calcul.");
	System.err.println("No problem for this scheme !.\nStop.");
	System.exit(-1);
	}
	
	if (stepcontrol == null) stepcontrol = new NoStepsizeControl(this);
	else stepcontrol.setTS(this); 
	/*if (npast()<0) 
	{
	System.err.println("Erreur le nombre de valeurs temporelles  stocker est ngatif!\nArret anormal du calcul.");
	System.exit(-1);
	}
	
	int i;
	past = new DVect[npast()];
	for (i=0;i<npast();i++)
		past[i] = new DVect(1+pb.dim());*/
	present = new DVect(pb.dim());
	past = new DVect(pb.dim());
	present.set(pb.getInit());
	past.set(pb.getInit());
	dt = (tmax-tinit)/(Nstep-1);
	if (dt < dtmin) dt = dtmin;
	else if (dt>dtmax) dt = dtmax;
	tc = tinit;

	dpresent = new DVect(pb.dim());
	dpast = new DVect(pb.dim());
	dpresent.set(pb.getEqn().derivs(time(),present()));
	dpast.set(dpresent);
	}

/**
* Up to date the time scheme : to increment the time step, store the vectors etc.... This method must  be called at each end of time step in a solver routine : for example, the class Problem uses this method.
*/
public boolean update()
	{
	//Debug.print("Dans TimeScheme.update() dt = "+dt);
	dpresent.set(pb.getEqn().derivs(time(),present()));
	//Debug.print ("epsilon = "+errorEstimate());
	if (stepcontrol.update() == true)
		{
		tc+=dt;
		past.set(present);
		dpast.set(dpresent);
		return true;
		}
	else
		{
		Stdio.println("Step time refused. New step time : dt = "+dt);
		present.set(past);
		dpresent.set(dpast);
		return false;
		}
	/*present().set(0,tc);
	int i,np=npast();
	if (npast()>0)
		{
		for (i=0;i<np-1;i++)
			past[np-1-i].set(past[np-i-2]);
		past[0].set(present());
		}*/
	}	
	
public int readKeyWord(String word,StreamReader f,OpaleSet ode) 
  throws java.io.IOException, InvalidFormatException
	{
	int type,i;
	if ( word.equals("N"))
		{
		Nstep = (int) f.nextDouble();
		return 0;
		}
	else if ( word.equals("tmin"))
		{
		tinit =  f.nextDouble();
		return 0;
		}
	else if ( word.equals("tmax"))
		{
		tmax = f.nextDouble();
		return 0;
		}
	 else if ( word.equals("stepcontrol"))
		 {
		String sval = f.next();
                 if ( (!ode.contains(sval)) || (ode.getObject(sval)) == null) throw new InvalidFormatException("Incorrect data file : indentifier "+sval+" doesn't exist !!");
                 if (!(ode.getObject(sval) instanceof StepsizeControl) ) throw new InvalidFormatException("Incorrect data file : indentifier "+sval+" is not a step control method !!");
                 stepcontrol = (StepsizeControl) ode.getObject(sval);
                 return 0;
                 }
	return -1;
	}
	
	
public void write(PrintWriter f, OpaleSet ode)
	{
	f.println("{");
	f.println("tmin "+tinit);
	f.println("tmax "+tmax);
	f.println("N "+Nstep);
        if (stepcontrol !=null )
		if (stepcontrol.getId()!=null) f.println("stepcontrol "+stepcontrol.getId());
	f.println("}");
	}


	
}
