/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */

package opale.matrix;



 

import java.io.*;
import opale.tools.*;
import opale.mathtools.DVect;

/**
* This class implements a full matrix with real coefficients.
* @since Opale-Matrix 0.1
* @author J.M., O.C.
* @date 03/2001
*/
public final class Matrix  extends OpaleObject implements  Cloneable
{
private double[][] a;
private int n,m;	//row and column dimension



/** 
* Constructs an matrix without dimensions neither defined double arrays. 
*/

public Matrix () 
	{
	super();
	this.m = -1;
	this.n = -1;
	a = null;
	}



/** 
* Construct an n X m matrix of zeros. 
* @param int n, row dimension.
* @param int m, colum dimension.
*/

public Matrix (int n, int m) 
	{
	super();
	this.m = m;
	this.n = n;
	a = new double[n][m];
	}

/** 
* Constructs a matrix from a double array.
* @param double[][], the coefficients
*/

public Matrix(double[][] x)
	{ 
	super();
	a = new double [n=x.length][m=x[0].length] ;
	if (Debug.On) Debug.print("n = "+n);
	if (Debug.On) Debug.print("m = "+m);
	
	
	for (int i=0;i<n;i++)
		for (int j=0;j<m;j++)
			this.a[i][j]= x[i][j];
	} 

/**
* Get the number of rows.
* @return int, the row dimension.
*/
public int getRowDim() { return n; }
/**
* Get the number of columns.
* @return int, the column dimension.
*/
public int getColumnDim() { return m; }

/** 
* Get a coefficient of the matrix.
* @param int i, row index.
* @param int j, column index.
* @return double, coeff(i,j)
*/
public double get(int i, int j) 
	{
	if (Assert.enabled) Assert.check((i<n) && (j<m));

	return a[i][j];
	}
	
	
/** 
* Set a coefficient of the matrix.
* @param int i, row index.
* @param int j, column index.
* @param double x, the value of coeff(i,j)
*/
public void set(int i, int j, double x) 
	{
	if (Assert.enabled) Assert.check((i<n) && (j<m));
	a[i][j] = x;
	}

/** 
* Make a copy of the current matrix.
* @return Matrix, the copy
*/
public Matrix copy () 
	{
	Matrix C = new Matrix(n,m);
	for (int i = 0; i < n; i++) 
		for (int j = 0; j < m; j++) 
			C.a[i][j] = a[i][j];
	return C;
	}
/**
* Add an another matrix A to the current matrix.
* @param Matrix A, an another matrix
* @return Matrix, the result
*/
public Matrix add(Matrix A)
	{
	if (Assert.enabled) Assert.check((n == A.n) && (m == A.m));
	for (int i=0;i<n;i++)
		for (int j=0;j<m;j++)
			a[i][j]+=A.a[i][j];
	return this;
	}

/**
* Multiply the current matrix by a vector.
* @param DVect x, a vector
* @param DVect, the result A*x
*/
public DVect mul(final DVect x)
	{
	if (Assert.enabled) Assert.check(m == x.size());
	
	DVect y = new DVect(n);
	
	for (int i=0;i<n;i++)
		{
		double s=0.;
		for (int j=0;j<m;j++)
			s+=a[i][j]*x.get(j);
		y.set(i,s);
		}
	return y;

	}
	
	
/**
* Return the matrix transpose.
* @return Matrix.
*/
public Matrix transpose()
	{
	Matrix t = new Matrix (n,m);
       	for (int i = 0; i < n; i++) 
		for (int j = 0;j<m;j++) 
			t.a[i][j]=this.a[j][i];
	return t;				        
	
	}


/**
* Get the internal two-dimensional array.
*  @return double[][],    Pointer to the array of matrix elements.
*/

public double[][] getInternalArray() 
	{
	return a;
	}
	
/**
* Get a copy of array of matrix elements.
*  @return double[][],  the array of matrix elements.
*/

public double[][] getArrayCopy() 
	{
	double[][] tab = new double[n][m];
	for (int i = 0; i < n; i++) 
		for (int j = 0; j < m; j++) 
			tab[i][j] = a[i][j];
	return tab;
	}

/**
* This method applies a function on all the matrix elements.
* @return IyFx f, a function.
* @since Opale-matrix 0.11
*/
public void appFunc(final opale.mathtools.IyFx f) 
	{
	for (int i = 0; i < n; i++) 
		for (int j = 0; j < m; j++) 
			a[i][j] = f.val(a[i][j]);
	}


/**
* Print the matrix.
*/
public String toString()
	{
	int i,j;
	double aff;
	//String s="";
	StringBuffer s=new StringBuffer(n*m); // bad approximation of size !!
	
	for (i = 0; i <n; i++) 
		{
		for ( j = 0;j<m;j++) 
			{
			aff=a[i][j];
//	     		s+=aff;
//			s+=" ";
			s.append(aff+" ");
			}
//		s+="\n";
		s.append("\n");
		}
	return s.toString();
	}
	
	
	
/**
* Create and return a copy of this object.
* @return Object a copy of this object.
* @exception  OutOfMemoryError if not enough memory.
* @see        java.lang.Cloneable
* @since Opale-Matrix 0.11
*/
public Object clone()
	{
		Matrix copy = (Matrix) super.clone();
		copy.n = n;
		copy.m = m;
		copy.a = getArrayCopy();
		
		return copy;
/*	catch (CloneNotSupportedException e) // ne devrait jamais arriver
		{
		throw new InternalError();
		}
*/   	}
	
	
/**
* Reads a matrix in a stream.
* @see <code>opale.tools.OpaleObject</code>
*/
public int readKeyWord(String word,StreamReader f,OpaleSet ode)  throws java.io.IOException,InvalidFormatException
	{
	if ( word.equals("dim"))
		{
		n = (int) f.nextDouble();
		m = (int) f.nextDouble();
		if (n<1 || m<1) throw new InvalidFormatException("Matrix dimensions must be positive integer values !!!");
		a = new double[n][m];
		return 0;
		}
	else if (word.equals("values"))
		{
		if(n<1 || m<1)  throw new InvalidFormatException("No matrix dimensions are defined !!!");
		for (int i=0;i<n;i++)
			{
			for (int j=0;j<m;j++)
				a[i][j] = f.nextDouble();
			}
		return 0;
		}		
	return -1;
	}



/**
* Writes a problem in a stream. The format is : 
* <P> { dim n m <br> values <br>  a00 a01 a02 .... a0m <br> a10 a11 ... a1m <br>...<br> an0 an1 ... anm<P>
* @see <code>opale.tools.OpaleObject</code>
*/
public void write(PrintWriter f, OpaleSet ode) throws InvalidFormatException
	{
	f.println("{");
	f.println("dim "+n+" "+m);
	f.println("values");
	for (int i=0;i<n;i++)
		{
		for (int j=0;j<m;j++)
			f.print(a[i][j]+" ");
		f.println("");
		}
	f.println("}");
	}

/**
* Tests if the specified object is an instance of Matrix and if it's equaled to this matrix.
* @param Object obj, an object
* @return <code>true</code> if the two objects are the same, false otherwise
* @since Opale-Matrix 0.13
*/
public boolean equals(Object obj)
	{
	if (obj instanceof Matrix)
		{
        	Matrix p = (Matrix) obj;
		if (n!= p.n || m!=p.m) return false;
		boolean test=true;
		for (int i=0;i<n;i++)
			for (int j=0;j<m;j++)
				test = test && (a[i][j] == p.a[i][j]);
				
		return test;
        	}
    	return false;
    }


	
/*public static void main(String[] arg)
	{
	Matrix m =new Matrix(2,3);
	System.out.println(m);
	Matrix mm = (Matrix) m.clone();
	System.out.println(m);
	}	*/
}
