/*
Copyright (C) 2000  Groupe Opale (http://www.opale.ovh.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You can visit the web site http://www.opale.ovh.org to obtain more informations about this program and/or to contact the coders.
*/
package opale.ode;

import opale.mathtools.*;
import opale.tools.*;
import java.io.*;
import java.util.Vector;

/**
* This class represents an ODE problem. An instance of this class must be associated (UML sens) to an instance of th class <code>TimeScheme</code> and to an instance of <code>Equation</code>. 
* <P> The method <code>solve()</code> runs the resolution of the problem
* @since Opale-ODE 0.1
* @author O.C.
*/

final public class Problem extends ObjectODE
{
public final boolean PRINT = false;
private TimeScheme sch;
private Equation eqn;
private DVect cdinit;
//private DVect[] inco;
private Vector inco;
//private double[] times;
private Vector times;
private int dim;
private boolean issolve = false;

/**
* Initialize an one dimensional problem.
*/
public Problem()
	{
	dim = 1;
	cdinit = new DVect(dim);
	}	

/**
* Constructs a problem with a given dimension.
* @param int dim, la dimension du problme.
*/
public Problem(int dim)
	{
	this.dim = Math.max(dim,1);
	cdinit = new DVect(this.dim);
	}	

	
/**
* Runs the resolution after verifying all the elments (time schem, equation...).
*/
public void solve()
	{
	issolve = false;	Stdio.printerrln("\n**************************************************");
//	Stdio.printerrln("************ Rsolution d'un problme ************");
	Stdio.printerrln("************ Solving a problem *******************");
	Stdio.printerrln("**************************************************");
	//Vrification des composants du problme.
//	Stdio.printerrln("*** Vrification des composants du problme ***");
	Stdio.printerrln("*** Check the problem ***");
	
	if (eqn == null)
		{
//		Stdio.printerrln("Vous n'avez pas associ d'quation au problme.\nArret anormal du calcul.");
		Stdio.printerrln("No equation for this problem !!\n Stop.");
		System.exit(-1);
		}
	if ( dim!= eqn.dim())
		{
//		Stdio.printerrln("L'quation et le problme n'ont pas la mme dimension.\nArret anormal.");
		Stdio.printerrln("Equation and problem have not same dimension !!.\n Stop.");
		System.exit(-1);
		}
		
	Stdio.printerrln(" --> Equation OK");

	if (sch == null)
		{
//		Stdio.printerrln("Vous n'avez pas associ de schma en temps au problme.\nArret anormal du calcul.");
		Stdio.printerrln("No time scheme !!\n Stop.");
		System.exit(-1);
		}
	Stdio.printerrln(" --> Scheme : "+sch);
		
	
	int i;
//	Stdio.printerrln(" --> Conditions initiales : ");
	Stdio.printerrln(" --> Initial condition : ");
		for (i=0;i<dim;i++)
			Stdio.printerrln("\t"+cdinit.get(i));
		
	Stdio.printerrln("*** Problem OK ***\n");
	
	
	//Rservation mmoire pour l'inconnue
	
//	Stdio.printerrln("*** Rservation mmoire pour l'inconnu ***\n");
	Stdio.printerrln("*** Memory allocation  ***\n");
//	inco = new DVect[sch.getNstep()];
	inco = new Vector(sch.getNstep());
//	for (i=0;i<sch.getNstep();i++)
		//inco[i] = new DVect(dim);
	//times = new double[sch.getNstep()];	
	times = new Vector(sch.getNstep());	

//	Stdio.printerrln("*** Initialisation du schma en temps ***\n");
	Stdio.printerrln("*** Time scheme initialisation ***\n");
	sch.init();
//	inco[0].set(sch.present());
//	times[0] = sch.tmin();
	inco.add(sch.present().clone());
	times.add(new Double(sch.tmin()));
	
//	Stdio.printerrln("*** Dbut du calcul ***");
	Stdio.printerrln("*** Starting the resolution ***");

	
	Chronometer.start();
		
//	for(i=1;(i<sch.getNstep()) && (sch.time() < sch.tmax());i++)
	for(i=1;sch.time() < sch.tmax();i++)
		{
		if (PRINT) 	
			{
			Stdio.println("->>Time : "+sch.time());
			Stdio.println("->>Step time : "+sch.dt());
			}
		sch.forward(eqn);
		boolean b = sch.update();
		if (b == true)
			{
//			inco[i].set(sch.present());
//			times[i] = sch.time();
			inco.add((DVect) sch.present().clone());
			times.add(new Double(sch.time()));
			}
		//else i--;
		if (PRINT) 	
			Stdio.println("->>Step time valid ?? "+b+"\n");
		
		//Stdio.printerrln("t = "+sch.time()+"\n"+sch.present());
		}
	Chronometer.stop();
//	Stdio.printerrln("*** Fin du calcul ***");
//	Stdio.printerrln("*** End of the calculation ***");
//	Stdio.printerrln("*** Temps pass pour ce calcul : "+Chronometer.time()+" ms\n");
	Stdio.printerrln("*** elapsed time : "+Chronometer.time()+" ms\n");
	Stdio.printerrln("**********************************************");
//	Stdio.printerrln("************ Fin de la rsolution ************");
	Stdio.printerrln("************ It's the end ! Ciao *************");
	Stdio.printerrln("**********************************************\n");
	issolve = true;
	}

/**
* Get the solution in a formatted String. The first column is the time, and the others the values x, y, z of the solution 
* @return String, the solution.
*/
public String printSol()
	{
	StringBuffer sol=new StringBuffer();
	if (issolve)
	{
	int i,j;
/*	for (i=0;i<sch.getNstep();i++)
		{
		sol.append( times[i]+" ");
		for (j=0;j<dim;j++)
			sol.append(inco[i].get(j)+" "); 
		sol.append("\n");
		}*/
	for (i=0;i<times.size();i++)
		{
		sol.append( ((Double)times.get(i)).doubleValue()+" ");
		DVect v = (DVect) inco.get(i);
		for (j=0;j<dim;j++)
			{
			sol.append(v.get(j)+" "); 
			}
		sol.append("\n");
		}
	}
//	else sol.append("Problme pas encore rsolu\n");
	else sol.append("Problem not solved \n");
	return sol.toString();	
	}
	
/**
* Get the solution in two dimensionnal array of double tab[dim+1][N+1]. 
* <br>dim is the dimension of the problem ; N is the number of time step.
* <br> t[0][j],0 <= j <= N represent the time steps ; t[i>=1][j],0 <= j <= N represent the composants of the solution.
* @return double[][], the solution , null if the problem is not solved.
*/
public double[][] toTab()
	{
	if (issolve)
	{
/*	double[][] tab = new double[dim+1][sch.getNstep()];
	for (i=0;i<sch.getNstep();i++)
		{
		tab[0][i]=times[i];
		for (j=1;j<=dim;j++)
			tab[j][i] = inco[i].get(j-1); 
		}*/
	double[][] tab = new double[dim+1][times.size()];
	for (int i=0;i<times.size();i++)
		{
		tab[0][i]=((Double)times.get(i)).doubleValue();
			
		DVect v = (DVect) inco.get(i);
		for (int j=1;j<=dim;j++)
			{
			tab[j][i] = v.get(j-1); 
			}
		}
	return tab;	
	}
	return null;
	}
	
public void writeSol(WFile wf)
	{
	wf.writeString(printSol());
	}
	
/**
* get the dimension problem.
* @return int, the dimension.
*/
public int dim()
	{
	return dim;
	}

/**
* Set the initial condition.
* @param DVect, the vector for the initial condition.
*/	
public void setInit(DVect init)
	{
	cdinit.set(init);
	}

/**
* Returns the initial condition.
* @return DVect, a vector.
*/
public DVect getInit()
	{
	return cdinit;
	}

/**
* Returns the number of step computed, -1 if the problem is not solved.
* @return int
* @since Opale-ODE 0.12.
*/
public int getNStep()
	{
	if (issolve)
	return times.size();
	else return -1;
	}

/**
* Associate an equation to this problem. The method verifies if the equation and the problem have same dimension.
* @param Equation, an equation.
*/
public void setEqn(Equation eq)
	{
	if ( dim!= eq.dim())
		{
//		Stdio.printerrln("L'quation et le problme n'ont pas la mme dimension.\nArret anormal.");
		Stdio.printerrln("Equation and problem have not same dimension !!.\n Stop.");
		System.exit(-1);
		}
	eqn = eq;
	}
	
/**
* Returns the equation of this problem.
* @return Equation, the equation.
*/
public Equation getEqn()
	{
	return eqn;
	}
	
/**
* Set a time scheme for the resolution of this problem. 
* @param TimeScheme, a time scheme.
*/
public void setTS(TimeScheme sch)
	{
	this.sch = sch;
	sch.setPb(this);
	}		
/**
* Get the time scheme of this problem. 
* @return TimeScheme, the time scheme.
*/
public TimeScheme getTS()
	{
	return sch;
	}		



/**
* Read a problem in a File
*/
public int readKeyWord(String word,RFile f,ODE ode)  throws java.io.IOException,MalformedFileException
	{
	int type,i;
	if ( word.equals("dim"))
		{
		type = f.nextToken();
		if (type != StreamTokenizer.TT_NUMBER) return -1;
		this.dim = (int) f.nval;
		cdinit = new DVect(this.dim);
		return 0;
		}
	else if (word.equals("y0"))
		{
		for(i=0;i<dim;i++)
			{
			type = f.nextToken();
			if (type != StreamTokenizer.TT_NUMBER) return -1;
			cdinit.set(i,f.nval);
			}
		return 0;
		}
		
	else if ( word.equals("sch"))
		{
		type = f.nextToken();
		if (type != StreamTokenizer.TT_WORD) throw new MalformedFileException("Incorrect data file : Error reading a time scheme in a problem");
		if ( (!ode.contains(f.sval)) || (ode.fromId(f.sval)) == null) throw new MalformedFileException("Incorrect data file : indentifier "+f.sval+" doesn't exist !!");
		if (!(ode.fromId(f.sval) instanceof TimeScheme) ) throw new MalformedFileException("Incorrect data file : indentifier "+f.sval+" is not a time scheme !!");
		setTS((TimeScheme) ode.fromId(f.sval));
		return 0;
		}
	else if ( word.equals("eqn"))
		{
		type = f.nextToken();
		if (type != StreamTokenizer.TT_WORD) throw new MalformedFileException("Incorrect data file : Error reading an equation in a problem");
		if ( (!ode.contains(f.sval)) || (ode.fromId(f.sval)) == null) throw new MalformedFileException("Incorrect data file : indentifier "+f.sval+"  doesn't exist!!");
		if (!(ode.fromId(f.sval) instanceof Equation) ) throw new MalformedFileException("Incorrect data file : indentifier  "+f.sval+" isn't an equation !!");
		setEqn((Equation) ode.fromId(f.sval));
		return 0;
		}
		
	return -1;
	}



/**
* Write a problem in a text file.
*/
public void writeFile(WFile f, ODE ode) throws MalformedFileException
	{
	f.writeln("{");
	if (!ode.contains(sch)) throw new MalformedFileException("Error writing a file : the time scheme "+sch+" doesn't exist !!");
	if (!(eqn instanceof ObjectODE))  throw new MalformedFileException("Error writing a file : the equation "+eqn+" of type "+eqn.getClass()+" can't write in a file !!");
	if (!ode.contains((ObjectODE) eqn)) throw new MalformedFileException("Error writing a file : the equation "+eqn+" doesn't exist !!");
	f.writeln("dim "+dim);
	f.writeString("y0");
	int i;
	for (i=0;i<dim;i++)
		f.writeString(" "+cdinit.get(i));
	f.writeln("");
	f.writeln("sch "+ode.getId(sch));
	f.writeln("eqn "+ode.getId((SEquation) eqn));
	
	f.writeln("}");
	}


/**
* Returns the stribng "Problem of ODE". Not very interesting !!
*/
public String toString()
	{
	return "Problem of ODE";
	}



}


	
