/*
Copyright (C) 2000  Olivier Cioni

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You can visit the web site http://www.opale.ovh.org to obtain more informations about this program and/or to contact the coders.
*/


package opale.module2d.object2d;
import opale.module2d.noyau2d.*;
import opale.tools.*;

/**
* Cette classe reprsente uen droite d'quation ax+by+c=0. Elle tend la classe <code>Object2D</code>
* @author O.C.
* @version 0.1
*/

public class Line2D extends Object2D implements Cloneable
{
public final static double EPSILON = 1e-15;
protected double a,b,c;	//les coeffs de l'equation ax+by+c=0


/**
* Constrcuteur par dfaut. Construit la droite d'quation x = 0.
* @since Opale 0.1
*/
public Line2D()
	{
	super();
	}
 
/**
* Constrcuteurde cope.
* @param Line2D l, la droite  copier.
* @since Opale 0.11
*/
public Line2D(Line2D l)
	{
	super(l.getOIJ());
	a = l.a;
	b = l.b;
	c = l.c;
	}
 

/**
* Constructeur d'une instance de Line2D  associ  un  rpre OIJ et d'quation donne.
* @param OIJ rep, le repre de dfinition de la droite.
* @param double a,b,c, les coefficients de l'quation ax+by+c=0.
* @since Opale 0.1
*/
public Line2D(OIJ rep,double a, double b, double c) throws LineException
	{
	super(rep);
	if ( (Math.abs(a)<=EPSILON) && (Math.abs(b)<=EPSILON) && (Math.abs(c)>EPSILON) )
		throw new LineException("Impossible de construire une droite d'quation c = 0, avec c<>0");

	this.a =a;
	this.b =b;
	this.c =c;
	}
	
/**
* Constructeur d'une instance de Line2D  associ  un  rpre OIJ et passant par deux points donns.
* @param OIJ rep, le repre de dfinition de la droite.
* @param double x1,y1,x2,y2, les coordonnes des deux points.
* @since Opale 0.1
*/
public Line2D(OIJ rep,double x1, double y1, double x2, double y2) throws LineException
	{
	super(rep);
	this.a =y2-y1;
	this.b =x1-x2;
	this.c =x2*y1-x1*y2;
	if ( (Math.abs(a)<=EPSILON) && (Math.abs(b)<=EPSILON) )
		throw new LineException("Impossible de construire une droite d'quation c = 0");
	}

/**
* Constructeur d'une instance de Line2D passant par deux points donns. Le repre de la droite est celui du premier point.
* @param Point2D p, Point2D q, les deux points.
* @since Opale 0.1
*/
public Line2D(Point2D p, Point2D q) throws LineException
	{
	super(p.getOIJ());
	Point2D pp = (Point2D) q.clone();
	pp.changeOIJ(p.getOIJ());
	
	this.a =pp.getY()-p.getY();
	this.b =p.getX()-pp.getX();
	this.c =pp.getX()*p.getY()-p.getX()*pp.getY();

	if ( (Math.abs(a)<=EPSILON) && (Math.abs(b)<=EPSILON) )
		throw new LineException("Impossible de construire une droite d'quation c = 0");
	}
	

/**
* Constructeur d'une instance de Line2D passant par un point et de vecteur directeur donn. Le repre de la droite est du premier point.
* @param Point2D p, le point.
* @param Vector2D v, le vecteur directeur.
* @since Opale 0.1
*/
public Line2D(Point2D p, Vector2D v)
	{
	super(p.getOIJ());
	Vector2D vv = (Vector2D) v.clone();
	vv.changeOIJ(p.getOIJ());
	
	this.a =vv.getY();
	this.b =-vv.getX();
	this.c =vv.getX()*p.getY()-p.getX()*vv.getY();
	}




/**
* Change l'quation de la droite. 
* @param double a,b,c, les coefficients de l'quation ax+by+c=0.
* @since Opale 0.1
*/
public void set(double a, double b, double c) throws LineException
	{
	if ( (Math.abs(a)<=EPSILON) && (Math.abs(b)<=EPSILON) )
		throw new LineException("Impossible de construire une droite d'quation c = 0");

	this.a =a;
	this.b =b;
	this.c =c;
	}

/**
* Change l'quation de la droite avec deux points.
* @param double x1,y1,x2,y2, les coordonnes des deux points.
* @since Opale 0.1
*/
public void set(double x1, double y1, double x2, double y2) throws LineException
	{
	double a = y2-y1;
	double b = x1-x2;
	double c = x2*y1-x1*y2;
	if ( (Math.abs(a)<=EPSILON) && (Math.abs(b)<=EPSILON))
		throw new LineException("Impossible de construire une droite d'quation c = 0");
	
	this.a =a;
	this.b =b;
	this.c =c;
	}

/**
* Change l'quation de la droite avec deux points.
* @param Point2D p,q les deux points.
* @since Opale 0.1
*/
public void set(Point2D p, Point2D q) throws LineException
	{
	Point2D pp = (Point2D) p.clone();
	Point2D qq = (Point2D) q.clone();
	
	pp.changeOIJ(getOIJ());
	qq.changeOIJ(getOIJ());
	
	double a =qq.getY()-pp.getY();
	double b =pp.getX()-qq.getX();
	double c =qq.getX()*pp.getY()-pp.getX()*qq.getY();
	if ( (Math.abs(a)<=EPSILON) && (Math.abs(b)<=EPSILON))
		throw new LineException("Impossible de construire une droite d'quation c = 0");

	this.a =a;
	this.b =b;
	this.c =c;
	}

/**
* Recopie une droite dans la droite courante.
* @param Line2D d, la droite  recopier.
* @since Opale 0.1
*/
public void set(Line2D d)
	{
	setOIJ(d.getOIJ());
	this.a =d.a;
	this.b =d.b;
	this.c =d.c;
	}

/**
* Renvoie la valeur des coeeficients dans l'quation de la droite.
* @param int i, le numro du coefficient dsir (entre 0 et 2).
* @return double, le coefficient.
*/
public final double get(int i)	//pb a recoder!!!
	{
	if (i == 0)
		return a;
	else if (i == 1)
		return b;
	else if (i==2)
		return c;
	return -1;
	}

/**
* Renvoie une instance de Line2D perpendiculaire  la droite courante et passant par un point donn.
* @param double x, y, les coordonnes du point.
* @since Opale 0.1
*/
public void orthoLine(double x, double y, Line2D d)
	{
	
	try
	{
	d.setOIJ(getOIJ());
	d.set(-b,a,b*x-a*y);
	}
	catch(LineException e)
	{
	System.out.println("Dans Line2D.java::orthoLine()  Ne devrait jamais arriver !");
	System.exit(-1);
	}
	
	}
	
/**
* Renvoie une instance de Line2D perpendiculaire  la droite courante et passant par un point donn.
* @param Point2D p, le point.
* @since Opale 0.1
*/
public void orthoLine(Point2D p, Line2D d)
	{
	Point2D q = (Point2D) p.clone();
	q.changeOIJ(getOIJ());
	

	try
	{
	d.setOIJ(getOIJ());
	d.set(-b,a,b*q.getX()-a*q.getY());
	}
	catch(LineException e)
	{
	System.out.println("Dans Line2D.java::orthoLine()  Ne devrait jamais arriver !");
	System.exit(-1);
	}

	}


/**
* Renvoie une instance de Line2D parallle  la droite courante et passant par un point donn.
* @param double x, y, les coordonnes du point.
* @since Opale 0.1
*/
public void parallLine(double x, double y, Line2D d)
	{
	
	try
	{
	d.setOIJ(getOIJ());
	d.set(a,b,-a*x-b*y);
	}
	catch(LineException e)
	{
	System.out.println("Dans Line2D.java::parallLine()  Ne devrait jamais arriver !");
	System.exit(-1);
	}

	}
	
/**
* Renvoie une instance de Line2D perpendiculaire  la droite courante et passant par un point donn.
* @param Point2D p, le point.
* @since Opale 0.1
*/
public void parallLine(Point2D p, Line2D d)
	{
	Point2D q = (Point2D) p.clone();
	q.changeOIJ(getOIJ());
	try
	{
	d.setOIJ(getOIJ());
	d.set(a,b,-a*q.getX()-b*q.getY());
	}
	catch(LineException e)
	{
	System.out.println("Dans Line2D.java::parallLine()  Ne devrait jamais arriver !");
	System.exit(-1);
	}
	}

/**
* Renvoie un vecteur orthogonal  la droite.
* @return Vector2D, un vecteur orthogonal.
* @since Opale 0.1
*/
public Vector2D orthoVect()
	{
	return new Vector2D(getOIJ(),a,b);
	}
	
/**
* Renvoie un vecteur directeur  la droite.
* @return Vector2D, un vecteur directeur.
* @since Opale 0.1
*/
public final Vector2D dirVect()
	{
	return new Vector2D(getOIJ(),-b,a);
	}
	
public double distSq(double x, double y)
	{
	double norm = dirVect().norm();
	//double norm = Math.sqrt(a*a+b*b);
	
	if (norm > EPSILON)
		return Math.abs(a*x+b*y+c)/norm;
	else
		return -1;
	}
	
public double distSq(Point2D p)
	{
	double norm = dirVect().norm();

	Point2D q= (Point2D) p.clone();
	q.changeOIJ(getOIJ());

	if (norm > EPSILON)
		return Math.abs(a*q.getX()+b*q.getY()+c)/norm;
	else
		return -1;
	}
	
	
/*public final double distSq(Line2D d)
	{
	return 0;
	}
*/

/**
* Teste si la droite est parallle  une droite donne.
* @param Line2D d, la droite  tester.
* @return boolean, true si les droites sont parallles, false sinon.
* @since Opale 0.1
*/
public final boolean isParall(Line2D d)
	{
	return ( (Math.abs(a*d.b-b*d.a) <= EPSILON) );
	}
	
/**
* Teste si la droite est orthogonale  une droite donne.
* @param Line2D d, la droite  tester.
* @return boolean, true si les droites sont orthogonales, false sinon.
* @since Opale 0.1
*/
public final boolean isOrtho(Line2D d)
	{
	return ( (Math.abs(a*d.a+b*d.b) <= EPSILON) );
	}
	

/**
* Cre un nouvel objet de mme classe et de mme contenu.
* @return Object un clone de l'objet.
* @exception  OutOfMemoryError s'il n'y a pas assez de mmoire.
* @see        java.lang.Cloneable
* @since Opale 0.1
*/
public Object clone()
	{
	Line2D d = (Line2D) super.clone();
	d.a = a;
	d.b = b;
	d.c = c;
	return d;
	}
	
/**
* Cette mthode teste si la droite est gal  un objet pass en argument.
* @param Object obj un objet  comparer avec la droite.
* @return <code>true</code> si l'objet  comparer est une instance de Line2D et
est gal  la droite courante. <code>false</code> sinon.
* @since Opale 0.1
*/public boolean equals(Object obj)
	{
	if (obj instanceof Line2D)
		{
        	Line2D d = (Line2D) obj;
		return (Math.abs(a * d.b - d.a*b) <= EPSILON) && (Math.abs(c *
		d.b - d.c*b) <= EPSILON) && (getOIJ().equals(d.getOIJ()));
        	}
    	return false;
	}
	

public void changeOIJ(OIJ newrep)
	{
	Vector2D v = dirVect();
	Point2D p = new Point2D(getOIJ());

	if ( Math.abs(a) > EPSILON)
		p.setLocation(-c/a,0);
	else if ( Math.abs(b) > EPSILON)
		p.setLocation(0,-c/b);
	
	v.changeOIJ(newrep);
	p.changeOIJ(newrep);
	try
	{
	set(p.getX(),p.getY(),p.getX()+v.getX(),p.getY()+v.getY());	
	}
	catch(LineException e)
	{
	System.out.println("Dans Line2D.java::changeOIJ()  Ne devrait jamais arriver !");
	System.exit(-1);
	}
	setOIJ(newrep);
	}
	
public String toString()
	{
	return "Droite: "+a+"x + "+b+"y + "+c+" = 0";
	}

public void writeFileMore(WFile f, Pack2D p)
	{
	f.writeln("a " +a);
	f.writeln("b " +b);
	f.writeln("c " +c);
	}	
	
public void readFile(RFile f, Pack2D p) throws java.io.IOException, MalformedFileException
	{
	super.readFile(f,p);
	try
	{
	set(a,b,c);
	}
	catch (LineException e)
	{
	throw new MalformedFileException("Impossible de construire la droite "+this);
	}
	}

public int readKeyWord(String word, RFile f, Pack2D p) throws java.io.IOException, MalformedFileException
	{
	int type;
	if ( word.equals("a"))
		{
		type = f.next();
		if (type != RFile.RF_DOUBLE) return -1;
		a = f.dval;
		return 0;
		}
	else if ( word.equals("b"))
		{
		type = f.next();
		if (type != RFile.RF_DOUBLE) return -1;
		b = f.dval;
		return 0;
		}
	else if ( word.equals("c"))
		{
		type = f.next();
		if (type != RFile.RF_DOUBLE) return -1;
		c = f.dval;
		return 0;
		}
	return -1;
	}

}

	
	
