/*
Copyright (C) 2000  Olivier Cioni

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You can visit the web site http://www.opale.ovh.org to obtain more informations about this program and/or to contact the coders.
*/


package opale.module2d.object2d;
import opale.tools.*;
import opale.module2d.noyau2d.*;
import java.util.Vector;


/**
* Cette classe modlise un nuage de points. Elle tend <code>Object2D</code> et implmente <code>Transformable</code>.
* Elle reprsente un nuage de points dans le plan et possede des mthodes de calculs sur ces points (drive, interpolation...)
* @author D.K. ; O.C.
* @version 0.1
*/
public class Data2D extends Object2D implements Transformable, Cloneable
{
protected double[] x,y;
protected int nbPts;

/**
* Constructeur par dfaut.
* @since Opale 0.1
*/
public Data2D()
	{
	super();
	System.out.println("Dans Data2D()" + nbPts);
	x = new double[1];
	y = new double[1];
	
	}

/**
* Initialise l'objet avec 0 points et un repre OIJ donn.
* @param OIJ rep, le repre.
* @since Opale 0.1
*/
public Data2D(OIJ rep)
        {
        super(rep);
	x = new double[1];
	y = new double[1];
        }

/**
* Initialise l'objet avec n points et un repre OIJ donn.
* @param OIJ rep, le repre.
* @param int n, le nombre de points
* @since Opale 0.1
*/
public Data2D(OIJ rep, int n)
        {
        super(rep);
	nbPts = Math.abs(n);
	x = new double[nbPts];
	y = new double[nbPts];
        }

/**
* Le constructeur de copie.
* @param Data2D le nuage  copier.
* @since Opale 0.1
*/
public Data2D(Data2D d)
        {
        super(d.getOIJ());
	int i;
	
        this.nbPts = d.nbPts;
	x = new double[nbPts];
	y = new double[nbPts];
	
	for (i=0;i<nbPts;i++)
		{
		x[i] = d.x[i];
		y[i] = d.y[i];
		}
	}
	
/**	
* Remplit l'objet avec le tableau de points pass en paramtre.
* @param int n, la taille des tableaux.
* @apram double[] x, tableau des x.
* @apram double[] y, tableau des y.
* @since Opale 0.1
*/
public void set(int n, double[] x, double[] y)
	{
	int i;
	
        this.nbPts = n;
	x = new double[nbPts];
	y = new double[nbPts];
	
	for (i=0;i<nbPts;i++)
		{
		this.x[i] = x[i];
		this.y[i] = y[i];
		}
	}
	
public void writeFileMore(WFile f, Pack2D p)
	{
	int i;
	f.writeln("N "+nbPts);
	f.writeln("Points ");
	for (i=0;i<nbPts;i++)
		f.writeln(x[i]+" "+y[i]);
	}	
	
public int readKeyWord(String word, RFile f, Pack2D p)  throws java.io.IOException, MalformedFileException
	{
	int type,i;
	if ( word.equals("N"))
		{
		type = f.next();
		if (type != RFile.RF_DOUBLE) return -1;
		nbPts = (int) f.dval;
		x = new double[nbPts];
		y = new double[nbPts];
		return 0;
		}
	else if ( word.equals("Points"))
		{
		for (i=0;i<nbPts;i++)
			{
			type = f.next();
			if (type != RFile.RF_DOUBLE) return -1;
			x[i] = f.dval;
			type = f.next();
			if (type != RFile.RF_DOUBLE) return -1;
			y[i] = f.dval;
			}
		return 0;
		}
	return -1;
		
	}
/**
* Cette mthode lit directement un fichier de donnes numriques et remplit l'objet courant en consquence.
* @param RFile f, le fichier de donnes.
* @since Opale 0.1
*/	
public void readData(RFile f)  throws java.io.IOException
	{
	int type,i;
	Vector xtemp = new Vector(100,20);
	Vector ytemp = new Vector(100,20);
	Double[] xtab, ytab;
	
	type = f.next();
	while (type != RFile.RF_EOF)
		{
		if (type == RFile.RF_DOUBLE) 
			xtemp.add(new Double(f.dval));
		type = f.next();
		if (type == RFile.RF_DOUBLE) ;
			ytemp.add(new Double(f.dval));
		type = f.next();
		}
	nbPts = xtemp.size();
	x = new double[nbPts];
	y = new double[nbPts];
	
	xtab = new Double[nbPts];
	ytab = new Double[nbPts];
	
	xtemp.copyInto(xtab);
	ytemp.copyInto(ytab);
	
	for (i=0;i<nbPts;i++)
		{
		x[i] = xtab[i].doubleValue();	
		y[i] = ytab[i].doubleValue();	
		}
	}
	
// les mthodes de l'interface Transformable.

public void translate(double dx,double dy)
	{
	int i;
	for (i=0;i<nbPts;i++)
		{
		this.x[i]+=dx;
		this.y[i]+=dy;
		}
	}
	

public void rotate(double theta)
	{
	int i;
	double costh=Math.cos(theta);
	double sinth=Math.sin(theta);

	for (i=0;i<nbPts;i++)
		{
		double x=this.x[i];
		this.x[i]=x*costh-y[i]*sinth ;
		this.y[i]=x*sinth+this.y[i]*costh ;
		}
	}

public void rotate(double oriX, double oriY,double theta)
	{
	this.translate(-oriX,-oriY);
	this.rotate(theta);
	this.translate(oriX,oriY);
	}




/**
* Effectue un changement de repere de l'objet.
* @param OIJ rep, le nouveau repere.
* @since Opale 0.1
*/
public void changeOIJ(OIJ rep)
	{
	int i;
	MVPass mv = new MVPass();
	getOIJ().matPassage(rep,mv);
	
	for (i=0;i<nbPts;i++)
		{
		Point2D p = mv.compute(x[i],y[i]);
		x[i] = p.getX();
		y[i] = p.getY();
		}
	setOIJ(rep);
	}
	
/**
* Cre un nouvel objet de mme classe et de mme contenu.
* @return Object un clone de l'objet.
* @exception  OutOfMemoryError s'il n'y a pas assez de mmoire.
* @see        java.lang.Cloneable
* @since Opale 0.1
*/
public Object clone()
	{
	int i;
	Data2D d = (Data2D) super.clone();
	for (i=0;i<nbPts;i++)
		{
		d.x[i] = x[i];
		d.y[i] = y[i];
		}
	return d;
   	}

/**
* Cette mthode teste si le point est gal  un objet pass en argument.
* @param Object obj un objet  comparer avec le point.
* @return <code>true</code> si l'objet  comparer est une instance de Point2D et
est gal au point courant; <code>false</code> sinon.
* @since Opale 0.1
*/
public boolean equals(Object obj)
	{
	if (obj instanceof Data2D)
		{
        	Data2D d = (Data2D) obj;
		int i;
		boolean test=false;
		if ( (test=(nbPts == d.nbPts) ) )
			{
			for (i=0;i<nbPts;i++)
				{
				test = test && (x[i] == d.x[i]) && (y[i] == d.y[i]);
				}
			}
			
		return test && (getOIJ().equals(d.getOIJ()));
        	}
    	return false;
    }


public String toString()
	{
	String s="Data2D";
	return s;
	}
	

/**
* Drive le nuage de points en utilisant des diffrences 'avance'.
* @return Data2D, le rsultat obtenu.
* @since Opale 0.1
*/	
public Data2D forwardDiff()
	{
	int i;
	double h;
	Data2D a = new Data2D(getOIJ(),nbPts-1);
	for(i=0;i<nbPts-1;i++)
		{
		a.x[i] = x[i];
		if ( (h=x[i+1]-x[i]) != 0)
			a.y[i] = (y[i+1]-y[i])/h;
		else {}//exception
			
		}
	return a;
	}
		
	

/**
* Drive le nuage de points en utilisant des diffrences 'retard'.
* @return Data2D, le rsultat obtenu.
* @since Opale 0.1
*/	
public Data2D backwardDiff()
	{
	int i;
	double h;
	Data2D a = new Data2D(getOIJ(),nbPts-1);
	for(i=1;i<nbPts;i++)
		{
		a.x[i] = x[i];
		if ( (h=x[i]-x[i-1]) != 0)
			a.y[i] = (y[i]-y[i-1])/h;
		else {}//exception
			
		}
	return a;
	}

/**
* Drive le nuage de points en utilisant des diffrences 'centres'.
* @return Data2D, le rsultat obtenu.
* @since Opale 0.1
*/	
public Data2D centeredDiff()
	{
	int i;
	double h;
	Data2D a = new Data2D(getOIJ(),nbPts-1);
	for(i=1;i<nbPts-1;i++)
		{
		a.x[i] = x[i];
		if ( (h=x[i+1]-x[i-1]) != 0)
			a.y[i] = (y[i+1]-y[i-1])/h;
		else {}//exception
			
		}
	return a;
	}

/**
* Calcule l'aire sou sle nuage de points par la mthode des rectangles 'avants'.
* @return double, le rsultat obtenu.
* @since Opale 0.1
*/	
public double sumRectForward()
	{
	double sum=0.;
	int i;

	for(i=0;i<nbPts-1;i++)	
		sum+=y[i]*(x[i+1]-x[i]);
	
	return sum;	
	}
	
/**
* Calcule l'aire sou sle nuage de points par la mthode des rectangles 'retards'.
* @return double, le rsultat obtenu.
* @since Opale 0.1
*/	
public double sumRectBackward()
	{
	double sum=0.;
	int i;

	for(i=1;i<nbPts;i++)	
		sum+=y[i]*(x[i]-x[i-1]);
	
	return sum;	
	}

/**
* Calcule l'aire sous le nuage de points par la mthode des rectangles 'centrs'.
* Cette mthode est quivalente  la mthode des trapzes.
* @return double, le rsultat obtenu.
* @since Opale 0.1
*/	
public double sumRectCentered()
	{
	double sum=y[0]*(x[1]-x[0])/2;
	int i;

	for(i=1;i<nbPts-1;i++)	
		sum+=y[i]*(x[i+1]-x[i-1])/2;
	
	sum+=y[nbPts-1]*(x[nbPts-1]-x[nbPts-2])/2;
	return sum;	
	}

/**
* Calcule l'aire sous le nuage de points par la mthode des trapzes.
* @return double, le rsultat obtenu.
* @since Opale 0.1
*/	
public double sumTrapez()
	{
	double sum=0.;
	int i;

	for(i=0;i<nbPts-1;i++)	
		sum+=(y[i]+y[i+1])/2*(x[i+1]-x[i]);
	
	return sum;	
	}

		
}
