#!/usr/bin/perl -w
#
# Copyright (C) 2003 Dmitry Fedorov <fedorov@inp.nsk.su>
#
# This file is part of Offmirror.
#
# Offmirror is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Offmirror is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


=head1 NAME

offmirror-untar - synchronize destination files tree with source one

=head1 SYNOPSIS

offmirror-untar src-list dst-list tar-file [options]

=head1 DESCRIPTION

This program is part of files tree offline mirroring utility package,
which is used to synchronize file trees without any direct connection
between them.

This program should be runned on destination side.
It unpacks tar archive with copied source-side files,
change attributes of files, delete files unexistent on source.

=cut

require 5.003;
use strict;
use Getopt::Long;

use File::Basename;
use Cwd 'abs_path';

use FindBin;
use lib "$FindBin::Bin";
use OffMirror::FileList;


my $progname = 'offmirror-untar';

sub usage
{
	warn "\n".join(" ", @_)."\n" if @_;
	warn <<EOF;

Usage: $progname src-list dst-list tar-file			\
			[--tar-program=/usr/gnu/bin/tar]	\
			[--tar-options='options']		\
			[--show-cmd] [--verbose] [--dry-run] [--help]

For details, see man page.
EOF
	exit(1);
}


=head1 OPTIONS

=over 4

=item src-list[.gz|.bz2|.Z|.zip]

Specifies file with source files tree list.

The file can be compressed with gzip or bzip2 or compress or
zip (single file only).

=item dst-list[.gz|.bz2|.Z|.zip]

Specifies file with destination files tree list.

The file can be compressed with gzip or bzip2 or compress or
zip (single file only).

=item tar-file.tar[.gz|.bz2]

Specifies file name for the tar archive.
Name can have suffix '.gz' or '.bz2'
which implied proper GNU tar decompression option.

=item --tar-program=/usr/gnu/bin/tar

Specifies tar program to run.

=item --tar-options='options'

Specifies additional tar program options.

=item --show-cmd

Show running command line.

=item --verbose

Show what to do.

=item --dry-run

Simulate.

=item --help

Display a usage summary.

=back

=head1 SEE ALSO

L<offmirror(1)>

L<offmirror-tar(1)>

=cut


my $tar_prog='tar';
my $tar_opts='';
my $show_cmd;
my $verbose;
my $dry_run;
my $force_chown=0;
my $force_chmod=0;
my $force_mtime=0;
my $help;

GetOptions(
	'tar-program=s'	=> \$tar_prog,
	'tar-options=s'	=> \$tar_opts,
	'show-cmd'	=> \$show_cmd,
	'verbose'	=> \$verbose,
	'dry-run'	=> \$dry_run,
	'force-chown'	=> \$force_chown,
	'force-chmod'	=> \$force_chmod,
	'force-mtime'	=> \$force_mtime,
	'help'		=> \$help
) or usage;

usage if $help;

my $chattr_opts = '';
$chattr_opts .= ' --verbose'     if $verbose;
$chattr_opts .= ' --dry-run'     if $dry_run;
$chattr_opts .= ' --force-chown' if $force_chown;
$chattr_opts .= ' --force-chmod' if $force_chmod;
$chattr_opts .= ' --force-mtime' if $force_mtime;

$tar_opts    .= ' --verbose' if $verbose;


my $src_list = shift @ARGV or  usage("no src-list file specified");
my $dst_list = shift @ARGV or  usage("no dst-list file specified");
my $tar_file = shift @ARGV or  usage("no tar-file file specified");
               shift @ARGV and usage("extra parameter specified");


my $tar_zopt='';

if    ( $tar_file =~ m/(gz)|(zip)$/ )
{
    $tar_zopt = '--gzip';
}
elsif ( $tar_file =~ m/bz2$/ )
{
    $tar_zopt = '--bzip2';
}
elsif ( $tar_file =~ m/\.tar$/ )
{
    $tar_zopt = '';
}
else
{
    die "unknown tar file suffix: $tar_file";
}


my $basedir = basedir_from_file($dst_list);

umask 0; # to allow to create files/dirs with any permissions


$tar_file = abs_path(dirname($tar_file)) . '/' . basename($tar_file);

my $cmd = "$tar_prog -C $basedir -xpf $tar_file $tar_zopt $tar_opts";

print "$cmd\n" if $show_cmd;

unless ( $dry_run )
{
    if ( system("$cmd") != 0 )
    {
	die "tar failed: $?";
    }
}

print "\n\n" if $verbose;

my $  diffcmd = "$FindBin::Bin/offmirror-diff $src_list $dst_list";
my $chattrcmd = "$FindBin::Bin/offmirror-chattr $chattr_opts";

$cmd = "$diffcmd | $chattrcmd";

print "$cmd\n" if $show_cmd;

if ( system("$cmd") != 0 )
{
    die "chattr script failed: $?";
}


exit 0;


=head1 AUTHOR

Dmitry Fedorov <fedorov@inp.nsk.su>

=head1 COPYRIGHT

Copyright (C) 2003 Dmitry Fedorov <fedorov@inp.nsk.su>

=head1 LICENSE

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License,
or (at your option) any later version.

=head1 DISCLAIMER

The author disclaims any responsibility for any mangling of your system
etc, that this script may cause.

=cut

