/*  GNU Moe - My Own Editor
    Copyright (C) 2005 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <algorithm>
#include <string>
#include <vector>

#include "buffer.h"
#include "block.h"
#include "rc.h"


namespace Block {

Buffer *_bufferp = 0, *saved_bufferp = 0;
Point anchor, _begin, _end, saved_begin, saved_end;
bool _marking = false;


void enable_marking( Buffer & buffer, const Point & p ) throw()
  {
  _bufferp = &buffer; _begin = p; anchor = p; _end = Point(); _marking = true;
  }

} // end namespace Block


Buffer & Block::buffer() throw() { return *_bufferp; }

bool Block::same_buffer( const Buffer & buffer ) throw()
  { return &buffer == _bufferp; }

Point & Block::begin() throw() { return _begin; }

Point & Block::end() throw() { return _end; }


bool Block::in_block( const Buffer & buffer, const Point & p ) throw()
  { return ( &buffer == _bufferp && _begin.line >= 0 && p >= _begin && p < _end ); }


bool Block::in_block_or_end( const Buffer & buffer, const Point & p ) throw()
  {
  return ( &buffer == _bufferp && _begin.line >= 0 && _begin < _end &&
           p >= _begin && p <= _end );
  }


bool Block::valid() throw()
  { return ( _bufferp != 0 && _begin.line >= 0 && _begin < _end ); }


void Block::save_block_position() throw()
  { saved_bufferp = _bufferp; saved_begin = _begin; saved_end = _end; }


void Block::restore_block_position() throw()
  { _bufferp = saved_bufferp; _begin = saved_begin; _end = saved_end; }


void Block::reset() throw()
  { _bufferp = 0; _begin = _end = Point(); _marking = false; }


void Block::set_block( Buffer & buffer, const Point & p1, const Point & p2 ) throw()
  { _bufferp = &buffer; _begin = p1; _end = p2; _marking = false; }


void Block::set_begin( Buffer & buffer, const Point & p ) throw()
  {
  _begin = p; _marking = false;
  if( &buffer != _bufferp ) { _bufferp = &buffer; _end = Point(); }
  }


void Block::set_end( Buffer & buffer, const Point & p ) throw()
  {
  _end = p; _marking = false;
  if( &buffer != _bufferp ) { _bufferp = &buffer; _begin = Point(); }
  }


bool Block::follow_marking( const Buffer & buffer, const Point & p ) throw()
  {
  static Point old_p;
  if( _marking && same_buffer( buffer ) && p != old_p )
    {
    if( p < anchor ) _begin = p; else _begin = anchor;
    if( p > anchor ) _end = p; else _end = anchor;
    old_p = p;
    return true;
    }
  return false;
  }


const char * Block::toggle_marking( Buffer & buffer, const Point & p ) throw()
  {
  const char * const str[2] = { "Selection started.", "Selection cleared." };
  const char * msg = 0;
  if( !_marking )
    {
    if( in_block_or_end( buffer, p ) ) { reset(); msg = str[1]; }
    else { enable_marking( buffer, p ); msg = str[0]; }
    }
  else
    {
    if( same_buffer( buffer ) ) { if( valid() ) _marking = false; else reset(); }
    else { enable_marking( buffer, p ); msg = str[0]; }
    }
  return msg;
  }


bool Block::copy_block( Buffer & buffer, const Point & p ) throw()
  {
  if( buffer.options.read_only || !buffer.pisvalid( p ) || !valid() ||
      !_bufferp->pisvalid( _begin ) || !_bufferp->pisvalid( _end ) )
    return false;
  Basic_buffer tmp( *_bufferp, _begin, _end );
  Point p2 = p;
  buffer.reset_appendable();
  bool done = buffer.pputb( p2, tmp, tmp.bof(), tmp.eof() );
  buffer.reset_appendable();
  if( done )
    { if( RC::options().auto_unmark ) reset(); else set_block( buffer, p, p2 ); }
  return done;
  }


bool Block::delete_block() throw()
  {
  if( valid() )
    {
    Buffer & buffer = *_bufferp;
    _bufferp = 0;			// disable adjust functions
    buffer.reset_appendable();
    const bool result = buffer.pdelb( _begin, _end );
    buffer.reset_appendable();
    _bufferp = &buffer; _end = _begin; _marking = false;
    return result;
    }
  return false;
  }


bool Block::move_block( Buffer & buffer, const Point & p ) throw()
  {
  if( buffer.options.read_only || !buffer.pisvalid( p ) ||
      !valid() || _bufferp->options.read_only ||
      !_bufferp->pisvalid( _begin ) || !_bufferp->pisvalid( _end ) )
    return false;
  Basic_buffer tmp( *_bufferp, _begin, _end );
  const bool same = same_buffer( buffer );
  delete_block();
  if( same ) buffer.force_append();
  else buffer.reset_appendable();
  Point p2 = p;
  bool done = buffer.pputb( p2, tmp, tmp.bof(), tmp.eof() );
  buffer.reset_appendable();
  if( done )
    { if( RC::options().auto_unmark ) reset(); else set_block( buffer, p, p2 ); }
  return done;
  }
